@extends('layouts.admin')

@section('title', 'User Management | IMMIA FINANCE LIMITED')

@section('page-title', 'Users')
@section('page-description', 'Manage system users and permissions')

@section('content')
<!-- Header with Add Button -->
<div class="flex justify-between items-center mb-6">
    <h2 class="text-xl font-semibold text-gray-800">Users</h2>
    <button onclick="openCreateModal()" class="bg-blue-600 hover:bg-blue-700 text-white px-4 py-2 rounded-lg font-semibold flex items-center space-x-2 transition-colors">
        <i class="fas fa-plus"></i>
        <span>New User</span>
    </button>
</div>

<!-- Users Table -->
<div class="bg-white rounded-xl shadow-sm border overflow-hidden">
    <div class="overflow-x-auto">
        <table class="min-w-full divide-y divide-gray-200">
            <thead class="bg-gray-50">
                <tr>
                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">User Details</th>
                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Branch</th>
                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                        Permissions
                        <i class="fas fa-info-circle text-gray-400 ml-1" title="Permissions can be assigned after user creation"></i>
                    </th>
                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Status</th>
                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Last Login</th>
                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Actions</th>
                </tr>
            </thead>
            <tbody class="bg-white divide-y divide-gray-200">
                @forelse($users as $user)
                <tr class="hover:bg-gray-50 transition-colors">
                    <td class="px-6 py-4">
                        <div class="flex items-center space-x-3">
                            <div class="w-10 h-10 bg-blue-100 rounded-full flex items-center justify-center">
                                <i class="fas fa-user text-blue-600"></i>
                            </div>
                            <div>
                                <div class="text-sm font-medium text-gray-900">{{ $user->name }}</div>
                                <div class="text-sm text-gray-500">{{ $user->email }}</div>
                                <div class="text-xs text-gray-400">
                                    Created: {{ $user->created_at->format('M d, Y') }}
                                    @if($user->creator)
                                    by {{ $user->creator->name }}
                                    @endif
                                </div>
                            </div>
                        </div>
                    </td>
                    <td class="px-6 py-4">
                        <div class="text-sm text-gray-900">{{ $user->branch->branchname ?? 'N/A' }}</div>
                        <div class="text-xs text-gray-500">{{ $user->branch->branchcode ?? '' }}</div>
                    </td>
                    <td class="px-6 py-4">
                        <div class="flex flex-wrap gap-1 max-w-xs">
                            @forelse($user->permissions as $permission)
                            <span class="inline-flex items-center px-2 py-1 rounded-full text-xs font-medium bg-purple-100 text-purple-800">
                                {{ $permission->name }}
                            </span>
                            @empty
                            <span class="inline-flex items-center px-2 py-1 rounded-full text-xs font-medium bg-gray-100 text-gray-800">
                                No Permissions
                            </span>
                            @endforelse
                        </div>
                    </td>
                    <td class="px-6 py-4 whitespace-nowrap">
                        <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium {{ $user->status ? 'bg-green-100 text-green-800' : 'bg-red-100 text-red-800' }}">
                            <i class="fas fa-circle mr-1" style="font-size: 6px;"></i>
                            {{ $user->status_text }}
                        </span>
                    </td>
                    <td class="px-6 py-4 whitespace-nowrap">
                        <div class="text-sm text-gray-900">
                            @if($user->last_login)
                                {{ $user->last_login->format('M d, Y H:i') }}
                            @else
                                <span class="text-gray-400">Never</span>
                            @endif
                        </div>
                    </td>
                    <td class="px-6 py-4 whitespace-nowrap text-sm font-medium">
                        <div class="flex items-center space-x-3">
                            <button onclick="openEditModal({{ $user->id }})" class="text-blue-600 hover:text-blue-900 transition-colors" title="Edit User">
                                <i class="fas fa-edit"></i>
                            </button>
                            <button onclick="openPasswordModal({{ $user->id }})" class="text-yellow-600 hover:text-yellow-900 transition-colors" title="Reset Password">
                                <i class="fas fa-key"></i>
                            </button>
                            <button onclick="resendPassword({{ $user->id }})" class="text-purple-600 hover:text-purple-900 transition-colors" title="Resend Password Email">
                                <i class="fas fa-envelope"></i>
                            </button>
                            <button onclick="toggleStatus({{ $user->id }})" class="text-{{ $user->status ? 'yellow' : 'green' }}-600 hover:text-{{ $user->status ? 'yellow' : 'green' }}-900 transition-colors" title="{{ $user->status ? 'Deactivate' : 'Activate' }} User">
                                <i class="fas fa-{{ $user->status ? 'pause' : 'play' }}"></i>
                            </button>
                            <button onclick="confirmDelete({{ $user->id }})" class="text-red-600 hover:text-red-900 transition-colors" title="Delete User">
                                <i class="fas fa-trash"></i>
                            </button>
                        </div>
                    </td>
                </tr>
                @empty
                <tr>
                    <td colspan="6" class="px-6 py-8 text-center">
                        <div class="flex flex-col items-center justify-center text-gray-500">
                            <i class="fas fa-users text-4xl mb-4 text-gray-300"></i>
                            <p class="text-lg font-medium mb-2">No users found</p>
                            <p class="text-sm mb-4">Get started by creating your first user</p>
                            <button onclick="openCreateModal()" class="bg-blue-600 hover:bg-blue-700 text-white px-4 py-2 rounded-lg font-medium transition-colors">
                                <i class="fas fa-plus mr-2"></i>
                                Create First User
                            </button>
                        </div>
                    </td>
                </tr>
                @endforelse
            </tbody>
        </table>
    </div>
</div>

<!-- Create User Modal -->
<div id="createModal" class="fixed inset-0 bg-gray-600 bg-opacity-50 overflow-y-auto h-full w-full hidden z-50 transition-opacity">
    <div class="relative top-20 mx-auto p-5 border w-full max-w-md shadow-lg rounded-md bg-white transform transition-all">
        <div class="mt-3">
            <div class="flex justify-between items-center mb-4 pb-3 border-b">
                <h3 class="text-lg font-semibold text-gray-900">Create New User</h3>
                <button onclick="closeCreateModal()" class="text-gray-400 hover:text-gray-600 transition-colors">
                    <i class="fas fa-times text-lg"></i>
                </button>
            </div>
            
            <form id="createUserForm">
                @csrf
                <div class="space-y-4">
                    <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                        <div>
                            <label for="firstname" class="block text-sm font-medium text-gray-700 mb-1">First Name *</label>
                            <input type="text" name="firstname" id="firstname" required 
                                   class="w-full border border-gray-300 rounded-lg shadow-sm py-2 px-3 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-colors"
                                   placeholder="e.g., John">
                        </div>
                        
                        <div>
                            <label for="middlename" class="block text-sm font-medium text-gray-700 mb-1">Middle Name</label>
                            <input type="text" name="middlename" id="middlename"
                                   class="w-full border border-gray-300 rounded-lg shadow-sm py-2 px-3 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-colors"
                                   placeholder="e.g., Michael">
                        </div>
                    </div>
                    
                    <div>
                        <label for="lastname" class="block text-sm font-medium text-gray-700 mb-1">Last Name *</label>
                        <input type="text" name="lastname" id="lastname" required 
                               class="w-full border border-gray-300 rounded-lg shadow-sm py-2 px-3 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-colors"
                               placeholder="e.g., Doe">
                    </div>
                    
                    <div>
                        <label for="email" class="block text-sm font-medium text-gray-700 mb-1">Email Address *</label>
                        <input type="email" name="email" id="email" required 
                               class="w-full border border-gray-300 rounded-lg shadow-sm py-2 px-3 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-colors"
                               placeholder="e.g., john.doe@immiafinance.com">
                    </div>
                    
                    <div>
                        <label for="branchid" class="block text-sm font-medium text-gray-700 mb-1">Branch *</label>
                        <select name="branchid" id="branchid" required 
                                class="w-full border border-gray-300 rounded-lg shadow-sm py-2 px-3 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-colors">
                            <option value="">Select Branch</option>
                            @foreach($branches as $branch)
                                <option value="{{ $branch->id }}">{{ $branch->branchname }} ({{ $branch->branchcode }})</option>
                            @endforeach
                        </select>
                    </div>

                    <!-- Account Information Note -->
                    <div class="bg-blue-50 border border-blue-200 rounded-lg p-3">
                        <div class="flex items-start space-x-2">
                            <i class="fas fa-info-circle text-blue-500 mt-0.5"></i>
                            <div>
                                <p class="text-sm text-blue-800 font-medium">Account Information</p>
                                <p class="text-xs text-blue-700">
                                    A strong password will be automatically generated and sent to the user's email address. 
                                    Permissions can be assigned after creation using the edit function.
                                </p>
                            </div>
                        </div>
                    </div>
                </div>
                
                <div class="flex justify-end space-x-3 mt-6 pt-4 border-t">
                    <button type="button" onclick="closeCreateModal()" class="bg-gray-300 hover:bg-gray-400 text-gray-800 py-2 px-4 rounded-lg font-medium transition-colors">
                        Cancel
                    </button>
                    <button type="submit" class="bg-blue-600 hover:bg-blue-700 text-white py-2 px-4 rounded-lg font-medium flex items-center space-x-2 transition-colors">
                        <i class="fas fa-save"></i>
                        <span>Create User</span>
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Edit User Modal -->
<div id="editModal" class="fixed inset-0 bg-gray-600 bg-opacity-50 overflow-y-auto h-full w-full hidden z-50 transition-opacity">
    <div class="relative top-20 mx-auto p-5 border w-full max-w-2xl shadow-lg rounded-md bg-white transform transition-all">
        <div class="mt-3">
            <div class="flex justify-between items-center mb-4 pb-3 border-b">
                <h3 class="text-lg font-semibold text-gray-900">Edit User & Permissions</h3>
                <button onclick="closeEditModal()" class="text-gray-400 hover:text-gray-600 transition-colors">
                    <i class="fas fa-times text-lg"></i>
                </button>
            </div>
            
            <form id="editUserForm">
                @csrf
                @method('PUT')
                <input type="hidden" name="user_id" id="edit_user_id">
                
                <div class="grid grid-cols-1 md:grid-cols-2 gap-4 mb-4">
                    <div>
                        <label for="edit_firstname" class="block text-sm font-medium text-gray-700 mb-1">First Name *</label>
                        <input type="text" name="firstname" id="edit_firstname" required 
                               class="w-full border border-gray-300 rounded-lg shadow-sm py-2 px-3 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-colors">
                    </div>
                    
                    <div>
                        <label for="edit_middlename" class="block text-sm font-medium text-gray-700 mb-1">Middle Name</label>
                        <input type="text" name="middlename" id="edit_middlename"
                               class="w-full border border-gray-300 rounded-lg shadow-sm py-2 px-3 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-colors">
                    </div>
                    
                    <div>
                        <label for="edit_lastname" class="block text-sm font-medium text-gray-700 mb-1">Last Name *</label>
                        <input type="text" name="lastname" id="edit_lastname" required 
                               class="w-full border border-gray-300 rounded-lg shadow-sm py-2 px-3 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-colors">
                    </div>
                    
                    <div>
                        <label for="edit_branchid" class="block text-sm font-medium text-gray-700 mb-1">Branch *</label>
                        <select name="branchid" id="edit_branchid" required 
                                class="w-full border border-gray-300 rounded-lg shadow-sm py-2 px-3 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-colors">
                            <option value="">Select Branch</option>
                            @foreach($branches as $branch)
                                <option value="{{ $branch->id }}">{{ $branch->branchname }} ({{ $branch->branchcode }})</option>
                            @endforeach
                        </select>
                    </div>
                    
                    <div class="md:col-span-2">
                        <label for="edit_email" class="block text-sm font-medium text-gray-700 mb-1">Email Address *</label>
                        <input type="email" name="email" id="edit_email" required 
                               class="w-full border border-gray-300 rounded-lg shadow-sm py-2 px-3 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-colors">
                    </div>
                </div>
                
                <!-- Permissions Section -->
                <div class="mb-4">
                    <label class="block text-sm font-medium text-gray-700 mb-2">Permissions</label>
                    <div class="grid grid-cols-2 md:grid-cols-3 gap-2 max-h-48 overflow-y-auto p-3 border border-gray-200 rounded-lg" id="edit_permissions_container">
                        @foreach($permissions as $permission)
                        <label class="flex items-center space-x-2 text-sm">
                            <input type="checkbox" name="permissions[]" value="{{ $permission->id }}" 
                                   class="rounded border-gray-300 text-blue-600 focus:ring-blue-500 permission-checkbox">
                            <span class="text-gray-700">{{ $permission->name }}</span>
                        </label>
                        @endforeach
                    </div>
                    <p class="text-xs text-gray-500 mt-1">Select permissions to assign to this user</p>
                </div>
                
                <div class="flex justify-end space-x-3 mt-6 pt-4 border-t">
                    <button type="button" onclick="closeEditModal()" class="bg-gray-300 hover:bg-gray-400 text-gray-800 py-2 px-4 rounded-lg font-medium transition-colors">
                        Cancel
                    </button>
                    <button type="submit" class="bg-blue-600 hover:bg-blue-700 text-white py-2 px-4 rounded-lg font-medium flex items-center space-x-2 transition-colors">
                        <i class="fas fa-save"></i>
                        <span>Update User</span>
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Change Password Modal -->
<div id="passwordModal" class="fixed inset-0 bg-gray-600 bg-opacity-50 overflow-y-auto h-full w-full hidden z-50 transition-opacity">
    <div class="relative top-20 mx-auto p-5 border w-full max-w-md shadow-lg rounded-md bg-white transform transition-all">
        <div class="mt-3">
            <div class="flex justify-between items-center mb-4 pb-3 border-b">
                <h3 class="text-lg font-semibold text-gray-900">Change Password</h3>
                <button onclick="closePasswordModal()" class="text-gray-400 hover:text-gray-600 transition-colors">
                    <i class="fas fa-times text-lg"></i>
                </button>
            </div>
            
            <form id="passwordForm">
                @csrf
                <input type="hidden" name="user_id" id="password_user_id">
                
                <div class="space-y-4">
                    <div>
                        <label for="new_password" class="block text-sm font-medium text-gray-700 mb-1">New Password *</label>
                        <input type="password" name="password" id="new_password" required 
                               class="w-full border border-gray-300 rounded-lg shadow-sm py-2 px-3 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-colors"
                               placeholder="Enter new password">
                    </div>
                    
                    <div>
                        <label for="new_password_confirmation" class="block text-sm font-medium text-gray-700 mb-1">Confirm Password *</label>
                        <input type="password" name="password_confirmation" id="new_password_confirmation" required 
                               class="w-full border border-gray-300 rounded-lg shadow-sm py-2 px-3 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-colors"
                               placeholder="Confirm new password">
                    </div>
                </div>
                
                <div class="flex justify-end space-x-3 mt-6 pt-4 border-t">
                    <button type="button" onclick="closePasswordModal()" class="bg-gray-300 hover:bg-gray-400 text-gray-800 py-2 px-4 rounded-lg font-medium transition-colors">
                        Cancel
                    </button>
                    <button type="submit" class="bg-blue-600 hover:bg-blue-700 text-white py-2 px-4 rounded-lg font-medium flex items-center space-x-2 transition-colors">
                        <i class="fas fa-key"></i>
                        <span>Update Password</span>
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Loading Overlay for Edit Modal -->
<div id="editLoading" class="fixed inset-0 bg-gray-600 bg-opacity-50 overflow-y-auto h-full w-full hidden z-50">
    <div class="relative top-20 mx-auto p-5 border w-full max-w-md shadow-lg rounded-md bg-white">
        <div class="flex flex-col items-center justify-center py-8">
            <i class="fas fa-spinner fa-spin text-3xl text-blue-600 mb-4"></i>
            <p class="text-gray-700 font-medium">Loading user data...</p>
        </div>
    </div>
</div>

<!-- Success Alert Modal -->
<div id="successAlert" class="fixed inset-0 bg-gray-600 bg-opacity-50 overflow-y-auto h-full w-full hidden z-50 transition-opacity">
    <div class="relative top-20 mx-auto p-5 border w-full max-w-sm shadow-lg rounded-md bg-white transform transition-all">
        <div class="mt-3 text-center">
            <div class="mx-auto flex items-center justify-center h-12 w-12 rounded-full bg-green-100 mb-4">
                <i class="fas fa-check text-green-600 text-xl"></i>
            </div>
            <h3 class="text-lg font-medium text-gray-900 mb-2" id="successTitle">Success!</h3>
            <p class="text-sm text-gray-500 mb-4" id="successMessage">Operation completed successfully.</p>
            <div class="px-4 py-3">
                <button onclick="closeSuccessAlert()" class="w-full bg-green-600 hover:bg-green-700 text-white py-2 px-4 rounded-lg font-medium transition-colors">
                    Continue
                </button>
            </div>
        </div>
    </div>
</div>

<!-- Error Alert Modal -->
<div id="errorAlert" class="fixed inset-0 bg-gray-600 bg-opacity-50 overflow-y-auto h-full w-full hidden z-50 transition-opacity">
    <div class="relative top-20 mx-auto p-5 border w-full max-w-sm shadow-lg rounded-md bg-white transform transition-all">
        <div class="mt-3 text-center">
            <div class="mx-auto flex items-center justify-center h-12 w-12 rounded-full bg-red-100 mb-4">
                <i class="fas fa-exclamation-triangle text-red-600 text-xl"></i>
            </div>
            <h3 class="text-lg font-medium text-gray-900 mb-2" id="errorTitle">Error</h3>
            <p class="text-sm text-gray-500 mb-4" id="errorMessage">An error occurred.</p>
            <div class="px-4 py-3">
                <button onclick="closeErrorAlert()" class="w-full bg-red-600 hover:bg-red-700 text-white py-2 px-4 rounded-lg font-medium transition-colors">
                    Try Again
                </button>
            </div>
        </div>
    </div>
</div>

<!-- Confirmation Modal -->
<div id="confirmationModal" class="fixed inset-0 bg-gray-600 bg-opacity-50 overflow-y-auto h-full w-full hidden z-50 transition-opacity">
    <div class="relative top-20 mx-auto p-5 border w-full max-w-sm shadow-lg rounded-md bg-white transform transition-all">
        <div class="mt-3 text-center">
            <div class="mx-auto flex items-center justify-center h-12 w-12 rounded-full bg-yellow-100 mb-4">
                <i class="fas fa-question text-yellow-600 text-xl"></i>
            </div>
            <h3 class="text-lg font-medium text-gray-900 mb-2" id="confirmationTitle">Confirm Action</h3>
            <p class="text-sm text-gray-500 mb-4" id="confirmationMessage">Are you sure you want to proceed?</p>
            <div class="flex space-x-3 px-4 py-3">
                <button onclick="closeConfirmationModal(false)" class="flex-1 bg-gray-300 hover:bg-gray-400 text-gray-800 py-2 px-4 rounded-lg font-medium transition-colors">
                    Cancel
                </button>
                <button onclick="closeConfirmationModal(true)" class="flex-1 bg-blue-600 hover:bg-blue-700 text-white py-2 px-4 rounded-lg font-medium transition-colors">
                    Confirm
                </button>
            </div>
        </div>
    </div>
</div>
@endsection

@section('scripts')
<script>
    // Global variables
    let currentLoadingButton = null;
    let confirmationCallback = null;
    const baseUrl = '{{ url("/") }}';

    // Alert Modal Functions
    function showSuccessAlert(title, message) {
        document.getElementById('successTitle').textContent = title;
        document.getElementById('successMessage').textContent = message;
        document.getElementById('successAlert').classList.remove('hidden');
    }

    function closeSuccessAlert() {
        document.getElementById('successAlert').classList.add('hidden');
    }

    function showErrorAlert(title, message) {
        document.getElementById('errorTitle').textContent = title;
        document.getElementById('errorMessage').textContent = message;
        document.getElementById('errorAlert').classList.remove('hidden');
    }

    function closeErrorAlert() {
        document.getElementById('errorAlert').classList.add('hidden');
    }

    function showConfirmation(title, message, callback) {
        document.getElementById('confirmationTitle').textContent = title;
        document.getElementById('confirmationMessage').textContent = message;
        document.getElementById('confirmationModal').classList.remove('hidden');
        confirmationCallback = callback;
    }

    function closeConfirmationModal(confirmed) {
        document.getElementById('confirmationModal').classList.add('hidden');
        if (confirmed && confirmationCallback) {
            confirmationCallback();
        }
        confirmationCallback = null;
    }

    // Modal functions
    function openCreateModal() {
        document.getElementById('createModal').classList.remove('hidden');
    }

    function closeCreateModal() {
        document.getElementById('createModal').classList.add('hidden');
        document.getElementById('createUserForm').reset();
    }

    function openEditModal(userId) {
        currentLoadingButton = event.target.closest('button');
        const originalHTML = currentLoadingButton.innerHTML;
        currentLoadingButton.innerHTML = '<i class="fas fa-spinner fa-spin"></i>';
        currentLoadingButton.disabled = true;

        document.getElementById('editLoading').classList.remove('hidden');

        fetch(`${baseUrl}/admin/users/${userId}`, {
            headers: {
                'Accept': 'application/json',
                'X-Requested-With': 'XMLHttpRequest'
            }
        })
        .then(response => {
            if (!response.ok) throw new Error(`HTTP error! status: ${response.status}`);
            return response.json();
        })
        .then(data => {
            if (!data.user) throw new Error('Failed to load user data');

            const user = data.user;
            
            // Populate form fields
            document.getElementById('edit_user_id').value = user.id;
            document.getElementById('edit_firstname').value = user.firstname;
            document.getElementById('edit_middlename').value = user.middlename || '';
            document.getElementById('edit_lastname').value = user.lastname;
            document.getElementById('edit_email').value = user.email;
            document.getElementById('edit_branchid').value = user.branchid;

            // Clear all permission checkboxes first
            document.querySelectorAll('.permission-checkbox').forEach(checkbox => {
                checkbox.checked = false;
            });

            // Check the user's permissions
            if (user.permissions && user.permissions.length > 0) {
                user.permissions.forEach(permission => {
                    const checkbox = document.querySelector(`.permission-checkbox[value="${permission.id}"]`);
                    if (checkbox) checkbox.checked = true;
                });
            }

            document.getElementById('editLoading').classList.add('hidden');
            document.getElementById('editModal').classList.remove('hidden');
            
            currentLoadingButton.innerHTML = originalHTML;
            currentLoadingButton.disabled = false;
            currentLoadingButton = null;
        })
        .catch(error => {
            console.error('Error loading user:', error);
            document.getElementById('editLoading').classList.add('hidden');
            showErrorAlert('Load Error', 'Failed to load user data: ' + error.message);
            
            if (currentLoadingButton) {
                currentLoadingButton.innerHTML = originalHTML;
                currentLoadingButton.disabled = false;
                currentLoadingButton = null;
            }
        });
    }

    function closeEditModal() {
        document.getElementById('editModal').classList.add('hidden');
        if (currentLoadingButton && currentLoadingButton.disabled) {
            currentLoadingButton.innerHTML = '<i class="fas fa-edit"></i>';
            currentLoadingButton.disabled = false;
            currentLoadingButton = null;
        }
    }

    function openPasswordModal(userId) {
        document.getElementById('password_user_id').value = userId;
        document.getElementById('passwordModal').classList.remove('hidden');
    }

    function closePasswordModal() {
        document.getElementById('passwordModal').classList.add('hidden');
        document.getElementById('passwordForm').reset();
    }

    // Form submissions
    document.addEventListener('DOMContentLoaded', function() {
        // Create User Form
        const createForm = document.getElementById('createUserForm');
        if (createForm) {
            createForm.addEventListener('submit', function(e) {
                e.preventDefault();
                
                const submitButton = this.querySelector('button[type="submit"]');
                const originalText = submitButton.innerHTML;
                
                submitButton.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Creating...';
                submitButton.disabled = true;

                const formData = new FormData(this);
                
                fetch(`${baseUrl}/admin/users`, {
                    method: 'POST',
                    body: formData,
                    headers: {
                        'X-CSRF-TOKEN': '{{ csrf_token() }}',
                        'Accept': 'application/json'
                    }
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        if (data.email_sent) {
                            showSuccessAlert('Success', data.message);
                        } else {
                            showSuccessAlert('User Created - Email Failed', data.message + ' Please provide this password to the user manually: ' + data.generated_password);
                        }
                        closeCreateModal();
                        setTimeout(() => location.reload(), 2000);
                    } else {
                        showErrorAlert('Error', data.message);
                    }
                })
                .catch(error => {
                    console.error('Error:', error);
                    showErrorAlert('Error', 'Failed to create user: ' + error.message);
                })
                .finally(() => {
                    submitButton.innerHTML = originalText;
                    submitButton.disabled = false;
                });
            });
        }
        
        // Edit User Form
        const editForm = document.getElementById('editUserForm');
        if (editForm) {
            editForm.addEventListener('submit', function(e) {
                e.preventDefault();
                
                const userId = document.getElementById('edit_user_id')?.value;
                if (!userId) {
                    showErrorAlert('Error', 'User ID not found.');
                    return;
                }
                
                const submitButton = this.querySelector('button[type="submit"]');
                const originalText = submitButton.innerHTML;
                
                submitButton.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Updating...';
                submitButton.disabled = true;

                const formData = new FormData(this);
                
                fetch(`${baseUrl}/admin/users/${userId}`, {
                    method: 'POST',
                    body: formData,
                    headers: {
                        'X-CSRF-TOKEN': '{{ csrf_token() }}',
                        'X-HTTP-Method-Override': 'PUT',
                        'Accept': 'application/json'
                    }
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        showSuccessAlert('Success', data.message);
                        closeEditModal();
                        setTimeout(() => location.reload(), 1500);
                    } else {
                        showErrorAlert('Error', data.message);
                    }
                })
                .catch(error => {
                    console.error('Error:', error);
                    showErrorAlert('Error', 'Failed to update user: ' + error.message);
                })
                .finally(() => {
                    submitButton.innerHTML = originalText;
                    submitButton.disabled = false;
                });
            });
        }
        
        // Password Form
        const passwordForm = document.getElementById('passwordForm');
        if (passwordForm) {
            passwordForm.addEventListener('submit', function(e) {
                e.preventDefault();
                
                const userId = document.getElementById('password_user_id')?.value;
                if (!userId) {
                    showErrorAlert('Error', 'User ID not found.');
                    return;
                }
                
                const submitButton = this.querySelector('button[type="submit"]');
                const originalText = submitButton.innerHTML;
                
                submitButton.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Updating...';
                submitButton.disabled = true;

                const formData = new FormData(this);
                
                fetch(`${baseUrl}/admin/users/${userId}/password`, {
                    method: 'POST',
                    body: formData,
                    headers: {
                        'X-CSRF-TOKEN': '{{ csrf_token() }}',
                        'X-HTTP-Method-Override': 'PUT',
                        'Accept': 'application/json'
                    }
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        showSuccessAlert('Success', data.message);
                        closePasswordModal();
                    } else {
                        showErrorAlert('Error', data.message);
                    }
                })
                .catch(error => {
                    console.error('Error:', error);
                    showErrorAlert('Error', 'Failed to update password: ' + error.message);
                })
                .finally(() => {
                    submitButton.innerHTML = originalText;
                    submitButton.disabled = false;
                });
            });
        }
    });

    // Resend Password Function
    function resendPassword(userId) {
        showConfirmation(
            'Resend Password',
            'Are you sure you want to generate a new password and resend it to this user?',
            () => {
                const button = event.target.closest('button');
                const originalHTML = button.innerHTML;
                button.innerHTML = '<i class="fas fa-spinner fa-spin"></i>';
                button.disabled = true;

                fetch(`${baseUrl}/admin/users/${userId}/resend-password`, {
                    method: 'POST',
                    headers: {
                        'X-CSRF-TOKEN': '{{ csrf_token() }}',
                        'Content-Type': 'application/json',
                        'Accept': 'application/json'
                    }
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        showSuccessAlert('Success', data.message);
                    } else {
                        if (data.generated_password) {
                            showErrorAlert('Email Failed', data.message + ' Please provide this password to the user manually: ' + data.generated_password);
                        } else {
                            showErrorAlert('Error', data.message);
                        }
                    }
                })
                .catch(error => {
                    console.error('Error:', error);
                    showErrorAlert('Error', 'Failed to resend password: ' + error.message);
                })
                .finally(() => {
                    button.innerHTML = originalHTML;
                    button.disabled = false;
                });
            }
        );
    }

    function toggleStatus(userId) {
        showConfirmation(
            'Change User Status',
            'Are you sure you want to change the status of this user?',
            () => {
                const button = event.target.closest('button');
                const originalHTML = button.innerHTML;
                button.innerHTML = '<i class="fas fa-spinner fa-spin"></i>';
                button.disabled = true;

                fetch(`${baseUrl}/admin/users/${userId}/toggle-status`, {
                    method: 'POST',
                    headers: {
                        'X-CSRF-TOKEN': '{{ csrf_token() }}',
                        'Content-Type': 'application/json',
                        'Accept': 'application/json'
                    }
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        showSuccessAlert('Success', data.message);
                        setTimeout(() => location.reload(), 1500);
                    } else {
                        showErrorAlert('Error', data.message);
                    }
                })
                .catch(error => {
                    console.error('Error:', error);
                    showErrorAlert('Error', 'Failed to update user status: ' + error.message);
                })
                .finally(() => {
                    button.innerHTML = originalHTML;
                    button.disabled = false;
                });
            }
        );
    }

    function confirmDelete(userId) {
        showConfirmation(
            'Delete User',
            'Are you sure you want to delete this user? This action cannot be undone.',
            () => {
                const button = event.target.closest('button');
                const originalHTML = button.innerHTML;
                button.innerHTML = '<i class="fas fa-spinner fa-spin"></i>';
                button.disabled = true;

                fetch(`${baseUrl}/admin/users/${userId}`, {
                    method: 'DELETE',
                    headers: {
                        'X-CSRF-TOKEN': '{{ csrf_token() }}',
                        'Content-Type': 'application/json',
                        'Accept': 'application/json'
                    }
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        showSuccessAlert('Success', data.message);
                        setTimeout(() => location.reload(), 1500);
                    } else {
                        showErrorAlert('Error', data.message);
                    }
                })
                .catch(error => {
                    console.error('Error:', error);
                    showErrorAlert('Error', 'Failed to delete user: ' + error.message);
                })
                .finally(() => {
                    button.innerHTML = originalHTML;
                    button.disabled = false;
                });
            }
        );
    }

    // Close modals when clicking outside
    document.addEventListener('click', function(event) {
        const modals = [
            'createModal', 'editModal', 'passwordModal', 'editLoading', 
            'successAlert', 'errorAlert', 'confirmationModal'
        ];
        
        modals.forEach(modalId => {
            const modal = document.getElementById(modalId);
            if (event.target === modal) {
                if (modalId === 'successAlert') closeSuccessAlert();
                else if (modalId === 'errorAlert') closeErrorAlert();
                else if (modalId === 'confirmationModal') closeConfirmationModal(false);
                else if (modalId === 'editLoading') {
                    modal.classList.add('hidden');
                    closeEditModal();
                }
                else if (modalId === 'createModal') closeCreateModal();
                else if (modalId === 'editModal') closeEditModal();
                else if (modalId === 'passwordModal') closePasswordModal();
            }
        });
    });

    // Close modals with Escape key
    document.addEventListener('keydown', function(event) {
        if (event.key === 'Escape') {
            closeCreateModal();
            closeEditModal();
            closePasswordModal();
            closeSuccessAlert();
            closeErrorAlert();
            closeConfirmationModal(false);
            document.getElementById('editLoading').classList.add('hidden');
        }
    });
</script>
@endsection