<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\DisbursedLoan;
use App\Models\PaymentSchedule;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class InitializeAccrualPeriods extends Command
{
    protected $signature = 'accruals:initialize 
                            {--loan= : Specific loan ID to process}
                            {--all : Process all loans}
                            {--dry-run : Show what would be done without making changes}
                            {--force : Skip confirmation}
                            {--chunk=100 : Number of loans to process at once}';
    
    protected $description = 'Initialize accrual periods for existing loans';

    public function handle()
    {
        $this->info('=== ACCRUAL PERIODS INITIALIZATION ===');
        $this->info('Starting at: ' . Carbon::now()->format('Y-m-d H:i:s'));
        $this->line('');
        
        $specificLoanId = $this->option('loan');
        $processAll = $this->option('all');
        $dryRun = $this->option('dry-run');
        $force = $this->option('force');
        $chunkSize = (int)$this->option('chunk');
        
        if (!$specificLoanId && !$processAll) {
            $this->error('ERROR: Please specify --loan=<id> or --all');
            $this->line('');
            $this->info('Usage examples:');
            $this->info('  php artisan accruals:initialize --all --dry-run          # Dry run for all loans');
            $this->info('  php artisan accruals:initialize --all --force            # Process all loans');
            $this->info('  php artisan accruals:initialize --loan=123 --dry-run     # Dry run for loan 123');
            $this->info('  php artisan accruals:initialize --loan=123 --force       # Process loan 123');
            $this->info('  php artisan accruals:initialize --all --dry-run --chunk=50 # Process in chunks of 50');
            return 1;
        }
        
        // Get loans to process
        $loansQuery = DisbursedLoan::where('status', 'active')
            ->where('principalbalance', '>', 0)
            ->with(['paymentSchedules' => function($query) {
                $query->orderBy('paymentdate', 'asc');
            }]);
        
        if ($specificLoanId) {
            $loansQuery->where('loanid', $specificLoanId);
        }
        
        $totalLoansCount = $loansQuery->count();
        
        if ($totalLoansCount === 0) {
            $this->error('No active loans found matching criteria');
            return 1;
        }
        
        $this->info("Found {$totalLoansCount} loan(s) to process");
        
        // Show warning for large operations
        if ($totalLoansCount > 100 && $processAll && !$force && !$dryRun) {
            $this->warn("WARNING: You are about to process {$totalLoansCount} loans.");
            $this->warn("Consider running with --dry-run first to see what will be done.");
            $this->line('');
        }
        
        if (!$force && !$dryRun) {
            if (!$this->confirm("Do you want to initialize accrual periods for {$totalLoansCount} loan(s)?")) {
                $this->info('Operation cancelled');
                return 0;
            }
        }
        
        // Initialize counters
        $totalSchedulesUpdated = 0;
        $totalLoansProcessed = 0;
        $totalErrors = 0;
        $results = [];
        
        // Process in chunks for memory efficiency
        $this->info("Processing loans in chunks of {$chunkSize}...");
        $this->line('');
        
        $loansQuery->chunk($chunkSize, function($loansChunk) use (&$results, &$totalSchedulesUpdated, &$totalLoansProcessed, &$totalErrors, $dryRun, $totalLoansCount) {
            
            foreach ($loansChunk as $loan) {
                try {
                    $this->line("Processing loan: {$loan->loannumber} (ID: {$loan->loanid})...");
                    
                    $result = $this->initializeLoanAccruals($loan, $dryRun);
                    
                    if ($result['success']) {
                        $totalLoansProcessed++;
                        $totalSchedulesUpdated += $result['schedules_updated'];
                        
                        $results[$loan->loannumber] = [
                            'status' => 'success',
                            'schedules_updated' => $result['schedules_updated'],
                            'accrual_period' => $result['accrual_period'],
                            'message' => $dryRun ? 'Would be updated' : 'Updated successfully',
                        ];
                        
                        $status = $dryRun ? 'DRY RUN' : 'SUCCESS';
                        $this->info("  ✓ {$status}: {$result['schedules_updated']} schedule(s) - {$result['accrual_period']}");
                    } else {
                        $totalErrors++;
                        $results[$loan->loannumber] = [
                            'status' => 'error',
                            'message' => $result['message'],
                        ];
                        
                        $this->error("  ✗ ERROR: {$result['message']}");
                    }
                    
                } catch (\Exception $e) {
                    $totalErrors++;
                    $results[$loan->loannumber] = [
                        'status' => 'exception',
                        'message' => $e->getMessage(),
                    ];
                    
                    Log::error("Error initializing accruals for loan {$loan->loannumber}: " . $e->getMessage());
                    $this->error("  ✗ EXCEPTION: " . $e->getMessage());
                }
            }
            
            // Show progress
            $progress = round(($totalLoansProcessed + $totalErrors) / $totalLoansCount * 100, 1);
            $this->line("");
            $this->info("Progress: {$progress}% ({$totalLoansProcessed} processed, {$totalErrors} errors)");
            $this->line("");
        });
        
        // Display summary
        $this->line('');
        $this->info('=== SUMMARY ===');
        $this->line('');
        
        $summaryTable = [
            ['Total Loans Found', $totalLoansCount],
            ['Total Loans Processed', $totalLoansProcessed],
            ['Total Schedules Updated', $totalSchedulesUpdated],
            ['Total Errors', $totalErrors],
            ['Success Rate', $totalLoansCount > 0 ? round(($totalLoansProcessed / $totalLoansCount) * 100, 1) . '%' : 'N/A'],
            ['Dry Run', $dryRun ? 'Yes' : 'No'],
            ['Completed At', Carbon::now()->format('Y-m-d H:i:s')],
        ];
        
        $this->table(['Metric', 'Value'], $summaryTable);
        
        if ($dryRun) {
            $this->line('');
            $this->warn('⚠️  DRY RUN: No changes were made to the database.');
            $this->warn('   Run without --dry-run flag to apply changes.');
        } else {
            $this->line('');
            $this->info('✅ Accrual periods initialized successfully!');
        }
        
        // Save results to log file
        $this->saveResultsToLog($results, $dryRun);
        
        // Show sample results
        if ($totalLoansProcessed > 0) {
            $this->line('');
            $this->info('=== SAMPLE RESULTS (First 3 loans) ===');
            $this->line('');
            
            $sampleCount = 0;
            foreach ($results as $loanNumber => $result) {
                if ($sampleCount >= 3) break;
                
                if ($result['status'] === 'success') {
                    $this->line("Loan: {$loanNumber}");
                    $this->line("  Status: " . ($dryRun ? 'Would update' : 'Updated'));
                    $this->line("  Schedules: {$result['schedules_updated']}");
                    $this->line("  Accrual Period: {$result['accrual_period']}");
                    $this->line("");
                    $sampleCount++;
                }
            }
        }
        
        if ($totalErrors > 0) {
            $this->line('');
            $this->error('=== ERRORS DETECTED ===');
            $this->line('');
            
            $errorCount = 0;
            foreach ($results as $loanNumber => $result) {
                if ($result['status'] !== 'success' && $errorCount < 5) {
                    $this->line("Loan: {$loanNumber}");
                    $this->line("  Error: {$result['message']}");
                    $this->line("");
                    $errorCount++;
                }
            }
            
            if ($totalErrors > 5) {
                $this->info("... and " . ($totalErrors - 5) . " more error(s)");
            }
            
            $this->line("Check storage/logs/accruals-initialization.log for full details");
        }
        
        return $totalErrors > 0 ? 1 : 0;
    }
    
    private function initializeLoanAccruals($loan, $dryRun = false)
    {
        $schedules = $loan->paymentSchedules;
        
        if ($schedules->isEmpty()) {
            return [
                'success' => false,
                'message' => 'No payment schedules found',
                'schedules_updated' => 0,
                'accrual_period' => 'N/A',
            ];
        }
        
        $disbursementDate = Carbon::parse($loan->disburseddate);
        $updatedCount = 0;
        $firstAccrualStart = null;
        $lastAccrualEnd = null;
        
        if (!$dryRun) {
            DB::beginTransaction();
        }
        
        try {
            foreach ($schedules as $index => $schedule) {
                $paymentDate = Carbon::parse($schedule->paymentdate);
                
                // Determine accrual period
                if ($index === 0) {
                    // First schedule: accrual starts on disbursement date
                    $accrualStartDate = $disbursementDate->format('Y-m-d');
                } else {
                    // Subsequent schedules: start day after previous schedule's payment date
                    $previousSchedule = $schedules[$index - 1];
                    $previousPaymentDate = Carbon::parse($previousSchedule->paymentdate);
                    $accrualStartDate = $previousPaymentDate->copy()->addDay()->format('Y-m-d');
                }
                
                // Accrual ends on the payment date
                $accrualEndDate = $paymentDate->format('Y-m-d');
                
                if ($index === 0) {
                    $firstAccrualStart = $accrualStartDate;
                }
                if ($index === count($schedules) - 1) {
                    $lastAccrualEnd = $accrualEndDate;
                }
                
                // Check if accrual dates already exist and are correct
                $needsUpdate = false;
                
                if ($schedule->accrual_start_date !== $accrualStartDate || 
                    $schedule->accrual_end_date !== $accrualEndDate ||
                    $schedule->actualaccruedinterest === null ||
                    $schedule->actualaccruedprocessingfee === null) {
                    
                    $needsUpdate = true;
                }
                
                if ($needsUpdate && !$dryRun) {
                    $schedule->update([
                        'accrual_start_date' => $accrualStartDate,
                        'accrual_end_date' => $accrualEndDate,
                        'actualaccruedinterest' => $schedule->actualaccruedinterest ?? 0.00,
                        'actualaccruedprocessingfee' => $schedule->actualaccruedprocessingfee ?? 0.00,
                    ]);
                    
                    $updatedCount++;
                } elseif ($needsUpdate && $dryRun) {
                    // Count as would-be updated in dry run
                    $updatedCount++;
                }
            }
            
            if (!$dryRun) {
                DB::commit();
            }
            
            $accrualPeriod = "{$firstAccrualStart} to {$lastAccrualEnd}";
            
            Log::info("Accrual periods initialized for loan {$loan->loannumber}, {$updatedCount} schedules updated");
            
            return [
                'success' => true,
                'schedules_updated' => $updatedCount,
                'accrual_period' => $accrualPeriod,
            ];
            
        } catch (\Exception $e) {
            if (!$dryRun) {
                DB::rollBack();
            }
            
            throw $e;
        }
    }
    
    private function saveResultsToLog($results, $dryRun)
    {
        $logFile = storage_path('logs/accruals-initialization-' . Carbon::now()->format('Y-m-d-His') . '.log');
        
        $logContent = "=== ACCRUAL PERIODS INITIALIZATION LOG ===\n";
        $logContent .= "Date: " . Carbon::now()->format('Y-m-d H:i:s') . "\n";
        $logContent .= "Dry Run: " . ($dryRun ? 'Yes' : 'No') . "\n";
        $logContent .= "Total Loans: " . count($results) . "\n";
        $logContent .= "\n";
        
        $successCount = 0;
        $errorCount = 0;
        
        foreach ($results as $loanNumber => $result) {
            $logContent .= "Loan: {$loanNumber}\n";
            $logContent .= "Status: {$result['status']}\n";
            
            if ($result['status'] === 'success') {
                $successCount++;
                $logContent .= "Schedules Updated: {$result['schedules_updated']}\n";
                $logContent .= "Accrual Period: {$result['accrual_period']}\n";
            } else {
                $errorCount++;
                $logContent .= "Error: {$result['message']}\n";
            }
            
            $logContent .= "---\n";
        }
        
        $logContent .= "\n=== SUMMARY ===\n";
        $logContent .= "Successful: {$successCount}\n";
        $logContent .= "Errors: {$errorCount}\n";
        $logContent .= "Success Rate: " . round(($successCount / count($results)) * 100, 1) . "%\n";
        
        file_put_contents($logFile, $logContent);
        
        $this->line("Detailed log saved to: {$logFile}");
    }
}