<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class MigrateInterestIncome extends Command
{
    /**
     * The name and signature of the console command.
     * @var string
     */
    protected $signature = 'migrate:interestincome {file=interestincome.csv}';

    /**
     * The console command description.
     * @var string
     */
    protected $description = 'Migrate data from interestincome CSV to interestincome table';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $fileName = $this->argument('file');
        $filePath = base_path($fileName);

        if (!file_exists($filePath)) {
            $this->error("File not found at: {$filePath}");
            return 1;
        }

        $file = fopen($filePath, 'r');
        $header = fgetcsv($file);
        $headers = array_flip($header);
        
        $this->info("Starting interest income migration process...");
        $count = 0;

        DB::beginTransaction();

        try {
            // Disable foreign key checks temporarily if needed
            DB::statement('SET FOREIGN_KEY_CHECKS=0;');

            while (($row = fgetcsv($file)) !== false) {
                
                // Helper function: Handles empty strings and literal "NULL" text
                $get = function($key) use ($headers, $row) {
                    if (!isset($headers[$key])) return null;
                    $val = trim($row[$headers[$key]] ?? '');
                    if ($val === '' || strcasecmp($val, 'null') === 0) return null;
                    return $val;
                };

                // Parse date - handle different formats if needed
                $chargedDate = $get('chargeddate');
                if ($chargedDate) {
                    try {
                        $parsedDate = Carbon::createFromFormat('Y-m-d', $chargedDate);
                    } catch (\Exception $e) {
                        // Try alternative format if needed
                        $parsedDate = Carbon::parse($chargedDate);
                    }
                } else {
                    $parsedDate = now();
                }

                DB::table('interestincome')->insert([
                    'id' => $get('id'),
                    'iloanid' => (int) $get('iloanid') ?? 0,
                    'customerid' => (int) $get('customerid') ?? null,
                    'ipid' => (int) $get('ipid') ?? null,
                    'charge' => (float) $get('charge') ?? 0.00,
                    'chargeddate' => $parsedDate,
                    'poster' => (int) $get('poster') ?? null,
                    'branchid' => (int) $get('branchid') ?? null,
                    'companyid' => (int) $get('companyid') ?? null,
                    'created_at' => $parsedDate,
                    'updated_at' => $parsedDate,
                ]);

                $count++;
                
                // Progress output for large files
                if ($count % 1000 === 0) {
                    $this->info("Processed {$count} records...");
                }
            }

            DB::statement('SET FOREIGN_KEY_CHECKS=1;');
            DB::commit();
            $this->info("Successfully migrated {$count} interest income records.");
            
        } catch (\Exception $e) {
            DB::rollBack();
            DB::statement('SET FOREIGN_KEY_CHECKS=1;');
            $this->error("Migration failed: " . $e->getMessage());
            $this->error("Error on record #{$count}");
            $this->error("Row data: " . json_encode($row ?? []));
            throw $e;
        }

        fclose($file);
    }
}