<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class MigrateLoans extends Command
{
    /**
     * The name and signature of the console command.
     * @var string
     */
    protected $signature = 'migrate:loans {file=loanform.csv}';

    /**
     * The console command description.
     * @var string
     */
    protected $description = 'Migrate data from old system CSV to applications table including oldloanstatus';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $fileName = $this->argument('file');
        $filePath = base_path($fileName);

        if (!file_exists($filePath)) {
            $this->error("File not found at: {$filePath}");
            return 1;
        }

        $file = fopen($filePath, 'r');
        $header = fgetcsv($file);
        $headers = array_flip($header);
        
        $this->info("Starting migration process...");
        $count = 0;

        // Mapping for the Laravel ENUM 'status' field
        $statusMap = [
            'Closed'               => 'disbursed',
            'Disbursed'            => 'disbursed',
            'Non Performing Loans' => 'disbursed',
            'Rolled over'          => 'disbursed',
            'Pending'              => 'submitted',
            'Sent For Approve'     => 'under_review',
            'Sent Back'            => 'draft',
            'Declined'             => 'rejected',
        ];

        DB::beginTransaction();

        try {
            while (($row = fgetcsv($file)) !== false) {
                
                // Helper function: Handles empty strings and literal "NULL" text
                $get = function($key) use ($headers, $row) {
                    if (!isset($headers[$key])) return null;
                    $val = trim($row[$headers[$key]]);
                    if ($val === '' || strcasecmp($val, 'null') === 0) return null;
                    return $val;
                };

                $loanAmount = (float) ($get('amount') ?? 0);
                $adminAmt   = (float) ($get('adminamount') ?? 0);
                $insAmt     = (float) ($get('insuranceamount') ?? 0);
                $rawStatus  = $get('status');

                DB::table('applications')->insert([
                    'id'                 => $get('loanid'),
                    'application_number' => 'APP-' . str_pad($get('loanid'), 6, '0', STR_PAD_LEFT),
                    'application_date'   => $get('pdate'),
                    'customer_id'        => $get('customerid'),
                    'product_id'         => $get('productid'),
                    'currency_id'        => 57,
                    
                    // Financials
                    'loan_amount'               => $loanAmount,
                    'net_disbursement'          => $loanAmount - ($adminAmt + $insAmt),
                    'loan_tenure'               => $get('duration') ?? 0,
                    'interest_rate'             => $get('rate') ?? 0,
                    'interest_method'           => $get('ratemethod'),
                    'estimated_monthly_payment' => $get('cyclepayment'),
                    'installment_amount'        => $get('cyclepayment'),
                    'payment_frequency'         => $get('pcycle') ?? 'Monthly',
                    
                    // Fees
                    'adminfee'              => $get('adminfee'), 
                    'admin_fee_amount'      => $adminAmt,
                    'insurancefee'          => $get('insurancefee'),
                    'insurance_fee_amount'  => $insAmt,
                    'processing_fee_amount' => 0.00,
                    
                    // Income
                    'gross_salary' => $get('income'),
                    
                    // Status Workflow
                    'status'           => $statusMap[$rawStatus] ?? 'draft',
                    'oldloanstatus'   => $rawStatus, // Stores the exact text from CSV
                    'submitted_at'     => $get('pdate'),
                    'reviewed_at'      => $get('approvaldate'),
                    'approved_at'      => $get('approvaldate'),
                    'disbursed_at'     => $get('disburseddate'),
                    'rejection_reason' => $get('reason'),
                    'admin_notes'      => $get('cdescr'), 
                    
                    // Meta
                    'branch_id'               => $get('branchid') ?? 1,
                    'company_id'              => $get('companyid') ?? 1,
                    'created_by'              => $get('userid'),
                    'source'                  => 'Migration',
                    'penalty_rate'            => $get('discount') ?? 0,
                    'suggestedpaymentmethod'  => $get('disbursedmethod'),
                    'loan_id'                 => $get('loanid'),
                    'kin_country'             => 'Zambia',
                    'applicationtype'         => 'Normal',
                    
                    // Timestamps
                    'created_at' => $get('pdate') ? Carbon::parse($get('pdate')) : now(),
                    'updated_at' => $get('pdate') ? Carbon::parse($get('pdate')) : now(),
                ]);

                $count++;
            }

            DB::commit();
            $this->info("Successfully migrated {$count} records.");
            
        } catch (\Exception $e) {
            DB::rollBack();
            $this->error("Migration failed: " . $e->getMessage());
        }

        fclose($file);
    }
}