<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class MigratePaymentSchedules extends Command
{
    /**
     * The name and signature of the console command.
     * @var string
     */
    protected $signature = 'migrate:payment-schedules {file=paymentschedule.csv}';

    /**
     * The console command description.
     * @var string
     */
    protected $description = 'Migrate data from paymentschedule CSV to the new paymentschedule table';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $fileName = $this->argument('file');
        $filePath = base_path($fileName);

        if (!file_exists($filePath)) {
            $this->error("File not found at: {$filePath}");
            return 1;
        }

        $file = fopen($filePath, 'r');
        $header = fgetcsv($file);
        $headers = array_flip($header);
        
        $this->info("Starting payment schedule migration process...");
        $count = 0;

        DB::beginTransaction();

        try {
            // Disable foreign key checks for the migration
            DB::statement('SET FOREIGN_KEY_CHECKS=0;');

            while (($row = fgetcsv($file)) !== false) {
                
                // Helper to get CSV values by header name
                $get = function($key) use ($headers, $row) {
                    if (!isset($headers[$key])) return null;
                    $val = trim($row[$headers[$key]] ?? '');
                    if ($val === '' || strcasecmp($val, 'null') === 0) return null;
                    return $val;
                };

                // Parse Dates
                $paymentDate = $get('paymentdate') ? Carbon::parse($get('paymentdate')) : now();
                $systemDate = $get('systemdate') ? Carbon::parse($get('systemdate')) : now();

                // Check if record already exists
                $existing = DB::table('paymentschedule')->where('scheduleid', $get('paymentid'))->exists();
                
                if (!$existing) {
                    $principal = (float) $get('principal') ?? 0.00;
                    $interest = (float) $get('interestcharge') ?? 0.00;
                    $paidPrincipal = (float) $get('actualppay') ?? 0.00;
                    $paidInterest = (float) $get('actuallipay') ?? 0.00;

                    DB::table('paymentschedule')->insert([
                        'scheduleid'           => $get('paymentid'),
                        'loanid'               => $get('ploanid'),
                        'loannumber'           => 'LN-' . $get('ploanid'), // Generating a placeholder number
                        'processingfeesamount' => 0.00,
                        'interestamount'       => $interest,
                        'principalamount'      => $principal,
                        'totalamount'          => (float) $get('amountpaid') ?? 0.00,
                        'paymentdate'          => $paymentDate,
                        'transactiontype'      => 'Repayment',
                        'status'               => ((float)$get('actualpay') > 0) ? 'paid' : 'scheduled',
                        
                        // Financial breakdowns
                        'outstanding_principal' => $principal - $paidPrincipal,
                        'outstanding_interest'  => $interest - $paidInterest,
                        'paid_principal'        => $paidPrincipal,
                        'paid_interest'         => $paidInterest,
                        'total_paid'            => (float) $get('actualpay') ?? 0.00,
                        
                        // Metadata
                        'branchid'             => (int) $get('branchid') ?? 1,
                        'companyid'            => (int) $get('companyid') ?? 1,
                        'createdby'            => (int) $get('poster') ?? 1,
                        'updatedon'            => $systemDate,
                        'created_at'           => $systemDate,
                        'updated_at'           => $systemDate,
                    ]);

                    $count++;
                    
                    if ($count % 100 === 0) {
                        $this->info("Processed {$count} records...");
                    }
                }
            }

            DB::statement('SET FOREIGN_KEY_CHECKS=1;');
            DB::commit();
            $this->info("Successfully migrated {$count} payment schedule records.");
            
        } catch (\Exception $e) {
            DB::rollBack();
            DB::statement('SET FOREIGN_KEY_CHECKS=1;');
            $this->error("Migration failed: " . $e->getMessage());
            throw $e;
        }

        fclose($file);
    }
}