<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class MigrateRepayments extends Command
{
    /**
     * The name and signature of the console command.
     * @var string
     */
    protected $signature = 'migrate:repayments {file=repayments.csv}';

    /**
     * The console command description.
     * @var string
     */
    protected $description = 'Migrate data from repayments CSV to repayments table';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $fileName = $this->argument('file');
        $filePath = base_path($fileName);

        if (!file_exists($filePath)) {
            $this->error("File not found at: {$filePath}");
            return 1;
        }

        $file = fopen($filePath, 'r');
        $header = fgetcsv($file);
        $headers = array_flip($header);
        
        $this->info("Starting repayments migration process...");
        $count = 0;

        DB::beginTransaction();

        try {
            // Disable foreign key checks temporarily if needed
            DB::statement('SET FOREIGN_KEY_CHECKS=0;');

            while (($row = fgetcsv($file)) !== false) {
                
                // Helper function: Handles empty strings and literal "NULL" text
                $get = function($key) use ($headers, $row) {
                    if (!isset($headers[$key])) return null;
                    $val = trim($row[$headers[$key]] ?? '');
                    if ($val === '' || strcasecmp($val, 'null') === 0) return null;
                    return $val;
                };

                // Parse date - handle different formats if needed
                $rdate = $get('rdate');
                if ($rdate) {
                    try {
                        $parsedDate = Carbon::createFromFormat('Y-m-d', $rdate);
                    } catch (\Exception $e) {
                        // Try alternative format if needed
                        $parsedDate = Carbon::parse($rdate);
                    }
                } else {
                    $parsedDate = now();
                }

                // Check if record already exists to avoid duplicates
                $existing = DB::table('repayments')->where('id', $get('id'))->exists();
                
                if (!$existing) {
                    DB::table('repayments')->insert([
                        'id' => $get('id'),
                        'customerid' => (int) $get('customerid') ?? null,
                        'rpid' => (int) $get('rpid') ?? null,
                        'rloanid' => (int) $get('rloanid') ?? 0,
                        'rtotal' => (float) $get('rtotal') ?? 0.00,
                        'rinterest' => (float) $get('rinterest') ?? 0.00,
                        'processing_fees_amount_paid' => 0.00, // Default value
                        'penalties' => 0.00, // Default value
                        'rprincipal' => (float) $get('rprincipal') ?? 0.00,
                        'rdate' => $parsedDate,
                        'userid' => (int) $get('userid') ?? null,
                        'docpath' => $get('docpath'),
                        'class' => $get('class'),
                        'psid' => (int) $get('psid') ?? null,
                        'branchid' => (int) $get('branchid') ?? null,
                        'companyid' => (int) $get('companyid') ?? null,
                        'pmethod' => $get('pmethod'),
                        'paccount' => (int) $get('paccount') ?? null,
                        'transtype' => $get('transtype'),
                        'status' => $get('status'),
                        'created_at' => $parsedDate,
                        'updated_at' => $parsedDate,
                        'paymentreference' => null,
                        'bank_name' => null,
                        'account_holder' => null,
                        'account_number' => null,
                        'bank_branch' => null,
                        'swift_code' => null,
                        'mobile_channel' => null,
                        'mobile_number' => null,
                        'payment_notes' => null,
                        'allocation_mode' => 'auto'
                    ]);

                    $count++;
                    
                    // Progress output for large files
                    if ($count % 100 === 0) {
                        $this->info("Processed {$count} records...");
                    }
                }
            }

            DB::statement('SET FOREIGN_KEY_CHECKS=1;');
            DB::commit();
            $this->info("Successfully migrated {$count} repayment records.");
            
        } catch (\Exception $e) {
            DB::rollBack();
            DB::statement('SET FOREIGN_KEY_CHECKS=1;');
            $this->error("Migration failed: " . $e->getMessage());
            $this->error("Error on record #{$count}");
            $this->error("Row data: " . json_encode($row ?? []));
            throw $e;
        }

        fclose($file);
    }
}