<?php

namespace App\Http\Controllers\Accounting;

use App\Http\Controllers\Controller;
use App\Models\ApplicationRequirement;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;

class ApplicationRequirementController extends Controller
{
    /**
     * Display a listing of application requirements.
     */
    public function index()
    {
        $requirements = ApplicationRequirement::forCompany()
            ->orderBy('documentname')
            ->paginate(20);
        
        return view('accounting.applicationrequirements.index', [
            'requirements' => $requirements,
            'stats' => $this->getStatistics(),
            'documentTypes' => $this->getDocumentTypes(),
        ]);
    }

    /**
     * Show the form for creating a new application requirement.
     */
    public function create()
    {
        return view('accounting.applicationrequirements.create');
    }

    /**
     * Store a newly created application requirement.
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'documentname' => 'required|string|max:255|unique:applicationrequirements,documentname,NULL,id,companyid,' . auth()->user()->company_id,
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        try {
            DB::beginTransaction();

            $requirement = ApplicationRequirement::create([
                'documentname' => $request->documentname,
                'companyid' => auth()->user()->company_id,
                'createdby' => auth()->user()->username ?? auth()->user()->email,
            ]);

            DB::commit();

            return redirect()->route('admin.accounting.application-requirements.index')
                ->with('success', 'Application requirement added successfully!');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()
                ->with('error', 'Error creating requirement: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * Show the form for editing the specified application requirement.
     */
    public function edit($id)
    {
        $requirement = ApplicationRequirement::forCompany()->findOrFail($id);
        
        return view('accounting.applicationrequirements.edit', [
            'requirement' => $requirement,
        ]);
    }

    /**
     * Update the specified application requirement.
     */
    public function update(Request $request, $id)
    {
        $requirement = ApplicationRequirement::forCompany()->findOrFail($id);
        
        $validator = Validator::make($request->all(), [
            'documentname' => 'required|string|max:255|unique:applicationrequirements,documentname,' . $id . ',id,companyid,' . auth()->user()->company_id,
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        try {
            DB::beginTransaction();

            $requirement->update([
                'documentname' => $request->documentname,
            ]);

            DB::commit();

            return redirect()->route('admin.accounting.application-requirements.index')
                ->with('success', 'Application requirement updated successfully!');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()
                ->with('error', 'Error updating requirement: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * Delete application requirement.
     */
    public function destroy($id)
    {
        $requirement = ApplicationRequirement::forCompany()->findOrFail($id);
        
        $requirement->delete();
        
        return redirect()->route('admin.accounting.application-requirements.index')
            ->with('success', 'Application requirement deleted successfully!');
    }

    /**
     * Get statistics for application requirements.
     */
    private function getStatistics()
    {
        return [
            'total' => ApplicationRequirement::forCompany()->count(),
            'identification' => ApplicationRequirement::forCompany()
                ->where('documentname', 'like', '%id%')
                ->orWhere('documentname', 'like', '%identification%')
                ->count(),
            'income' => ApplicationRequirement::forCompany()
                ->where('documentname', 'like', '%pay%')
                ->orWhere('documentname', 'like', '%salary%')
                ->orWhere('documentname', 'like', '%income%')
                ->count(),
            'bank' => ApplicationRequirement::forCompany()
                ->where('documentname', 'like', '%bank%')
                ->orWhere('documentname', 'like', '%statement%')
                ->count(),
        ];
    }

    /**
     * Get document type categories.
     */
    private function getDocumentTypes()
    {
        return [
            'Identification' => ['National ID', 'Passport', 'Driver\'s License'],
            'Proof of Income' => ['Payslip', 'Bank Statement', 'Employment Letter'],
            'Proof of Address' => ['Utility Bill', 'Rental Agreement', 'Council Bill'],
            'Business Documents' => ['Business License', 'Tax Clearance', 'Registration'],
            'Other' => ['Photograph', 'Reference Letter', 'Application Form'],
        ];
    }

    /**
     * Bulk import requirements.
     */
    public function bulkImport(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'requirements' => 'required|string',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->with('error', 'Please enter at least one requirement');
        }

        try {
            DB::beginTransaction();

            $requirements = explode("\n", $request->requirements);
            $addedCount = 0;
            $skippedCount = 0;

            foreach ($requirements as $requirement) {
                $requirement = trim($requirement);
                
                if (empty($requirement)) {
                    continue;
                }

                // Check if requirement already exists
                $exists = ApplicationRequirement::forCompany()
                    ->where('documentname', $requirement)
                    ->exists();

                if (!$exists) {
                    ApplicationRequirement::create([
                        'documentname' => $requirement,
                        'companyid' => auth()->user()->company_id,
                        'createdby' => auth()->user()->username ?? auth()->user()->email,
                    ]);
                    $addedCount++;
                } else {
                    $skippedCount++;
                }
            }

            DB::commit();

            $message = "Added {$addedCount} new requirements";
            if ($skippedCount > 0) {
                $message .= ", {$skippedCount} already existed";
            }

            return redirect()->route('admin.accounting.application-requirements.index')
                ->with('success', $message);
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()
                ->with('error', 'Error importing requirements: ' . $e->getMessage());
        }
    }
}