<?php

namespace App\Http\Controllers\Accounting;

use App\Http\Controllers\Controller;
use App\Models\CashBankAccount;
use App\Models\ChartOfAccount;
use App\Models\AccountType;
use App\Models\GeneralLedgerEntry;
use App\Models\BusinessProcessMapping;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class CashBankAccountController extends Controller
{
    public function __construct()
    {
        // Removed middleware permission check
    }

    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        // Get the user's company ID
        $companyId = auth()->user()->companyid;
        
        // Check if company ID exists
        if (!$companyId) {
            return redirect()->back()
                ->with('error', 'User is not associated with a company. Please contact administrator.');
        }

        $accounts = CashBankAccount::with(['glAccount', 'creator'])
            ->where('companyid', $companyId)
            ->orderBy('accountType')
            ->orderBy('accountName')
            ->get();

        return view('accounting.cash-banking.accounts.index', compact('accounts'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        // Get the user's company ID
        $companyId = auth()->user()->companyid;
        
        // Check if company ID exists
        if (!$companyId) {
            return redirect()->back()
                ->with('error', 'User is not associated with a company. Please contact administrator.');
        }

        // Get appropriate GL accounts (Asset accounts)
        $glAccounts = ChartOfAccount::where('company_id', $companyId)
            ->where('is_active', true)
            ->where('account_type', 'Assets')
            ->orderBy('code')
            ->orderBy('name')
            ->get();

        $currencies = DB::table('currencies')->get();

        // Get Account Types for GL account creation
        $accountTypes = AccountType::where('is_active', true)
            ->with(['accountGroup.primaryCategory'])
            ->whereHas('accountGroup.primaryCategory', function($query) {
                $query->whereIn('name', ['Assets', 'Current Assets', 'Cash and Cash Equivalents']);
            })
            ->orderBy('name')
            ->get();

        return view('accounting.cash-banking.accounts.create', 
            compact('glAccounts', 'currencies', 'accountTypes'));
    }

    /**
     * Store a newly created resource in storage.
     */
  public function store(Request $request)
{
    // Get the user's company ID
    $companyId = auth()->user()->companyid;
    
    if (!$companyId) {
        return redirect()->back()
            ->with('error', 'User is not associated with a company. Please contact administrator.');
    }

    $validated = $request->validate([
        'accountType' => 'required|in:Cash,Mobile Money,Bank',
        'accountName' => 'required|string|max:100',
        'currency' => 'required|string|max:10',
        'bankAccountName' => 'nullable|string|max:100',
        'category' => 'nullable|string|max:50',
        'defaultPaymentMethod' => 'nullable|string|max:50',
        'bankName' => 'nullable|string|max:100',
        'accountNumber' => 'nullable|string|max:50',
        'branchName' => 'nullable|string|max:100',
        'description' => 'nullable|string',
        'openingBalance' => 'required|numeric|min:0',
        'openingBalanceDate' => 'required|date',
        'glAccountId' => 'nullable|exists:chart_of_accounts,id',
        'isActive' => 'boolean',
        'isDefault' => 'boolean',
        // New fields for GL account creation
        'create_gl_account' => 'boolean',
        'gl_account_type_id' => 'nullable|required_if:create_gl_account,1|exists:account_types,id',
        'gl_account_code' => 'nullable|string|max:50',
        'gl_account_description' => 'nullable|string',
    ]);

    // Use transaction for data consistency
    return DB::transaction(function () use ($validated, $request, $companyId) {
        try {
            // Ensure only one default account per company
            if ($request->has('isDefault') && $request->isDefault) {
                CashBankAccount::where('companyid', $companyId)
                    ->update(['isDefault' => false]);
            }

            // GL account variable
            $glAccount = null;

            // Create GL account if requested
            if ($request->has('create_gl_account') && $request->gl_account_type_id) {
                $accountType = AccountType::with(['accountGroup.primaryCategory'])->find($request->gl_account_type_id);
                
                if (!$accountType) {
                    throw new \Exception('Selected account type not found.');
                }

                // Generate account code if not provided
                $accountCode = $request->gl_account_code;
                if (!$accountCode) {
                    $prefix = $validated['accountType'] === 'Cash' ? 'CASH' : 
                             ($validated['accountType'] === 'Bank' ? 'BANK' : 'MM');
                    $cleanName = strtoupper(preg_replace('/[^A-Z0-9]/', '_', $validated['accountName']));
                    $accountCode = $prefix . '_' . $cleanName;
                }

                // Check if code already exists
                $existingAccount = ChartOfAccount::where('code', $accountCode)
                    ->where('company_id', $companyId)
                    ->first();
                
                if ($existingAccount) {
                    $accountCode = $accountCode . '_' . time();
                }

                // Create the GL account
                $glAccount = ChartOfAccount::create([
                    'company_id' => $companyId,
                    'branch_id' => 1, // Default branch
                    'account_type_id' => $request->gl_account_type_id,
                    'code' => $accountCode,
                    'name' => $validated['accountName'] . ' (' . $validated['accountType'] . ')',
                    'description' => $request->gl_account_description ?? $validated['description'] ?? null,
                    'account_type' => $accountType->accountGroup->primaryCategory->name ?? 'Assets',
                    'opening_balance' => $validated['openingBalance'],
                    'current_balance' => $validated['openingBalance'],
                    'is_active' => $request->has('isActive'),
                    'created_by' => auth()->id(),
                ]);

                // Use this GL account ID
                $glAccountId = $glAccount->id;
            } else {
                // Use existing GL account if provided
                $glAccountId = $validated['glAccountId'] ?? null;
                if ($glAccountId) {
                    $glAccount = ChartOfAccount::find($glAccountId);
                }
            }

            // Create the cash/bank account
            $account = CashBankAccount::create([
                'accountType' => $validated['accountType'],
                'accountName' => $validated['accountName'],
                'currency' => $validated['currency'],
                'bankAccountName' => $validated['bankAccountName'] ?? null,
                'category' => $validated['category'] ?? null,
                'defaultPaymentMethod' => $validated['defaultPaymentMethod'] ?? null,
                'bankName' => $validated['bankName'] ?? null,
                'accountNumber' => $validated['accountNumber'] ?? null,
                'branchName' => $validated['branchName'] ?? null,
                'description' => $validated['description'] ?? null,
                'openingBalance' => $validated['openingBalance'],
                'openingBalanceDate' => $validated['openingBalanceDate'],
                'glAccountId' => $glAccountId ?? null,
                'current_balance' => $validated['openingBalance'],
                'userid' => auth()->id(),
                'companyid' => $companyId,
                'isActive' => $request->has('isActive'),
                'isDefault' => $request->has('isDefault')
            ]);

            // Record opening balance in CashBook
            if ($validated['openingBalance'] > 0) {
                \App\Models\CashBook::recordOpeningBalance(
                    $account, 
                    $validated['openingBalance'],
                    "Opening balance for new {$validated['accountType']} account: {$validated['accountName']}"
                );
            }

            // Post opening balance to General Ledger if GL account exists
            if ($glAccount && $validated['openingBalance'] > 0) {
                $this->postOpeningBalanceToGL($account, $glAccount, $validated['openingBalance']);
            }

            DB::commit();

            $successMessage = 'Cash/Bank account created successfully.';
            if ($glAccount) {
                $successMessage .= ' Corresponding GL account ' . 
                    ($request->has('create_gl_account') ? 'created and ' : '') . 
                    'opening balance posted to General Ledger.';
            }
            
            $successMessage .= ' Opening balance recorded in cash book.';

            return redirect()->route('admin.accounting.cash-banking.accounts.index')
                ->with('success', $successMessage);

        } catch (\Exception $e) {
            DB::rollBack();
            
            return redirect()->back()
                ->withInput()
                ->with('error', 'Error creating account: ' . $e->getMessage());
        }
    });
}



    /**
     * Post opening balance to General Ledger
     */
    private function postOpeningBalanceToGL(CashBankAccount $cashAccount, ChartOfAccount $glAccount, $amount)
    {
        try {
            // Get the opening balance mapping from business process mappings
            $openingBalanceMapping = BusinessProcessMapping::where('nameofthebusinessprocess', 'Opening Balances')
                ->first();

            if (!$openingBalanceMapping) {
                throw new \Exception('Opening Balances business process mapping not found. Please configure it in Business Process Mappings.');
            }

            $correspondingAccountId = $openingBalanceMapping->accountid;
            $correspondingAccount = ChartOfAccount::find($correspondingAccountId);

            if (!$correspondingAccount) {
                throw new \Exception("Corresponding account (ID: {$correspondingAccountId}) for opening balances not found.");
            }

            // Get the account types with their normal_balance values
            $accountType = AccountType::find($glAccount->account_type_id);
            $correspondingAccountType = AccountType::find($correspondingAccount->account_type_id);

            if (!$accountType || !$correspondingAccountType) {
                throw new \Exception("Account type information not found for one of the accounts.");
            }

            // Use the actual normal_balance field from account_types table
            $newAccountNormalBalance = strtolower($accountType->normal_balance);
            $correspondingAccountNormalBalance = strtolower($correspondingAccountType->normal_balance);
            
            $entryTypeForAccount = null;
            $entryTypeForOpeningBalanceAccount = null;

            // Set amount values based on normal_balance (positive = debit, negative = credit)
            if ($newAccountNormalBalance === 'debit') {
                // Account with debit normal balance: Positive amount (debit)
                $newAccountAmount = $amount; // Positive = Debit
                $correspondingAccountAmount = -$amount; // Negative = Credit
                $entryTypeForAccount = 'debit';
                $entryTypeForOpeningBalanceAccount = 'credit';
                
            } elseif ($newAccountNormalBalance === 'credit') {
                // Account with credit normal balance: Negative amount (credit)
                $newAccountAmount = -$amount; // Negative = Credit
                $correspondingAccountAmount = $amount; // Positive = Debit
                $entryTypeForAccount = 'credit';
                $entryTypeForOpeningBalanceAccount = 'debit';
                
            } else {
                throw new \Exception("Invalid normal_balance value for account type: {$newAccountNormalBalance}");
            }

            // Create the journal entry
            $transactionId = 'CASH_OB_' . time() . '_' . uniqid();
            $currentUserId = auth()->id();
            $currentTime = now();
            $transType = 'Cash/Bank Account Opening Balance';
            $description = "Opening balance for cash/bank account: {$cashAccount->accountName}";

            // Entry for the cash/bank GL account (DEBIT)
            GeneralLedgerEntry::create([
                'accountid' => $glAccount->id,
                'entrydate' => $currentTime,
                'entrytype' => $entryTypeForAccount,
                'amount' => $newAccountAmount,
                'description' => $description,
                'transactionid' => $transactionId,
                'transtype' => $transType,
                'companyid' => $glAccount->company_id,
                'branchid' => $glAccount->branch_id,
                'createdby' => $currentUserId,
                'created_at' => $currentTime,
                'updated_at' => $currentTime,
            ]);

            // Corresponding entry for the opening balance account (CREDIT)
            GeneralLedgerEntry::create([
                'accountid' => $correspondingAccountId,
                'entrydate' => $currentTime,
                'entrytype' => $entryTypeForOpeningBalanceAccount,
                'amount' => $correspondingAccountAmount,
                'description' => "Corresponding entry for: {$description}",
                'transactionid' => $transactionId,
                'transtype' => $transType,
                'companyid' => $glAccount->company_id,
                'branchid' => $glAccount->branch_id,
                'createdby' => $currentUserId,
                'created_at' => $currentTime,
                'updated_at' => $currentTime,
            ]);

            // Update current balances for both accounts
            $this->updateAccountBalance($glAccount);
            $this->updateAccountBalance($correspondingAccount);

        } catch (\Exception $e) {
            throw new \Exception('Failed to post opening balance to General Ledger: ' . $e->getMessage());
        }
    }

    /**
     * Update account current balance based on general ledger entries
     */
    private function updateAccountBalance(ChartOfAccount $account)
    {
        $balance = $account->generalLedgerEntries()
            ->selectRaw('SUM(amount) as balance')
            ->value('balance') ?? 0;

        $account->update(['current_balance' => $balance]);
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        // Get the user's company ID
        $companyId = auth()->user()->companyid;
        
        // Check if company ID exists
        if (!$companyId) {
            return redirect()->back()
                ->with('error', 'User is not associated with a company. Please contact administrator.');
        }

        $account = CashBankAccount::with([
            'glAccount', 
            'outgoingTransfers.fromAccount', 
            'outgoingTransfers.toAccount',
            'incomingTransfers.fromAccount',
            'incomingTransfers.toAccount',
            'pettyCashTransactions',
            'glAccount.generalLedgerEntries' => function($query) {
                $query->where('transtype', 'like', '%Opening Balance%')
                      ->orderBy('entrydate', 'desc');
            }
        ])
        ->where('cashBankId', $id)
        ->where('companyid', $companyId)
        ->firstOrFail();

        return view('accounting.cash-banking.accounts.show', compact('account'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        // Get the user's company ID
        $companyId = auth()->user()->companyid;
        
        // Check if company ID exists
        if (!$companyId) {
            return redirect()->back()
                ->with('error', 'User is not associated with a company. Please contact administrator.');
        }

        $account = CashBankAccount::where('cashBankId', $id)
            ->where('companyid', $companyId)
            ->firstOrFail();

        // Get appropriate GL accounts (Asset accounts)
        $glAccounts = ChartOfAccount::where('company_id', $companyId)
            ->where('is_active', true)
            ->where('account_type', 'Assets')
            ->orderBy('code')
            ->orderBy('name')
            ->get();

        $currencies = DB::table('currencies')->get();

        // Get Account Types for GL account creation
        $accountTypes = AccountType::where('is_active', true)
            ->with(['accountGroup.primaryCategory'])
            ->whereHas('accountGroup.primaryCategory', function($query) {
                $query->whereIn('name', ['Assets', 'Current Assets', 'Cash and Cash Equivalents']);
            })
            ->orderBy('name')
            ->get();

        return view('accounting.cash-banking.accounts.edit', 
            compact('account', 'glAccounts', 'currencies', 'accountTypes'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        // Get the user's company ID
        $companyId = auth()->user()->companyid;
        
        if (!$companyId) {
            return redirect()->back()
                ->with('error', 'User is not associated with a company. Please contact administrator.');
        }

        $account = CashBankAccount::where('cashBankId', $id)
            ->where('companyid', $companyId)
            ->firstOrFail();

        $validated = $request->validate([
            'accountType' => 'required|in:Cash,Mobile Money,Bank',
            'accountName' => 'required|string|max:100',
            'currency' => 'required|string|max:10',
            'bankAccountName' => 'nullable|string|max:100',
            'category' => 'nullable|string|max:50',
            'defaultPaymentMethod' => 'nullable|string|max:50',
            'bankName' => 'nullable|string|max:100',
            'accountNumber' => 'nullable|string|max:50',
            'branchName' => 'nullable|string|max:100',
            'description' => 'nullable|string',
            'openingBalance' => 'required|numeric|min:0',
            'openingBalanceDate' => 'required|date',
            'glAccountId' => 'nullable|exists:chart_of_accounts,id',
            'isActive' => 'boolean',
            'isDefault' => 'boolean',
        ]);

        // Check if opening balance changed
        $openingBalanceChanged = $account->openingBalance != $validated['openingBalance'];
        $oldOpeningBalance = $account->openingBalance;

        // Ensure only one default account per company
        if ($request->has('isDefault') && $request->isDefault && !$account->isDefault) {
            CashBankAccount::where('companyid', $companyId)
                ->update(['isDefault' => false]);
        }

        // Update the account
        $account->update([
            'accountType' => $validated['accountType'],
            'accountName' => $validated['accountName'],
            'currency' => $validated['currency'],
            'bankAccountName' => $validated['bankAccountName'] ?? null,
            'category' => $validated['category'] ?? null,
            'defaultPaymentMethod' => $validated['defaultPaymentMethod'] ?? null,
            'bankName' => $validated['bankName'] ?? null,
            'accountNumber' => $validated['accountNumber'] ?? null,
            'branchName' => $validated['branchName'] ?? null,
            'description' => $validated['description'] ?? null,
            'openingBalance' => $validated['openingBalance'],
            'openingBalanceDate' => $validated['openingBalanceDate'],
            'glAccountId' => $validated['glAccountId'] ?? $account->glAccountId,
            'current_balance' => $account->current_balance + ($validated['openingBalance'] - $oldOpeningBalance),
            'isActive' => $request->has('isActive'),
            'isDefault' => $request->has('isDefault')
        ]);

        // If GL account is linked and opening balance changed, create adjustment entry
        if ($account->glAccount && $openingBalanceChanged) {
            $this->createOpeningBalanceAdjustment($account, $oldOpeningBalance, $validated['openingBalance']);
        }

        return redirect()->route('admin.accounting.cash-banking.accounts.index')
            ->with('success', 'Cash/Bank account updated successfully.' . 
                   ($openingBalanceChanged && $account->glAccount ? ' Opening balance adjustment posted to GL.' : ''));
    }

    /**
     * Create adjustment entry for opening balance change
     */
    private function createOpeningBalanceAdjustment(CashBankAccount $account, $oldBalance, $newBalance)
    {
        try {
            $adjustmentAmount = $newBalance - $oldBalance;
            
            if ($adjustmentAmount == 0) {
                return;
            }

            // Get the GL account
            $glAccount = ChartOfAccount::find($account->glAccountId);
            if (!$glAccount) {
                return;
            }

            // Get the opening balance mapping account
            $openingBalanceMapping = BusinessProcessMapping::where('nameofthebusinessprocess', 'Opening Balances')
                ->first();

            if (!$openingBalanceMapping) {
                throw new \Exception('Opening Balances business process mapping not found.');
            }

            $correspondingAccount = ChartOfAccount::find($openingBalanceMapping->accountid);
            if (!$correspondingAccount) {
                throw new \Exception("Corresponding account for opening balances not found.");
            }

            // Get account types
            $accountType = AccountType::find($glAccount->account_type_id);
            $correspondingAccountType = AccountType::find($correspondingAccount->account_type_id);

            if (!$accountType || !$correspondingAccountType) {
                throw new \Exception("Account type information not found.");
            }

            // Determine entry types based on normal balance and adjustment amount
            $newAccountNormalBalance = strtolower($accountType->normal_balance);
            
            $glAccountEntryType = '';
            $correspondingEntryType = '';
            $glAccountAmount = 0;
            $correspondingAmount = 0;

            if ($newAccountNormalBalance === 'debit') {
                if ($adjustmentAmount > 0) {
                    // Increase in debit account = Debit the GL account, Credit the corresponding
                    $glAccountEntryType = 'debit';
                    $correspondingEntryType = 'credit';
                    $glAccountAmount = $adjustmentAmount;
                    $correspondingAmount = -$adjustmentAmount;
                } else {
                    // Decrease in debit account = Credit the GL account, Debit the corresponding
                    $glAccountEntryType = 'credit';
                    $correspondingEntryType = 'debit';
                    $glAccountAmount = $adjustmentAmount; // Negative
                    $correspondingAmount = -$adjustmentAmount; // Positive
                }
            } else {
                if ($adjustmentAmount > 0) {
                    // Increase in credit account = Credit the GL account, Debit the corresponding
                    $glAccountEntryType = 'credit';
                    $correspondingEntryType = 'debit';
                    $glAccountAmount = -$adjustmentAmount;
                    $correspondingAmount = $adjustmentAmount;
                } else {
                    // Decrease in credit account = Debit the GL account, Credit the corresponding
                    $glAccountEntryType = 'debit';
                    $correspondingEntryType = 'credit';
                    $glAccountAmount = -$adjustmentAmount; // Positive
                    $correspondingAmount = $adjustmentAmount; // Negative
                }
            }

            // Create adjustment journal entry
            $transactionId = 'CASH_ADJ_' . time() . '_' . uniqid();
            $currentUserId = auth()->id();
            $currentTime = now();
            $description = "Opening balance adjustment for {$account->accountName}: {$oldBalance} to {$newBalance}";

            // Adjustment entry for the GL account
            GeneralLedgerEntry::create([
                'accountid' => $glAccount->id,
                'entrydate' => $currentTime,
                'entrytype' => $glAccountEntryType,
                'amount' => $glAccountAmount,
                'description' => $description,
                'transactionid' => $transactionId,
                'transtype' => 'Opening Balance Adjustment',
                'companyid' => $glAccount->company_id,
                'branchid' => $glAccount->branch_id,
                'createdby' => $currentUserId,
                'created_at' => $currentTime,
                'updated_at' => $currentTime,
            ]);

            // Corresponding adjustment entry
            GeneralLedgerEntry::create([
                'accountid' => $correspondingAccount->id,
                'entrydate' => $currentTime,
                'entrytype' => $correspondingEntryType,
                'amount' => $correspondingAmount,
                'description' => "Adjustment for: {$description}",
                'transactionid' => $transactionId,
                'transtype' => 'Opening Balance Adjustment',
                'companyid' => $glAccount->company_id,
                'branchid' => $glAccount->branch_id,
                'createdby' => $currentUserId,
                'created_at' => $currentTime,
                'updated_at' => $currentTime,
            ]);

            // Update current balances
            $this->updateAccountBalance($glAccount);
            $this->updateAccountBalance($correspondingAccount);

        } catch (\Exception $e) {
            // Log error but don't fail the update
            \Log::error('Failed to create opening balance adjustment: ' . $e->getMessage());
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        // Get the user's company ID
        $companyId = auth()->user()->companyid;
        
        // Check if company ID exists
        if (!$companyId) {
            return redirect()->back()
                ->with('error', 'User is not associated with a company. Please contact administrator.');
        }

        $account = CashBankAccount::where('cashBankId', $id)
            ->where('companyid', $companyId)
            ->firstOrFail();

        // Check if account has transactions
        if ($account->outgoingTransfers()->exists() || 
            $account->incomingTransfers()->exists() || 
            $account->pettyCashTransactions()->exists()) {
            return redirect()->back()
                ->with('error', 'Cannot delete account that has transactions. Deactivate it instead.');
        }

        // Use transaction for consistency
        return DB::transaction(function () use ($account, $companyId) {
            try {
                // Store GL account info before deletion for reversal
                $glAccount = $account->glAccount;
                $glAccountId = $account->glAccountId;
                $openingBalance = $account->openingBalance;

                // Delete the cash/bank account
                $account->delete();

                // If there was a GL account with opening balance, create reversal entry
                if ($glAccount && $openingBalance > 0) {
                    $this->reverseOpeningBalanceEntry($glAccount, $openingBalance, $account->accountName);
                }

                DB::commit();

                return redirect()->route('admin.accounting.cash-banking.accounts.index')
                    ->with('success', 'Cash/Bank account deleted successfully.' . 
                           ($glAccount ? ' Opening balance reversed in General Ledger.' : ''));

            } catch (\Exception $e) {
                DB::rollBack();
                
                return redirect()->back()
                    ->with('error', 'Error deleting account: ' . $e->getMessage());
            }
        });
    }

    /**
     * Reverse opening balance entry when account is deleted
     */
    private function reverseOpeningBalanceEntry(ChartOfAccount $glAccount, $amount, $accountName)
    {
        try {
            // Get the opening balance mapping account
            $openingBalanceMapping = BusinessProcessMapping::where('nameofthebusinessprocess', 'Opening Balances')
                ->first();

            if (!$openingBalanceMapping) {
                throw new \Exception('Opening Balances business process mapping not found.');
            }

            $correspondingAccount = ChartOfAccount::find($openingBalanceMapping->accountid);
            if (!$correspondingAccount) {
                throw new \Exception("Corresponding account for opening balances not found.");
            }

            // Get account types
            $accountType = AccountType::find($glAccount->account_type_id);
            $correspondingAccountType = AccountType::find($correspondingAccount->account_type_id);

            if (!$accountType || !$correspondingAccountType) {
                throw new \Exception("Account type information not found.");
            }

            // Reverse the original entry - opposite of original posting
            $newAccountNormalBalance = strtolower($accountType->normal_balance);
            
            $glAccountEntryType = '';
            $correspondingEntryType = '';
            $glAccountAmount = 0;
            $correspondingAmount = 0;

            if ($newAccountNormalBalance === 'debit') {
                // Original was debit to GL account, credit to corresponding
                // Reverse: credit GL account, debit corresponding
                $glAccountEntryType = 'credit';
                $correspondingEntryType = 'debit';
                $glAccountAmount = -$amount; // Negative (credit)
                $correspondingAmount = $amount; // Positive (debit)
            } else {
                // Original was credit to GL account, debit to corresponding
                // Reverse: debit GL account, credit corresponding
                $glAccountEntryType = 'debit';
                $correspondingEntryType = 'credit';
                $glAccountAmount = $amount; // Positive (debit)
                $correspondingAmount = -$amount; // Negative (credit)
            }

            // Create reversal journal entry
            $transactionId = 'CASH_REV_' . time() . '_' . uniqid();
            $currentUserId = auth()->id();
            $currentTime = now();
            $description = "Reversal of opening balance for deleted account: {$accountName}";

            // Reversal entry for the GL account
            GeneralLedgerEntry::create([
                'accountid' => $glAccount->id,
                'entrydate' => $currentTime,
                'entrytype' => $glAccountEntryType,
                'amount' => $glAccountAmount,
                'description' => $description,
                'transactionid' => $transactionId,
                'transtype' => 'Opening Balance Reversal',
                'companyid' => $glAccount->company_id,
                'branchid' => $glAccount->branch_id,
                'createdby' => $currentUserId,
                'created_at' => $currentTime,
                'updated_at' => $currentTime,
            ]);

            // Corresponding reversal entry
            GeneralLedgerEntry::create([
                'accountid' => $correspondingAccount->id,
                'entrydate' => $currentTime,
                'entrytype' => $correspondingEntryType,
                'amount' => $correspondingAmount,
                'description' => "Reversal for: {$description}",
                'transactionid' => $transactionId,
                'transtype' => 'Opening Balance Reversal',
                'companyid' => $glAccount->company_id,
                'branchid' => $glAccount->branch_id,
                'createdby' => $currentUserId,
                'created_at' => $currentTime,
                'updated_at' => $currentTime,
            ]);

            // Update current balances
            $this->updateAccountBalance($glAccount);
            $this->updateAccountBalance($correspondingAccount);

        } catch (\Exception $e) {
            \Log::error('Failed to reverse opening balance: ' . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Toggle account status
     */
    public function toggleStatus(string $id)
    {
        // Get the user's company ID
        $companyId = auth()->user()->companyid;
        
        // Check if company ID exists
        if (!$companyId) {
            return redirect()->back()
                ->with('error', 'User is not associated with a company. Please contact administrator.');
        }

        $account = CashBankAccount::where('cashBankId', $id)
            ->where('companyid', $companyId)
            ->firstOrFail();

        $account->update(['isActive' => !$account->isActive]);

        // Also update linked GL account status if exists
        if ($account->glAccount) {
            $account->glAccount->update(['is_active' => $account->isActive]);
        }

        $status = $account->isActive ? 'activated' : 'deactivated';

        return redirect()->back()
            ->with('success', "Account {$status} successfully." . 
                   ($account->glAccount ? ' Linked GL account also updated.' : ''));
    }

    /**
     * Get accounts for API/select2
     */
    public function getAccounts(Request $request)
    {
        // Get the user's company ID
        $companyId = auth()->user()->companyid;
        
        if (!$companyId) {
            return response()->json(['error' => 'User not associated with a company'], 400);
        }

        $accounts = CashBankAccount::where('companyid', $companyId)
            ->where('isActive', true)
            ->when($request->has('q'), function ($query) use ($request) {
                $query->where('accountName', 'like', '%' . $request->q . '%')
                      ->orWhere('accountNumber', 'like', '%' . $request->q . '%');
            })
            ->orderBy('accountName')
            ->get(['cashBankId as id', 'accountName as text', 'currency', 'current_balance']);

        return response()->json($accounts);
    }
}