<?php

namespace App\Http\Controllers\Accounting;

use App\Http\Controllers\Controller;
use App\Models\CashBook;
use App\Models\CashBankAccount;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class CashBookController extends Controller
{
    /**
     * Display cash book for all accounts or specific account
     */
    public function index(Request $request)
    {
        $user = auth()->user();
        $companyId = $user->companyid;
        
        if (!$companyId) {
            return redirect()->back()
                ->with('error', 'User is not associated with a company.');
        }

        // Get filter parameters
        $accountId = $request->get('account_id');
        $startDate = $request->get('start_date', date('Y-m-01'));
        $endDate = $request->get('end_date', date('Y-m-t'));
        $transactionType = $request->get('transaction_type');
        
        // Get all accounts for filter dropdown
        $accounts = CashBankAccount::where('companyid', $companyId)
            ->orderBy('accountType')
            ->orderBy('accountName')
            ->get();
        
        // Build query
        $query = CashBook::with(['account'])
            ->where('companyid', $companyId)
            ->whereBetween('transaction_date', [$startDate, $endDate]);
        
        if ($accountId) {
            $query->where('account_id', $accountId);
        }
        
        if ($transactionType) {
            $query->where('transaction_type', $transactionType);
        }
        
        $transactions = $query->orderBy('transaction_date', 'desc')
            ->orderBy('created_at', 'desc')
            ->paginate(50);
        
        // Summary statistics
        $summary = DB::table('cash_book')
            ->select(
                DB::raw('SUM(CASE WHEN amount > 0 THEN amount ELSE 0 END) as total_debits'),
                DB::raw('SUM(CASE WHEN amount < 0 THEN ABS(amount) ELSE 0 END) as total_credits'),
                DB::raw('COUNT(*) as transaction_count')
            )
            ->where('companyid', $companyId)
            ->whereBetween('transaction_date', [$startDate, $endDate])
            ->when($accountId, function($q) use ($accountId) {
                return $q->where('account_id', $accountId);
            })
            ->when($transactionType, function($q) use ($transactionType) {
                return $q->where('transaction_type', $transactionType);
            })
            ->first();
        
        // Transaction types for filter
        $transactionTypes = [
            CashBook::TRANSACTION_OPENING_BALANCE,
            CashBook::TRANSACTION_TRANSFER_IN,
            CashBook::TRANSACTION_TRANSFER_OUT,
            CashBook::TRANSACTION_PETTY_CASH_EXPENSE,
            CashBook::TRANSACTION_PETTY_CASH_INCOME,
            CashBook::TRANSACTION_EXPENSE,
            CashBook::TRANSACTION_INCOME,
            CashBook::TRANSACTION_ADJUSTMENT,
            CashBook::TRANSACTION_REVERSAL,
            CashBook::TRANSACTION_OTHER,
        ];
        
        return view('accounting.cash-banking.cash-book.index', compact(
            'transactions',
            'accounts',
            'accountId',
            'startDate',
            'endDate',
            'transactionType',
            'transactionTypes',
            'summary'
        ));
    }
    
    /**
     * Export cash book to CSV
     */
    public function export(Request $request)
    {
        $user = auth()->user();
        $companyId = $user->companyid;
        
        // Get filter parameters
        $accountId = $request->get('account_id');
        $startDate = $request->get('start_date', date('Y-m-01'));
        $endDate = $request->get('end_date', date('Y-m-t'));
        
        $query = CashBook::with(['account'])
            ->where('companyid', $companyId)
            ->whereBetween('transaction_date', [$startDate, $endDate]);
        
        if ($accountId) {
            $query->where('account_id', $accountId);
        }
        
        $transactions = $query->orderBy('transaction_date', 'asc')
            ->orderBy('created_at', 'asc')
            ->get();
        
        $filename = 'cash-book-' . date('Y-m-d') . '.csv';
        
        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => 'attachment; filename="' . $filename . '"',
        ];
        
        $callback = function() use ($transactions) {
            $file = fopen('php://output', 'w');
            
            // Add BOM for UTF-8
            fwrite($file, "\xEF\xBB\xBF");
            
            // Headers
            fputcsv($file, [
                'Date',
                'Account',
                'Type',
                'Reference',
                'Description',
                'Debit',
                'Credit',
                'Balance Before',
                'Balance After',
                'Currency',
                'Status'
            ]);
            
            // Data
            foreach ($transactions as $transaction) {
                fputcsv($file, [
                    $transaction->transaction_date,
                    $transaction->account->accountName ?? 'N/A',
                    $transaction->transaction_type,
                    $transaction->reference_number,
                    $transaction->description,
                    $transaction->amount > 0 ? $transaction->amount : '',
                    $transaction->amount < 0 ? abs($transaction->amount) : '',
                    $transaction->balance_before,
                    $transaction->balance_after,
                    $transaction->currency,
                    $transaction->status
                ]);
            }
            
            fclose($file);
        };
        
        return response()->stream($callback, 200, $headers);
    }
}