<?php

namespace App\Http\Controllers\Accounting;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\ChartOfAccount;
use App\Models\AccountGroup;
use App\Models\AccountType;
use App\Models\GeneralLedgerEntry;
use App\Models\BusinessProcessMapping;
use Illuminate\Support\Facades\DB;

class ChartOfAccountController extends Controller
{
    public function index()
    {
        $accounts = ChartOfAccount::with(['accountType', 'accountType.accountGroup'])
            ->orderBy('code')
            ->get();
            
        return view('accounting.chart-of-accounts.index', compact('accounts'));
    }

    public function create()
    {
        $accountTypes = AccountType::with('accountGroup')->get();
        
        return view('accounting.chart-of-accounts.create', compact('accountTypes'));
    }

    public function store(Request $request)
    {
        // Validate the request
        $validated = $request->validate([
            'code' => 'required|unique:chart_of_accounts,code',
            'name' => 'required|string|max:255',
            'account_type_id' => 'required|exists:account_types,id',
            'description' => 'nullable|string',
            'opening_balance' => 'required|numeric|min:0',
            'is_active' => 'sometimes|boolean',
        ]);

        // Use database transaction to ensure data consistency
        return DB::transaction(function () use ($validated, $request) {
            try {
                // Get the account type to determine account_type category and normal_balance
                $accountType = AccountType::with('accountGroup.primaryCategory')->find($validated['account_type_id']);
                
                // Create the account
                $account = ChartOfAccount::create([
                    'code' => $validated['code'],
                    'name' => $validated['name'],
                    'account_type_id' => $validated['account_type_id'],
                    'description' => $validated['description'] ?? null,
                    'account_type' => $accountType->accountGroup->primaryCategory->name ?? 'asset',
                    'opening_balance' => $validated['opening_balance'],
                    'current_balance' => $validated['opening_balance'],
                    'company_id' => 1,
                    'branch_id' => 1,
                    'is_active' => $validated['is_active'] ?? true,
                    'created_by' => auth()->id(),
                ]);

                // Create general ledger entry for opening balance
                if ($validated['opening_balance'] > 0) {
                    $this->createOpeningBalanceEntries(
                        $account, 
                        $validated['opening_balance'],
                        "Opening balance for new account: {$account->name}"
                    );
                }

                DB::commit();

                return redirect()->route('admin.accounting.chart-of-accounts.index')
                    ->with('success', 'Account created successfully!');

            } catch (\Exception $e) {
                DB::rollBack();
                
                return redirect()->back()
                    ->withInput()
                    ->with('error', 'Error creating account: ' . $e->getMessage());
            }
        });
    }

    public function show($id)
    {
        $account = ChartOfAccount::with([
            'accountType', 
            'accountType.accountGroup',
            'generalLedgerEntries' => function($query) {
                $query->orderBy('entrydate', 'desc')->orderBy('entryid', 'desc');
            }
        ])->findOrFail($id);
        
        return view('accounting.chart-of-accounts.show', compact('account'));
    }

    public function edit($id)
    {
        $account = ChartOfAccount::findOrFail($id);
        $accountTypes = AccountType::with('accountGroup')->get();
        
        return view('accounting.chart-of-accounts.edit', compact('account', 'accountTypes'));
    }

    public function update(Request $request, $id)
    {
        $account = ChartOfAccount::findOrFail($id);
        
        $validated = $request->validate([
            'code' => 'required|unique:chart_of_accounts,code,' . $id,
            'name' => 'required|string|max:255',
            'account_type_id' => 'required|exists:account_types,id',
            'description' => 'nullable|string',
            'is_active' => 'sometimes|boolean',
        ]);

        try {
            // Get the account type to determine account_type category
            $accountType = AccountType::with('accountGroup.primaryCategory')->find($validated['account_type_id']);

            $account->update([
                'code' => $validated['code'],
                'name' => $validated['name'],
                'account_type_id' => $validated['account_type_id'],
                'description' => $validated['description'] ?? null,
                'account_type' => $accountType->accountGroup->primaryCategory->name ?? 'asset',
                'is_active' => $validated['is_active'] ?? $account->is_active,
                'updated_by' => auth()->id(),
            ]);

            return redirect()->route('admin.accounting.chart-of-accounts.index')
                ->with('success', 'Account updated successfully!');

        } catch (\Exception $e) {
            return redirect()->back()
                ->withInput()
                ->with('error', 'Error updating account: ' . $e->getMessage());
        }
    }

    public function destroy($id)
    {
        $account = ChartOfAccount::findOrFail($id);
        
        // Check if account has children
        if ($account->children()->count() > 0) {
            return redirect()->route('admin.accounting.chart-of-accounts.index')
                ->with('error', 'Cannot delete account that has sub-accounts.');
        }

        // Check if account has general ledger entries
        if ($account->generalLedgerEntries()->count() > 0) {
            return redirect()->route('admin.accounting.chart-of-accounts.index')
                ->with('error', 'Cannot delete account that has transaction history.');
        }

        try {
            $account->delete();

            return redirect()->route('admin.accounting.chart-of-accounts.index')
                ->with('success', 'Account deleted successfully!');

        } catch (\Exception $e) {
            return redirect()->route('admin.accounting.chart-of-accounts.index')
                ->with('error', 'Error deleting account: ' . $e->getMessage());
        }
    }

    /**
     * Create opening balance entries with corresponding entry
     * based on business process mapping for "Opening Balances"
     */
    private function createOpeningBalanceEntries(ChartOfAccount $account, $amount, $description)
    {
        // Get the opening balance mapping from business process mappings
        $openingBalanceMapping = BusinessProcessMapping::where('nameofthebusinessprocess', 'Opening Balances')
            ->first();

        if (!$openingBalanceMapping) {
            throw new \Exception('Opening Balances business process mapping not found. Please configure it in Business Process Mappings.');
        }

        $correspondingAccountId = $openingBalanceMapping->accountid;
        $correspondingAccount = ChartOfAccount::find($correspondingAccountId);

        if (!$correspondingAccount) {
            throw new \Exception("Corresponding account (ID: {$correspondingAccountId}) for opening balances not found.");
        }

        // Get the account types with their normal_balance values
        $accountType = AccountType::find($account->account_type_id);
        $correspondingAccountType = AccountType::find($correspondingAccount->account_type_id);

        if (!$accountType || !$correspondingAccountType) {
            throw new \Exception("Account type information not found for one of the accounts.");
        }

        // Use the actual normal_balance field from account_types table
        $newAccountNormalBalance = strtolower($accountType->normal_balance);
        $correspondingAccountNormalBalance = strtolower($correspondingAccountType->normal_balance);
        
        $entrtypefortheaccount=null;
        $entrtypefortheopeningbalanceaccount=null;

        // Set amount values based on normal_balance (positive = debit, negative = credit)
        if ($newAccountNormalBalance === 'debit') {
            // Account with debit normal balance: Positive amount (debit)
            $newAccountAmount = $amount; // Positive = Debit
            $correspondingAccountAmount = -$amount; // Negative = Credit
            $entrtypefortheaccount='debit';
            $entrtypefortheopeningbalanceaccount='credit';
            
        } elseif ($newAccountNormalBalance === 'credit') {
            // Account with credit normal balance: Negative amount (credit)
            $newAccountAmount = -$amount; // Negative = Credit
            $correspondingAccountAmount = $amount; // Positive = Debit
            $entrtypefortheaccount='credit';
            $entrtypefortheopeningbalanceaccount='debit';
            
        } else {
            throw new \Exception("Invalid normal_balance value for account type: {$newAccountNormalBalance}");
        }

        // Create the journal entry
        $transactionId = 'OB_' . time() . '_' . uniqid();
        $currentUserId = auth()->id();
        $currentTime = now();
        $transType = 'Opening balance';

        // Entry for the new account
        GeneralLedgerEntry::create([
            'accountid' => $account->id,
            'entrydate' => $currentTime,
            'entrytype' =>$entrtypefortheaccount,
            'amount' => $newAccountAmount, // Positive for debit, negative for credit
            'description' => $description,
            'transactionid' => $transactionId,
            'transtype' => $transType,
            'companyid' => $account->company_id,
            'branchid' => $account->branch_id,
            'createdby' => $currentUserId,
            'created_at' => $currentTime,
            'updated_at' => $currentTime,
        ]);

        // Corresponding entry for the opening balance account
        GeneralLedgerEntry::create([
            'accountid' => $correspondingAccountId,
            'entrydate' => $currentTime,
            'entrytype' =>$entrtypefortheopeningbalanceaccount,
            'amount' => $correspondingAccountAmount, // Opposite of new account
            'description' => "Corresponding entry for: {$description}",
            'transactionid' => $transactionId,
            'transtype' => $transType,
            'companyid' => $account->company_id,
            'branchid' => $account->branch_id,
            'createdby' => $currentUserId,
            'created_at' => $currentTime,
            'updated_at' => $currentTime,
        ]);

        // Update current balances for both accounts
        $this->updateAccountBalance($account);
        $this->updateAccountBalance($correspondingAccount);
    }

    /**
     * Update account current balance based on general ledger entries
     */
    private function updateAccountBalance(ChartOfAccount $account)
    {
        $balance = $account->generalLedgerEntries()
            ->selectRaw('SUM(amount) as balance')
            ->value('balance') ?? 0;

        $account->update(['current_balance' => $balance]);
    }

    /**
     * Get account balance history
     */
    public function getBalanceHistory($id)
    {
        $account = ChartOfAccount::findOrFail($id);
        
        $balanceHistory = GeneralLedgerEntry::where('accountid', $id)
            ->selectRaw('DATE(entrydate) as date, SUM(amount) as daily_balance')
            ->groupBy('date')
            ->orderBy('date', 'desc')
            ->get();

        return response()->json([
            'account' => $account,
            'balance_history' => $balanceHistory
        ]);
    }

    /**
     * Toggle account status
     */
    public function toggleStatus($id)
    {
        $account = ChartOfAccount::findOrFail($id);
        
        try {
            $account->update([
                'is_active' => !$account->is_active,
                'updated_by' => auth()->id(),
            ]);

            $status = $account->is_active ? 'activated' : 'deactivated';
            
            return redirect()->route('admin.accounting.chart-of-accounts.index')
                ->with('success', "Account {$status} successfully!");

        } catch (\Exception $e) {
            return redirect()->route('admin.accounting.chart-of-accounts.index')
                ->with('error', 'Error updating account status: ' . $e->getMessage());
        }
    }

    /**
     * Bulk update accounts
     */
    public function bulkUpdate(Request $request)
    {
        $validated = $request->validate([
            'account_ids' => 'required|array',
            'account_ids.*' => 'exists:chart_of_accounts,id',
            'action' => 'required|in:activate,deactivate',
        ]);

        try {
            $isActive = $request->action === 'activate';
            
            ChartOfAccount::whereIn('id', $validated['account_ids'])
                ->update([
                    'is_active' => $isActive,
                    'updated_by' => auth()->id(),
                    'updated_at' => now(),
                ]);

            $action = $isActive ? 'activated' : 'deactivated';
            
            return redirect()->route('admin.accounting.chart-of-accounts.index')
                ->with('success', count($validated['account_ids']) . " accounts {$action} successfully!");

        } catch (\Exception $e) {
            return redirect()->route('admin.accounting.chart-of-accounts.index')
                ->with('error', 'Error performing bulk update: ' . $e->getMessage());
        }
    }
}