<?php

namespace App\Http\Controllers\Accounting;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\CurrencyRate;
use App\Models\Currency;

class CurrencyRateController extends Controller
{
    public function index()
    {
        // Using get() instead of paginate() since you don't have pagination in your view
        $currencyRates = CurrencyRate::with(['baseCurrency', 'targetCurrency'])
            ->orderBy('rate_date', 'desc')
            ->orderBy('created_at', 'desc')
            ->get();
        $currencies = Currency::where('is_active', true)->get();
        
        return view('accounting.settings.currency-rates.index', compact('currencyRates', 'currencies'));
    }

    public function create()
    {
        $currencies = Currency::where('is_active', true)->get();
        return view('accounting.settings.currency-rates.create', compact('currencies'));
    }

    public function store(Request $request)
    {
        \Log::debug('Currency Rate Store Request:', $request->all());

        $validated = $request->validate([
            'base_currency_id' => 'required|exists:currencies,id',
            'target_currency_id' => 'required|exists:currencies,id|different:base_currency_id',
            'rate_date' => 'required|date',
            'rate' => 'required|numeric|min:0.000001',
        ]);

        \Log::debug('Currency Rate Validated Data:', $validated);

        try {
            CurrencyRate::create([
                'base_currency_id' => $validated['base_currency_id'],
                'target_currency_id' => $validated['target_currency_id'],
                'rate' => $validated['rate'],
                'rate_date' => $validated['rate_date'],
                'company_id' => 1, // Default company
                'branch_id' => 1, // Default branch
                'created_by' => auth()->id(),
            ]);

            return redirect()->route('admin.accounting.settings.currency-rates.index')
                ->with('success', 'Currency rate created successfully.');

        } catch (\Exception $e) {
            \Log::error('Currency Rate Store Error: ' . $e->getMessage());
            return redirect()->back()
                ->with('error', 'Failed to create currency rate. Please try again.')
                ->withInput();
        }
    }

    public function show($id)
    {
        $currencyRate = CurrencyRate::with(['baseCurrency', 'targetCurrency', 'creator'])
            ->findOrFail($id);
        return view('accounting.settings.currency-rates.show', compact('currencyRate'));
    }

    public function edit($id)
    {
        $currencyRate = CurrencyRate::findOrFail($id);
        $currencies = Currency::where('is_active', true)->get();
        
        return view('accounting.settings.currency-rates.edit', compact('currencyRate', 'currencies'));
    }

    public function update(Request $request, $id)
    {
        $currencyRate = CurrencyRate::findOrFail($id);
        
        $validated = $request->validate([
            'base_currency_id' => 'required|exists:currencies,id',
            'target_currency_id' => 'required|exists:currencies,id|different:base_currency_id',
            'rate_date' => 'required|date',
            'rate' => 'required|numeric|min:0.000001',
        ]);

        try {
            $currencyRate->update([
                'base_currency_id' => $validated['base_currency_id'],
                'target_currency_id' => $validated['target_currency_id'],
                'rate' => $validated['rate'],
                'rate_date' => $validated['rate_date'],
            ]);

            return redirect()->route('admin.accounting.settings.currency-rates.index')
                ->with('success', 'Currency rate updated successfully.');

        } catch (\Exception $e) {
            \Log::error('Currency Rate Update Error: ' . $e->getMessage());
            return redirect()->back()
                ->with('error', 'Failed to update currency rate. Please try again.')
                ->withInput();
        }
    }

    public function destroy($id)
    {
        try {
            $currencyRate = CurrencyRate::findOrFail($id);
            $currencyRate->delete();

            return redirect()->route('admin.accounting.settings.currency-rates.index')
                ->with('success', 'Currency rate deleted successfully.');

        } catch (\Exception $e) {
            \Log::error('Currency Rate Delete Error: ' . $e->getMessage());
            return redirect()->route('admin.accounting.settings.currency-rates.index')
                ->with('error', 'Failed to delete currency rate. Please try again.');
        }
    }

    // Optional: If you want to add pagination later, use this index method instead:
    /*
    public function index()
    {
        $currencyRates = CurrencyRate::with(['baseCurrency', 'targetCurrency'])
            ->orderBy('rate_date', 'desc')
            ->orderBy('created_at', 'desc')
            ->paginate(10); // 10 records per page
        
        $currencies = Currency::where('is_active', true)->get();
        
        return view('accounting.settings.currency-rates.index', compact('currencyRates', 'currencies'));
    }
    */
}