<?php

namespace App\Http\Controllers\Accounting;

use App\Http\Controllers\Controller;
use App\Models\ExpenseTransaction;
use App\Models\ExpenseItem;
use App\Models\Vendor;
use App\Models\CashBankAccount;
use App\Models\ChartOfAccount;
use App\Models\Currency;
use App\Models\Company;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class ExpenseTransactionController extends Controller
{
    /**
     * Display a listing of the resource.
     */
   /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $companyId = auth()->user()->companyid;
        
        if (!$companyId) {
            return redirect()->back()
                ->with('error', 'User is not associated with a company. Please contact administrator.');
        }

        $query = ExpenseTransaction::with(['vendor', 'account', 'creator', 'items', 'items.chartOfAccount'])
            ->where('company_id', $companyId)
            ->latest();

        // Search functionality
        if ($request->has('search') && $request->search != '') {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('transaction_number', 'like', "%{$search}%")
                  ->orWhere('description', 'like', "%{$search}%")
                  ->orWhere('reference_number', 'like', "%{$search}%")
                  ->orWhereHas('vendor', function($q) use ($search) {
                      $q->where('display_name', 'like', "%{$search}%")
                        ->orWhere('company_name', 'like', "%{$search}%");
                  });
            });
        }

        // Filter by status
        if ($request->has('status') && $request->status != '') {
            $query->where('status', $request->status);
        }

        // Filter by vendor
        if ($request->has('vendor_id') && $request->vendor_id != '') {
            $query->where('vendor_id', $request->vendor_id);
        }

        // Filter by date range
        if ($request->has('start_date') && $request->start_date != '') {
            $query->where('transaction_date', '>=', $request->start_date);
        }
        if ($request->has('end_date') && $request->end_date != '') {
            $query->where('transaction_date', '<=', $request->end_date);
        }

        // Filter by payment method
        if ($request->has('payment_method') && $request->payment_method != '') {
            $query->where('payment_method', $request->payment_method);
        }

        // Get statistics
        $summary = [
            'total_transactions' => (clone $query)->count(),
            'total_amount' => (clone $query)->where('status', 'posted')->sum('base_amount'),
            'draft_transactions' => ExpenseTransaction::where('company_id', $companyId)
                ->where('status', 'draft')->count(),
            'posted_transactions' => ExpenseTransaction::where('company_id', $companyId)
                ->where('status', 'posted')->count(),
        ];

        // Get filter data
        $vendors = Vendor::where('company_id', $companyId)
            ->where('active', true)
            ->orderBy('display_name')
            ->get();

        $transactions = $query->paginate(25);

        if ($request->wantsJson() || $request->ajax()) {
            return response()->json([
                'transactions' => $transactions,
                'summary' => $summary,
                'filters' => $request->all(),
            ]);
        }

        return view('accounting.expenses.transactions.index', compact(
            'transactions',
            'summary',
            'vendors'
        ));
    }

    /**
     * Show the form for creating a new resource.
     */
      /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $companyId = auth()->user()->companyid;
        
        if (!$companyId) {
            return redirect()->back()
                ->with('error', 'User is not associated with a company. Please contact administrator.');
        }

        // Get company with reporting currency
        $company = Company::with('reportingCurrency')->find($companyId);
        
        if (!$company) {
            return redirect()->back()
                ->with('error', 'Company not found.');
        }

        // Get active vendors
        $vendors = Vendor::where('company_id', $companyId)
            ->where('active', true)
            ->orderBy('display_name')
            ->get();

        // Get payment accounts (Cash, Bank, Mobile Money)
        $accounts = CashBankAccount::where('companyid', $companyId)
            ->where('isActive', true)
            ->whereIn('accountType', ['Cash', 'Bank', 'Mobile Money'])
            ->orderBy('accountType')
            ->orderBy('accountName')
            ->get();

        // Get expense chart of accounts
        $chartOfAccounts = ChartOfAccount::where('company_id', $companyId)
            ->where('is_active', true)
            ->where('account_type', 'expense')
            ->orderBy('name')
            ->get();

        // Get currencies
        $currencies = Currency::where('company_id', $companyId)
            ->where('is_active', true)
            ->orderBy('code')
            ->get();

        // Get reporting currency
        $reportingCurrency = $company->reportingCurrency;

        return view('accounting.expenses.transactions.create', compact(
            'vendors',
            'accounts',
            'chartOfAccounts',
            'currencies',
            'reportingCurrency'
        ));
    }


    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $companyId = auth()->user()->companyid;
        
        if (!$companyId) {
            return redirect()->back()
                ->with('error', 'User is not associated with a company. Please contact administrator.');
        }

        $validated = $request->validate([
            'transaction_date' => 'required|date',
            'vendor_id' => 'nullable|exists:vendors,id',
            'description' => 'required|string|max:500',
            'currency' => 'required|string|max:10',
            'exchange_rate' => 'required|numeric|min:0.0001',
            'payment_method' => 'required|in:cash,check,credit_card,bank_transfer,mobile_money',
            'check_number' => 'nullable|string|max:50',
            'reference_number' => 'nullable|string|max:100',
            'account_id' => 'nullable|exists:cash_bank_accounts,cashBankId',
            'notes' => 'nullable|string|max:1000',
            'items' => 'required|array|min:1',
            'items.*.item_description' => 'required|string|max:500',
            'items.*.amount' => 'required|numeric|min:0.01',
            'items.*.chart_of_account_id' => 'required|exists:chart_of_accounts,id',
        ]);

        // Calculate total amount from items
        $totalAmount = 0;
        foreach ($validated['items'] as $itemData) {
            $totalAmount += $itemData['amount'];
        }
        
        $baseAmount = $totalAmount * $validated['exchange_rate'];
        
        // Validate that total amount is greater than 0
        if ($totalAmount <= 0) {
            return redirect()->back()
                ->withInput()
                ->with('error', 'Total amount must be greater than 0.');
        }

        // Validate account balance for cash, bank transfer, and mobile money payments
        if (in_array($validated['payment_method'], ['cash', 'bank_transfer', 'mobile_money']) && $validated['account_id']) {
            $account = CashBankAccount::find($validated['account_id']);
            if ($account && $account->current_balance < $totalAmount) {
                return redirect()->back()
                    ->withInput()
                    ->with('error', 'Insufficient balance in selected account. Available balance: ' . 
                        number_format($account->current_balance, 2) . ' ' . $account->currency);
            }
        }

        DB::transaction(function () use ($validated, $companyId, $totalAmount, $baseAmount) {
            // Create the expense transaction
            $transaction = ExpenseTransaction::create([
                'transaction_date' => $validated['transaction_date'],
                'vendor_id' => $validated['vendor_id'] ?? null,
                'description' => $validated['description'],
                'amount' => $totalAmount,
                'currency' => $validated['currency'],
                'exchange_rate' => $validated['exchange_rate'],
                'base_amount' => $baseAmount,
                'payment_method' => $validated['payment_method'],
                'check_number' => $validated['check_number'] ?? null,
                'reference_number' => $validated['reference_number'] ?? null,
                'account_id' => $validated['account_id'] ?? null,
                'notes' => $validated['notes'] ?? null,
                'status' => 'draft',
                'company_id' => $companyId,
                'branch_id' => auth()->user()->branchid ?? 1,
                'created_by' => auth()->id(),
            ]);

            // Create items
            foreach ($validated['items'] as $itemData) {
                $transaction->items()->create([
                    'item_description' => $itemData['item_description'],
                    'amount' => $itemData['amount'],
                    'chart_of_account_id' => $itemData['chart_of_account_id'],
                ]);
            }
        });

        return redirect()->route('accounting.expenses.transactions.index')
            ->with('success', 'Expense transaction created successfully.');
    }

    /**
     * Display the specified resource.
     */
         /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $companyId = auth()->user()->companyid;
        
        if (!$companyId) {
            return redirect()->back()
                ->with('error', 'User is not associated with a company. Please contact administrator.');
        }

        $transaction = ExpenseTransaction::with([
            'vendor', 
            'account', 
            'items',
            'items.chartOfAccount',
            'creator',
            'reverser',
            'company',
            'branch'
        ])
        ->where('expense_transaction_id', $id)
        ->where('company_id', $companyId)
        ->firstOrFail();

        return view('accounting.expenses.transactions.show', compact('transaction'));
    }

    /**
     * Show the form for editing the specified resource.
     */

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        $companyId = auth()->user()->companyid;
        
        if (!$companyId) {
            return redirect()->back()
                ->with('error', 'User is not associated with a company. Please contact administrator.');
        }

        $transaction = ExpenseTransaction::with(['items'])
            ->where('expense_transaction_id', $id)
            ->where('company_id', $companyId)
            ->where('status', 'draft')
            ->firstOrFail();

        // Get company with reporting currency
        $company = Company::with('reportingCurrency')->find($companyId);
        
        if (!$company) {
            return redirect()->back()
                ->with('error', 'Company not found.');
        }

        // Get active vendors
        $vendors = Vendor::where('company_id', $companyId)
            ->where('active', true)
            ->orderBy('display_name')
            ->get();

        // Get payment accounts
        $accounts = CashBankAccount::where('companyid', $companyId)
            ->where('isActive', true)
            ->whereIn('accountType', ['Cash', 'Bank', 'Mobile Money'])
            ->orderBy('accountType')
            ->orderBy('accountName')
            ->get();

        // Get chart of accounts for expenses
        $chartOfAccounts = ChartOfAccount::where('company_id', $companyId)
            ->where('is_active', true)
            ->where('account_type', 'expense')
            ->orderBy('name')
            ->get();

        // Get currencies
        $currencies = Currency::where('company_id', $companyId)
            ->where('is_active', true)
            ->orderBy('code')
            ->get();

        // Get reporting currency
        $reportingCurrency = $company->reportingCurrency;

        return view('accounting.expenses.transactions.edit', compact(
            'transaction',
            'vendors',
            'accounts',
            'chartOfAccounts',
            'currencies',
            'reportingCurrency'
        ));
    }

    /**
     * Update the specified resource in storage.
     */
         /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $companyId = auth()->user()->companyid;
        
        if (!$companyId) {
            return redirect()->back()
                ->with('error', 'User is not associated with a company. Please contact administrator.');
        }

        $transaction = ExpenseTransaction::where('expense_transaction_id', $id)
            ->where('company_id', $companyId)
            ->where('status', 'draft')
            ->firstOrFail();

        $validated = $request->validate([
            'transaction_date' => 'required|date',
            'vendor_id' => 'nullable|exists:vendors,id',
            'description' => 'required|string|max:500',
            'currency' => 'required|string|max:10',
            'exchange_rate' => 'required|numeric|min:0.0001',
            'payment_method' => 'required|in:cash,check,credit_card,bank_transfer,mobile_money',
            'check_number' => 'nullable|string|max:50',
            'reference_number' => 'nullable|string|max:100',
            'account_id' => 'nullable|exists:cash_bank_accounts,cashBankId',
            'notes' => 'nullable|string|max:1000',
            'items' => 'required|array|min:1',
            'items.*.item_description' => 'required|string|max:500',
            'items.*.amount' => 'required|numeric|min:0.01',
            'items.*.chart_of_account_id' => 'required|exists:chart_of_accounts,id',
        ]);

        DB::transaction(function () use ($transaction, $validated) {
            // Delete existing items
            $transaction->items()->delete();

            // Create new items
            $totalAmount = 0;
            foreach ($validated['items'] as $itemData) {
                $item = $transaction->items()->create([
                    'item_description' => $itemData['item_description'],
                    'amount' => $itemData['amount'],
                    'chart_of_account_id' => $itemData['chart_of_account_id'],
                ]);
                
                $totalAmount += $item->amount;
            }

            // Calculate base amount
            $baseAmount = $totalAmount * $validated['exchange_rate'];

            // Update the transaction
            $transaction->update([
                'transaction_date' => $validated['transaction_date'],
                'vendor_id' => $validated['vendor_id'] ?? null,
                'description' => $validated['description'],
                'amount' => $totalAmount,
                'currency' => $validated['currency'],
                'exchange_rate' => $validated['exchange_rate'],
                'base_amount' => $baseAmount,
                'payment_method' => $validated['payment_method'],
                'check_number' => $validated['check_number'] ?? null,
                'reference_number' => $validated['reference_number'] ?? null,
                'account_id' => $validated['account_id'] ?? null,
                'notes' => $validated['notes'] ?? null,
            ]);
        });

        return redirect()->route('admin.accounting.expenses.transactions.index')
            ->with('success', 'Expense transaction updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
         /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        $companyId = auth()->user()->companyid;
        
        if (!$companyId) {
            return redirect()->back()
                ->with('error', 'User is not associated with a company. Please contact administrator.');
        }

        $transaction = ExpenseTransaction::where('expense_transaction_id', $id)
            ->where('company_id', $companyId)
            ->where('status', 'draft')
            ->firstOrFail();

        $transaction->delete();

        return redirect()->route('admin.accounting.expenses.transactions.index')
            ->with('success', 'Expense transaction deleted successfully.');
    }

    /**
     * Post an expense transaction
     */
    public function post(string $id)
    {
        $companyId = auth()->user()->companyid;
        
        if (!$companyId) {
            return response()->json([
                'success' => false,
                'message' => 'User is not associated with a company.'
            ], 400);
        }

        $transaction = ExpenseTransaction::where('expense_transaction_id', $id)
            ->where('company_id', $companyId)
            ->firstOrFail();

        if ($transaction->post()) {
            return response()->json([
                'success' => true,
                'message' => 'Transaction posted successfully.'
            ]);
        }

        return response()->json([
            'success' => false,
            'message' => 'Failed to post transaction.'
        ], 500);
    }

    /**
     * Reverse an expense transaction
     */
    public function reverse(Request $request, string $id)
    {
        $companyId = auth()->user()->companyid;
        
        if (!$companyId) {
            return response()->json([
                'success' => false,
                'message' => 'User is not associated with a company.'
            ], 400);
        }

        $validated = $request->validate([
            'reversal_reason' => 'required|string|max:1000'
        ]);

        $transaction = ExpenseTransaction::where('expense_transaction_id', $id)
            ->where('company_id', $companyId)
            ->firstOrFail();

        if ($transaction->reverse($validated['reversal_reason'])) {
            return response()->json([
                'success' => true,
                'message' => 'Transaction reversed successfully.'
            ]);
        }

        return response()->json([
            'success' => false,
            'message' => 'Failed to reverse transaction.'
        ], 500);
    }

    /**
     * Export expense transactions
     */
    public function export(Request $request)
    {
        $companyId = auth()->user()->companyid;
        
        if (!$companyId) {
            return redirect()->back()
                ->with('error', 'User is not associated with a company. Please contact administrator.');
        }

        $query = ExpenseTransaction::with(['vendor', 'account', 'creator'])
            ->where('company_id', $companyId);

        // Apply filters
        if ($request->has('start_date') && $request->start_date != '') {
            $query->where('transaction_date', '>=', $request->start_date);
        }
        if ($request->has('end_date') && $request->end_date != '') {
            $query->where('transaction_date', '<=', $request->end_date);
        }
        if ($request->has('status') && $request->status != '') {
            $query->where('status', $request->status);
        }
        if ($request->has('vendor_id') && $request->vendor_id != '') {
            $query->where('vendor_id', $request->vendor_id);
        }

        $transactions = $query->orderBy('transaction_date', 'desc')->get();

        $fileName = 'expense-transactions-' . date('Y-m-d') . '.csv';
        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => "attachment; filename=\"{$fileName}\"",
        ];

        $callback = function() use ($transactions) {
            $file = fopen('php://output', 'w');
            
            // Add BOM for UTF-8
            fwrite($file, "\xEF\xBB\xBF");
            
            // Headers
            fputcsv($file, [
                'Transaction Number',
                'Transaction Date',
                'Vendor',
                'Description',
                'Total Amount',
                'Currency',
                'Base Amount',
                'Exchange Rate',
                'Payment Method',
                'Check Number',
                'Reference Number',
                'Account',
                'Status',
                'Created By',
                'Created Date',
                'Posted Date',
                'Reversed Date',
                'Reversal Reason',
                'Notes'
            ]);

            // Data
            foreach ($transactions as $item) {
                fputcsv($file, [
                    $item->transaction_number,
                    $item->transaction_date->format('Y-m-d'),
                    $item->vendor->display_name ?? 'N/A',
                    $item->description,
                    $item->amount,
                    $item->currency,
                    $item->base_amount,
                    $item->exchange_rate,
                    ucfirst(str_replace('_', ' ', $item->payment_method)),
                    $item->check_number ?? 'N/A',
                    $item->reference_number ?? 'N/A',
                    $item->account->accountName ?? 'N/A',
                    ucfirst($item->status),
                    $item->creator->name ?? 'N/A',
                    $item->created_at->format('Y-m-d H:i:s'),
                    $item->posted_at ? $item->posted_at->format('Y-m-d H:i:s') : 'N/A',
                    $item->reversed_at ? $item->reversed_at->format('Y-m-d H:i:s') : 'N/A',
                    $item->reversal_reason ?? 'N/A',
                    $item->notes ?? 'N/A'
                ]);
            }

            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }

    /**
     * Get expense statistics
     */
    public function statistics(Request $request)
    {
        $companyId = auth()->user()->companyid;
        
        if (!$companyId) {
            return response()->json([
                'success' => false,
                'message' => 'User is not associated with a company.'
            ], 400);
        }

        $query = ExpenseTransaction::where('company_id', $companyId)
            ->where('status', 'posted');

        if ($request->has('start_date')) {
            $query->where('transaction_date', '>=', $request->start_date);
        }
        if ($request->has('end_date')) {
            $query->where('transaction_date', '<=', $request->end_date);
        }

        $stats = [
            'total_expenses' => $query->sum('base_amount'),
            'total_transactions' => $query->count(),
            'average_expense' => $query->avg('base_amount'),
            'by_vendor' => $query->select('vendor_id', DB::raw('SUM(base_amount) as total'))
                ->groupBy('vendor_id')
                ->with('vendor')
                ->orderBy('total', 'desc')
                ->limit(10)
                ->get()
                ->map(function($item) {
                    return [
                        'vendor' => $item->vendor->display_name ?? 'Unknown Vendor',
                        'total' => $item->total,
                        'percentage' => 0 // Will be calculated below
                    ];
                }),
            'by_payment_method' => $query->select('payment_method', DB::raw('COUNT(*) as count'))
                ->groupBy('payment_method')
                ->get()
                ->mapWithKeys(function($item) {
                    return [ucfirst(str_replace('_', ' ', $item->payment_method)) => $item->count];
                }),
        ];

        // Calculate percentages
        $totalAmount = $stats['total_expenses'];
        if ($totalAmount > 0) {
            foreach ($stats['by_vendor'] as &$vendor) {
                $vendor['percentage'] = round(($vendor['total'] / $totalAmount) * 100, 2);
            }
        }

        return response()->json([
            'success' => true,
            'statistics' => $stats,
            'period' => [
                'start_date' => $request->start_date ?? date('Y-m-01'),
                'end_date' => $request->end_date ?? date('Y-m-d')
            ]
        ]);
    }
}