<?php

namespace App\Http\Controllers\Accounting;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\JournalEntry;
use App\Models\JournalEntryItem;
use App\Models\ChartOfAccount;
use App\Models\Currency;
use App\Models\FinancialYear;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class JournalEntryController extends Controller
{
    /**
     * Display a listing of journal entries.
     */
    public function index()
    {
        $status = request('status', 'all');
        $search = request('search', '');
        $date_from = request('date_from');
        $date_to = request('date_to');
        
        $query = JournalEntry::with(['currency', 'creator', 'poster', 'items.account'])
            ->orderBy('entry_date', 'desc')
            ->orderBy('created_at', 'desc');

        // Filter by status
        if ($status !== 'all') {
            $query->where('status', $status);
        }

        // Search functionality
        if (!empty($search)) {
            $query->where(function($q) use ($search) {
                $q->where('entry_number', 'like', "%{$search}%")
                  ->orWhere('description', 'like', "%{$search}%")
                  ->orWhere('reference', 'like', "%{$search}%")
                  ->orWhereHas('creator', function($q) use ($search) {
                      $q->where('name', 'like', "%{$search}%");
                  });
            });
        }

        // Date range filter
        if (!empty($date_from)) {
            $query->where('entry_date', '>=', $date_from);
        }
        if (!empty($date_to)) {
            $query->where('entry_date', '<=', $date_to);
        }

        $journalEntries = $query->paginate(20);
        
        return view('accounting.journals.index', compact('journalEntries', 'status', 'search', 'date_from', 'date_to'));
    }

    /**
     * Show the form for creating a new journal entry.
     */
    public function create()
    {
        $accounts = ChartOfAccount::where('is_active', true)
            ->with('accountType')
            ->orderBy('code')
            ->get();
            
        $currencies = Currency::where('is_active', true)->get();
        $financialYears = FinancialYear::where('is_active', true)->get();
        
        // Generate next journal number
        $lastJournal = JournalEntry::where('entry_number', 'like', 'JE-%')
            ->orderBy('id', 'desc')
            ->first();
            
        $nextNumber = 'JE-' . date('Ymd') . '-' . str_pad(
            ($lastJournal ? (int) substr($lastJournal->entry_number, -3) + 1 : 1), 
            3, '0', STR_PAD_LEFT
        );
        
        return view('accounting.journals.create', compact(
            'accounts', 
            'currencies', 
            'financialYears',
            'nextNumber'
        ));
    }

    /**
     * Store a newly created journal entry.
     */
    public function store(Request $request)
    {
        $validated = $this->validateJournalEntry($request);

        DB::beginTransaction();
        
        try {
            $journalEntry = JournalEntry::create([
                'company_id' => 1, // Default company ID
                'branch_id' => 1, // Default branch ID
                'entry_number' => $validated['entry_number'],
                'entry_date' => $validated['entry_date'],
                'financial_year_id' => $validated['financial_year_id'],
                'reference' => $validated['reference'],
                'description' => $validated['description'],
                'currency_id' => $validated['currency_id'],
                'exchange_rate' => $validated['exchange_rate'],
                'total_debit' => $validated['total_debit'],
                'total_credit' => $validated['total_credit'],
                'status' => $validated['status'],
                'created_by' => auth()->id(),
            ]);

            foreach ($validated['items'] as $item) {
                JournalEntryItem::create([
                    'company_id' => 1, // Default company ID
                    'branch_id' => 1, // Default branch ID
                    'journal_entry_id' => $journalEntry->id,
                    'account_id' => $item['account_id'],
                    'debit' => $item['debit'] ?? 0,
                    'credit' => $item['credit'] ?? 0,
                    'description' => $item['description'],
                    'created_by' => auth()->id(),
                ]);
            }

            // If status is posted, update account balances
            if ($validated['status'] === 'posted') {
                $this->postJournalEntry($journalEntry);
            }

            DB::commit();

            $message = 'Journal entry created successfully.' . 
                      ($validated['status'] === 'posted' ? ' Entry has been posted.' : ' Entry saved as draft.');

            return redirect()->route('admin.accounting.journals.show', $journalEntry->id)
                ->with('success', $message);

        } catch (\Exception $e) {
            DB::rollBack();
            
            logger()->error('Journal entry creation failed: ' . $e->getMessage(), [
                'user_id' => auth()->id(),
                'entry_number' => $validated['entry_number'] ?? 'unknown',
                'trace' => $e->getTraceAsString()
            ]);

            return back()->withErrors(['error' => 'Failed to create journal entry: ' . $e->getMessage()])->withInput();
        }
    }

    /**
     * Display the specified journal entry.
     */
    public function show($id)
    {
        $journalEntry = JournalEntry::with([
            'currency', 
            'creator', 
            'poster', 
            'updater',
            'items.account.accountType',
            'financialYear'
        ])->findOrFail($id);
        
        // Calculate if the entry is balanced
        $isBalanced = abs($journalEntry->total_debit - $journalEntry->total_credit) < 0.01;
        
        return view('accounting.journals.show', compact('journalEntry', 'isBalanced'));
    }

    /**
     * Show the form for editing the specified journal entry.
     */
    public function edit($id)
    {
        $journalEntry = JournalEntry::with(['items.account'])->findOrFail($id);
        
        // Only allow editing of draft entries
        if ($journalEntry->status !== 'draft') {
            return redirect()->route('admin.accounting.journals.show', $id)
                ->with('error', 'Only draft journal entries can be edited.');
        }
        
        $accounts = ChartOfAccount::where('is_active', true)
            ->with('accountType')
            ->orderBy('code')
            ->get();
            
        $currencies = Currency::where('is_active', true)->get();
        $financialYears = FinancialYear::where('is_active', true)->get();
        
        return view('accounting.journals.edit', compact(
            'journalEntry', 
            'accounts', 
            'currencies', 
            'financialYears'
        ));
    }

    /**
     * Update the specified journal entry.
     */
    public function update(Request $request, $id)
    {
        $journalEntry = JournalEntry::with(['items'])->findOrFail($id);
        
        // Only allow updating of draft entries
        if ($journalEntry->status !== 'draft') {
            return redirect()->route('admin.accounting.journals.show', $id)
                ->with('error', 'Only draft journal entries can be updated.');
        }
        
        $validated = $this->validateJournalEntry($request, $journalEntry->id);

        DB::beginTransaction();
        
        try {
            $journalEntry->update([
                'entry_number' => $validated['entry_number'],
                'entry_date' => $validated['entry_date'],
                'financial_year_id' => $validated['financial_year_id'],
                'reference' => $validated['reference'],
                'description' => $validated['description'],
                'currency_id' => $validated['currency_id'],
                'exchange_rate' => $validated['exchange_rate'],
                'total_debit' => $validated['total_debit'],
                'total_credit' => $validated['total_credit'],
                'status' => $validated['status'],
                'updated_by' => auth()->id(),
            ]);

            // Delete existing items and create new ones
            $journalEntry->items()->delete();
            
            foreach ($validated['items'] as $item) {
                JournalEntryItem::create([
                    'company_id' => 1, // Default company ID
                    'branch_id' => 1, // Default branch ID
                    'journal_entry_id' => $journalEntry->id,
                    'account_id' => $item['account_id'],
                    'debit' => $item['debit'] ?? 0,
                    'credit' => $item['credit'] ?? 0,
                    'description' => $item['description'],
                    'created_by' => auth()->id(),
                ]);
            }

            // If status is posted, update account balances
            if ($validated['status'] === 'posted') {
                $this->postJournalEntry($journalEntry);
            }

            DB::commit();

            $message = 'Journal entry updated successfully.' . 
                      ($validated['status'] === 'posted' ? ' Entry has been posted.' : '');

            return redirect()->route('admin.accounting.journals.show', $journalEntry->id)
                ->with('success', $message);

        } catch (\Exception $e) {
            DB::rollBack();
            
            logger()->error('Journal entry update failed: ' . $e->getMessage(), [
                'user_id' => auth()->id(),
                'journal_entry_id' => $id,
                'trace' => $e->getTraceAsString()
            ]);

            return back()->withErrors(['error' => 'Failed to update journal entry: ' . $e->getMessage()])->withInput();
        }
    }

    /**
     * Remove the specified journal entry.
     */
    public function destroy($id)
    {
        $journalEntry = JournalEntry::findOrFail($id);
        
        // Only allow deletion of draft entries
        if ($journalEntry->status !== 'draft') {
            return redirect()->route('admin.accounting.journals.show', $id)
                ->with('error', 'Only draft journal entries can be deleted.');
        }

        DB::beginTransaction();
        
        try {
            $journalEntry->items()->delete();
            $journalEntry->delete();
            
            DB::commit();

            return redirect()->route('admin.accounting.journals.index')
                ->with('success', 'Journal entry deleted successfully.');

        } catch (\Exception $e) {
            DB::rollBack();
            
            logger()->error('Journal entry deletion failed: ' . $e->getMessage(), [
                'user_id' => auth()->id(),
                'journal_entry_id' => $id
            ]);

            return back()->withErrors(['error' => 'Failed to delete journal entry: ' . $e->getMessage()]);
        }
    }

    /**
     * Post a journal entry (update account balances)
     */
    public function post($id)
    {
        $journalEntry = JournalEntry::with(['items.account'])->findOrFail($id);
        
        // Only allow posting of draft entries
        if ($journalEntry->status !== 'draft') {
            return redirect()->route('admin.accounting.journals.show', $id)
                ->with('error', 'Only draft journal entries can be posted.');
        }

        // Validate that the entry is balanced
        if (!$this->isJournalBalanced($journalEntry)) {
            return redirect()->route('admin.accounting.journals.show', $id)
                ->with('error', 'Cannot post an unbalanced journal entry. Total debits must equal total credits.');
        }

        DB::beginTransaction();
        
        try {
            $this->postJournalEntry($journalEntry);
            
            DB::commit();

            return redirect()->route('admin.accounting.journals.show', $journalEntry->id)
                ->with('success', 'Journal entry posted successfully.');

        } catch (\Exception $e) {
            DB::rollBack();
            
            logger()->error('Journal entry posting failed: ' . $e->getMessage(), [
                'user_id' => auth()->id(),
                'journal_entry_id' => $id,
                'trace' => $e->getTraceAsString()
            ]);

            return back()->withErrors(['error' => 'Failed to post journal entry: ' . $e->getMessage()]);
        }
    }

    /**
     * Cancel a posted journal entry (reverse the transaction)
     */
    public function cancel($id)
    {
        $journalEntry = JournalEntry::with(['items.account'])->findOrFail($id);
        
        // Only allow cancellation of posted entries
        if ($journalEntry->status !== 'posted') {
            return redirect()->route('admin.accounting.journals.show', $id)
                ->with('error', 'Only posted journal entries can be cancelled.');
        }

        DB::beginTransaction();
        
        try {
            $this->reverseJournalEntry($journalEntry);
            
            DB::commit();

            return redirect()->route('admin.accounting.journals.show', $journalEntry->id)
                ->with('success', 'Journal entry cancelled successfully.');

        } catch (\Exception $e) {
            DB::rollBack();
            
            logger()->error('Journal entry cancellation failed: ' . $e->getMessage(), [
                'user_id' => auth()->id(),
                'journal_entry_id' => $id,
                'trace' => $e->getTraceAsString()
            ]);

            return back()->withErrors(['error' => 'Failed to cancel journal entry: ' . $e->getMessage()]);
        }
    }

    /**
     * Get journal entries for approval
     */
    public function approvalIndex()
    {
        $search = request('search', '');
        $date_from = request('date_from');
        $date_to = request('date_to');
        
        $query = JournalEntry::with(['currency', 'creator', 'items.account'])
            ->where('status', 'draft')
            ->orderBy('entry_date', 'desc')
            ->orderBy('created_at', 'desc');

        // Search functionality
        if (!empty($search)) {
            $query->where(function($q) use ($search) {
                $q->where('entry_number', 'like', "%{$search}%")
                  ->orWhere('description', 'like', "%{$search}%")
                  ->orWhere('reference', 'like', "%{$search}%")
                  ->orWhereHas('creator', function($q) use ($search) {
                      $q->where('name', 'like', "%{$search}%");
                  });
            });
        }

        // Date range filter
        if (!empty($date_from)) {
            $query->where('entry_date', '>=', $date_from);
        }
        if (!empty($date_to)) {
            $query->where('entry_date', '<=', $date_to);
        }

        $journalEntries = $query->paginate(20);
        
        return view('accounting.journals.approval-index', compact('journalEntries', 'search', 'date_from', 'date_to'));
    }

    /**
     * Bulk approve journal entries
     */
    public function bulkApprove(Request $request)
    {
        $validated = $request->validate([
            'journal_entries' => 'required|array|min:1',
            'journal_entries.*' => 'exists:journal_entries,id',
        ]);

        $approvedCount = 0;
        $errors = [];
        
        foreach ($validated['journal_entries'] as $journalId) {
            DB::beginTransaction();
            
            try {
                $journalEntry = JournalEntry::with(['items.account'])->find($journalId);
                
                if ($journalEntry && $journalEntry->status === 'draft') {
                    
                    // Validate that the entry is balanced
                    if (!$this->isJournalBalanced($journalEntry)) {
                        $errors[] = "Journal {$journalEntry->entry_number} is not balanced and cannot be posted.";
                        DB::rollBack();
                        continue;
                    }
                    
                    $this->postJournalEntry($journalEntry);
                    
                    DB::commit();
                    $approvedCount++;
                }
            } catch (\Exception $e) {
                DB::rollBack();
                $errors[] = "Failed to approve journal {$journalEntry->entry_number}: " . $e->getMessage();
                
                logger()->error('Bulk approval failed for journal: ' . $e->getMessage(), [
                    'user_id' => auth()->id(),
                    'journal_entry_id' => $journalId
                ]);
            }
        }

        $response = redirect()->route('admin.accounting.journals.approval-index');
        
        if ($approvedCount > 0) {
            $response->with('success', $approvedCount . ' journal entries approved and posted successfully.');
        }
        
        if (!empty($errors)) {
            $response->with('errors', $errors);
        }

        return $response;
    }

    /**
     * Show only draft journal entries
     */
    public function draft()
    {
        return $this->indexWithStatus('draft');
    }

    /**
     * Show only posted journal entries
     */
    public function posted()
    {
        return $this->indexWithStatus('posted');
    }

    /**
     * Show only cancelled journal entries
     */
    public function cancelled()
    {
        return $this->indexWithStatus('cancelled');
    }

    /**
     * Validate journal entry number uniqueness
     */
    public function validateEntryNumber(Request $request)
    {
        $request->validate([
            'entry_number' => 'required|string',
            'exclude_id' => 'nullable|exists:journal_entries,id'
        ]);

        $query = JournalEntry::where('entry_number', $request->entry_number);
        
        if ($request->exclude_id) {
            $query->where('id', '!=', $request->exclude_id);
        }

        $exists = $query->exists();

        return response()->json([
            'valid' => !$exists,
            'message' => $exists ? 'Journal entry number already exists.' : 'Journal entry number is available.'
        ]);
    }

    /**
     * Common validation for journal entries
     */
    private function validateJournalEntry(Request $request, $excludeId = null)
    {
        $rules = [
            'entry_number' => 'required|unique:journal_entries,entry_number,' . $excludeId,
            'entry_date' => 'required|date',
            'financial_year_id' => 'required|exists:financial_years,id',
            'reference' => 'nullable|string|max:255',
            'description' => 'required|string|max:1000',
            'currency_id' => 'required|exists:currencies,id',
            'exchange_rate' => 'required|numeric|min:0.000001',
            'status' => 'required|in:draft,posted',
            'items' => 'required|array|min:2',
            'items.*.account_id' => 'required|exists:chart_of_accounts,id',
            'items.*.debit' => 'required_without:items.*.credit|numeric|min:0',
            'items.*.credit' => 'required_without:items.*.debit|numeric|min:0',
            'items.*.description' => 'nullable|string|max:255',
        ];

        $validated = $request->validate($rules);

        // Additional validation for line items
        $totalDebit = 0;
        $totalCredit = 0;

        foreach ($validated['items'] as $index => $item) {
            // Validate that each line has either debit or credit, not both
            if ((empty($item['debit']) || $item['debit'] == 0) && (empty($item['credit']) || $item['credit'] == 0)) {
                throw \Illuminate\Validation\ValidationException::withMessages([
                    'items' => "Line " . ($index + 1) . " must have either debit or credit amount."
                ]);
            }
            
            if (!empty($item['debit']) && $item['debit'] > 0 && !empty($item['credit']) && $item['credit'] > 0) {
                throw \Illuminate\Validation\ValidationException::withMessages([
                    'items' => "Line " . ($index + 1) . " cannot have both debit and credit amounts."
                ]);
            }

            $totalDebit += $item['debit'] ?? 0;
            $totalCredit += $item['credit'] ?? 0;
        }

        // Validate debit and credit totals match
        if (abs($totalDebit - $totalCredit) > 0.01) {
            throw \Illuminate\Validation\ValidationException::withMessages([
                'items' => 'Total debits must equal total credits. Difference: ' . number_format(abs($totalDebit - $totalCredit), 2)
            ]);
        }

        $validated['total_debit'] = $totalDebit;
        $validated['total_credit'] = $totalCredit;

        return $validated;
    }

    /**
     * Post journal entry and update account balances
     */
    private function postJournalEntry(JournalEntry $journalEntry)
    {
        // Update account balances for each item
        foreach ($journalEntry->items as $item) {
            $account = $item->account;
            
            if ($item->debit > 0) {
                // For debit entries, increase asset/expense accounts, decrease liability/equity/revenue accounts
                if (in_array($account->account_type, ['asset', 'expense'])) {
                    $account->current_balance += $item->debit;
                } else {
                    $account->current_balance -= $item->debit;
                }
            }
            
            if ($item->credit > 0) {
                // For credit entries, decrease asset/expense accounts, increase liability/equity/revenue accounts
                if (in_array($account->account_type, ['asset', 'expense'])) {
                    $account->current_balance -= $item->credit;
                } else {
                    $account->current_balance += $item->credit;
                }
            }
            
            $account->save();
        }

        // Update journal entry status
        $journalEntry->update([
            'status' => 'posted',
            'posted_by' => auth()->id(),
            'posted_at' => now(),
        ]);
    }

    /**
     * Reverse a posted journal entry (cancel)
     */
    private function reverseJournalEntry(JournalEntry $journalEntry)
    {
        // Reverse account balances for each item
        foreach ($journalEntry->items as $item) {
            $account = $item->account;
            
            if ($item->debit > 0) {
                // Reverse debit entries
                if (in_array($account->account_type, ['asset', 'expense'])) {
                    $account->current_balance -= $item->debit;
                } else {
                    $account->current_balance += $item->debit;
                }
            }
            
            if ($item->credit > 0) {
                // Reverse credit entries
                if (in_array($account->account_type, ['asset', 'expense'])) {
                    $account->current_balance += $item->credit;
                } else {
                    $account->current_balance -= $item->credit;
                }
            }
            
            $account->save();
        }

        // Update journal entry status
        $journalEntry->update([
            'status' => 'cancelled',
            'updated_by' => auth()->id(),
        ]);
    }

    /**
     * Check if journal entry is balanced
     */
    private function isJournalBalanced(JournalEntry $journalEntry)
    {
        return abs($journalEntry->total_debit - $journalEntry->total_credit) < 0.01;
    }

    /**
     * Common method for status-based index views
     */
    private function indexWithStatus($status)
    {
        request()->merge(['status' => $status]);
        return $this->index();
    }
}