<?php

namespace App\Http\Controllers\Accounting;

use App\Http\Controllers\Controller;
use App\Models\Company;
use App\Models\Employee;
use App\Models\LeaveType;
use App\Models\LeaveRequest;
use App\Models\LeaveBalance;
use App\Models\User;
use App\Models\LeaveAccrualTransaction;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class LeaveController extends Controller
{
    // Dashboard/Index
    public function index()
    {
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        
        // Get leave requests statistics
        $totalRequests = LeaveRequest::where('company_id', $company->id)->count();
        $pendingRequests = LeaveRequest::where('company_id', $company->id)
            ->where('status', 'pending')
            ->count();
        $approvedRequests = LeaveRequest::where('company_id', $company->id)
            ->where('status', 'approved')
            ->count();
        $takenLeaves = LeaveRequest::where('company_id', $company->id)
            ->where('status', 'approved')
            ->where('start_date', '<=', today())
            ->sum('total_days');
        
        // Get recent leave requests
        $recentRequests = LeaveRequest::where('company_id', $company->id)
            ->with(['employee', 'leaveType', 'approver'])
            ->orderBy('created_at', 'desc')
            ->take(10)
            ->get();
        
        // Get leave types
        $leaveTypes = LeaveType::where('company_id', $company->id)
            ->where('is_active', true)
            ->get();
        
        return view('accounting.payroll.leave.index', compact(
            'company',
            'totalRequests',
            'pendingRequests',
            'approvedRequests',
            'takenLeaves',
            'recentRequests',
            'leaveTypes'
        ));
    }
    
    // Leave Requests
    public function requestsIndex(Request $request)
    {
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        
        $query = LeaveRequest::where('company_id', $company->id)
            ->with(['employee', 'leaveType', 'approver']);
        
        // Apply filters
        if ($request->has('status') && $request->status != 'all') {
            $query->where('status', $request->status);
        }
        
        if ($request->has('leave_type_id') && $request->leave_type_id != 'all') {
            $query->where('leave_type_id', $request->leave_type_id);
        }
        
        if ($request->has('employee_id') && $request->employee_id != 'all') {
            $query->where('employee_id', $request->employee_id);
        }
        
        if ($request->has('date_from') && $request->date_from) {
            $query->whereDate('start_date', '>=', $request->date_from);
        }
        
        if ($request->has('date_to') && $request->date_to) {
            $query->whereDate('end_date', '<=', $request->date_to);
        }
        
        $leaveRequests = $query->orderBy('created_at', 'desc')
            ->paginate(20);
        
        $employees = Employee::where('company_id', $company->id)
            ->where('is_active', true)
            ->orderBy('first_name')
            ->get();
        
        $leaveTypes = LeaveType::where('company_id', $company->id)
            ->where('is_active', true)
            ->get();
        
        return view('accounting.payroll.leave.requests.index', compact(
            'company',
            'leaveRequests',
            'employees',
            'leaveTypes'
        ));
    }
    
    public function requestsCreate()
    {
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        
        $employees = Employee::where('company_id', $company->id)
            ->where('is_active', true)
            ->orderBy('first_name')
            ->get();
        
        $leaveTypes = LeaveType::where('company_id', $company->id)
            ->where('is_active', true)
            ->get();
        
        return view('accounting.payroll.leave.requests.create', compact(
            'company',
            'employees',
            'leaveTypes'
        ));
    }
    
    public function requestsStore(Request $request)
    {
        $request->validate([
            'employee_id' => 'required|exists:employees,id',
            'leave_type_id' => 'required|exists:leave_types,id',
            'start_date' => 'required|date',
            'end_date' => 'required|date|after_or_equal:start_date',
            'reason' => 'required|string|max:500',
            'requested_by' => 'nullable|exists:users,id'
        ]);
        
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        
        // Check if employee belongs to company
        $employee = Employee::where('company_id', $company->id)
            ->findOrFail($request->employee_id);
        
        // Check if leave type belongs to company
        $leaveType = LeaveType::where('company_id', $company->id)
            ->findOrFail($request->leave_type_id);
        
        // Calculate total days
        $startDate = Carbon::parse($request->start_date);
        $endDate = Carbon::parse($request->end_date);
        $totalDays = $startDate->diffInDays($endDate) + 1; // Inclusive
        
        // Check leave balance
        $leaveBalance = LeaveBalance::where('company_id', $company->id)
            ->where('employee_id', $employee->id)
            ->where('leave_type_id', $leaveType->id)
            ->where('year', date('Y'))
            ->first();
        
        $availableBalance = $leaveBalance ? $leaveBalance->available_balance : 0;
        
        if ($totalDays > $availableBalance && !$leaveType->can_carry_forward) {
            return redirect()->back()
                ->with('error', "Insufficient leave balance. Available: {$availableBalance} days, Requested: {$totalDays} days.")
                ->withInput();
        }
        
        DB::beginTransaction();
        
        try {
            $leaveRequest = LeaveRequest::create([
                'company_id' => $company->id,
                'employee_id' => $employee->id,
                'leave_type_id' => $leaveType->id,
                'start_date' => $request->start_date,
                'end_date' => $request->end_date,
                'total_days' => $totalDays,
                'reason' => $request->reason,
                'status' => $leaveType->requires_approval ? 'pending' : 'approved',
                'requested_by' => $request->requested_by ?? auth()->id()
            ]);
            
            // If no approval required, auto-approve
            if (!$leaveType->requires_approval) {
                $leaveRequest->approve(auth()->user());
                
                // Deduct from leave balance
                if ($leaveBalance) {
                    $leaveBalance->addTaken($totalDays);
                }
            }
            
            DB::commit();
            
            return redirect()->route('admin.accounting.payroll.leave.requests.index')
                ->with('success', 'Leave request submitted successfully.');
                
        } catch (\Exception $e) {
            DB::rollBack();
            
            return redirect()->back()
                ->with('error', 'Failed to submit leave request: ' . $e->getMessage())
                ->withInput();
        }
    }
    
    public function requestsShow($id)
    {
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        
        $leaveRequest = LeaveRequest::where('company_id', $company->id)
            ->with(['employee', 'leaveType', 'approver', 'rejector', 'requester'])
            ->findOrFail($id);
        
        return view('accounting.payroll.leave.requests.show', compact(
            'company',
            'leaveRequest'
        ));
    }
    
    public function requestsApprove(Request $request, $id)
    {
        $request->validate([
            'approval_notes' => 'nullable|string|max:500'
        ]);
        
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        
        $leaveRequest = LeaveRequest::where('company_id', $company->id)
            ->findOrFail($id);
        
        if (!$leaveRequest->canApprove()) {
            return redirect()->back()
                ->with('error', 'This leave request cannot be approved.');
        }
        
        // Check leave balance before approving
        $leaveBalance = LeaveBalance::where('company_id', $company->id)
            ->where('employee_id', $leaveRequest->employee_id)
            ->where('leave_type_id', $leaveRequest->leave_type_id)
            ->where('year', date('Y'))
            ->first();
        
        $availableBalance = $leaveBalance ? $leaveBalance->available_balance : 0;
        
        if ($leaveRequest->total_days > $availableBalance) {
            return redirect()->back()
                ->with('error', "Cannot approve. Insufficient leave balance. Available: {$availableBalance} days, Requested: {$leaveRequest->total_days} days.");
        }
        
        DB::beginTransaction();
        
        try {
            $leaveRequest->approve(auth()->user(), $request->approval_notes);
            
            // Deduct from leave balance
            if ($leaveBalance) {
                $leaveBalance->addTaken($leaveRequest->total_days);
            }
            
            DB::commit();
            
            return redirect()->route('admin.accounting.payroll.leave.requests.show', $leaveRequest->id)
                ->with('success', 'Leave request approved successfully.');
                
        } catch (\Exception $e) {
            DB::rollBack();
            
            return redirect()->back()
                ->with('error', 'Failed to approve leave request: ' . $e->getMessage());
        }
    }
    
    public function requestsReject(Request $request, $id)
    {
        $request->validate([
            'rejection_reason' => 'required|string|max:500'
        ]);
        
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        
        $leaveRequest = LeaveRequest::where('company_id', $company->id)
            ->findOrFail($id);
        
        if ($leaveRequest->status !== 'pending') {
            return redirect()->back()
                ->with('error', 'Only pending leave requests can be rejected.');
        }
        
        DB::beginTransaction();
        
        try {
            $leaveRequest->reject(auth()->user(), $request->rejection_reason);
            
            DB::commit();
            
            return redirect()->route('admin.accounting.payroll.leave.requests.show', $leaveRequest->id)
                ->with('success', 'Leave request rejected successfully.');
                
        } catch (\Exception $e) {
            DB::rollBack();
            
            return redirect()->back()
                ->with('error', 'Failed to reject leave request: ' . $e->getMessage());
        }
    }
    
    public function requestsCancel($id)
    {
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        
        $leaveRequest = LeaveRequest::where('company_id', $company->id)
            ->findOrFail($id);
        
        if (!$leaveRequest->canCancel()) {
            return redirect()->back()
                ->with('error', 'This leave request cannot be cancelled.');
        }
        
        DB::beginTransaction();
        
        try {
            // If it was approved, add back the leave balance
            if ($leaveRequest->status === 'approved') {
                $leaveBalance = LeaveBalance::where('company_id', $company->id)
                    ->where('employee_id', $leaveRequest->employee_id)
                    ->where('leave_type_id', $leaveRequest->leave_type_id)
                    ->where('year', date('Y'))
                    ->first();
                
                if ($leaveBalance) {
                    $leaveBalance->balance += $leaveRequest->total_days;
                    $leaveBalance->total_taken -= $leaveRequest->total_days;
                    $leaveBalance->save();
                }
            }
            
            $leaveRequest->cancel(auth()->user());
            
            DB::commit();
            
            return redirect()->route('admin.accounting.payroll.leave.requests.show', $leaveRequest->id)
                ->with('success', 'Leave request cancelled successfully.');
                
        } catch (\Exception $e) {
            DB::rollBack();
            
            return redirect()->back()
                ->with('error', 'Failed to cancel leave request: ' . $e->getMessage());
        }
    }
    
    // Leave Pending Review
    public function pendingReview()
    {
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        
        $pendingRequests = LeaveRequest::where('company_id', $company->id)
            ->where('status', 'pending')
            ->with(['employee', 'leaveType'])
            ->orderBy('created_at', 'desc')
            ->paginate(20);
        
        return view('accounting.payroll.leave.pending-review', compact(
            'company',
            'pendingRequests'
        ));
    }
    
    // Approved Leave
    public function approvedLeave()
    {
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        
        $approvedRequests = LeaveRequest::where('company_id', $company->id)
            ->where('status', 'approved')
            ->with(['employee', 'leaveType', 'approver'])
            ->orderBy('approved_at', 'desc')
            ->paginate(20);
        
        return view('accounting.payroll.leave.approved-leave', compact(
            'company',
            'approvedRequests'
        ));
    }
    
    // Taken Leaves
    public function takenLeaves()
    {
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        
        $takenLeaves = LeaveRequest::where('company_id', $company->id)
            ->where('status', 'approved')
            ->where('start_date', '<=', today()) // Leaves that have started
            ->with(['employee', 'leaveType', 'approver'])
            ->orderBy('start_date', 'desc')
            ->paginate(20);
        
        // Calculate statistics
        $totalTakenDays = LeaveRequest::where('company_id', $company->id)
            ->where('status', 'approved')
            ->where('start_date', '<=', today())
            ->sum('total_days');
        
        $currentMonthTaken = LeaveRequest::where('company_id', $company->id)
            ->where('status', 'approved')
            ->whereYear('start_date', date('Y'))
            ->whereMonth('start_date', date('m'))
            ->sum('total_days');
        
        return view('accounting.payroll.leave.taken-leaves', compact(
            'company',
            'takenLeaves',
            'totalTakenDays',
            'currentMonthTaken'
        ));
    }
    
    // Leave Settings (Leave Types)
    public function settingsIndex()
    {
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        
        $leaveTypes = LeaveType::where('company_id', $company->id)
            ->orderBy('name')
            ->paginate(20);
        
        return view('accounting.payroll.leave.settings.index', compact(
            'company',
            'leaveTypes'
        ));
    }
    
    public function settingsCreate()
    {
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        
        return view('accounting.payroll.leave.settings.create', compact('company'));
    }
    
    public function settingsStore(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'code' => 'required|string|max:50|unique:leave_types,code',
            'description' => 'nullable|string',
            'max_days_per_year' => 'nullable|integer|min:0',
            'accrual_rate_days' => 'nullable|numeric|min:0',
            'is_paid' => 'required|boolean',
            'requires_approval' => 'required|boolean',
            'can_carry_forward' => 'required|boolean',
            'max_carry_forward_days' => 'nullable|integer|min:0',
        ]);
        
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        
        try {
            LeaveType::create([
                'company_id' => $company->id,
                'name' => $request->name,
                'code' => strtoupper($request->code),
                'description' => $request->description,
                'max_days_per_year' => $request->max_days_per_year,
                'accrual_rate_days' => $request->accrual_rate_days,
                'is_paid' => $request->is_paid,
                'requires_approval' => $request->requires_approval,
                'can_carry_forward' => $request->can_carry_forward,
                'max_carry_forward_days' => $request->max_carry_forward_days,
                'is_active' => true
            ]);
            
            return redirect()->route('admin.accounting.payroll.leave.settings.index')
                ->with('success', 'Leave type created successfully.');
                
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to create leave type: ' . $e->getMessage())
                ->withInput();
        }
    }
    
    public function settingsEdit($id)
    {
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        
        $leaveType = LeaveType::where('company_id', $company->id)
            ->findOrFail($id);
        
        return view('accounting.payroll.leave.settings.edit', compact(
            'company',
            'leaveType'
        ));
    }
    
    public function settingsUpdate(Request $request, $id)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'code' => 'required|string|max:50|unique:leave_types,code,' . $id,
            'description' => 'nullable|string',
            'max_days_per_year' => 'nullable|integer|min:0',
            'accrual_rate_days' => 'nullable|numeric|min:0',
            'is_paid' => 'required|boolean',
            'requires_approval' => 'required|boolean',
            'can_carry_forward' => 'required|boolean',
            'max_carry_forward_days' => 'nullable|integer|min:0',
            'is_active' => 'required|boolean'
        ]);
        
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        
        $leaveType = LeaveType::where('company_id', $company->id)
            ->findOrFail($id);
        
        try {
            $leaveType->update([
                'name' => $request->name,
                'code' => strtoupper($request->code),
                'description' => $request->description,
                'max_days_per_year' => $request->max_days_per_year,
                'accrual_rate_days' => $request->accrual_rate_days,
                'is_paid' => $request->is_paid,
                'requires_approval' => $request->requires_approval,
                'can_carry_forward' => $request->can_carry_forward,
                'max_carry_forward_days' => $request->max_carry_forward_days,
                'is_active' => $request->is_active
            ]);
            
            return redirect()->route('admin.accounting.payroll.leave.settings.index')
                ->with('success', 'Leave type updated successfully.');
                
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to update leave type: ' . $e->getMessage())
                ->withInput();
        }
    }
    
    public function settingsDestroy($id)
    {
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        
        $leaveType = LeaveType::where('company_id', $company->id)
            ->findOrFail($id);
        
        // Check if leave type has associated requests
        if ($leaveType->leaveRequests()->count() > 0) {
            return redirect()->back()
                ->with('error', 'Cannot delete leave type that has associated leave requests.');
        }
        
        try {
            $leaveType->delete();
            
            return redirect()->route('admin.accounting.payroll.leave.settings.index')
                ->with('success', 'Leave type deleted successfully.');
                
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to delete leave type: ' . $e->getMessage());
        }
    }
    
    // Leave Balance Management
    public function balancesIndex()
    {
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        
        $employees = Employee::where('company_id', $company->id)
            ->where('is_active', true)
            ->with(['leaveBalances.leaveType'])
            ->orderBy('first_name')
            ->paginate(20);
        
        $leaveTypes = LeaveType::where('company_id', $company->id)
            ->where('is_active', true)
            ->get();
        
        // Calculate total balances
        $totalBalances = [];
        foreach ($leaveTypes as $leaveType) {
            $totalBalances[$leaveType->id] = LeaveBalance::where('company_id', $company->id)
                ->where('leave_type_id', $leaveType->id)
                ->where('year', date('Y'))
                ->sum('balance');
        }
        
        return view('accounting.payroll.leave.balances.index', compact(
            'company',
            'employees',
            'leaveTypes',
            'totalBalances'
        ));
    }
    
    public function balancesAdjust(Request $request)
    {
        $request->validate([
            'employee_id' => 'required|exists:employees,id',
            'leave_type_id' => 'required|exists:leave_types,id',
            'adjustment_type' => 'required|in:add_accrual,add_taken,adjust_balance',
            'days' => 'required|numeric',
            'notes' => 'nullable|string|max:500'
        ]);
        
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        
        // Check if employee belongs to company
        $employee = Employee::where('company_id', $company->id)
            ->findOrFail($request->employee_id);
        
        // Check if leave type belongs to company
        $leaveType = LeaveType::where('company_id', $company->id)
            ->findOrFail($request->leave_type_id);
        
        DB::beginTransaction();
        
        try {
            // Get or create leave balance
            $leaveBalance = LeaveBalance::firstOrCreate(
                [
                    'company_id' => $company->id,
                    'employee_id' => $employee->id,
                    'leave_type_id' => $leaveType->id,
                    'year' => date('Y')
                ],
                [
                    'total_accrued' => 0,
                    'total_taken' => 0,
                    'balance' => 0,
                    'carry_forward_balance' => 0
                ]
            );
            
            // Apply adjustment based on type
            switch ($request->adjustment_type) {
                case 'add_accrual':
                    $leaveBalance->addAccrual($request->days);
                    $transactionType = 'accrual';
                    break;
                    
                case 'add_taken':
                    $leaveBalance->addTaken($request->days);
                    $transactionType = 'taken';
                    break;
                    
                case 'adjust_balance':
                    $leaveBalance->balance += $request->days;
                    $leaveBalance->save();
                    $transactionType = 'adjustment';
                    break;
            }
            
            // Create leave accrual transaction
            LeaveAccrualTransaction::create([
                'employee_id' => $employee->id,
                'company_id' => $company->id,
                'period_date' => today(),
                'accrued_days' => $request->adjustment_type === 'add_accrual' ? $request->days : 0,
                'taken_days' => $request->adjustment_type === 'add_taken' ? $request->days : 0,
                'balance_days' => $leaveBalance->balance,
                'created_by' => auth()->id(),
                'notes' => $request->notes
            ]);
            
            DB::commit();
            
            return redirect()->route('admin.accounting.payroll.leave.balances.index')
                ->with('success', 'Leave balance adjusted successfully.');
                
        } catch (\Exception $e) {
            DB::rollBack();
            
            return redirect()->back()
                ->with('error', 'Failed to adjust leave balance: ' . $e->getMessage())
                ->withInput();
        }
    }
    
    public function processAccrual(Request $request)
    {
        $request->validate([
            'accrue_for_all' => 'required|boolean',
            'employee_ids' => 'required_if:accrue_for_all,false|array',
            'employee_ids.*' => 'exists:employees,id',
            'leave_type_id' => 'required|exists:leave_types,id',
            'accrual_date' => 'required|date'
        ]);
        
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        
        // Check if leave type belongs to company
        $leaveType = LeaveType::where('company_id', $company->id)
            ->findOrFail($request->leave_type_id);
        
        if (!$leaveType->accrual_rate_days || $leaveType->accrual_rate_days <= 0) {
            return redirect()->back()
                ->with('error', 'This leave type does not have an accrual rate configured.');
        }
        
        // Get employees to process
        if ($request->accrue_for_all) {
            $employees = Employee::where('company_id', $company->id)
                ->where('is_active', true)
                ->where('employment_type', 'permanent') // Typically only permanent employees get leave accrual
                ->get();
        } else {
            $employees = Employee::where('company_id', $company->id)
                ->whereIn('id', $request->employee_ids)
                ->get();
        }
        
        DB::beginTransaction();
        
        try {
            $processedCount = 0;
            $totalAccruedDays = 0;
            
            foreach ($employees as $employee) {
                // Get or create leave balance
                $leaveBalance = LeaveBalance::firstOrCreate(
                    [
                        'company_id' => $company->id,
                        'employee_id' => $employee->id,
                        'leave_type_id' => $leaveType->id,
                        'year' => date('Y', strtotime($request->accrual_date))
                    ],
                    [
                        'total_accrued' => 0,
                        'total_taken' => 0,
                        'balance' => 0,
                        'carry_forward_balance' => 0
                    ]
                );
                
                // Check if accrual already exists for this period
                $existingAccrual = LeaveAccrualTransaction::where('employee_id', $employee->id)
                    ->where('leave_type_id', $leaveType->id)
                    ->where('period_date', $request->accrual_date)
                    ->where('accrued_days', '>', 0)
                    ->exists();
                
                if (!$existingAccrual) {
                    // Add accrual
                    $leaveBalance->addAccrual($leaveType->accrual_rate_days);
                    
                    // Create accrual transaction
                    LeaveAccrualTransaction::create([
                        'employee_id' => $employee->id,
                        'company_id' => $company->id,
                        'period_date' => $request->accrual_date,
                        'accrued_days' => $leaveType->accrual_rate_days,
                        'balance_days' => $leaveBalance->balance,
                        'created_by' => auth()->id(),
                        'notes' => 'Monthly leave accrual'
                    ]);
                    
                    $processedCount++;
                    $totalAccruedDays += $leaveType->accrual_rate_days;
                }
            }
            
            DB::commit();
            
            return redirect()->route('admin.accounting.payroll.leave.balances.index')
                ->with('success', 'Leave accrual processed for ' . $processedCount . ' employees. Total ' . $totalAccruedDays . ' days accrued.');
                
        } catch (\Exception $e) {
            DB::rollBack();
            
            return redirect()->back()
                ->with('error', 'Failed to process leave accrual: ' . $e->getMessage());
        }
    }
    
    // Reports
    public function reportsIndex()
    {
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        
        return view('accounting.payroll.leave.reports.index', compact('company'));
    }
    
    public function reportsLeaveUtilization()
    {
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        
        // Get leave utilization by department/employee
        $utilization = DB::table('leave_requests')
            ->select(
                'employees.id',
                'employees.first_name',
                'employees.last_name',
                'leave_types.name as leave_type',
                DB::raw('SUM(leave_requests.total_days) as total_days_taken'),
                DB::raw('COUNT(leave_requests.id) as total_requests')
            )
            ->join('employees', 'leave_requests.employee_id', '=', 'employees.id')
            ->join('leave_types', 'leave_requests.leave_type_id', '=', 'leave_types.id')
            ->where('leave_requests.company_id', $company->id)
            ->where('leave_requests.status', 'approved')
            ->whereYear('leave_requests.start_date', date('Y'))
            ->groupBy('employees.id', 'leave_types.id')
            ->orderBy('employees.first_name')
            ->get();
        
        // Get leave balance summary
        $balanceSummary = DB::table('leave_balances')
            ->select(
                'leave_types.name as leave_type',
                DB::raw('SUM(leave_balances.total_accrued) as total_accrued'),
                DB::raw('SUM(leave_balances.total_taken) as total_taken'),
                DB::raw('SUM(leave_balances.balance) as total_balance')
            )
            ->join('leave_types', 'leave_balances.leave_type_id', '=', 'leave_types.id')
            ->where('leave_balances.company_id', $company->id)
            ->where('leave_balances.year', date('Y'))
            ->groupBy('leave_types.id')
            ->get();
        
        return view('accounting.payroll.leave.reports.leave-utilization', compact(
            'company',
            'utilization',
            'balanceSummary'
        ));
    }
    
    public function reportsLeaveCalendar()
    {
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        
        $leaveRequests = LeaveRequest::where('company_id', $company->id)
            ->where('status', 'approved')
            ->with(['employee', 'leaveType'])
            ->get();
        
        // Format for calendar
        $calendarEvents = [];
        foreach ($leaveRequests as $request) {
            $calendarEvents[] = [
                'title' => $request->employee->full_name . ' - ' . $request->leaveType->name,
                'start' => $request->start_date,
                'end' => Carbon::parse($request->end_date)->addDay()->format('Y-m-d'), // Add one day for fullCalendar
                'color' => $this->getLeaveTypeColor($request->leaveType->id),
                'extendedProps' => [
                    'employee' => $request->employee->full_name,
                    'leave_type' => $request->leaveType->name,
                    'duration' => $request->total_days . ' days',
                    'status' => $request->status
                ]
            ];
        }
        
        return view('accounting.payroll.leave.reports.leave-calendar', compact(
            'company',
            'calendarEvents'
        ));
    }
    
    private function getLeaveTypeColor($leaveTypeId)
    {
        $colors = [
            1 => '#3498db', // Blue
            2 => '#2ecc71', // Green
            3 => '#e74c3c', // Red
            4 => '#f39c12', // Orange
            5 => '#9b59b6', // Purple
            6 => '#1abc9c', // Teal
            7 => '#34495e', // Dark Blue
            8 => '#d35400', // Dark Orange
        ];
        
        return $colors[$leaveTypeId % count($colors)] ?? '#3498db';
    }
    
    // API Methods
    public function getLeaveBalance($employeeId, $leaveTypeId)
    {
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        
        $leaveBalance = LeaveBalance::where('company_id', $company->id)
            ->where('employee_id', $employeeId)
            ->where('leave_type_id', $leaveTypeId)
            ->where('year', date('Y'))
            ->first();
        
        return response()->json([
            'success' => true,
            'balance' => $leaveBalance ? $leaveBalance->available_balance : 0,
            'total_accrued' => $leaveBalance ? $leaveBalance->total_accrued : 0,
            'total_taken' => $leaveBalance ? $leaveBalance->total_taken : 0
        ]);
    }
    
    public function getEmployeeLeaveRequests($employeeId)
    {
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        
        $leaveRequests = LeaveRequest::where('company_id', $company->id)
            ->where('employee_id', $employeeId)
            ->with(['leaveType', 'approver'])
            ->orderBy('created_at', 'desc')
            ->get();
        
        return response()->json([
            'success' => true,
            'requests' => $leaveRequests
        ]);
    }
    
    // Dashboard Statistics
    public function getLeaveStats()
    {
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        
        $stats = [
            'total_requests' => LeaveRequest::where('company_id', $company->id)->count(),
            'pending_requests' => LeaveRequest::where('company_id', $company->id)
                ->where('status', 'pending')
                ->count(),
            'approved_this_month' => LeaveRequest::where('company_id', $company->id)
                ->where('status', 'approved')
                ->whereMonth('approved_at', date('m'))
                ->whereYear('approved_at', date('Y'))
                ->count(),
            'taken_this_month' => LeaveRequest::where('company_id', $company->id)
                ->where('status', 'approved')
                ->where('start_date', '<=', today())
                ->whereMonth('start_date', date('m'))
                ->sum('total_days')
        ];
        
        return response()->json([
            'success' => true,
            'stats' => $stats
        ]);
    }
    
    public static function getPendingCount()
   {
    if (!auth()->check()) {
        return 0;
    }
    
    $company = Company::where('id', auth()->user()->companyid)->first();
    
    if (!$company) {
        return 0;
    }
    
    return LeaveRequest::where('company_id', $company->id)
        ->where('status', 'pending')
        ->count();
   }
}