<?php

namespace App\Http\Controllers\Accounting;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Accounting\PaymentVoucher;
use App\Models\Accounting\Supplier;
use App\Models\Accounting\ChartOfAccount;

class PaymentVoucherController extends Controller
{
    public function index()
    {
        $vouchers = PaymentVoucher::with(['supplier'])->orderBy('voucher_date', 'desc')->get();
        $suppliers = Supplier::where('is_active', true)->get();
        $accounts = ChartOfAccount::where('is_active', true)->get();
        
        return view('accounting.payables.payment-vouchers.index', compact('vouchers', 'suppliers', 'accounts'));
    }

    public function create()
    {
        $suppliers = Supplier::where('is_active', true)->get();
        $accounts = ChartOfAccount::where('is_active', true)->get();
        
        return view('accounting.payables.payment-vouchers.create', compact('suppliers', 'accounts'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'voucher_number' => 'required|unique:payment_vouchers,voucher_number',
            'voucher_date' => 'required|date',
            'supplier_id' => 'required|exists:suppliers,id',
            'amount' => 'required|numeric|min:0',
            'payment_method' => 'required|in:cash,bank_transfer,cheque',
            'description' => 'nullable',
            'status' => 'required|in:draft,approved,paid,cancelled',
            'account_id' => 'required|exists:chart_of_accounts,id',
        ]);

        PaymentVoucher::create($validated);

        return redirect()->route('admin.accounting.payables.payment-vouchers.index')
            ->with('success', 'Payment voucher created successfully.');
    }

    public function show($id)
    {
        $voucher = PaymentVoucher::with(['supplier', 'account'])->findOrFail($id);
        return view('accounting.payables.payment-vouchers.show', compact('voucher'));
    }

    public function edit($id)
    {
        $voucher = PaymentVoucher::findOrFail($id);
        $suppliers = Supplier::where('is_active', true)->get();
        $accounts = ChartOfAccount::where('is_active', true)->get();
        
        return view('accounting.payables.payment-vouchers.edit', compact('voucher', 'suppliers', 'accounts'));
    }

    public function update(Request $request, $id)
    {
        $voucher = PaymentVoucher::findOrFail($id);
        
        $validated = $request->validate([
            'voucher_number' => 'required|unique:payment_vouchers,voucher_number,' . $id,
            'voucher_date' => 'required|date',
            'supplier_id' => 'required|exists:suppliers,id',
            'amount' => 'required|numeric|min:0',
            'payment_method' => 'required|in:cash,bank_transfer,cheque',
            'description' => 'nullable',
            'status' => 'required|in:draft,approved,paid,cancelled',
            'account_id' => 'required|exists:chart_of_accounts,id',
        ]);

        $voucher->update($validated);

        return redirect()->route('admin.accounting.payables.payment-vouchers.index')
            ->with('success', 'Payment voucher updated successfully.');
    }

    public function destroy($id)
    {
        $voucher = PaymentVoucher::findOrFail($id);
        $voucher->delete();

        return redirect()->route('admin.accounting.payables.payment-vouchers.index')
            ->with('success', 'Payment voucher deleted successfully.');
    }
}