<?php

namespace App\Http\Controllers\Accounting;

use App\Http\Controllers\Controller;
use App\Models\PettyCash;
use App\Models\CashBankAccount;
use App\Models\Currency;
use App\Models\Company;
use App\Models\ChartOfAccount;
use App\Models\PettyCashItem;
use App\Models\GeneralLedgerEntry;
use App\Models\CashBook;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class PettyCashController extends Controller
{
    // REMOVED THE CONSTRUCTOR - this was causing the error
    
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        // Get the user's company ID
        $companyId = auth()->user()->companyid;
        
        if (!$companyId) {
            return redirect()->back()
                ->with('error', 'User is not associated with a company. Please contact administrator.');
        }

        $query = PettyCash::with(['sourceAccount', 'receiptingAccount', 'chartOfAccount', 'requester', 'items'])
            ->where('company_id', $companyId)
            ->latest();

        // Search functionality
        if ($request->has('search') && $request->search != '') {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('petty_cash_number', 'like', "%{$search}%")
                  ->orWhere('payee', 'like', "%{$search}%")
                  ->orWhere('description', 'like', "%{$search}%")
                  ->orWhere('receipt_number', 'like', "%{$search}%");
            });
        }

        // Filter by status
        if ($request->has('status') && $request->status != '') {
            $query->where('status', $request->status);
        }

        // Filter by date range
        if ($request->has('start_date') && $request->start_date != '') {
            $query->where('transaction_date', '>=', $request->start_date);
        }
        if ($request->has('end_date') && $request->end_date != '') {
            $query->where('transaction_date', '<=', $request->end_date);
        }

        $pettyCash = $query->paginate(25);

        return view('accounting.cash-banking.petty-cash.index', compact('pettyCash'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $companyId = auth()->user()->companyid;
        
        if (!$companyId) {
            return redirect()->back()
                ->with('error', 'User is not associated with a company. Please contact administrator.');
        }

        // Get company with reporting currency
        $company = Company::with('reportingCurrency')->find($companyId);
        
        if (!$company) {
            return redirect()->back()
                ->with('error', 'Company not found.');
        }

        // Get ALL cash/bank accounts for source (Cash, Bank, Mobile Money)
        $sourceAccounts = CashBankAccount::where('companyid', $companyId)
            ->where('isActive', true)
            ->whereIn('accountType', ['Cash', 'Bank', 'Mobile Money'])
            ->orderBy('accountType')
            ->orderBy('accountName')
            ->get();

        // Get expense-related chart of accounts
        $chartOfAccounts = ChartOfAccount::where('company_id', $companyId)
            ->where('is_active', true)
            ->orderBy('name')
            ->get();

        // Get currencies from Currency model
        $currencies = Currency::where('company_id', $companyId)
            ->where('is_active', true)
            ->orderBy('code')
            ->get();

        // Get reporting currency
        $reportingCurrency = $company->reportingCurrency;

        return view('accounting.cash-banking.petty-cash.create', compact(
            'sourceAccounts', 
            'chartOfAccounts',
            'currencies',
            'reportingCurrency'
        ));
    }
    
    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $companyId = auth()->user()->companyid;
        
        if (!$companyId) {
            return redirect()->back()
                ->with('error', 'User is not associated with a company. Please contact administrator.');
        }

        $validated = $request->validate([
            'transaction_date' => 'required|date',
            'payee' => 'required|string|max:255',
            'description' => 'nullable|string|max:1000',
            'currency' => 'required|string|max:10',
            'exchange_rate' => 'required|numeric|min:0.0001',
            'source_account_id' => 'required|exists:cash_bank_accounts,cashBankId',
            'receipting_account_id' => 'required|exists:cash_bank_accounts,cashBankId',
            'chart_of_account_id' => 'nullable|exists:chart_of_accounts,id',
            'receipt_number' => 'nullable|string|max:100',
            'items' => 'required|array|min:1',
            'items.*.item_description' => 'required|string|max:500',
            'items.*.amount' => 'required|numeric|min:0.01',
            'items.*.chart_of_account_id' => 'nullable|exists:chart_of_accounts,id',
        ]);

        // Validate that source and receipting accounts are Cash accounts
        $sourceAccount = CashBankAccount::find($validated['source_account_id']);
        $receiptingAccount = CashBankAccount::find($validated['receipting_account_id']);
        
        if (!$sourceAccount) {
            return redirect()->back()
                ->withInput()
                ->with('error', 'Source account not found.');
        }
        
        if (!$receiptingAccount) {
            return redirect()->back()
                ->withInput()
                ->with('error', 'Receipting account not found.');
        }
        
        if ($sourceAccount->accountType !== 'Cash') {
            return redirect()->back()
                ->withInput()
                ->with('error', 'Source account must be a Cash account.');
        }
        
        if ($receiptingAccount->accountType !== 'Cash') {
            return redirect()->back()
                ->withInput()
                ->with('error', 'Receipting account must be a Cash account.');
        }

        // Calculate total amount from items BEFORE creating the record
        $totalAmount = 0;
        foreach ($validated['items'] as $itemData) {
            $totalAmount += $itemData['amount'];
        }
        
        $baseAmount = $totalAmount * $validated['exchange_rate'];
        
        // Validate that total amount is greater than 0
        if ($totalAmount <= 0) {
            return redirect()->back()
                ->withInput()
                ->with('error', 'Total amount must be greater than 0.');
        }

        // Validate account balance
        if ($sourceAccount->current_balance < $totalAmount) {
            return redirect()->back()
                ->withInput()
                ->with('error', 'Insufficient balance in source account. Available balance: ' . 
                    number_format($sourceAccount->current_balance, 2) . ' ' . $sourceAccount->currency);
        }

        DB::transaction(function () use ($validated, $companyId, $totalAmount, $baseAmount) {
            // Create the petty cash header WITH the calculated amounts
            $pettyCash = PettyCash::create([
                'transaction_date' => $validated['transaction_date'],
                'payee' => $validated['payee'],
                'description' => $validated['description'] ?? null,
                'amount' => $totalAmount,
                'currency' => $validated['currency'],
                'exchange_rate' => $validated['exchange_rate'],
                'base_amount' => $baseAmount,
                'source_account_id' => $validated['source_account_id'],
                'receipting_account_id' => $validated['receipting_account_id'],
                'chart_of_account_id' => $validated['chart_of_account_id'] ?? null,
                'receipt_number' => $validated['receipt_number'] ?? null,
                'status' => 'Draft',
                'requested_by' => auth()->id(),
                'company_id' => $companyId,
                'branch_id' => auth()->user()->branchid ?? 1,
                'created_by' => auth()->id(),
            ]);

            // Create items
            foreach ($validated['items'] as $itemData) {
                $pettyCash->items()->create([
                    'item_description' => $itemData['item_description'],
                    'amount' => $itemData['amount'],
                    'chart_of_account_id' => $itemData['chart_of_account_id'] ?? $validated['chart_of_account_id'] ?? null,
                ]);
            }
        });

        return redirect()->route('admin.accounting.cash-banking.petty-cash.index')
            ->with('success', 'Petty cash request created successfully.');
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $companyId = auth()->user()->companyid;
        
        if (!$companyId) {
            return redirect()->back()
                ->with('error', 'User is not associated with a company. Please contact administrator.');
        }

        $pettyCash = PettyCash::with([
            'items', 
            'items.chartOfAccount',
            'sourceAccount', 
            'receiptingAccount',
            'chartOfAccount', 
            'requester', 
            'approver',
            'creator',
            'updater',
            'reimbursedBy',
            'reversedBy',
            'company',
            'branch'
        ])
        ->where('petty_cash_id', $id)
        ->where('company_id', $companyId)
        ->firstOrFail();

        // Poweruser can view everything - no restrictions needed
        return view('accounting.cash-banking.petty-cash.show', compact('pettyCash'));
    }
    
    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        $companyId = auth()->user()->companyid;
        
        if (!$companyId) {
            return redirect()->back()
                ->with('error', 'User is not associated with a company. Please contact administrator.');
        }

        $pettyCash = PettyCash::with(['items', 'items.chartOfAccount'])
            ->where('petty_cash_id', $id)
            ->where('company_id', $companyId)
            ->whereIn('status', ['Draft', 'Rejected'])
            ->firstOrFail();

        // Get company with reporting currency
        $company = Company::with('reportingCurrency')->find($companyId);
        
        if (!$company) {
            return redirect()->back()
                ->with('error', 'Company not found.');
        }

        // Get ALL cash/bank accounts for source (Cash, Bank, Mobile Money)
        $sourceAccounts = CashBankAccount::where('companyid', $companyId)
            ->where('isActive', true)
            ->whereIn('accountType', ['Cash', 'Bank', 'Mobile Money'])
            ->orderBy('accountType')
            ->orderBy('accountName')
            ->get();

        // Get expense-related chart of accounts
        $chartOfAccounts = ChartOfAccount::where('company_id', $companyId)
            ->where('is_active', true)
            ->orderBy('name')
            ->get();

        // Get currencies from Currency model
        $currencies = Currency::where('company_id', $companyId)
            ->where('is_active', true)
            ->orderBy('code')
            ->get();

        // Get reporting currency
        $reportingCurrency = $company->reportingCurrency;

        return view('accounting.cash-banking.petty-cash.edit', compact(
            'pettyCash',
            'sourceAccounts', 
            'chartOfAccounts',
            'currencies',
            'reportingCurrency'
        ));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $companyId = auth()->user()->companyid;
        
        if (!$companyId) {
            return redirect()->back()
                ->with('error', 'User is not associated with a company. Please contact administrator.');
        }

        $pettyCash = PettyCash::where('petty_cash_id', $id)
            ->where('company_id', $companyId)
            ->whereIn('status', ['Draft', 'Rejected'])
            ->firstOrFail();

        $validated = $request->validate([
            'transaction_date' => 'required|date',
            'payee' => 'required|string|max:255',
            'description' => 'nullable|string|max:1000',
            'currency' => 'required|string|max:10',
            'exchange_rate' => 'required|numeric|min:0.0001',
            'source_account_id' => 'required|exists:cash_bank_accounts,cashBankId',
            'receipting_account_id' => 'required|exists:cash_bank_accounts,cashBankId',
            'chart_of_account_id' => 'nullable|exists:chart_of_accounts,id',
            'receipt_number' => 'nullable|string|max:100',
            'items' => 'required|array|min:1',
            'items.*.item_description' => 'required|string|max:500',
            'items.*.amount' => 'required|numeric|min:0.01',
            'items.*.chart_of_account_id' => 'nullable|exists:chart_of_accounts,id',
        ]);

        // Validate that source and receipting accounts are Cash accounts
        $sourceAccount = CashBankAccount::find($validated['source_account_id']);
        $receiptingAccount = CashBankAccount::find($validated['receipting_account_id']);
        
        if ($sourceAccount->accountType !== 'Cash') {
            return redirect()->back()
                ->withInput()
                ->with('error', 'Source account must be a Cash account.');
        }
        
        if ($receiptingAccount->accountType !== 'Cash') {
            return redirect()->back()
                ->withInput()
                ->with('error', 'Receipting account must be a Cash account.');
        }

        DB::transaction(function () use ($pettyCash, $validated) {
            // Update the petty cash header
            $pettyCash->update([
                'transaction_date' => $validated['transaction_date'],
                'payee' => $validated['payee'],
                'description' => $validated['description'] ?? null,
                'currency' => $validated['currency'],
                'exchange_rate' => $validated['exchange_rate'],
                'source_account_id' => $validated['source_account_id'],
                'receipting_account_id' => $validated['receipting_account_id'],
                'chart_of_account_id' => $validated['chart_of_account_id'] ?? null,
                'receipt_number' => $validated['receipt_number'] ?? null,
                'updated_by' => auth()->id(),
            ]);

            // Delete existing items
            $pettyCash->items()->delete();

            // Create new items
            $totalAmount = 0;
            foreach ($validated['items'] as $itemData) {
                $item = $pettyCash->items()->create([
                    'item_description' => $itemData['item_description'],
                    'amount' => $itemData['amount'],
                    'chart_of_account_id' => $itemData['chart_of_account_id'] ?? $validated['chart_of_account_id'] ?? null,
                ]);
                
                $totalAmount += $item->amount;
            }

            // Update the header with calculated totals
            $pettyCash->update([
                'amount' => $totalAmount,
                'base_amount' => $totalAmount * $validated['exchange_rate'],
            ]);
        });

        return redirect()->route('admin.accounting.cash-banking.petty-cash.index')
            ->with('success', 'Petty cash request updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        $companyId = auth()->user()->companyid;
        
        if (!$companyId) {
            return redirect()->back()
                ->with('error', 'User is not associated with a company. Please contact administrator.');
        }

        $pettyCash = PettyCash::where('petty_cash_id', $id)
            ->where('company_id', $companyId)
            ->whereIn('status', ['Draft', 'Rejected'])
            ->firstOrFail();

        // Items will be automatically deleted due to cascade foreign key
        $pettyCash->delete();

        return redirect()->route('admin.accounting.cash-banking.petty-cash.index')
            ->with('success', 'Petty cash request deleted successfully.');
    }

    /**
     * Submit petty cash for approval
     */
    public function submit(string $id)
    {
        $companyId = auth()->user()->companyid;
        
        if (!$companyId) {
            return redirect()->back()
                ->with('error', 'User is not associated with a company. Please contact administrator.');
        }

        $pettyCash = PettyCash::where('petty_cash_id', $id)
            ->where('company_id', $companyId)
            ->where('status', 'Draft')
            ->firstOrFail();

        $pettyCash->update([
            'status' => 'Submitted',
            'updated_by' => auth()->id()
        ]);

        return redirect()->back()->with('success', 'Petty cash submitted for approval.');
    }

    /**
     * Approve petty cash
     */
    public function approve(Request $request, string $id)
    {
        $companyId = auth()->user()->companyid;
        
        if (!$companyId) {
            return redirect()->back()
                ->with('error', 'User is not associated with a company. Please contact administrator.');
        }

        $pettyCash = PettyCash::where('petty_cash_id', $id)
            ->where('company_id', $companyId)
            ->where('status', 'Submitted')
            ->firstOrFail();

        $validated = $request->validate([
            'approval_notes' => 'nullable|string|max:1000'
        ]);

        $pettyCash->update([
            'status' => 'Approved',
            'approved_by' => auth()->id(),
            'approved_at' => now(),
            'approval_notes' => $validated['approval_notes'] ?? null,
            'updated_by' => auth()->id()
        ]);

        return redirect()->back()->with('success', 'Petty cash approved successfully.');
    }

    /**
     * Reject petty cash
     */
    public function reject(Request $request, string $id)
    {
        $companyId = auth()->user()->companyid;
        
        if (!$companyId) {
            return redirect()->back()
                ->with('error', 'User is not associated with a company. Please contact administrator.');
        }

        $pettyCash = PettyCash::where('petty_cash_id', $id)
            ->where('company_id', $companyId)
            ->where('status', 'Submitted')
            ->firstOrFail();

        $validated = $request->validate([
            'rejection_reason' => 'required|string|max:1000'
        ]);

        $pettyCash->update([
            'status' => 'Rejected',
            'rejection_reason' => $validated['rejection_reason'],
            'updated_by' => auth()->id()
        ]);

        return redirect()->back()->with('success', 'Petty cash rejected.');
    }

    /**
     * Mark petty cash as paid
     */
    public function markAsPaid(string $id)
    {
        $companyId = auth()->user()->companyid;
        
        if (!$companyId) {
            return redirect()->back()
                ->with('error', 'User is not associated with a company. Please contact administrator.');
        }

        $pettyCash = PettyCash::where('petty_cash_id', $id)
            ->where('company_id', $companyId)
            ->where('status', 'Approved')
            ->firstOrFail();

        DB::transaction(function () use ($pettyCash) {
            $pettyCash->update([
                'status' => 'Paid',
                'updated_by' => auth()->id()
            ]);

            // Update source account balance (decrease)
            $sourceAccount = $pettyCash->sourceAccount;
            $sourceAccount->updateBalance($pettyCash->amount, 'credit');

            // Update receipting account balance (increase)
            $receiptingAccount = $pettyCash->receiptingAccount;
            $receiptingAccount->updateBalance($pettyCash->amount, 'debit');

            // Create journal entry for the payment
            $this->createPaymentJournalEntry($pettyCash);
            
            // Record transaction in cash book
            $this->recordPaymentInCashBook($pettyCash);
        });

        return redirect()->back()->with('success', 'Petty cash marked as paid.');
    }

    /**
     * Reimburse petty cash
     */
    public function reimburse(string $id)
    {
        $companyId = auth()->user()->companyid;
        
        if (!$companyId) {
            return redirect()->back()
                ->with('error', 'User is not associated with a company. Please contact administrator.');
        }

        $pettyCash = PettyCash::where('petty_cash_id', $id)
            ->where('company_id', $companyId)
            ->where('status', 'Paid')
            ->firstOrFail();

        DB::transaction(function () use ($pettyCash) {
            $pettyCash->update([
                'status' => 'Reimbursed',
                'reimbursed_at' => now(),
                'reimbursed_by' => auth()->id(),
                'updated_by' => auth()->id()
            ]);

            // Update source account balance (add back the amount)
            $sourceAccount = $pettyCash->sourceAccount;
            $sourceAccount->updateBalance($pettyCash->amount, 'debit');

            // Update receipting account balance (decrease)
            $receiptingAccount = $pettyCash->receiptingAccount;
            $receiptingAccount->updateBalance($pettyCash->amount, 'credit');

            // Create reimbursement journal entry (reverse of payment)
            $this->createReimbursementJournalEntry($pettyCash);
            
            // Record reimbursement in cash book
            $this->recordReimbursementInCashBook($pettyCash);
        });

        return redirect()->back()->with('success', 'Petty cash reimbursed successfully.');
    }

    /**
     * Reverse petty cash transaction
     */
    public function reverse(Request $request, string $id)
    {
        $companyId = auth()->user()->companyid;
        
        if (!$companyId) {
            return redirect()->back()
                ->with('error', 'User is not associated with a company. Please contact administrator.');
        }

        $validated = $request->validate([
            'reversal_reason' => 'required|string|max:1000'
        ]);

        $pettyCash = PettyCash::where('petty_cash_id', $id)
            ->where('company_id', $companyId)
            ->whereIn('status', ['Paid', 'Reimbursed'])
            ->firstOrFail();

        DB::transaction(function () use ($pettyCash, $validated) {
            // Store original status
            $originalStatus = $pettyCash->status;
            
            // Update petty cash status
            $pettyCash->update([
                'status' => 'Reversed',
                'reversal_reason' => $validated['reversal_reason'],
                'reversed_at' => now(),
                'reversed_by' => auth()->id(),
                'updated_by' => auth()->id()
            ]);

            // Reverse the accounting entries
            $this->reverseAccountingEntries($pettyCash, $originalStatus);
            
            // Update source account balance based on original status
            $sourceAccount = $pettyCash->sourceAccount;
            $receiptingAccount = $pettyCash->receiptingAccount;
            
            if ($originalStatus === 'Paid') {
                // If it was paid: 
                // - Add back to source account (reverse the credit)
                // - Decrease from receipting account (reverse the debit)
                $sourceAccount->updateBalance($pettyCash->amount, 'debit');
                $receiptingAccount->updateBalance($pettyCash->amount, 'credit');
            } elseif ($originalStatus === 'Reimbursed') {
                // If it was reimbursed:
                // - Subtract from source account (reverse the debit)
                // - Add back to receipting account (reverse the credit)
                $sourceAccount->updateBalance($pettyCash->amount, 'credit');
                $receiptingAccount->updateBalance($pettyCash->amount, 'debit');
            }
            
            // Record reversal in cash book
            $this->recordReversalInCashBook($pettyCash, $originalStatus);
        });

        return redirect()->back()->with('success', 'Petty cash transaction reversed successfully.');
    }

    /**
     * Create payment journal entry
     */
    private function createPaymentJournalEntry(PettyCash $pettyCash)
    {
        // Get source account's GL account
        $sourceGlAccount = $pettyCash->sourceAccount->glAccount;
        $receiptingGlAccount = $pettyCash->receiptingAccount->glAccount;
        $expenseGlAccount = $pettyCash->chartOfAccount;
        
        if (!$sourceGlAccount || !$receiptingGlAccount || !$expenseGlAccount) {
            \Log::warning('GL accounts not properly linked for petty cash: ' . $pettyCash->petty_cash_id);
            return;
        }

        // Create journal entries for payment
        $transactionId = 'PC_PAY_' . time() . '_' . uniqid();
        $currentUserId = auth()->id();
        $currentTime = now();
        $description = "Payment for petty cash: {$pettyCash->petty_cash_number} - {$pettyCash->description}";

        // 1. Debit the expense account
        GeneralLedgerEntry::create([
            'accountid' => $expenseGlAccount->id,
            'entrydate' => $currentTime,
            'entrytype' => 'debit',
            'amount' => $pettyCash->base_amount,
            'description' => $description,
            'transactionid' => $transactionId,
            'transtype' => 'Petty Cash Payment',
            'companyid' => $pettyCash->company_id,
            'branchid' => $pettyCash->branch_id,
            'createdby' => $currentUserId,
            'created_at' => $currentTime,
            'updated_at' => $currentTime,
        ]);

        // 2. Credit the source account
        GeneralLedgerEntry::create([
            'accountid' => $sourceGlAccount->id,
            'entrydate' => $currentTime,
            'entrytype' => 'credit',
            'amount' => $pettyCash->base_amount,
            'description' => $description,
            'transactionid' => $transactionId,
            'transtype' => 'Petty Cash Payment',
            'companyid' => $pettyCash->company_id,
            'branchid' => $pettyCash->branch_id,
            'createdby' => $currentUserId,
            'created_at' => $currentTime,
            'updated_at' => $currentTime,
        ]);

        // 3. Debit the receipting account
        GeneralLedgerEntry::create([
            'accountid' => $receiptingGlAccount->id,
            'entrydate' => $currentTime,
            'entrytype' => 'debit',
            'amount' => $pettyCash->base_amount,
            'description' => $description . ' (Receipting)',
            'transactionid' => $transactionId . '_REC',
            'transtype' => 'Petty Cash Receipting',
            'companyid' => $pettyCash->company_id,
            'branchid' => $pettyCash->branch_id,
            'createdby' => $currentUserId,
            'created_at' => $currentTime,
            'updated_at' => $currentTime,
        ]);
    }

    /**
     * Record payment in cash book
     */
    private function recordPaymentInCashBook(PettyCash $pettyCash)
    {
        try {
            // Record payment entry (decrease in source account)
            CashBook::recordTransaction(
                $pettyCash->source_account_id,
                -$pettyCash->base_amount,
                'credit',
                'Petty Cash Payment',
                "Payment for petty cash: {$pettyCash->petty_cash_number} - {$pettyCash->description}",
                $pettyCash->petty_cash_number,
                $pettyCash->company_id,
                $pettyCash->branch_id,
                $pettyCash->currency
            );

            // Record receipting entry (increase in receipting account)
            CashBook::recordTransaction(
                $pettyCash->receipting_account_id,
                $pettyCash->base_amount,
                'debit',
                'Petty Cash Receipting',
                "Receipting for petty cash: {$pettyCash->petty_cash_number} - {$pettyCash->description}",
                $pettyCash->petty_cash_number . '_REC',
                $pettyCash->company_id,
                $pettyCash->branch_id,
                $pettyCash->currency
            );
        } catch (\Exception $e) {
            \Log::error('Failed to record payment in cash book: ' . $e->getMessage());
        }
    }

    /**
     * Create reimbursement journal entry
     */
    private function createReimbursementJournalEntry(PettyCash $pettyCash)
    {
        // Get GL accounts
        $sourceGlAccount = $pettyCash->sourceAccount->glAccount;
        $receiptingGlAccount = $pettyCash->receiptingAccount->glAccount;
        $expenseGlAccount = $pettyCash->chartOfAccount;
        
        if (!$sourceGlAccount || !$receiptingGlAccount || !$expenseGlAccount) {
            \Log::warning('GL accounts not properly linked for petty cash reimbursement: ' . $pettyCash->petty_cash_id);
            return;
        }

        // Create reversal journal entries (opposite of payment)
        $transactionId = 'PC_REIMB_' . time() . '_' . uniqid();
        $currentUserId = auth()->id();
        $currentTime = now();
        $description = "Reimbursement for petty cash: {$pettyCash->petty_cash_number}";

        // 1. Credit the expense account (reverse the debit)
        GeneralLedgerEntry::create([
            'accountid' => $expenseGlAccount->id,
            'entrydate' => $currentTime,
            'entrytype' => 'credit',
            'amount' => $pettyCash->base_amount,
            'description' => $description,
            'transactionid' => $transactionId,
            'transtype' => 'Petty Cash Reimbursement',
            'companyid' => $pettyCash->company_id,
            'branchid' => $pettyCash->branch_id,
            'createdby' => $currentUserId,
            'created_at' => $currentTime,
            'updated_at' => $currentTime,
        ]);

        // 2. Debit the source account (reverse the credit)
        GeneralLedgerEntry::create([
            'accountid' => $sourceGlAccount->id,
            'entrydate' => $currentTime,
            'entrytype' => 'debit',
            'amount' => $pettyCash->base_amount,
            'description' => $description,
            'transactionid' => $transactionId,
            'transtype' => 'Petty Cash Reimbursement',
            'companyid' => $pettyCash->company_id,
            'branchid' => $pettyCash->branch_id,
            'createdby' => $currentUserId,
            'created_at' => $currentTime,
            'updated_at' => $currentTime,
        ]);

        // 3. Credit the receipting account (reverse the debit)
        GeneralLedgerEntry::create([
            'accountid' => $receiptingGlAccount->id,
            'entrydate' => $currentTime,
            'entrytype' => 'credit',
            'amount' => $pettyCash->base_amount,
            'description' => $description . ' (Receipting Reversal)',
            'transactionid' => $transactionId . '_REC',
            'transtype' => 'Petty Cash Receipting Reversal',
            'companyid' => $pettyCash->company_id,
            'branchid' => $pettyCash->branch_id,
            'createdby' => $currentUserId,
            'created_at' => $currentTime,
            'updated_at' => $currentTime,
        ]);
    }

    /**
     * Record reimbursement in cash book
     */
    private function recordReimbursementInCashBook(PettyCash $pettyCash)
    {
        try {
            // Record reimbursement entry (increase in source account)
            CashBook::recordTransaction(
                $pettyCash->source_account_id,
                $pettyCash->base_amount,
                'debit',
                'Petty Cash Reimbursement',
                "Reimbursement for petty cash: {$pettyCash->petty_cash_number}",
                $pettyCash->petty_cash_number . '_REIMB',
                $pettyCash->company_id,
                $pettyCash->branch_id,
                $pettyCash->currency
            );

            // Record receipting reversal entry (decrease in receipting account)
            CashBook::recordTransaction(
                $pettyCash->receipting_account_id,
                -$pettyCash->base_amount,
                'credit',
                'Petty Cash Receipting Reversal',
                "Receipting reversal for petty cash reimbursement: {$pettyCash->petty_cash_number}",
                $pettyCash->petty_cash_number . '_REIMB_REC',
                $pettyCash->company_id,
                $pettyCash->branch_id,
                $pettyCash->currency
            );
        } catch (\Exception $e) {
            \Log::error('Failed to record reimbursement in cash book: ' . $e->getMessage());
        }
    }

    /**
     * Reverse accounting entries
     */
    private function reverseAccountingEntries(PettyCash $pettyCash, string $originalStatus)
    {
        $sourceGlAccount = $pettyCash->sourceAccount->glAccount;
        $receiptingGlAccount = $pettyCash->receiptingAccount->glAccount;
        $expenseGlAccount = $pettyCash->chartOfAccount;
        
        if (!$sourceGlAccount || !$receiptingGlAccount || !$expenseGlAccount) {
            \Log::warning('GL accounts not properly linked for petty cash reversal: ' . $pettyCash->petty_cash_id);
            return;
        }

        $transactionId = 'PC_REV_' . time() . '_' . uniqid();
        $currentUserId = auth()->id();
        $currentTime = now();
        $description = "Reversal of {$originalStatus} petty cash: {$pettyCash->petty_cash_number}";

        if ($originalStatus === 'Paid') {
            // Reverse payment: 
            // 1. Debit source (reverse credit), 2. Credit expense (reverse debit), 3. Credit receipting (reverse debit)
            GeneralLedgerEntry::create([
                'accountid' => $sourceGlAccount->id,
                'entrydate' => $currentTime,
                'entrytype' => 'debit',
                'amount' => $pettyCash->base_amount,
                'description' => $description . ' (Source)',
                'transactionid' => $transactionId,
                'transtype' => 'Petty Cash Reversal',
                'companyid' => $pettyCash->company_id,
                'branchid' => $pettyCash->branch_id,
                'createdby' => $currentUserId,
                'created_at' => $currentTime,
                'updated_at' => $currentTime,
            ]);

            GeneralLedgerEntry::create([
                'accountid' => $expenseGlAccount->id,
                'entrydate' => $currentTime,
                'entrytype' => 'credit',
                'amount' => $pettyCash->base_amount,
                'description' => $description . ' (Expense)',
                'transactionid' => $transactionId,
                'transtype' => 'Petty Cash Reversal',
                'companyid' => $pettyCash->company_id,
                'branchid' => $pettyCash->branch_id,
                'createdby' => $currentUserId,
                'created_at' => $currentTime,
                'updated_at' => $currentTime,
            ]);

            GeneralLedgerEntry::create([
                'accountid' => $receiptingGlAccount->id,
                'entrydate' => $currentTime,
                'entrytype' => 'credit',
                'amount' => $pettyCash->base_amount,
                'description' => $description . ' (Receipting)',
                'transactionid' => $transactionId . '_REC',
                'transtype' => 'Petty Cash Receipting Reversal',
                'companyid' => $pettyCash->company_id,
                'branchid' => $pettyCash->branch_id,
                'createdby' => $currentUserId,
                'created_at' => $currentTime,
                'updated_at' => $currentTime,
            ]);
        } elseif ($originalStatus === 'Reimbursed') {
            // Reverse reimbursement: 
            // 1. Credit source (reverse debit), 2. Debit expense (reverse credit), 3. Debit receipting (reverse credit)
            GeneralLedgerEntry::create([
                'accountid' => $sourceGlAccount->id,
                'entrydate' => $currentTime,
                'entrytype' => 'credit',
                'amount' => $pettyCash->base_amount,
                'description' => $description . ' (Source)',
                'transactionid' => $transactionId,
                'transtype' => 'Petty Cash Reversal',
                'companyid' => $pettyCash->company_id,
                'branchid' => $pettyCash->branch_id,
                'createdby' => $currentUserId,
                'created_at' => $currentTime,
                'updated_at' => $currentTime,
            ]);

            GeneralLedgerEntry::create([
                'accountid' => $expenseGlAccount->id,
                'entrydate' => $currentTime,
                'entrytype' => 'debit',
                'amount' => $pettyCash->base_amount,
                'description' => $description . ' (Expense)',
                'transactionid' => $transactionId,
                'transtype' => 'Petty Cash Reversal',
                'companyid' => $pettyCash->company_id,
                'branchid' => $pettyCash->branch_id,
                'createdby' => $currentUserId,
                'created_at' => $currentTime,
                'updated_at' => $currentTime,
            ]);

            GeneralLedgerEntry::create([
                'accountid' => $receiptingGlAccount->id,
                'entrydate' => $currentTime,
                'entrytype' => 'debit',
                'amount' => $pettyCash->base_amount,
                'description' => $description . ' (Receipting)',
                'transactionid' => $transactionId . '_REC',
                'transtype' => 'Petty Cash Receipting Reversal',
                'companyid' => $pettyCash->company_id,
                'branchid' => $pettyCash->branch_id,
                'createdby' => $currentUserId,
                'created_at' => $currentTime,
                'updated_at' => $currentTime,
            ]);
        }
    }

    /**
     * Record reversal in cash book
     */
    private function recordReversalInCashBook(PettyCash $pettyCash, string $originalStatus)
    {
        try {
            $transactionType = 'Petty Cash Reversal';
            $description = "Reversal of {$originalStatus} petty cash: {$pettyCash->petty_cash_number}";

            if ($originalStatus === 'Paid') {
                // Record as debit (increase) to source account
                CashBook::recordTransaction(
                    $pettyCash->source_account_id,
                    $pettyCash->base_amount,
                    'debit',
                    $transactionType,
                    $description . ' (Source)',
                    $pettyCash->petty_cash_number . '_REV',
                    $pettyCash->company_id,
                    $pettyCash->branch_id,
                    $pettyCash->currency
                );

                // Record as credit (decrease) to receipting account
                CashBook::recordTransaction(
                    $pettyCash->receipting_account_id,
                    -$pettyCash->base_amount,
                    'credit',
                    $transactionType . ' - Receipting',
                    $description . ' (Receipting)',
                    $pettyCash->petty_cash_number . '_REV_REC',
                    $pettyCash->company_id,
                    $pettyCash->branch_id,
                    $pettyCash->currency
                );
            } elseif ($originalStatus === 'Reimbursed') {
                // Record as credit (decrease) to source account
                CashBook::recordTransaction(
                    $pettyCash->source_account_id,
                    -$pettyCash->base_amount,
                    'credit',
                    $transactionType,
                    $description . ' (Source)',
                    $pettyCash->petty_cash_number . '_REV',
                    $pettyCash->company_id,
                    $pettyCash->branch_id,
                    $pettyCash->currency
                );

                // Record as debit (increase) to receipting account
                CashBook::recordTransaction(
                    $pettyCash->receipting_account_id,
                    $pettyCash->base_amount,
                    'debit',
                    $transactionType . ' - Receipting',
                    $description . ' (Receipting)',
                    $pettyCash->petty_cash_number . '_REV_REC',
                    $pettyCash->company_id,
                    $pettyCash->branch_id,
                    $pettyCash->currency
                );
            }
        } catch (\Exception $e) {
            \Log::error('Failed to record reversal in cash book: ' . $e->getMessage());
        }
    }

    /**
     * Export petty cash data
     */
    public function export(Request $request)
    {
        $companyId = auth()->user()->companyid;
        
        if (!$companyId) {
            return redirect()->back()
                ->with('error', 'User is not associated with a company. Please contact administrator.');
        }

        $query = PettyCash::with(['sourceAccount', 'receiptingAccount', 'chartOfAccount', 'requester'])
            ->where('company_id', $companyId);

        // Apply filters
        if ($request->has('start_date') && $request->start_date != '') {
            $query->where('transaction_date', '>=', $request->start_date);
        }
        if ($request->has('end_date') && $request->end_date != '') {
            $query->where('transaction_date', '<=', $request->end_date);
        }
        if ($request->has('status') && $request->status != '') {
            $query->where('status', $request->status);
        }

        $pettyCash = $query->get();

        $fileName = 'petty-cash-' . date('Y-m-d') . '.csv';
        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => "attachment; filename=\"{$fileName}\"",
        ];

        $callback = function() use ($pettyCash) {
            $file = fopen('php://output', 'w');
            
            // Add BOM for UTF-8
            fwrite($file, "\xEF\xBB\xBF");
            
            // Headers
            fputcsv($file, [
                'Petty Cash Number',
                'Transaction Date',
                'Payee',
                'Description',
                'Amount',
                'Currency',
                'Base Amount',
                'Exchange Rate',
                'Source Account',
                'Receipting Account',
                'Chart of Account',
                'Receipt Number',
                'Project Code',
                'Department',
                'Status',
                'Requested By',
                'Approved By',
                'Approval Date',
                'Reimbursed By',
                'Reimbursed Date',
                'Reversed By',
                'Reversed Date',
                'Reversal Reason'
            ]);

            // Data
            foreach ($pettyCash as $item) {
                fputcsv($file, [
                    $item->petty_cash_number,
                    $item->transaction_date->format('Y-m-d'),
                    $item->payee,
                    $item->description,
                    $item->amount,
                    $item->currency,
                    $item->base_amount,
                    $item->exchange_rate,
                    $item->sourceAccount->accountName ?? 'N/A',
                    $item->receiptingAccount->accountName ?? 'N/A',
                    $item->chartOfAccount->account_name ?? 'N/A',
                    $item->receipt_number ?? 'N/A',
                    $item->project_code ?? 'N/A',
                    $item->department ?? 'N/A',
                    $item->status,
                    $item->requester->name ?? 'N/A',
                    $item->approver->name ?? 'N/A',
                    $item->approved_at ? $item->approved_at->format('Y-m-d H:i:s') : 'N/A',
                    $item->reimbursedBy->name ?? 'N/A',
                    $item->reimbursed_at ? $item->reimbursed_at->format('Y-m-d H:i:s') : 'N/A',
                    $item->reversedBy->name ?? 'N/A',
                    $item->reversed_at ? $item->reversed_at->format('Y-m-d H:i:s') : 'N/A',
                    $item->reversal_reason ?? 'N/A'
                ]);
            }

            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }

    /**
     * Get source account balance
     */
    public function getAccountBalance($accountId)
    {
        $companyId = auth()->user()->companyid;
        
        if (!$companyId) {
            return response()->json(['error' => 'User not associated with a company'], 400);
        }

        $account = CashBankAccount::where('cashBankId', $accountId)
            ->where('companyid', $companyId)
            ->first();

        if (!$account) {
            return response()->json(['error' => 'Account not found'], 404);
        }

        return response()->json([
            'balance' => $account->current_balance,
            'currency' => $account->currency
        ]);
    }
}