<?php

namespace App\Http\Controllers\Accounting;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Receivable;
use App\Models\Customer;
use App\Models\ChartOfAccount;

class ReceivableController extends Controller
{
    public function index()
    {
        $receivables = Receivable::with(['customer', 'account'])
            ->orderBy('due_date')
            ->get();
            
        $customers = Customer::where('status', 'Active')->get();
        $accounts = ChartOfAccount::where('is_active', true)->get();
        
        return view('accounting.receivables.index', compact('receivables', 'customers', 'accounts'));
    }

    public function create()
    {
        $customers = Customer::where('status', 'Active')->get();
        $accounts = ChartOfAccount::where('is_active', true)->get();
        
        return view('accounting.receivables.create', compact('customers', 'accounts'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'customer_id' => 'required|exists:customers,id',
            'account_id' => 'required|exists:chart_of_accounts,id',
            'invoice_number' => 'required',
            'description' => 'required',
            'amount' => 'required|numeric|min:0',
            'due_date' => 'required|date',
            'status' => 'required|in:outstanding,paid,overdue,written_off',
            'notes' => 'nullable',
        ]);

        Receivable::create($validated);

        return redirect()->route('admin.accounting.receivables.index')
            ->with('success', 'Receivable created successfully.');
    }

    public function show($id)
    {
        $receivable = Receivable::with(['customer', 'account'])->findOrFail($id);
        
        return view('accounting.receivables.show', compact('receivable'));
    }

    public function edit($id)
    {
        $receivable = Receivable::findOrFail($id);
        $customers = Customer::where('status', 'Active')->get();
        $accounts = ChartOfAccount::where('is_active', true)->get();
        
        return view('accounting.receivables.edit', compact('receivable', 'customers', 'accounts'));
    }

    public function update(Request $request, $id)
    {
        $receivable = Receivable::findOrFail($id);
        
        $validated = $request->validate([
            'customer_id' => 'required|exists:customers,id',
            'account_id' => 'required|exists:chart_of_accounts,id',
            'invoice_number' => 'required',
            'description' => 'required',
            'amount' => 'required|numeric|min:0',
            'due_date' => 'required|date',
            'status' => 'required|in:outstanding,paid,overdue,written_off',
            'notes' => 'nullable',
        ]);

        $receivable->update($validated);

        return redirect()->route('admin.accounting.receivables.index')
            ->with('success', 'Receivable updated successfully.');
    }

    public function destroy($id)
    {
        $receivable = Receivable::findOrFail($id);
        $receivable->delete();

        return redirect()->route('admin.accounting.receivables.index')
            ->with('success', 'Receivable deleted successfully.');
    }
}