<?php

namespace App\Http\Controllers\Accounting;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Repayment;
use App\Models\DisbursedLoan;
use App\Models\PaymentSchedule;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class RepaymentController extends Controller
{
    public function store(Request $request)
    {
        $request->validate([
            'rloanid' => 'required|exists:disbursedloans,loanid',
            'rtotal' => 'required|numeric|min:0.01',
            'rdate' => 'required|date',
            'paccount' => 'required',
        ]);

        try {
            return DB::transaction(function () use ($request) {
                $loan = DisbursedLoan::lockForUpdate()->findOrFail($request->rloanid);
                $remainingPayment = (float)$request->rtotal;

                // Initialize allocation buckets
                $allocatedPrincipal = 0;
                $allocatedInterest = 0;
                $allocatedFees = 0;

                // Fetch unpaid schedules to allocate the payment
                $schedules = PaymentSchedule::where('loanid', $loan->loanid)
                    ->where('status', 'scheduled')
                    ->orderBy('paymentdate', 'asc')
                    ->get();

                foreach ($schedules as $schedule) {
                    if ($remainingPayment <= 0) break;

                    // 1. ALLOCATE TO PROCESSING FEES
                    $feeDue = (float)$schedule->processingfeesamount - (float)$schedule->paid_processingefees_amount;
                    if ($feeDue > 0 && $remainingPayment > 0) {
                        $paymentToFee = min($remainingPayment, $feeDue);
                        $allocatedFees += $paymentToFee;
                        $remainingPayment -= $paymentToFee;
                        $schedule->paid_processingefees_amount += $paymentToFee;
                    }

                    // 2. ALLOCATE TO INTEREST
                    $interestDue = (float)$schedule->interestamount - (float)$schedule->paid_interest;
                    if ($interestDue > 0 && $remainingPayment > 0) {
                        $paymentToInterest = min($remainingPayment, $interestDue);
                        $allocatedInterest += $paymentToInterest;
                        $remainingPayment -= $paymentToInterest;
                        $schedule->paid_interest += $paymentToInterest;
                    }

                    // 3. ALLOCATE TO PRINCIPAL
                    $principalDue = (float)$schedule->principalamount - (float)$schedule->paid_principal;
                    if ($principalDue > 0 && $remainingPayment > 0) {
                        $paymentToPrincipal = min($remainingPayment, $principalDue);
                        $allocatedPrincipal += $paymentToPrincipal;
                        $remainingPayment -= $paymentToPrincipal;
                        $schedule->paid_principal += $paymentToPrincipal;
                    }

                    // Update schedule status if fully paid
                    $totalDueOnSchedule = (float)$schedule->totalamount;
                    $totalPaidOnSchedule = (float)$schedule->paid_principal + (float)$schedule->paid_interest + (float)$schedule->paid_processingefees_amount;
                    
                    if ($totalPaidOnSchedule >= $totalDueOnSchedule) {
                        $schedule->status = 'paid';
                        $schedule->paid_date = $request->rdate;
                    }
                    $schedule->save();
                }

                // Create Repayment Record
                $repayment = Repayment::create([
                    'customerid' => $loan->customerid,
                    'rloanid' => $loan->loanid,
                    'rtotal' => $request->rtotal,
                    'rprincipal' => $allocatedPrincipal,
                    'rinterest' => $allocatedInterest,
                    'processing_fees_amount_paid' => $allocatedFees,
                    'rdate' => $request->rdate,
                    'userid' => auth()->id(),
                    'paccount' => $request->paccount,
                    'pmethod' => $request->pmethod,
                    'status' => 'paid'
                ]);

                // Update Loan Balances
                $loan->totalprincipalpaid += $allocatedPrincipal;
                $loan->totalinterestpaid += $allocatedInterest;
                $loan->totalpaid += $request->rtotal;
                $loan->principalbalance -= $allocatedPrincipal;
                $loan->totalbalance -= ($allocatedPrincipal + $allocatedInterest);
                
                if ($loan->principalbalance <= 0) {
                    $loan->status = 'closed';
                    $loan->closeddate = $request->rdate;
                }
                $loan->save();

                return redirect()->back()->with('success', 'Repayment processed and fees deducted successfully.');
            });
        } catch (\Exception $e) {
            Log::error("Repayment Error: " . $e->getMessage());
            return redirect()->back()->with('error', 'Failed to process repayment.');
        }
    }
}