<?php

namespace App\Http\Controllers\Accounting;

use App\Http\Controllers\Controller;
use App\Models\Company;
use App\Models\Employee;
use App\Models\PayrollEntry;
use App\Models\LeaveRequest;
use App\Models\LeaveBalance;
use App\Models\LeaveType;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;
use PDF;

class SelfServiceController extends Controller
{
    // Constructor removed - routes are already protected by 'auth' middleware in web.php
    
    /**
     * Self Service Dashboard
     */
    public function index()
    {
        $user = auth()->user();
        $company = Company::where('id', $user->companyid)->firstOrFail();
        
        // Get logged-in user's employee record
        $employee = Employee::where('company_id', $company->id)
            ->where('email', $user->email)
            ->firstOrFail();
        
        // Get current year stats
        $currentYear = date('Y');
        
        // Payroll stats for current year
        $payrollStats = PayrollEntry::where('company_id', $company->id)
            ->where('employee_id', $employee->id)
            ->whereYear('period_end_date', $currentYear)
            ->select(
                DB::raw('COUNT(*) as payrolls_done'),
                DB::raw('SUM(napsa_employee) as total_napsa'),
                DB::raw('SUM(nhima_employee) as total_nhima'),
                DB::raw('SUM(gross_earnings) as total_wages'),
                DB::raw('SUM(total_deductions) as total_deductions')
            )
            ->first();
        
        // Leave stats
        $leaveStats = LeaveBalance::where('company_id', $company->id)
            ->where('employee_id', $employee->id)
            ->where('year', $currentYear)
            ->select(
                DB::raw('SUM(total_accrued) as total_leave_days'),
                DB::raw('SUM(total_taken) as leave_days_taken'),
                DB::raw('SUM(balance) as leave_days_balance')
            )
            ->first();
        
        // Recent payrolls
        $recentPayrolls = PayrollEntry::where('company_id', $company->id)
            ->where('employee_id', $employee->id)
            ->with('payrollPeriod')
            ->orderBy('period_end_date', 'desc')
            ->take(5)
            ->get();
        
        // Recent leave requests
        $recentLeaveRequests = LeaveRequest::where('company_id', $company->id)
            ->where('employee_id', $employee->id)
            ->with('leaveType')
            ->orderBy('created_at', 'desc')
            ->take(5)
            ->get();
        
        // Current running leave (approved and ongoing)
        $runningLeaves = LeaveRequest::where('company_id', $company->id)
            ->where('employee_id', $employee->id)
            ->where('status', 'approved')
            ->where('end_date', '>=', today())
            ->where('start_date', '<=', today())
            ->with('leaveType')
            ->get();
        
        return view('accounting.self-service.index', compact(
            'company',
            'employee',
            'payrollStats',
            'leaveStats',
            'recentPayrolls',
            'recentLeaveRequests',
            'runningLeaves'
        ));
    }
    
    /**
     * Get dashboard stats via API
     */
    public function getStats()
    {
        $user = auth()->user();
        $company = Company::where('id', $user->companyid)->firstOrFail();
        
        $employee = Employee::where('company_id', $company->id)
            ->where('email', $user->email)
            ->firstOrFail();
        
        $currentYear = date('Y');
        
        $payrollStats = PayrollEntry::where('company_id', $company->id)
            ->where('employee_id', $employee->id)
            ->whereYear('period_end_date', $currentYear)
            ->select(
                DB::raw('COUNT(*) as payrolls_done'),
                DB::raw('SUM(napsa_employee) as total_napsa'),
                DB::raw('SUM(nhima_employee) as total_nhima'),
                DB::raw('SUM(gross_earnings) as total_wages'),
                DB::raw('SUM(total_deductions) as total_deductions')
            )
            ->first();
        
        $leaveStats = LeaveBalance::where('company_id', $company->id)
            ->where('employee_id', $employee->id)
            ->where('year', $currentYear)
            ->select(
                DB::raw('SUM(total_accrued) as total_leave_days'),
                DB::raw('SUM(total_taken) as leave_days_taken'),
                DB::raw('SUM(balance) as leave_days_balance')
            )
            ->first();
        
        return response()->json([
            'success' => true,
            'payroll_stats' => $payrollStats,
            'leave_stats' => $leaveStats
        ]);
    }
    
    /**
     * Payrolls Index - View all payrolls
     */
    public function payrollsIndex(Request $request)
    {
        $user = auth()->user();
        $company = Company::where('id', $user->companyid)->firstOrFail();
        
        $employee = Employee::where('company_id', $company->id)
            ->where('email', $user->email)
            ->firstOrFail();
        
        $query = PayrollEntry::where('company_id', $company->id)
            ->where('employee_id', $employee->id)
            ->with('payrollPeriod');
        
        // Apply filters
        if ($request->has('year') && $request->year != 'all') {
            $query->whereYear('period_end_date', $request->year);
        }
        
        if ($request->has('status') && $request->status != 'all') {
            $query->where('calculation_status', $request->status);
        }
        
        $payrolls = $query->orderBy('period_end_date', 'desc')
            ->paginate(20);
        
        // Get available years for filter
        $years = PayrollEntry::where('company_id', $company->id)
            ->where('employee_id', $employee->id)
            ->select(DB::raw('YEAR(period_end_date) as year'))
            ->distinct()
            ->orderBy('year', 'desc')
            ->pluck('year');
        
        return view('accounting.self-service.payrolls.index', compact(
            'company',
            'employee',
            'payrolls',
            'years'
        ));
    }
    
    /**
     * View specific payroll entry
     */
    public function payrollShow($entryId)
    {
        $user = auth()->user();
        $company = Company::where('id', $user->companyid)->firstOrFail();
        
        $employee = Employee::where('company_id', $company->id)
            ->where('email', $user->email)
            ->firstOrFail();
        
        $entry = PayrollEntry::where('company_id', $company->id)
            ->where('employee_id', $employee->id)
            ->where('id', $entryId)
            ->with(['payrollPeriod', 'employee'])
            ->firstOrFail();
        
        // Format earnings breakdown
        $earningsBreakdown = $entry->getEarningsBreakdownFormatted();
        
        return view('accounting.self-service.payrolls.show', compact(
            'company',
            'employee',
            'entry',
            'earningsBreakdown'
        ));
    }
    
    /**
     * Generate payslip PDF
     */
    public function generatePayslip($entryId)
    {
        $user = auth()->user();
        $company = Company::where('id', $user->companyid)->firstOrFail();
        
        $employee = Employee::where('company_id', $company->id)
            ->where('email', $user->email)
            ->firstOrFail();
        
        $entry = PayrollEntry::where('company_id', $company->id)
            ->where('employee_id', $employee->id)
            ->where('id', $entryId)
            ->with(['payrollPeriod', 'employee'])
            ->firstOrFail();
        
        // Format earnings breakdown
        $earningsBreakdown = $entry->getEarningsBreakdownFormatted();
        
        $data = [
            'company' => $company,
            'employee' => $employee,
            'entry' => $entry,
            'earningsBreakdown' => $earningsBreakdown,
            'generated_at' => now()->format('M d, Y H:i')
        ];
        
        $pdf = PDF::loadView('accounting.self-service.payrolls.payslip-pdf', $data);
        
        $filename = 'payslip-' . $employee->employee_number . '-' . $entry->payrollPeriod->period_name . '.pdf';
        
        return $pdf->download($filename);
    }
    
    /**
     * Leave Requests Index
     */
    public function leaveIndex(Request $request)
    {
        $user = auth()->user();
        $company = Company::where('id', $user->companyid)->firstOrFail();
        
        $employee = Employee::where('company_id', $company->id)
            ->where('email', $user->email)
            ->firstOrFail();
        
        $query = LeaveRequest::where('company_id', $company->id)
            ->where('employee_id', $employee->id)
            ->with('leaveType');
        
        // Apply filters
        if ($request->has('status') && $request->status != 'all') {
            $query->where('status', $request->status);
        }
        
        if ($request->has('leave_type_id') && $request->leave_type_id != 'all') {
            $query->where('leave_type_id', $request->leave_type_id);
        }
        
        if ($request->has('year') && $request->year != 'all') {
            $query->whereYear('start_date', $request->year);
        }
        
        $leaveRequests = $query->orderBy('created_at', 'desc')
            ->paginate(20);
        
        $leaveTypes = LeaveType::where('company_id', $company->id)
            ->where('is_active', true)
            ->get();
        
        // Get available years for filter
        $years = LeaveRequest::where('company_id', $company->id)
            ->where('employee_id', $employee->id)
            ->select(DB::raw('YEAR(start_date) as year'))
            ->distinct()
            ->orderBy('year', 'desc')
            ->pluck('year');
        
        // Get leave balances
        $leaveBalances = LeaveBalance::where('company_id', $company->id)
            ->where('employee_id', $employee->id)
            ->where('year', date('Y'))
            ->with('leaveType')
            ->get();
        
        return view('accounting.self-service.leave.index', compact(
            'company',
            'employee',
            'leaveRequests',
            'leaveTypes',
            'years',
            'leaveBalances'
        ));
    }
    
    /**
     * Create Leave Request
     */
    public function leaveCreate()
    {
        $user = auth()->user();
        $company = Company::where('id', $user->companyid)->firstOrFail();
        
        $employee = Employee::where('company_id', $company->id)
            ->where('email', $user->email)
            ->firstOrFail();
        
        $leaveTypes = LeaveType::where('company_id', $company->id)
            ->where('is_active', true)
            ->get();
        
        // Get leave balances for current year
        $leaveBalances = LeaveBalance::where('company_id', $company->id)
            ->where('employee_id', $employee->id)
            ->where('year', date('Y'))
            ->with('leaveType')
            ->get();
        
        return view('accounting.self-service.leave.create', compact(
            'company',
            'employee',
            'leaveTypes',
            'leaveBalances'
        ));
    }
    
    /**
     * Store Leave Request
     */
    public function leaveStore(Request $request)
    {
        $request->validate([
            'leave_type_id' => 'required|exists:leave_types,id',
            'start_date' => 'required|date',
            'end_date' => 'required|date|after_or_equal:start_date',
            'reason' => 'required|string|max:500'
        ]);
        
        $user = auth()->user();
        $company = Company::where('id', $user->companyid)->firstOrFail();
        
        $employee = Employee::where('company_id', $company->id)
            ->where('email', $user->email)
            ->firstOrFail();
        
        // Check if leave type belongs to company
        $leaveType = LeaveType::where('company_id', $company->id)
            ->findOrFail($request->leave_type_id);
        
        // Calculate total days
        $startDate = Carbon::parse($request->start_date);
        $endDate = Carbon::parse($request->end_date);
        $totalDays = $startDate->diffInDays($endDate) + 1; // Inclusive
        
        // Check leave balance
        $leaveBalance = LeaveBalance::where('company_id', $company->id)
            ->where('employee_id', $employee->id)
            ->where('leave_type_id', $leaveType->id)
            ->where('year', date('Y'))
            ->first();
        
        $availableBalance = $leaveBalance ? $leaveBalance->available_balance : 0;
        
        if ($totalDays > $availableBalance && !$leaveType->can_carry_forward) {
            return redirect()->back()
                ->with('error', "Insufficient leave balance. Available: {$availableBalance} days, Requested: {$totalDays} days.")
                ->withInput();
        }
        
        try {
            $leaveRequest = LeaveRequest::create([
                'company_id' => $company->id,
                'employee_id' => $employee->id,
                'leave_type_id' => $leaveType->id,
                'start_date' => $request->start_date,
                'end_date' => $request->end_date,
                'total_days' => $totalDays,
                'reason' => $request->reason,
                'status' => $leaveType->requires_approval ? 'pending' : 'approved',
                'requested_by' => $user->id
            ]);
            
            return redirect()->route('admin.self-service.leave.show', $leaveRequest->id)
                ->with('success', 'Leave request submitted successfully.');
                
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to submit leave request: ' . $e->getMessage())
                ->withInput();
        }
    }
    
    /**
     * View Leave Request
     */
    public function leaveShow($id)
    {
        $user = auth()->user();
        $company = Company::where('id', $user->companyid)->firstOrFail();
        
        $employee = Employee::where('company_id', $company->id)
            ->where('email', $user->email)
            ->firstOrFail();
        
        $leaveRequest = LeaveRequest::where('company_id', $company->id)
            ->where('employee_id', $employee->id)
            ->where('id', $id)
            ->with(['leaveType', 'approver', 'rejector'])
            ->firstOrFail();
        
        return view('accounting.self-service.leave.show', compact(
            'company',
            'employee',
            'leaveRequest'
        ));
    }
    
    /**
     * Cancel Leave Request
     */
    public function leaveCancel($id)
    {
        $user = auth()->user();
        $company = Company::where('id', $user->companyid)->firstOrFail();
        
        $employee = Employee::where('company_id', $company->id)
            ->where('email', $user->email)
            ->firstOrFail();
        
        $leaveRequest = LeaveRequest::where('company_id', $company->id)
            ->where('employee_id', $employee->id)
            ->where('id', $id)
            ->firstOrFail();
        
        if (!$leaveRequest->canCancel()) {
            return redirect()->back()
                ->with('error', 'This leave request cannot be cancelled.');
        }
        
        DB::beginTransaction();
        
        try {
            // If it was approved, add back the leave balance
            if ($leaveRequest->status === 'approved') {
                $leaveBalance = LeaveBalance::where('company_id', $company->id)
                    ->where('employee_id', $employee->id)
                    ->where('leave_type_id', $leaveRequest->leave_type_id)
                    ->where('year', date('Y'))
                    ->first();
                
                if ($leaveBalance) {
                    $leaveBalance->balance += $leaveRequest->total_days;
                    $leaveBalance->total_taken -= $leaveRequest->total_days;
                    $leaveBalance->save();
                }
            }
            
            $leaveRequest->update(['status' => 'cancelled']);
            
            DB::commit();
            
            return redirect()->route('admin.self-service.leave.show', $leaveRequest->id)
                ->with('success', 'Leave request cancelled successfully.');
                
        } catch (\Exception $e) {
            DB::rollBack();
            
            return redirect()->back()
                ->with('error', 'Failed to cancel leave request: ' . $e->getMessage());
        }
    }
    
    /**
     * Approved Leave (view only approved leave requests)
     */
    public function approvedLeave()
    {
        $user = auth()->user();
        $company = Company::where('id', $user->companyid)->firstOrFail();
        
        $employee = Employee::where('company_id', $company->id)
            ->where('email', $user->email)
            ->firstOrFail();
        
        $approvedRequests = LeaveRequest::where('company_id', $company->id)
            ->where('employee_id', $employee->id)
            ->where('status', 'approved')
            ->with('leaveType')
            ->orderBy('approved_at', 'desc')
            ->paginate(20);
        
        return view('accounting.self-service.leave.approved', compact(
            'company',
            'employee',
            'approvedRequests'
        ));
    }
    
    /**
     * Running Leave (current/taken leaves)
     */
    public function runningLeave()
    {
        $user = auth()->user();
        $company = Company::where('id', $user->companyid)->firstOrFail();
        
        $employee = Employee::where('company_id', $company->id)
            ->where('email', $user->email)
            ->firstOrFail();
        
        $runningLeaves = LeaveRequest::where('company_id', $company->id)
            ->where('employee_id', $employee->id)
            ->where('status', 'approved')
            ->where('end_date', '>=', today())
            ->where('start_date', '<=', today())
            ->with('leaveType')
            ->orderBy('start_date', 'desc')
            ->paginate(20);
        
        return view('accounting.self-service.leave.running', compact(
            'company',
            'employee',
            'runningLeaves'
        ));
    }
    
    /**
     * Get leave balance via API
     */
    public function getLeaveBalance(Request $request)
    {
        $user = auth()->user();
        $company = Company::where('id', $user->companyid)->firstOrFail();
        
        $employee = Employee::where('company_id', $company->id)
            ->where('email', $user->email)
            ->firstOrFail();
        
        $leaveBalance = LeaveBalance::where('company_id', $company->id)
            ->where('employee_id', $employee->id)
            ->where('leave_type_id', $request->leave_type_id)
            ->where('year', date('Y'))
            ->first();
        
        return response()->json([
            'success' => true,
            'balance' => $leaveBalance ? $leaveBalance->available_balance : 0,
            'total_accrued' => $leaveBalance ? $leaveBalance->total_accrued : 0,
            'total_taken' => $leaveBalance ? $leaveBalance->total_taken : 0
        ]);
    }
}