<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Application;
use App\Models\LoanType;
use App\Models\Currency;
use App\Models\DisbursedLoan;
use App\Models\Customer;
use App\Models\PaymentDisbursement;
use App\Services\SmsService;
use App\Services\PaymentScheduleService;
use App\Services\LipilaPaymentService;
use App\Services\LoanDisbursementService;
use App\Services\CreditScoreService;
use App\Models\LoanApplicationDocument;
use App\Models\ApplicationRequirement; // Add this import
use App\Models\ProductRequirement; // Add this import
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Response;
use Illuminate\Support\Facades\Storage; // Add this too if not present
use Illuminate\Support\Str; // ADD THIS LINE
use Illuminate\Support\Facades\Mail;
use Carbon\Carbon;
use Throwable;
use TCPDF;
use App\Mail\LoanApplicationSubmitted;
use App\Mail\LoanApplicationApproved;
use App\Mail\LoanApplicationRejected;
use App\Mail\LoanDisbursed;

class ApplicationController extends Controller
{
    private $smsService;
    private $paymentScheduleService;
    private $loanDisbursementService;
    private $creditScoreService;
    private $lipilaPaymentService;

    public function __construct(
        SmsService $smsService, 
        PaymentScheduleService $paymentScheduleService,
        LoanDisbursementService $loanDisbursementService,
        CreditScoreService $creditScoreService,
        LipilaPaymentService $lipilaPaymentService
    ) {
        $this->smsService = $smsService;
        $this->paymentScheduleService = $paymentScheduleService;
        $this->loanDisbursementService = $loanDisbursementService;
        $this->creditScoreService = $creditScoreService;
        $this->lipilaPaymentService = $lipilaPaymentService;
    }

    /**
     * Display all applications (excluding approved and disbursed ones)
     */
    /**
 * Display all applications (excluding approved and disbursed ones)
 */
/**
 * Display draft applications
 */
/**
 * Display ALL applications without status filtering
 */
public function index(Request $request) 
{
    if (!auth()->user()->hasAnyPermission(['applications.view', 'applications.approve', 'poweruser'])) { 
        abort(403, 'You do not have permission.'); 
    }
    
    // Show ALL applications - NO status filtering
    $query = Application::with(['customer', 'product', 'currency'])
        ->orderBy('created_at', 'desc');
    
    if ($request->has('search')) {
        $searchTerm = $request->search;
        $query->where(function ($q) use ($searchTerm) {
            $q->whereHas('customer', function($subQ) use ($searchTerm) {
                $subQ->where('first_name', 'like', "%{$searchTerm}%")
                     ->orWhere('surname', 'like', "%{$searchTerm}%");
            })->orWhere('application_number', 'like', "%{$searchTerm}%");
        });
    }
    
    $applications = $query->paginate(100);
    
    // Get all stats for sidebar
    $stats = $this->getApplicationStats();
    
    return view('admin.applications.index', compact('applications', 'stats'));
}

public function draft(Request $request) 
{
    if (!auth()->user()->hasAnyPermission(['applications.view', 'applications.approve', 'poweruser'])) { 
        abort(403, 'You do not have permission.'); 
    }
    
    // Show draft applications
    $query = Application::with(['customer', 'product', 'currency'])
        ->where('status', 'draft')
        ->orderBy('created_at', 'desc');
    
    if ($request->has('search')) {
        $searchTerm = $request->search;
        $query->where(function ($q) use ($searchTerm) {
            $q->whereHas('customer', function($subQ) use ($searchTerm) {
                $subQ->where('first_name', 'like', "%{$searchTerm}%")
                     ->orWhere('surname', 'like', "%{$searchTerm}%");
            })->orWhere('application_number', 'like', "%{$searchTerm}%");
        });
    }
    
    $applications = $query->paginate(20);
    
    // Get all stats for sidebar
    $stats = $this->getApplicationStats();
    
    return view('admin.applications.draft', compact('applications', 'stats'));
}
    /**
     * Display approved applications (excluding disbursed ones)
     */
    public function approved() 
    {
        if (!auth()->user()->hasAnyPermission(['applications.view', 'applications.approve', 'poweruser'])) { 
            abort(403); 
        }
        
        // Show ONLY approved but NOT disbursed applications
        $applications = Application::with(['customer', 'product', 'currency'])
            ->approved()        // Status = approved and approved_at is not null
            ->notDisbursed()    // Not yet disbursed
            ->orderBy('approved_at', 'desc')
            ->paginate(20);
        
        return view('admin.applications.approved', compact('applications'));
    }

    /**
     * Display disbursed applications
     */
    public function disbursed()
    {
        if (!auth()->user()->hasAnyPermission(['applications.view', 'applications.approve', 'poweruser'])) { 
            abort(403); 
        }
        
        // Show ONLY disbursed applications
        $applications = Application::with(['customer', 'product', 'currency', 'disbursedBy'])
            ->disbursed()  // Using scopeDisbursed
            ->orderBy('disbursed_at', 'desc')
            ->paginate(20);
        
        return view('admin.applications.disbursed', compact('applications'));
    }
    
    /**
 * Get application statistics for sidebar
 */
  public static function getApplicationStats()
  {
    $stats = [
        'draft_applications' => Application::where('status', 'draft')->count(),
        'pending_applications' => Application::where('status', 'pending')->count(),
        'pending_submitted_count' => Application::where('status', 'submitted')->count(),
        'approved_applications' => Application::approved()->count(),
        'rejected_applications' => Application::where('status', 'rejected')->count(),
        'disbursed_applications' => Application::disbursed()->count(),
    ];
    
    return $stats;
   }

    /**
     * Display rejected applications (excluding disbursed ones)
     */
    public function rejected()
    {
        if (!auth()->user()->hasAnyPermission(['applications.view', 'applications.approve', 'poweruser'])) { 
            abort(403); 
        }
        
        // Show rejected applications that are not disbursed
        $applications = Application::with(['customer', 'product', 'currency'])
            ->where('status', 'rejected')
            ->notDisbursed()  // Exclude disbursed rejected applications
            ->orderBy('reviewed_at', 'desc')
            ->paginate(20);
        
        return view('admin.applications.rejected', compact('applications'));
    }

    /**
     * Display pending applications (excluding disbursed ones)
     */
    public function pending()
    {
        if (!auth()->user()->hasAnyPermission(['applications.view', 'applications.approve', 'poweruser'])) { 
            abort(403); 
        }
        
        // Show pending applications that are not disbursed
        $applications = Application::with(['customer', 'product', 'currency'])
            ->where('status', 'submitted')
            ->notDisbursed()  // Exclude disbursed pending applications
            ->orderBy('submitted_at', 'desc')
            ->paginate(20);
        
        return view('admin.applications.pending', compact('applications'));
    }

    /**
     * Show the form for creating a new application
     */
    public function create()
    {
        if (!auth()->user()->hasPermission('applications.create')) { 
            abort(403); 
        }
        $customers = Customer::all();
        $currencies = Currency::all();
        $loanProducts = LoanType::all(); 
        return view('admin.applications.create', compact('customers', 'currencies', 'loanProducts'));
    }

    /**
     * Store a newly created application in storage.
     */
    /**
 * Store a newly created application in storage.
 */
public function store(Request $request)
{
    if (!auth()->user()->hasPermission('applications.create')) { 
        abort(403); 
    }
    
    \Log::info('=== APPLICATION FORM SUBMISSION DEBUG ===');
    \Log::info('Request Method: ' . $request->method());
    \Log::info('All Form Data:', $request->all());
    
    $validator = Validator::make($request->all(), [
        'customer_id' => 'required|exists:customer,id',
        'product_id' => 'required|exists:loantype,id',
        'currency_id' => 'required|exists:currencies,id',
        'loan_amount' => 'required|numeric|min:0',
        'loan_tenure' => 'required|integer|min:1',
        'payment_frequency' => 'required|string|in:one_off,daily,weekly,monthly,quarterly',
        'interest_method' => 'required|string|in:simple_interest,reducing_balance',
        'source' => 'required|string|in:online,branch,agent,mobile',
        'suggestedpaymentmethod' => 'required|string|in:Cash,Bank,Mobile',
        'mobile_number' => 'nullable|string|max:20',
        'mobile_channel' => 'required_if:suggestedpaymentmethod,Mobile|string|max:50',
        'application_date' => 'nullable|date|before_or_equal:today',
    ]);
    
    if ($validator->fails()) { 
        \Log::error('Validation Errors:', $validator->errors()->toArray());
        return redirect()->back()
            ->withErrors($validator)
            ->withInput();
    }
    
    try {
        return DB::transaction(function () use ($request) {
            $product = LoanType::findOrFail($request->product_id);
            $data = $request->all();
            
            // Generate application number
            $data['application_number'] = 'APP-' . time(); 
            
            // Determine status based on action
            $action = $request->input('action', 'submit');
            $isDraft = ($action === 'draft');
            $data['status'] = $isDraft ? 'draft' : 'submitted';
            
            // Handle application date (with backdating support)
            if ($request->filled('application_date')) {
                $applicationDate = Carbon::parse($request->application_date);
                
                if ($isDraft) {
                    // For drafts, we don't set submitted_at
                    $data['application_date'] = $applicationDate;
                } else {
                    // For submitted applications, set submitted_at
                    $data['submitted_at'] = $applicationDate;
                    
                    // Log backdating activity for audit trail
                    $daysAgo = $applicationDate->diffInDays(now());
                    if ($daysAgo > 0) {
                        Log::info('Backdated application created', [
                            'application_date' => $applicationDate->format('Y-m-d'),
                            'days_backdated' => $daysAgo,
                            'user_id' => auth()->id(),
                            'user_name' => auth()->user()->name,
                            'customer_id' => $request->customer_id,
                            'loan_amount' => $request->loan_amount,
                            'status' => $isDraft ? 'draft' : 'submitted'
                        ]);
                    }
                }
            } else {
                if (!$isDraft) {
                    $data['submitted_at'] = now();
                }
            }
            
            $data['created_by'] = auth()->id();
            $data['company_id'] = auth()->user()->company_id ?? 1;
            $data['branch_id'] = auth()->user()->branch_id ?? 1;
            
            // Calculate installment amount using controller's internal method
            $amount = $request->loan_amount;
            $interestRate = $request->interest_rate ?? $product->yearlyrate ?? 0;
            $tenure = $request->loan_tenure;
            $frequency = $request->payment_frequency;
            $interestMethod = $request->input('interest_method') ?? ($product->interest_method ?? 'reducing_balance');
            $processingFeeRate = $request->input('processing_fee') ?? $product->processing_fee ?? 0;
            $processingFeeBasis = $request->input('processing_fee_basis') ?? $product->processing_fee_basis ?? 'initial_amount';
            
            if ($interestMethod === 'simple_interest') {
                $data['installment_amount'] = $this->calculateSimpleInterestInstallment(
                    $amount, $interestRate, $tenure, $frequency, 
                    $processingFeeRate, $processingFeeBasis
                );
            } else {
                $data['installment_amount'] = $this->calculateReducingBalanceInstallment(
                    $amount, $interestRate, $tenure, $frequency,
                    $processingFeeRate, $processingFeeBasis
                );
            }
            
            // Use interest method from form
            $data['interest_method'] = $interestMethod;
            
            // Ensure fee values are properly set
            $data['processing_fee'] = $processingFeeRate;
            $data['processing_fee_basis'] = $processingFeeBasis;
            $data['adminfee'] = $request->input('adminfee') ?? $product->adminfee ?? 0;
            $data['insurancefee'] = $request->input('insurancefee') ?? $product->insurancefee ?? 0;
            $data['penalty_rate'] = $request->input('penalty_rate') ?? $product->penalty_rate ?? 0;
            $data['penalty_basis'] = $request->input('penalty_basis') ?? $product->penalty_basis ?? 'percentage_of_due';
            
            // Handle mobile money details if applicable
            if ($request->suggestedpaymentmethod === 'Mobile') {
                $data['mobile_number'] = $request->mobile_number;
                $data['mobile_channel'] = $request->mobile_channel;
            }
            
            // Handle contract expiry date formatting - only if provided
            if ($request->filled('contract_expiry_date')) {
                $data['contract_expiry_date'] = Carbon::parse($request->contract_expiry_date);
            } else {
                $data['contract_expiry_date'] = null;
            }
            
            \Log::info('Data to be saved:', $data);
            
            // Create the application with all fields
            $application = Application::create($data);
            
            \Log::info('Application created successfully:', [
                'id' => $application->id, 
                'application_number' => $application->application_number,
                'status' => $application->status
            ]);
            
            // Send notifications only for submitted applications (not drafts)
            if (!$isDraft) {
                // DEBUG: Log customer and notification settings check
                \Log::info('=== NOTIFICATION DEBUG INFO ===');
                \Log::info('Customer ID:', ['customer_id' => $application->customer_id]);
                
                // Send notifications
                $this->sendRequestNotifications($application);
            }
            
            // Redirect based on action
            if ($isDraft) {
                return redirect()->route('admin.applications.draft')
                    ->with('success', 'Application saved as draft successfully. You can edit and submit it later.');
            } else {
                return redirect()->route('admin.applications.index')
                    ->with('success', 'Application submitted successfully.');
            }
        });
    } catch (Throwable $e) {
        \Log::error("Application Store Error: " . $e->getMessage() . "\n" . $e->getTraceAsString());
        return redirect()->back()->with('error', 'Error saving application: ' . $e->getMessage())->withInput();
    }
}
    
    /**
     * Show the disbursement form for an approved application.
     */
    public function disburse($id)
    {
        if (!auth()->user()->hasPermission('applications.approve')) {
            abort(403, 'Unauthorized access to disbursement form.');
        }

        $application = Application::with(['customer', 'product', 'currency'])->findOrFail($id);
        
        // Check if application is approved and not disbursed
        if (!$application->isApproved() || $application->isDisbursed()) {
            return redirect()->back()->with('error', 'This application is not eligible for disbursement.');
        }

        // Fetch from CashBankAccount model
        $cashAccounts = \App\Models\CashBankAccount::active()
                        ->where('companyid', auth()->user()->companyid ?? $application->company_id)
                        ->orderBy('accountType')
                        ->orderBy('accountName')
                        ->get();

        return view('admin.applications.disburse', compact('application', 'cashAccounts'));
    }

    /**
     * Display the specified application.
     */
      /**
     * Display application details including documents with product-specific requirements
     */
    public function show($id)
    {
        if (!auth()->user()->hasAnyPermission(['applications.view', 'applications.approve', 'poweruser'])) { 
            abort(403); 
        }

        // Load application with all necessary relationships including documents
        $application = Application::with([
            'customer', 
            'product', 
            'currency', 
            'createdBy', 
            'reviewedBy',
            'documents', // This loads the documents relationship
            'documents.requirement' // Load the requirement for each document
        ])->findOrFail($id);
        
        // Get all product requirements for the specific loan product
        $productRequirements = ProductRequirement::where('product_id', $application->product_id)
            ->with('requirement')
            ->orderBy('created_at', 'desc')
            ->get();
        
        // Get the actual requirements with their mandatory status
        $requirementsWithStatus = [];
        foreach ($productRequirements as $productRequirement) {
            $requirement = $productRequirement->requirement;
            if (!$requirement) {
                continue; // Skip if requirement doesn't exist
            }
            
            $existingDocument = $application->documents->firstWhere('requirement_id', $requirement->id);
            
            $requirementsWithStatus[] = [
                'product_requirement_id' => $productRequirement->id,
                'requirement_id' => $requirement->id,
                'name' => $requirement->documentname,
                'description' => $requirement->description ?? 'No description available',
                'is_mandatory' => (bool)$productRequirement->is_mandatory,
                'has_document' => !is_null($existingDocument),
                'document' => $existingDocument,
                'status' => $existingDocument ? $existingDocument->status : 'missing',
                'status_text' => $existingDocument ? $existingDocument->getStatusTextAttribute() : 'Not Attached',
                'uploaded_at' => $existingDocument ? $existingDocument->created_at->format('M d, Y H:i') : null,
                'verified_by' => $existingDocument && $existingDocument->verified_by ? 
                    ($existingDocument->verifiedBy->name ?? 'Unknown') : null,
                'verified_at' => $existingDocument && $existingDocument->verified_at ? 
                    $existingDocument->verified_at->format('M d, Y H:i') : null,
                'rejection_reason' => $existingDocument ? $existingDocument->rejection_reason : null,
            ];
        }
        
        // Get document statistics
        $documentStats = $this->getDocumentStatistics($application, $productRequirements);
        
        // Load necessary data for the edit form
        $loanProducts = LoanType::all();
        $currencies = Currency::all();
        
        return view('admin.applications.show', compact(
            'application', 
            'loanProducts', 
            'currencies',
            'requirementsWithStatus',
            'documentStats'
        ));
    }

    
    /**
     * Show the form for editing the specified application.
     */
    /**
 * Show the form for editing the specified application.
 */
/**
 * Show the form for editing the specified application.
 */
public function edit($id)
{
    // Load application with all necessary relationships including documents
    $application = Application::with([
        'customer', 
        'product', 
        'currency', 
        'documents', // This loads the documents relationship
        'documents.requirement' // Load the requirement for each document
    ])->findOrFail($id);
    
    // Check if user has permission to edit
    $isDraft = $application->status === 'draft';
    
    // For draft applications, only allow edit if user created it or has special permissions
    if ($isDraft) {
        if (!auth()->user()->hasPermission('applications.edit') && 
            $application->created_by !== auth()->id()) {
            abort(403, 'You do not have permission to edit this draft application.');
        }
    } else {
        // For non-draft applications, require specific permissions
        if (!auth()->user()->hasAnyPermission(['applications.edit', 'applications.approve', 'poweruser'])) {
            abort(403, 'You do not have permission to edit this application.');
        }
        
        // Additional restrictions for submitted/approved applications
        if ($application->status === 'approved' && !auth()->user()->hasPermission('applications.approve')) {
            abort(403, 'Only approvers can edit approved applications.');
        }
    }
    
    $loanProducts = LoanType::all();
    $currencies = Currency::all();
    
    // Get all product requirements for the specific loan product
    $productRequirements = ProductRequirement::where('product_id', $application->product_id)
        ->with('requirement')
        ->orderBy('created_at', 'desc')
        ->get();
    
    // Get the actual requirements with their mandatory status
    $requirementsWithStatus = [];
    foreach ($productRequirements as $productRequirement) {
        $requirement = $productRequirement->requirement;
        if (!$requirement) {
            continue; // Skip if requirement doesn't exist
        }
        
        $existingDocument = $application->documents->firstWhere('requirement_id', $requirement->id);
        
        $requirementsWithStatus[] = [
            'product_requirement_id' => $productRequirement->id,
            'requirement_id' => $requirement->id,
            'name' => $requirement->documentname,
            'description' => $requirement->description ?? 'No description available',
            'is_mandatory' => (bool)$productRequirement->is_mandatory,
            'has_document' => !is_null($existingDocument),
            'document' => $existingDocument,
            'status' => $existingDocument ? $existingDocument->status : 'missing',
            'status_text' => $existingDocument ? $existingDocument->getStatusTextAttribute() : 'Not Attached',
            'uploaded_at' => $existingDocument ? $existingDocument->created_at->format('M d, Y H:i') : null,
            'verified_by' => $existingDocument && $existingDocument->verified_by ? 
                ($existingDocument->verifiedBy->name ?? 'Unknown') : null,
            'verified_at' => $existingDocument && $existingDocument->verified_at ? 
                $existingDocument->verified_at->format('M d, Y H:i') : null,
            'rejection_reason' => $existingDocument ? $existingDocument->rejection_reason : null,
        ];
    }
    
    // Get document statistics
    $documentStats = $this->getDocumentStatistics($application, $productRequirements);
    
    // Get customer details for form
    $customers = Customer::all();
    
    return view('admin.applications.edit', compact(
        'application', 
        'loanProducts', 
        'currencies',
        'customers',
        'requirementsWithStatus',
        'documentStats',
        'isDraft'
    ));
}

    /**
     * Update the application details.
     */
/**
 * Update the application details.
 */
public function update(Request $request, $id)
{
    $application = Application::findOrFail($id);
    
    // Check permissions based on application status
    $isDraft = $application->status === 'draft';
    
    if ($isDraft) {
        // For draft applications, only allow edit if user created it or has special permissions
        if (!auth()->user()->hasPermission('applications.edit') && 
            $application->created_by !== auth()->id()) {
            abort(403, 'You do not have permission to edit this draft application.');
        }
    } else {
        // For non-draft applications, require specific permissions
        if (!auth()->user()->hasAnyPermission(['applications.edit', 'applications.approve', 'poweruser'])) {
            abort(403, 'You do not have permission to edit this application.');
        }
        
        // Additional restrictions for submitted/approved applications
        if ($application->status === 'approved' && !auth()->user()->hasPermission('applications.approve')) {
            abort(403, 'Only approvers can edit approved applications.');
        }
    }
    
    // Determine validation rules based on status
    $validationRules = [
        'loan_amount' => 'required|numeric|min:0',
        'interest_rate' => 'required|numeric|min:0',
        'loan_tenure' => 'required|integer|min:1',
        'payment_frequency' => 'required|in:one_off,daily,weekly,monthly,quarterly',
        'interest_method' => 'required|in:simple_interest,reducing_balance',
        'processing_fee' => 'nullable|numeric|min:0',
        'processing_fee_basis' => 'nullable|in:initial_amount,outstanding_balance',
        'adminfee' => 'nullable|numeric|min:0',
        'insurancefee' => 'nullable|numeric|min:0',
        'mobile_number' => 'nullable|string|max:20',
        'mobile_channel' => 'nullable|string|max:50',
        'application_date' => 'nullable|date|before_or_equal:today',
    ];
    
    // Only allow status change for draft applications or users with approve permission
    if ($isDraft || auth()->user()->hasPermission('applications.approve')) {
        $validationRules['status'] = 'required|in:draft,submitted,approved,rejected';
    }
    
    // For draft applications, allow changing customer and product
    if ($isDraft) {
        $validationRules['customer_id'] = 'required|exists:customer,id';
        $validationRules['product_id'] = 'required|exists:loantype,id';
        $validationRules['currency_id'] = 'required|exists:currencies,id';
    }
    
    // Only allow approval date for approvers
    if (auth()->user()->hasPermission('applications.approve')) {
        $validationRules['approval_date'] = 'nullable|date|before_or_equal:today';
    }
    
    // Validate the update request
    $validator = Validator::make($request->all(), $validationRules);
    
    if ($validator->fails()) {
        return redirect()->back()->withErrors($validator)->withInput();
    }
    
    $data = $request->all();
    
    // Handle customer/product changes for draft applications
    if ($isDraft) {
        // Check if product changed
        if ($request->has('product_id') && $request->product_id != $application->product_id) {
            // Product changed - log this change
            Log::info('Product changed for draft application', [
                'application_id' => $application->id,
                'old_product_id' => $application->product_id,
                'new_product_id' => $request->product_id,
                'user_id' => auth()->id(),
                'user_name' => auth()->user()->name
            ]);
        }
    }
    
    // Handle application date (with backdating support)
    if ($request->filled('application_date')) {
        $applicationDate = Carbon::parse($request->application_date);
        
        if ($isDraft) {
            // For drafts, update application_date field
            $data['application_date'] = $applicationDate;
        } else {
            // For submitted applications, update submitted_at
            $data['submitted_at'] = $applicationDate;
            
            // Log backdating activity for audit trail
            $originalDate = $application->submitted_at;
            if ($originalDate && $applicationDate->ne($originalDate)) {
                $daysChanged = $applicationDate->diffInDays($originalDate);
                Log::info('Application date updated', [
                    'application_id' => $application->id,
                    'original_date' => $originalDate->format('Y-m-d'),
                    'new_date' => $applicationDate->format('Y-m-d'),
                    'days_changed' => $daysChanged,
                    'user_id' => auth()->id(),
                    'user_name' => auth()->user()->name,
                    'direction' => $applicationDate < $originalDate ? 'backdated' : 'forward-dated'
                ]);
            }
        }
    } else {
        if (!$isDraft) {
            // Keep existing submitted_at if application_date is not provided for non-draft
            unset($data['submitted_at']);
        }
    }
    
    // Handle status transitions
    if ($request->has('status') && $request->status !== $application->status) {
        $oldStatus = $application->status;
        $newStatus = $request->status;
        
        // Log status change
        Log::info('Application status changed', [
            'application_id' => $application->id,
            'old_status' => $oldStatus,
            'new_status' => $newStatus,
            'user_id' => auth()->id(),
            'user_name' => auth()->user()->name
        ]);
        
        // Handle specific status transitions
        if ($newStatus === 'submitted' && $oldStatus === 'draft') {
            // Draft is being submitted for the first time
            if (!$request->filled('submitted_at') && !isset($data['submitted_at'])) {
                $data['submitted_at'] = now();
            }
            
            // Set reviewed_by if not set
            if (!$application->reviewed_by) {
                $data['reviewed_by'] = auth()->id();
            }
        }
        
        // Handle approval date
        if ($newStatus === 'approved') {
            if ($request->filled('approval_date')) {
                $approvalDate = Carbon::parse($request->approval_date);
                
                // Validate approval date is not before application date
                $applicationDate = $request->filled('application_date') 
                    ? Carbon::parse($request->application_date) 
                    : $application->submitted_at;
                    
                if ($applicationDate && $approvalDate->lt($applicationDate)) {
                    return redirect()->back()
                        ->withErrors(['approval_date' => 'Approval date cannot be before the application date.'])
                        ->withInput();
                }
                
                $data['approved_at'] = $approvalDate;
                
                // Log approval date changes for audit trail
                $originalApprovalDate = $application->approved_at;
                if ($originalApprovalDate && $approvalDate->ne($originalApprovalDate)) {
                    $daysChanged = $approvalDate->diffInDays($originalApprovalDate);
                    Log::info('Approval date updated', [
                        'application_id' => $application->id,
                        'original_date' => $originalApprovalDate->format('Y-m-d'),
                        'new_date' => $approvalDate->format('Y-m-d'),
                        'days_changed' => $daysChanged,
                        'user_id' => auth()->id(),
                        'user_name' => auth()->user()->name,
                        'direction' => $approvalDate < $originalApprovalDate ? 'backdated' : 'forward-dated'
                    ]);
                }
            } else {
                // If status is approved but no approval date provided, set to now
                $data['approved_at'] = now();
            }
            
            // Set reviewed_by if not set
            if (!$application->reviewed_by) {
                $data['reviewed_by'] = auth()->id();
            }
        } elseif ($newStatus === 'rejected') {
            // Set reviewed_at and reviewed_by for rejection
            $data['reviewed_at'] = now();
            if (!$application->reviewed_by) {
                $data['reviewed_by'] = auth()->id();
            }
            
            // Clear approval date if set
            $data['approved_at'] = null;
        } elseif ($newStatus === 'draft') {
            // If changing back to draft, clear submission/approval dates
            $data['submitted_at'] = null;
            $data['approved_at'] = null;
            $data['reviewed_at'] = null;
            $data['reviewed_by'] = null;
        } else {
            // For other statuses, clear approval date
            $data['approved_at'] = null;
        }
    } elseif ($request->status === 'approved' && $application->status === 'approved') {
        // If already approved and staying approved, handle approval date updates
        if ($request->filled('approval_date')) {
            $approvalDate = Carbon::parse($request->approval_date);
            
            // Validate approval date is not before application date
            $applicationDate = $request->filled('application_date') 
                ? Carbon::parse($request->application_date) 
                : $application->submitted_at;
                
            if ($applicationDate && $approvalDate->lt($applicationDate)) {
                return redirect()->back()
                    ->withErrors(['approval_date' => 'Approval date cannot be before the application date.'])
                    ->withInput();
            }
            
            $data['approved_at'] = $approvalDate;
            
            // Log approval date changes for audit trail
            $originalApprovalDate = $application->approved_at;
            if ($originalApprovalDate && $approvalDate->ne($originalApprovalDate)) {
                $daysChanged = $approvalDate->diffInDays($originalApprovalDate);
                Log::info('Approval date updated', [
                    'application_id' => $application->id,
                    'original_date' => $originalApprovalDate->format('Y-m-d'),
                    'new_date' => $approvalDate->format('Y-m-d'),
                    'days_changed' => $daysChanged,
                    'user_id' => auth()->id(),
                    'user_name' => auth()->user()->name,
                    'direction' => $approvalDate < $originalApprovalDate ? 'backdated' : 'forward-dated'
                ]);
            }
        }
    }
    
    // Recalculate installment if loan details changed
    if ($request->hasAny(['loan_amount', 'interest_rate', 'loan_tenure', 'payment_frequency', 'interest_method', 
                        'processing_fee', 'processing_fee_basis', 'adminfee', 'insurancefee'])) {
        
        // Extract parameters for calculation
        $amount = $request->loan_amount;
        $interestRate = $request->interest_rate;
        $tenure = $request->loan_tenure;
        $frequency = $request->payment_frequency;
        $interestMethod = $request->interest_method;
        $processingFeeRate = $request->processing_fee ?? $application->processing_fee;
        $processingFeeBasis = $request->processing_fee_basis ?? $application->processing_fee_basis;
        $adminFee = $request->adminfee ?? $application->adminfee;
        $insuranceFee = $request->insurancefee ?? $application->insurancefee;
        
        // Calculate installment using the controller's internal method
        if ($interestMethod === 'simple_interest') {
            $data['installment_amount'] = $this->calculateSimpleInterestInstallment(
                $amount, $interestRate, $tenure, $frequency, 
                $processingFeeRate, $processingFeeBasis
            );
        } else {
            $data['installment_amount'] = $this->calculateReducingBalanceInstallment(
                $amount, $interestRate, $tenure, $frequency,
                $processingFeeRate, $processingFeeBasis
            );
        }
    }
    
    $application->update($data);
    
    // Send notifications based on status change
    if ($application->wasChanged('status')) {
        Log::info('Application status updated', [
            'application_id' => $application->id,
            'old_status' => $application->getOriginal('status'),
            'new_status' => $application->status,
            'user_id' => auth()->id(),
            'user_name' => auth()->user()->name,
            'approval_date' => $application->approved_at ? $application->approved_at->format('Y-m-d') : null
        ]);
        
        // Send notifications based on new status
        if ($application->status === 'approved') {
            $this->sendApprovalNotifications($application);
        } elseif ($application->status === 'rejected') {
            $this->sendRejectionNotifications($application);
        } elseif ($application->status === 'submitted' && $application->getOriginal('status') === 'draft') {
            // Only send submission notifications when transitioning from draft to submitted
            $this->sendRequestNotifications($application);
        }
    }
    
    // Determine success message based on action
    $successMessage = 'Application updated successfully.';
    if ($application->wasChanged('status')) {
        if ($application->status === 'draft') {
            $successMessage = 'Application saved as draft.';
        } elseif ($application->status === 'submitted') {
            $successMessage = 'Application submitted successfully.';
        } elseif ($application->status === 'approved') {
            $successMessage = 'Application approved successfully.';
        } elseif ($application->status === 'rejected') {
            $successMessage = 'Application rejected.';
        }
    }
    
    // Determine redirect based on status
    if ($application->status === 'draft') {
        return redirect()->route('admin.applications.draft')->with('success', $successMessage);
    } elseif ($application->status === 'submitted') {
        return redirect()->route('admin.applications.pending')->with('success', $successMessage);
    } elseif ($application->status === 'approved') {
        return redirect()->route('admin.applications.approved')->with('success', $successMessage);
    } elseif ($application->status === 'rejected') {
        return redirect()->route('admin.applications.rejected')->with('success', $successMessage);
    } else {
        return redirect()->route('admin.applications.show', $id)->with('success', $successMessage);
    }
}

    /**
     * Process the loan disbursement by moving application data to the disbursed loans table.
     */
    public function processDisbursement(Request $request, $id)
    {
        // Permission check
        if (!auth()->user()->hasPermission('applications.approve')) {
            abort(403, 'Unauthorized disbursement attempt.');
        }

        // Validation for disbursement specific fields
        $validator = Validator::make($request->all(), [
            'disbursement_date' => 'required|date',
            'payment_account_id' => 'required|exists:cash_bank_accounts,cashBankId',
            'disbursement_notes' => 'nullable|string|max:500',
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        try {
            return DB::transaction(function () use ($request, $id) {
                // Load application with necessary relationships
                $application = Application::with(['customer', 'product'])->findOrFail($id);

                // Guard against invalid status or double disbursement
                if (!$application->isApproved() || $application->isDisbursed()) {
                    throw new \Exception('This application is not in a state to be disbursed.');
                }

                $disbursementDate = Carbon::parse($request->disbursement_date);
                $paymentMethod = $application->suggestedpaymentmethod;

                // 1. Get the Cash/Bank Account (funding source)
                $cashBankAccount = \App\Models\CashBankAccount::where('cashBankId', $request->payment_account_id)
                    ->where('companyid', auth()->user()->companyid ?? $application->company_id)
                    ->where('isActive', true)
                    ->first();
                
                if (!$cashBankAccount) {
                    throw new \Exception('Selected payment account is not available or inactive.');
                }
                
                // === BYPASSED: REMOVED BALANCE CHECK FOR DEVELOPMENT ===
                Log::warning('BYPASSED FUNDS CHECK - Disbursement proceeding without balance verification', [
                    'account_id' => $cashBankAccount->cashBankId,
                    'account_name' => $cashBankAccount->accountName,
                    'account_type' => $cashBankAccount->accountType,
                    'current_balance' => $cashBankAccount->current_balance,
                    'loan_amount' => $application->loan_amount,
                    'application_id' => $id,
                    'user_id' => auth()->id(),
                    'bypass_reason' => 'Development mode - balance check disabled'
                ]);
                
                // 2. Get the GL Account ID from the CashBankAccount
                $paymentAccountId = $cashBankAccount->glAccountId;
                if (!$paymentAccountId) {
                    throw new \Exception('Payment account is not linked to a GL account.');
                }
                
                // 3. Find Loan Receivable Account
                $disbursementMapping = DB::table('business_process_mappings')
                    ->where('nameofthebusinessprocess', 'Loan Disbursement')
                    ->where('companyid', $application->company_id)
                    ->first();
                
                $loanReceivableAccountId = $disbursementMapping ? $disbursementMapping->accountid : null;
                if (!$loanReceivableAccountId) {
                    throw new \Exception('Loan Disbursement GL mapping not found.');
                }
                
                $loanReceivableAccount = \App\Models\ChartOfAccount::where('id', $loanReceivableAccountId)
                    ->where('is_active', true)
                    ->first();
                    
                if (!$loanReceivableAccount) {
                    throw new \Exception('Loan Receivable account is not available or inactive.');
                }

                // === MOBILE MONEY DISBURSEMENT (LIPILA) ===
                if ($paymentMethod === 'Mobile') {
                    Log::info('=== MOBILE MONEY DISBURSEMENT STARTED ===');
                    
                    // Validate mobile money details - use mobile_number field
                    if (!$application->mobile_number) {
                        throw new \Exception('Mobile number is required for mobile money disbursement. Please ensure the customer has provided a mobile number.');
                    }

                    if (!$application->mobile_channel) {
                        throw new \Exception('Mobile channel (e.g., Airtel Money, MTN Mobile Money) is required for mobile money disbursement.');
                    }

                    // Validate that a mobile money account is selected as funding source
                    if ($cashBankAccount->accountType !== 'Mobile Money') {
                        throw new \Exception('For mobile money disbursements, please select a Mobile Money account as the funding source.');
                    }

                    // Clean and format mobile number for Lipila API
                    $mobileNumber = $this->formatMobileNumberForLipila($application->mobile_number);
                    
                    Log::info('Mobile number formatted for Lipila:', [
                        'mobile_number' => $application->mobile_number,
                        'formatted' => $mobileNumber,
                        'mobile_channel' => $application->mobile_channel
                    ]);

                    // 5. Initiate Lipila disbursement
                    $lipilaResponse = null;
                    $lipilaData = [];
                    $lipilaError = null;
                    
                    try {
                        Log::info('Calling LipilaPaymentService::disburse()');
                        
                        $lipilaResponse = $this->lipilaPaymentService->disburse([
                            'amount' => $application->loan_amount,
                            'narration' => "Disbursement",
                            'account_number' => $mobileNumber,
                            'currency' => $application->currency->code ?? 'ZMW',
                        ], $application->company_id, auth()->id());

                        Log::info('Lipila API Response Received:', [
                            'successful' => $lipilaResponse ? $lipilaResponse->successful() : false,
                            'status' => $lipilaResponse ? $lipilaResponse->status() : 'NO_RESPONSE'
                        ]);

                        if ($lipilaResponse && $lipilaResponse->successful()) {
                            $lipilaData = $lipilaResponse->json();
                            Log::info('Lipila API Success:', $lipilaData);
                        } else {
                            $errorData = $lipilaResponse ? $lipilaResponse->json() : ['message' => 'No response from Lipila API'];
                            $lipilaError = $errorData['message'] ?? 'Lipila API request failed';
                            Log::error('Lipila API Failed:', $errorData);
                        }
                    } catch (\Exception $e) {
                        $lipilaError = $e->getMessage();
                        Log::error('Lipila Service Exception:', [
                            'message' => $lipilaError,
                            'trace' => $e->getTraceAsString()
                        ]);
                    }

                    if ($lipilaError) {
                        throw new \Exception('Lipila disbursement failed: ' . $lipilaError);
                    }

                    // Determine first payment date
                    $firstPaymentDate = $this->calculateFirstPaymentDate(
                        $disbursementDate, 
                        $application->payment_frequency, 
                        $application->loan_tenure
                    );

                    // Determine Maturity Date
                    $loanTenure = (int)$application->loan_tenure;
                    $maturityDate = ($application->payment_frequency === 'one_off') 
                        ? $firstPaymentDate->copy()
                        : $disbursementDate->copy()->addMonths($loanTenure);

                    // Get reference ID from response
                    $lipilaReferenceId = $lipilaData['referenceId'] ?? '';
                    $lipilaStatus = $lipilaData['status'] ?? '';

                    // 6. Create the Disbursed Loan record
                    $disbursedLoan = DisbursedLoan::create([
                        'application_id'      => $application->id,
                        'loannumber'          => DisbursedLoan::generateLoanNumber($application->company_id),
                        'customerid'          => $application->customer_id,
                        'loantypeid'          => $application->product_id,
                        'amount'              => $application->loan_amount,
                        'interestrate'        => $application->interest_rate,
                        'interest_method'     => $application->interest_method,
                        'loanterm'            => $loanTenure,
                        'paymentfrequency'    => $application->payment_frequency,
                        'installmentamount'   => $application->installment_amount,
                        
                        // Fees and Penalties
                        'processing_fee'      => $application->processing_fee,
                        'processing_fee_basis'=> $application->processing_fee_basis,
                        'adminfee'            => $application->adminfee,
                        'insurancefee'        => $application->insurancefee,
                        'penalty_rate'        => $application->penalty_rate,
                        'penalty_basis'       => $application->penalty_basis,

                        // Disbursement Information
                        'disburseddate'       => $disbursementDate,
                        'payment_account_id'  => $cashBankAccount->cashBankId,
                        'payment_account_type' => $cashBankAccount->accountType,
                        'disbursement_method' => 'Mobile',
                        'mobile_channel'      => $application->mobile_channel,
                        'mobile_number'       => $application->mobile_number,
                        'disbursement_notes'  => $request->disbursement_notes . " | Lipila Reference: " . $lipilaReferenceId . " | Status: " . $lipilaStatus,
                        'disbursement_reference' => $lipilaReferenceId,
                        'lipila_status' => $lipilaStatus,
                        'lipila_identifier' => $lipilaData['identifier'] ?? null,
                        'lipila_response' => json_encode($lipilaData),

                        // Financial Meta
                        'status'              => 'active',
                        'firstpaymentdate'    => $firstPaymentDate,
                        'maturitydate'        => $maturityDate,
                        'principalbalance'    => $application->loan_amount,
                        'interestbalance'     => 0.00,
                        'cumulative_interest_charged' => 0.00,
                        'totalbalance'        => $application->loan_amount,
                        'branchid'            => $application->branch_id,
                        'companyid'           => $application->company_id,
                        'disbursedby'         => auth()->id(),
                        'createdby'           => auth()->id(),
                        'updatedby'           => auth()->id(),
                    ]);

                    // 7. Save to payment_disbursements table
                    try {
                        $disbursementRecord = PaymentDisbursement::create([
                            'company_id' => $application->company_id,
                            'created_by' => auth()->id(),
                            'reference_id' => $lipilaReferenceId,
                            'identifier' => $lipilaData['identifier'] ?? $lipilaReferenceId,
                            'amount' => $application->loan_amount,
                            'currency' => $application->currency->code ?? 'ZMW',
                            'account_number' => $mobileNumber,
                            'status' => $lipilaStatus,
                            'payment_type' => 'mobile_money',
                            'narration' => "Loan Disbursement - {$application->application_number} - {$disbursedLoan->loannumber}",
                            'ip_address' => request()->ip(),
                            'api_created_at' => now(),
                        ]);
                        
                        Log::info('Payment disbursement record created', [
                            'disbursement_id' => $disbursementRecord->id,
                            'reference_id' => $lipilaReferenceId,
                            'loan_number' => $disbursedLoan->loannumber
                        ]);
                    } catch (\Exception $e) {
                        Log::error('Failed to create payment disbursement record', [
                            'error' => $e->getMessage(),
                            'reference_id' => $lipilaReferenceId
                        ]);
                        // Don't throw error, just log it since the disbursement itself was successful
                    }

                    // 8. Generate Payment Schedule
                    $installmentsCount = $this->paymentScheduleService->generatePaymentSchedule($disbursedLoan);

                    // 9. GL Posting (Double Entry) for mobile money
                    $description = "Loan disbursement via " . $cashBankAccount->accountName . " - {$disbursedLoan->loannumber} for {$application->customer->first_name} {$application->customer->surname}";

                    // DEBIT: Loan Receivable (Asset Account - Increases)
                    DB::table('generalledgerentries')->insert([
                        'entrydate' => $disbursementDate->format('Y-m-d'),
                        'description' => $description,
                        'accountid' => $loanReceivableAccountId,
                        'referencedocument' => 'Loan Disbursement - Mobile Money',
                        'documentno' => $disbursedLoan->loannumber,
                        'entrytype' => 'debit',
                        'amount' => $application->loan_amount,
                        'createdby' => auth()->id(),
                        'companyid' => $application->company_id,
                        'branchid' => $application->branch_id,
                        'created_at' => now(),
                        'updated_at' => now(),
                        'transtype' => 'loan_disbursement_mobile',
                        'loanid' => $disbursedLoan->loanid,
                    ]);

                    // CREDIT: Mobile Money GL Account (Asset Account - Decreases)
                    DB::table('generalledgerentries')->insert([
                        'entrydate' => $disbursementDate->format('Y-m-d'),
                        'description' => $description,
                        'accountid' => $paymentAccountId,
                        'referencedocument' => 'Loan Disbursement - Mobile Money',
                        'documentno' => $disbursedLoan->loannumber,
                        'entrytype' => 'credit',
                        'amount' => $application->loan_amount * -1,
                        'createdby' => auth()->id(),
                        'companyid' => $application->company_id,
                        'branchid' => $application->branch_id,
                        'created_at' => now(),
                        'updated_at' => now(),
                        'transtype' => 'loan_disbursement_mobile',
                        'loanid' => $disbursedLoan->loanid,
                    ]);

                    // 10. Update Account Balances (with bypassed check)
                    $newBalance = $cashBankAccount->current_balance - $application->loan_amount;
                    $cashBankAccount->update(['current_balance' => $newBalance]);
                    
                    Log::warning('UPDATED BALANCE WITH POTENTIAL NEGATIVE - Mobile Money Account', [
                        'account_id' => $cashBankAccount->cashBankId,
                        'account_name' => $cashBankAccount->accountName,
                        'old_balance' => $cashBankAccount->current_balance + $application->loan_amount,
                        'new_balance' => $newBalance,
                        'amount_deducted' => $application->loan_amount,
                        'loan_number' => $disbursedLoan->loannumber
                    ]);
                    
                    // Update GL account balances
                    DB::table('chart_of_accounts')
                        ->where('id', $paymentAccountId)
                        ->decrement('current_balance', $application->loan_amount);

                    DB::table('chart_of_accounts')
                        ->where('id', $loanReceivableAccountId)
                        ->increment('current_balance', $application->loan_amount);
                        
                    // 11. Create Cash Book entry for the disbursement
                    $balanceBefore = $cashBankAccount->current_balance + $application->loan_amount;
                    $balanceAfter = $cashBankAccount->current_balance;

                    \App\Models\CashBook::create([
                        'companyid' => $application->company_id,
                        'branchid' => $application->branch_id,
                        'account_id' => $cashBankAccount->cashBankId,
                        'transaction_date' => $disbursementDate,
                        'transaction_type' => 'Loan Disbursement',
                        'reference_number' => $disbursedLoan->loannumber,
                        'transfer_id' => null,
                        'description' => $description,
                        'amount' => -$application->loan_amount,
                        'balance_before' => $balanceBefore,
                        'balance_after' => $balanceAfter,
                        'currency' => $cashBankAccount->currency,
                        'exchange_rate' => 1.0000,
                        'status' => 'Posted',
                        'related_transaction_type' => 'Loan Disbursement',
                        'related_transaction_id' => $disbursedLoan->loanid,
                        'created_by' => auth()->id(),
                    ]);

                    // 12. Update original Application status
                    $application->update([
                        'status'            => 'disbursed',
                        'disbursed_at'      => $disbursementDate,
                        'disbursed_by'      => auth()->id(),
                        'disbursement_notes' => $request->disbursement_notes . " | Disbursed via " . $cashBankAccount->accountName . " - Lipila Reference: " . $lipilaReferenceId,
                        'loan_id'           => $disbursedLoan->loannumber,
                        'cash_bank_account_id' => $cashBankAccount->cashBankId,
                    ]);

                    // 13. Send Notifications
                    $this->sendMobileDisbursementNotifications($application, $disbursedLoan, $lipilaData, $cashBankAccount->accountName);

                    // Log success
                    Log::info('✅ Mobile money disbursement completed via ' . $cashBankAccount->accountName, [
                        'loan_number' => $disbursedLoan->loannumber,
                        'application_id' => $application->id,
                        'lipila_reference' => $lipilaReferenceId,
                        'mobile_channel' => $application->mobile_channel,
                        'funding_account' => $cashBankAccount->accountName,
                        'funding_account_id' => $cashBankAccount->cashBankId,
                        'amount' => $application->loan_amount,
                        'status' => $lipilaStatus,
                        'final_balance' => $cashBankAccount->current_balance,
                        'balance_check_bypassed' => true,
                    ]);

                    $successMessage = 'Mobile money disbursement completed successfully via ' . $cashBankAccount->accountName . '. Lipila Reference: ' . $lipilaReferenceId;

                    return redirect()->route('admin.applications.approved')->with('success', $successMessage);

                } 
                // === TRADITIONAL DISBURSEMENT (BANK/CASH) ===
                else {
                    Log::info('=== TRADITIONAL DISBURSEMENT STARTED ===');
                    
                    // Validate that a Bank or Cash account is selected
                    if ($paymentMethod === 'Bank' && $cashBankAccount->accountType !== 'Bank') {
                        throw new \Exception('For bank disbursements, please select a Bank account as the funding source.');
                    }
                    
                    if ($paymentMethod === 'Cash' && $cashBankAccount->accountType !== 'Cash') {
                        throw new \Exception('For cash disbursements, please select a Cash account as the funding source.');
                    }

                    // Determine first payment date
                    $firstPaymentDate = $this->calculateFirstPaymentDate(
                        $disbursementDate, 
                        $application->payment_frequency, 
                        $application->loan_tenure
                    );

                    // Determine Maturity Date
                    $loanTenure = (int)$application->loan_tenure;
                    $maturityDate = ($application->payment_frequency === 'one_off') 
                        ? $firstPaymentDate->copy()
                        : $disbursementDate->copy()->addMonths($loanTenure);

                    // 5. Create the Disbursed Loan record
                    $disbursedLoan = DisbursedLoan::create([
                        'application_id'      => $application->id,
                        'loannumber'          => DisbursedLoan::generateLoanNumber($application->company_id),
                        'customerid'          => $application->customer_id,
                        'loantypeid'          => $application->product_id,
                        'amount'              => $application->loan_amount,
                        'interestrate'        => $application->interest_rate,
                        'interest_method'     => $application->interest_method,
                        'loanterm'            => $loanTenure,
                        'paymentfrequency'    => $application->payment_frequency,
                        'installmentamount'   => $application->installment_amount,
                        
                        // Fees and Penalties
                        'processing_fee'      => $application->processing_fee,
                        'processing_fee_basis'=> $application->processing_fee_basis,
                        'adminfee'            => $application->adminfee,
                        'insurancefee'        => $application->insurancefee,
                        'penalty_rate'        => $application->penalty_rate,
                        'penalty_basis'       => $application->penalty_basis,

                        // Disbursement Information
                        'disburseddate'       => $disbursementDate,
                        'payment_account_id'  => $cashBankAccount->cashBankId,
                        'payment_account_type' => $cashBankAccount->accountType,
                        'disbursement_method' => $application->suggestedpaymentmethod,
                        'bank_name'           => $application->bank_name,
                        'account_holder'      => $application->account_holder,
                        'account_number'      => $application->account_number,
                        'bank_branch'         => $application->bank_branch,
                        'swift_code'          => $application->swift_code,
                        'mobile_channel'      => $application->mobile_channel,
                        'mobile_number'       => $application->mobile_number,
                        'disbursement_notes'  => $request->disbursement_notes,

                        // Financial Meta
                        'status'              => 'active',
                        'firstpaymentdate'    => $firstPaymentDate,
                        'maturitydate'        => $maturityDate,
                        'principalbalance'    => $application->loan_amount,
                        'interestbalance'     => 0.00,
                        'cumulative_interest_charged' => 0.00,
                        'totalbalance'        => $application->loan_amount,
                        'branchid'            => $application->branch_id,
                        'companyid'           => $application->company_id,
                        'disbursedby'         => auth()->id(),
                        'createdby'           => auth()->id(),
                        'updatedby'           => auth()->id(),
                    ]);

                    // 6. Save to payment_disbursements table for traditional disbursements
                    try {
                        $disbursementRecord = PaymentDisbursement::create([
                            'company_id' => $application->company_id,
                            'created_by' => auth()->id(),
                            'reference_id' => $disbursedLoan->loannumber,
                            'identifier' => 'TRAD-' . $disbursedLoan->loannumber,
                            'amount' => $application->loan_amount,
                            'currency' => $application->currency->code ?? 'ZMW',
                            'account_number' => $application->account_number ?? $application->mobile_number ?? 'N/A',
                            'status' => 'completed',
                            'payment_type' => strtolower($application->suggestedpaymentmethod),
                            'narration' => "Loan Disbursement - {$application->application_number} - {$disbursedLoan->loannumber}",
                            'ip_address' => request()->ip(),
                            'api_created_at' => now(),
                        ]);
                        
                        Log::info('Traditional payment disbursement record created', [
                            'disbursement_id' => $disbursementRecord->id,
                            'loan_number' => $disbursedLoan->loannumber
                        ]);
                    } catch (\Exception $e) {
                        Log::error('Failed to create traditional payment disbursement record', [
                            'error' => $e->getMessage(),
                            'loan_number' => $disbursedLoan->loannumber
                        ]);
                    }

                    // 7. Generate Payment Schedule using your service
                    $installmentsCount = $this->paymentScheduleService->generatePaymentSchedule($disbursedLoan);

                    // 8. GL Posting (Double Entry)
                    $description = "Loan disbursement - {$disbursedLoan->loannumber} for {$application->customer->first_name} {$application->customer->surname}";

                    // DEBIT: Loan Receivable (Asset Account - Increases)
                    DB::table('generalledgerentries')->insert([
                        'entrydate' => $disbursementDate->format('Y-m-d'),
                        'description' => $description,
                        'accountid' => $loanReceivableAccountId,
                        'referencedocument' => 'Loan Disbursement',
                        'documentno' => $disbursedLoan->loannumber,
                        'entrytype' => 'debit',
                        'amount' => $application->loan_amount,
                        'createdby' => auth()->id(),
                        'companyid' => $application->company_id,
                        'branchid' => $application->branch_id,
                        'created_at' => now(),
                        'updated_at' => now(),
                        'transtype' => 'loan_disbursement',
                        'loanid' => $disbursedLoan->loanid,
                    ]);

                    // CREDIT: Cash/Bank Account (Asset Account - Decreases)
                    DB::table('generalledgerentries')->insert([
                        'entrydate' => $disbursementDate->format('Y-m-d'),
                        'description' => $description,
                        'accountid' => $paymentAccountId,
                        'referencedocument' => 'Loan Disbursement',
                        'documentno' => $disbursedLoan->loannumber,
                        'entrytype' => 'credit',
                        'amount' => $application->loan_amount * -1,
                        'createdby' => auth()->id(),
                        'companyid' => $application->company_id,
                        'branchid' => $application->branch_id,
                        'created_at' => now(),
                        'updated_at' => now(),
                        'transtype' => 'loan_disbursement',
                        'loanid' => $disbursedLoan->loanid,
                    ]);

                    // 9. Update Account Balances (with bypassed check)
                    $newBalance = $cashBankAccount->current_balance - $application->loan_amount;
                    $cashBankAccount->update(['current_balance' => $newBalance]);
                    
                    Log::warning('UPDATED BALANCE WITH POTENTIAL NEGATIVE - Traditional Account', [
                        'account_id' => $cashBankAccount->cashBankId,
                        'account_name' => $cashBankAccount->accountName,
                        'account_type' => $cashBankAccount->accountType,
                        'old_balance' => $cashBankAccount->current_balance + $application->loan_amount,
                        'new_balance' => $newBalance,
                        'amount_deducted' => $application->loan_amount,
                        'loan_number' => $disbursedLoan->loannumber
                    ]);
                    
                    // Update GL account balances
                    DB::table('chart_of_accounts')
                        ->where('id', $paymentAccountId)
                        ->decrement('current_balance', $application->loan_amount);

                    DB::table('chart_of_accounts')
                        ->where('id', $loanReceivableAccountId)
                        ->increment('current_balance', $application->loan_amount);
                        
                    // 10. Create Cash Book entry for the disbursement
                    $balanceBefore = $cashBankAccount->current_balance + $application->loan_amount;
                    $balanceAfter = $balanceBefore - $application->loan_amount;

                    \App\Models\CashBook::create([
                        'companyid' => $application->company_id,
                        'branchid' => $application->branch_id,
                        'account_id' => $cashBankAccount->cashBankId,
                        'transaction_date' => $disbursementDate,
                        'transaction_type' => 'Loan Disbursement',
                        'reference_number' => $disbursedLoan->loannumber,
                        'transfer_id' => null,
                        'description' => $description,
                        'amount' => -$application->loan_amount,
                        'balance_before' => $balanceBefore,
                        'balance_after' => $balanceAfter,
                        'currency' => $cashBankAccount->currency,
                        'exchange_rate' => 1.0000,
                        'status' => 'Posted',
                        'related_transaction_type' => 'Loan Disbursement',
                        'related_transaction_id' => $disbursedLoan->loanid,
                        'created_by' => auth()->id(),
                    ]);

                    // 11. Update original Application status
                    $application->update([
                        'status'            => 'disbursed',
                        'disbursed_at'      => $disbursementDate,
                        'disbursed_by'      => auth()->id(),
                        'disbursement_notes' => $request->disbursement_notes,
                        'loan_id'           => $disbursedLoan->loannumber,
                        'cash_bank_account_id' => $cashBankAccount->cashBankId,
                    ]);

                    // 12. Send Notifications
                    $this->sendDisbursementNotifications($application, $disbursedLoan);

                    // Log success
                    Log::info('✅ Traditional disbursement completed successfully', [
                        'loan_number' => $disbursedLoan->loannumber,
                        'application_id' => $application->id,
                        'method' => $application->suggestedpaymentmethod,
                        'funding_account' => $cashBankAccount->accountName,
                        'funding_account_id' => $cashBankAccount->cashBankId,
                        'amount' => $application->loan_amount,
                        'final_balance' => $cashBankAccount->current_balance,
                        'balance_check_bypassed' => true
                    ]);

                    $successMessage = 'Loan has been successfully disbursed with GL entries posted.';

                    return redirect()->route('admin.applications.approved')->with('success', $successMessage);
                }
            });
        } catch (Throwable $e) {
            Log::error('=== DISBURSEMENT CRITICAL ERROR ===');
            Log::error('Error Message:', ['message' => $e->getMessage()]);
            Log::error('Application ID:', ['id' => $id]);
            Log::error('User ID:', ['user_id' => auth()->id()]);
            Log::error('Trace:', ['trace' => $e->getTraceAsString()]);
            Log::error('=== DISBURSEMENT ERROR END ===');
            
            return redirect()->back()->with('error', 'Critical Error during disbursement: ' . $e->getMessage());
        }
    }

    /**
     * Format mobile number for Lipila API
     */
    private function formatMobileNumberForLipila($number)
    {
        // Remove any non-digit characters (spaces, plus signs, dashes, etc.)
        $cleaned = preg_replace('/[^0-9]/', '', $number);
        
        // Case 1: 10 digits starting with 0 (0976379025 -> 260976379025)
        if (strlen($cleaned) === 10 && str_starts_with($cleaned, '0')) {
            return '260' . substr($cleaned, 1);
        }
        
        // Case 2: 9 digits (976379025 -> 260976379025)
        if (strlen($cleaned) === 9) {
            return '260' . $cleaned;
        }
        
        // Case 3: 12 digits starting with 260 (260976379025 -> keep as is)
        if (strlen($cleaned) === 12 && str_starts_with($cleaned, '260')) {
            return $cleaned;
        }
        
        // Case 4: 12 digits starting with 0 and then 260 (0260976379025 -> 260976379025)
        if (strlen($cleaned) === 12 && str_starts_with($cleaned, '0') && substr($cleaned, 1, 3) === '260') {
            return substr($cleaned, 1);
        }
        
        // Case 5: Contains 260 somewhere but not at start
        if (str_contains($cleaned, '260')) {
            $pos = strpos($cleaned, '260');
            return substr($cleaned, $pos);
        }
        
        // Fallback: Try to create a valid format
        $withoutLeadingZeros = ltrim($cleaned, '0');
        
        // If after removing zeros we have 9 digits, add 260
        if (strlen($withoutLeadingZeros) === 9) {
            return '260' . $withoutLeadingZeros;
        }
        
        // Last resort: Return as-is
        Log::warning('Unusual mobile number format, using as-is:', [
            'original' => $number,
            'cleaned' => $cleaned
        ]);
        
        return $cleaned;
    }

    /**
     * Get customer details for application form
     */
    public function getCustomerDetails(Request $request, $id)
    {
        $customer = Customer::find($id);
        if (!$customer) return response()->json(['error' => 'Not found'], 404);
        return response()->json([
            'id' => $customer->id, 'title' => $customer->title, 'first_name' => $customer->first_name, 
            'middle_name' => $customer->middle_name, 'surname' => $customer->surname, 
            'dob' => $customer->dob ? Carbon::parse($customer->dob)->format('Y-m-d') : '',
            'nrc_no' => $customer->number, 'email' => $customer->email, 'phone' => $customer->phone,
            'office_telephone' => $customer->office_phone, 'physical_address' => $customer->address,
            'postal_address' => $customer->postal_address, 'town' => $customer->town, 
            'province' => $customer->province, 'customer_number' => $customer->customer_number,
        ]);
    }

    /**
     * View payment schedule for application
     */
    public function viewSchedule($id)
    {
        $application = Application::with(['customer', 'product', 'currency'])->findOrFail($id);
        
        // Debug: Log application data
        Log::info('Viewing Schedule for Application:', [
            'id' => $application->id,
            'application_number' => $application->application_number
        ]);
        
        $scheduleData = $this->paymentScheduleService->generateAmortizationScheduleForApplication($application);
        
        // Debug: Log schedule data
        Log::info('Schedule Data for View:', [
            'count' => count($scheduleData),
            'data' => $scheduleData
        ]);
        
        return view('admin.applications.schedule', compact('application', 'scheduleData'));
    }

    /**
     * Generate PDF schedule
     */
    public function generateSchedulePdf($id)
    {
        try {
            $application = Application::with(['customer', 'product', 'currency'])->findOrFail($id);
            
            $scheduleData = $this->paymentScheduleService->generateAmortizationScheduleForApplication($application);
            
            // Create PDF with A4 size
            $pdf = new TCPDF('P', 'mm', 'A4', true, 'UTF-8', false);
            
            // Set document information
            $pdf->SetCreator('IMMIA Finance');
            $pdf->SetAuthor('IMMIA Finance');
            $pdf->SetTitle('Loan Schedule - ' . $application->application_number);
            $pdf->SetSubject('Loan Payment Schedule');
            
            // Remove default header/footer
            $pdf->setPrintHeader(false);
            $pdf->setPrintFooter(false);
            
            // Set margins
            $pdf->SetMargins(15, 15, 15);
            $pdf->SetAutoPageBreak(true, 15);
            
            // Add a page
            $pdf->AddPage();
            
            // Get the HTML content
            $html = $this->renderScheduleHtml($application, $scheduleData);
            
            // Write HTML content
            $pdf->writeHTML($html, true, false, true, false, '');
            
            // Output the PDF
            $filename = 'Loan_Schedule_' . $application->application_number . '.pdf';
            
            return Response::make($pdf->Output($filename, 'I'), 200, [
                'Content-Type' => 'application/pdf',
                'Content-Disposition' => 'inline; filename="' . $filename . '"'
            ]);
            
        } catch (\Exception $e) {
            Log::error('PDF Generation Error: ' . $e->getMessage(), [
                'trace' => $e->getTraceAsString()
            ]);
            
            // Return a simple error PDF
            $pdf = new TCPDF('P', 'mm', 'A4', true, 'UTF-8', false);
            $pdf->setPrintHeader(false);
            $pdf->setPrintFooter(false);
            $pdf->AddPage();
            
            $errorHtml = '<h1>Error Generating PDF</h1>
                         <p>There was an error generating the payment schedule PDF.</p>
                         <p>Error: ' . htmlspecialchars($e->getMessage()) . '</p>
                         <p>Please contact support.</p>';
            
            $pdf->writeHTML($errorHtml, true, false, true, false, '');
            
            return Response::make($pdf->Output('error.pdf', 'I'), 200, [
                'Content-Type' => 'application/pdf'
            ]);
        }
    }

    /**
     * Approve an application
     */
    public function approve($id) {
        $app = Application::findOrFail($id);
        $app->update(['status' => 'approved', 'approved_at' => now(), 'reviewed_by' => auth()->id()]);
        $this->sendApprovalNotifications($app);
        return redirect()->back()->with('success', 'Approved.');
    }

    /**
     * Reject an application
     */
    public function reject($id) {
        $app = Application::findOrFail($id);
        $app->update(['status' => 'rejected', 'reviewed_at' => now(), 'reviewed_by' => auth()->id()]);
        $this->sendRejectionNotifications($app);
        return redirect()->back()->with('success', 'Rejected.');
    }

    /**
     * Calculate installment amount based on loan parameters
     */
    public function calculateInstallment(Request $request)
    {
        try {
            $data = $request->validate([
                'loan_amount' => 'required|numeric|min:0',
                'interest_rate' => 'required|numeric|min:0',
                'loan_tenure' => 'required|integer|min:1',
                'payment_frequency' => 'required|in:one_off,daily,weekly,monthly,quarterly',
                'interest_method' => 'required|in:simple_interest,reducing_balance',
                'processing_fee' => 'nullable|numeric|min:0',
                'processing_fee_basis' => 'nullable|in:initial_amount,outstanding_balance',
                'adminfee' => 'nullable|numeric|min:0',
                'insurancefee' => 'nullable|numeric|min:0',
            ]);

            $amount = $data['loan_amount'];
            $interestRate = $data['interest_rate'];
            $tenure = $data['loan_tenure'];
            $frequency = $data['payment_frequency'];
            $interestMethod = $data['interest_method'];
            $processingFeeRate = $data['processing_fee'] ?? 0;
            $processingFeeBasis = $data['processing_fee_basis'] ?? 'initial_amount';
            $adminFee = $data['adminfee'] ?? 0;
            $insuranceFee = $data['insurancefee'] ?? 0;

            $installment = 0;
            
            if ($amount <= 0 || $tenure <= 0) {
                return response()->json([
                    'success' => true,
                    'installment' => '0.00',
                    'installment_raw' => 0,
                    'breakdown' => $this->getEmptyBreakdown()
                ]);
            }

            // Calculate fixed fees (admin + insurance)
            $fixedFees = $adminFee + $insuranceFee;

            if ($interestMethod === 'simple_interest') {
                // SIMPLE INTEREST METHOD
                $installment = $this->calculateSimpleInterestInstallment(
                    $amount, $interestRate, $tenure, $frequency, 
                    $processingFeeRate, $processingFeeBasis, $fixedFees
                );
            } else {
                // REDUCING BALANCE METHOD
                $installment = $this->calculateReducingBalanceInstallment(
                    $amount, $interestRate, $tenure, $frequency,
                    $processingFeeRate, $processingFeeBasis, $fixedFees
                );
            }

            // Calculate processing fee amount for breakdown
            $processingFeeAmount = $this->calculateProcessingFeeAmount(
                $amount, $processingFeeRate, $processingFeeBasis, 
                $interestMethod, $frequency, $tenure
            );

            return response()->json([
                'success' => true,
                'installment' => number_format(round($installment, 2), 2),
                'installment_raw' => round($installment, 2),
                'breakdown' => $this->getBreakdown(
                    $amount, $processingFeeAmount, $adminFee, $insuranceFee, 
                    $processingFeeRate, $processingFeeBasis, $installment
                )
            ]);
            
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Calculation error: ' . $e->getMessage()
            ], 400);
        }
    }

    /**
     * Calculate installment for Simple Interest method
     */
    private function calculateSimpleInterestInstallment($amount, $interestRate, $tenure, $frequency, 
                                                    $processingFeeRate, $processingFeeBasis)
    {
        if ($frequency === 'one_off') {
            // One-off payment
            $totalRate = $interestRate;
            
            // Add processing fee to rate if based on initial amount
            if ($processingFeeRate > 0 && $processingFeeBasis === 'initial_amount') {
                $totalRate += $processingFeeRate;
            }
            
            // For one_off, interest rate is applied per day
            $totalInterest = $amount * ($totalRate / 100) * $tenure;
            return $amount + $totalInterest;
        } else {
            // For regular installments with simple interest
            
            // Calculate number of periods
            $n = $this->getNumberOfPeriods($frequency, $tenure);
            
            // Interest per period
            $interestPerPeriod = $amount * ($interestRate / 100);
            
            // Principal repayment per period
            $principalPerPeriod = $amount / max($n, 1);
            
            // Processing fee per period
            $processingFeePerPeriod = 0;
            if ($processingFeeRate > 0) {
                if ($processingFeeBasis === 'initial_amount') {
                    $processingFeePerPeriod = $amount * ($processingFeeRate / 100);
                } else {
                    // For outstanding_balance basis in simple interest
                    $processingFeePerPeriod = $amount * ($processingFeeRate / 100);
                }
            }
            
            return $principalPerPeriod + $interestPerPeriod + $processingFeePerPeriod;
        }
    }

    /**
     * Calculate installment for Reducing Balance method
     */
    private function calculateReducingBalanceInstallment($amount, $interestRate, $tenure, $frequency,
                                                     $processingFeeRate, $processingFeeBasis)
    {
        if ($frequency === 'one_off') {
            // One-off payment
            $totalRate = $interestRate;
            
            // Add processing fee to rate if based on initial amount
            if ($processingFeeRate > 0 && $processingFeeBasis === 'initial_amount') {
                $totalRate += $processingFeeRate;
            }
            
            // For one_off, interest rate is applied per day
            $totalInterest = $amount * ($totalRate / 100) * $tenure;
            return $amount + $totalInterest;
        } else {
            // For reducing balance with regular installments
            $periodRate = $interestRate / 100;
            $n = $this->getNumberOfPeriods($frequency, $tenure);
            
            // Add processing fee to rate if based on outstanding balance
            if ($processingFeeRate > 0 && $processingFeeBasis === 'outstanding_balance') {
                $periodRate += ($processingFeeRate / 100);
            }
            
            if ($periodRate > 0) {
                $power = pow(1 + $periodRate, $n);
                $installment = ($amount * $periodRate * $power) / ($power - 1);
                
                // Add processing fee if based on initial amount
                if ($processingFeeRate > 0 && $processingFeeBasis === 'initial_amount') {
                    $processingFeePerPeriod = $amount * ($processingFeeRate / 100);
                    $installment += $processingFeePerPeriod;
                }
                
                return $installment;
            } else {
                // No interest or fees - just divide principal by periods
                return $amount / max($n, 1);
            }
        }
    }

    /**
     * Calculate processing fee amount based on basis
     */
    private function calculateProcessingFeeAmount($amount, $processingFeeRate, $processingFeeBasis, 
                                                 $interestMethod, $frequency, $tenure)
    {
        if ($processingFeeRate <= 0) {
            return 0;
        }
        
        if ($processingFeeBasis === 'initial_amount') {
            return $amount * ($processingFeeRate / 100);
        } else {
            if ($interestMethod === 'simple_interest') {
                return $amount * ($processingFeeRate / 100) * ($tenure / 365);
            } else {
                $periodsPerYear = $this->getPeriodsPerYear($frequency);
                $periodRate = ($processingFeeRate / 100) / $periodsPerYear;
                $n = $this->getNumberOfPeriods($frequency, $tenure);
                
                if ($periodRate > 0) {
                    $installmentContribution = ($amount * $periodRate) / (1 - pow(1 + $periodRate, -max($n, 1)));
                    return $installmentContribution * $n;
                } else {
                    return 0;
                }
            }
        }
    }

    /**
     * Get periods per year based on frequency
     */
    private function getPeriodsPerYear($frequency)
    {
        return match($frequency) {
            'weekly' => 52,
            'daily' => 365,
            'quarterly' => 4,
            default => 12,
        };
    }

    /**
     * Get number of periods based on frequency and tenure
     */
    private function getNumberOfPeriods($frequency, $tenure)
    {
        return match($frequency) {
            'weekly' => $tenure * 4.33,
            'daily' => $tenure * 30,
            'quarterly' => $tenure / 3,
            default => $tenure,
        };
    }

    /**
     * Get breakdown data for response
     */
    private function getBreakdown($amount, $processingFeeAmount, $adminFee, $insuranceFee, 
                                  $processingFeeRate, $processingFeeBasis, $installment)
    {
        $totalFees = $processingFeeAmount + $adminFee + $insuranceFee;
        $totalAmount = $amount + $totalFees;
        
        return [
            'principal' => round($amount, 2),
            'processing_fee_rate' => round($processingFeeRate, 2),
            'processing_fee_basis' => $processingFeeBasis,
            'processing_fee_amount' => round($processingFeeAmount, 2),
            'admin_fee' => round($adminFee, 2),
            'insurance_fee' => round($insuranceFee, 2),
            'total_fees' => round($totalFees, 2),
            'amount_with_fees' => round($totalAmount, 2),
            'installment_amount' => round($installment, 2)
        ];
    }

    /**
     * Get empty breakdown for zero amount
     */
    private function getEmptyBreakdown()
    {
        return [
            'principal' => 0,
            'processing_fee_rate' => 0,
            'processing_fee_basis' => 'initial_amount',
            'processing_fee_amount' => 0,
            'admin_fee' => 0,
            'insurance_fee' => 0,
            'total_fees' => 0,
            'amount_with_fees' => 0,
            'installment_amount' => 0
        ];
    }

    /**
     * Render HTML for PDF schedule
     */
    private function renderScheduleHtml($application, $scheduleData)
    {
        // Debug: Log the schedule data structure
        Log::info('Schedule Data Structure:', [
            'count' => count($scheduleData),
            'all_data' => $scheduleData
        ]);
        
        $currencySymbol = $application->currency->symbol ?? 'ZMW';
        
        $html = '<!DOCTYPE html>
        <html>
        <head>
            <meta charset="UTF-8">
            <title>Loan Payment Schedule - ' . $application->application_number . '</title>
            <style>
                /* A4 size: 210mm x 297mm */
                @page { size: A4; margin: 10mm; }
                body { 
                    font-family: Arial, sans-serif; 
                    margin: 0; 
                    padding: 0;
                    font-size: 10pt;
                    width: 210mm;
                    max-width: 210mm;
                }
                .header { 
                    text-align: center; 
                    margin-bottom: 8mm; 
                    border-bottom: 1pt solid #0077C5; 
                    padding-bottom: 5mm; 
                }
                .company-name { 
                    font-size: 14pt; 
                    font-weight: bold; 
                    color: #0077C5; 
                    margin-bottom: 1pt; 
                }
                .document-title { 
                    font-size: 12pt; 
                    margin: 2pt 0; 
                    color: #333; 
                }
                .loan-info { 
                    margin-bottom: 6mm; 
                    background-color: #f8f9fa; 
                    padding: 4mm; 
                    border-radius: 2pt; 
                    border: 0.5pt solid #dee2e6; 
                }
                .loan-info h3 { 
                    margin-top: 0; 
                    color: #0077C5; 
                    font-size: 10pt; 
                    margin-bottom: 3mm; 
                    text-align: left;
                }
                .loan-info table { 
                    width: 100%; 
                    border-collapse: collapse; 
                    font-size: 9pt; 
                    table-layout: fixed;
                }
                .loan-info td { 
                    padding: 3pt 4pt; 
                    border-bottom: 0.5pt solid #ddd; 
                    word-wrap: break-word;
                    overflow-wrap: break-word;
                    vertical-align: top;
                }
                .loan-info .label { 
                    font-weight: bold; 
                    background-color: #e9ecef; 
                    text-align: left;
                }
                .loan-info .value { 
                    text-align: left;
                }
                .loan-info .amount-value { 
                    text-align: right;
                    font-family: "Courier New", monospace;
                }
                /* Specific column widths for loan-info table - 4 equal columns */
                .loan-info .col-1 { width: 25%; }
                .loan-info .col-2 { width: 25%; }
                .loan-info .col-3 { width: 25%; }
                .loan-info .col-4 { width: 25%; }
                
                .schedule-table { 
                    width: 100%; 
                    border-collapse: collapse; 
                    margin-top: 6mm; 
                    font-size: 8pt; 
                    table-layout: fixed;
                }
                .schedule-table th, 
                .schedule-table td { 
                    border: 0.5pt solid #ddd; 
                    padding: 4pt 3pt; 
                    text-align: center; 
                    font-size: 8pt;
                    white-space: nowrap;
                    height: 16pt;
                    vertical-align: middle;
                    overflow: hidden;
                    box-sizing: border-box;
                }
                .schedule-table th { 
                    background-color: #0077C5; 
                    color: white; 
                    font-weight: bold;
                    border: 0.5pt solid #005a94;
                }
                .schedule-table .col-1 { width: 5%; min-width: 5%; max-width: 5%; }
                .schedule-table .col-2 { width: 12%; min-width: 12%; max-width: 12%; }
                .schedule-table .col-3 { width: 15%; min-width: 15%; max-width: 15%; }
                .schedule-table .col-4 { width: 12%; min-width: 12%; max-width: 12%; }
                .schedule-table .col-5 { width: 12%; min-width: 12%; max-width: 12%; }
                .schedule-table .col-6 { width: 12%; min-width: 12%; max-width: 12%; }
                .schedule-table .col-7 { width: 12%; min-width: 12%; max-width: 12%; }
                .schedule-table .col-8 { width: 12%; min-width: 12%; max-width: 12%; }
                .schedule-table .col-9 { width: 8%; min-width: 8%; max-width: 8%; }
                
                .schedule-table tr:nth-child(even) { 
                    background-color: #f9f9f9; 
                }
                .schedule-table .disbursement-row { 
                    background-color: #d4edda; 
                    font-weight: bold;
                    color: #155724;
                }
                .schedule-table .total-row { 
                    font-weight: bold; 
                    background-color: #cce5ff; 
                    color: #004085;
                }
                .schedule-table .total-label { 
                    text-align: left !important;
                    padding-left: 6pt !important;
                }
                .footer { 
                    margin-top: 10mm; 
                    text-align: center; 
                    font-size: 7pt; 
                    color: #666;
                    border-top: 0.5pt solid #ddd;
                    padding-top: 4mm;
                }
                .amount { 
                    text-align: right; 
                    font-family: "Courier New", monospace; 
                    font-size: 8pt;
                    padding-right: 6pt !important;
                }
                .negative-amount { color: #dc3545; }
                .positive-amount { color: #28a745; }
                .summary-box {
                    background-color: #f8f9fa;
                    padding: 6mm;
                    border-radius: 2pt;
                    border: 0.5pt solid #dee2e6;
                    margin-top: 8mm;
                    font-size: 9pt;
                }
                .summary-box h3 { 
                    color: #0077C5; 
                    margin-top: 0; 
                    border-bottom: 0.5pt solid #0077C5;
                    padding-bottom: 3mm;
                    font-size: 10pt;
                    margin-bottom: 4mm;
                    text-align: left;
                }
                .no-wrap { white-space: nowrap; }
                .text-left { text-align: left; }
                .text-right { text-align: right; }
                .text-center { text-align: center; }
                .currency-note {
                    font-size: 8pt;
                    color: #666;
                    text-align: center;
                    margin-top: 1mm;
                    font-style: italic;
                }
            </style>
        </head>
        <body>
            <div class="header">
                <div class="company-name">IMMIA FINANCE</div>
                <div class="document-title">LOAN PAYMENT SCHEDULE</div>
                <div style="font-size: 9pt; color: #666; margin-top: 1pt;">
                    Application: ' . $application->application_number . ' | Date: ' . date('d/m/Y H:i') . '
                </div>
            </div>
            
            <div class="loan-info">
                <h3>Loan Details</h3>
                <table>
                    <tr>
                        <td class="label col-1">Customer Name:</td>
                        <td class="value col-2">' . htmlspecialchars($application->customer->first_name . ' ' . $application->customer->surname) . '</td>
                        <td class="label col-3">Customer No:</td>
                        <td class="value col-4">' . htmlspecialchars($application->customer->customer_number) . '</td>
                    </tr>
                    <tr>
                        <td class="label col-1">Product:</td>
                        <td class="value col-2">' . htmlspecialchars($application->product->product ?? 'N/A') . '</td>
                        <td class="label col-3">Loan Amount:</td>
                        <td class="amount-value col-4"><strong>' . $currencySymbol . ' ' . number_format($application->loan_amount, 2) . '</strong></td>
                    </tr>
                    <tr>
                        <td class="label col-1">Interest Rate:</td>
                        <td class="value col-2">' . number_format($application->interest_rate, 2) . '% per ' . $application->payment_frequency . '</td>
                        <td class="label col-3">Term:</td>
                        <td class="value col-4">' . $application->loan_tenure . ' ' . ($application->payment_frequency === 'monthly' ? 'Months' : 'Periods') . '</td>
                    </tr>
                    <tr>
                        <td class="label col-1">Interest Method:</td>
                        <td class="value col-2">' . ucfirst(str_replace('_', ' ', $application->interest_method)) . '</td>
                        <td class="label col-3">Payment Freq:</td>
                        <td class="value col-4">' . ucfirst(str_replace('_', ' ', $application->payment_frequency)) . '</td>
                    </tr>
                    <tr>
                        <td class="label col-1">Monthly Installment:</td>
                        <td class="amount-value col-2"><strong>' . $currencySymbol . ' ' . number_format($application->installment_amount, 2) . '</strong></td>
                        <td class="label col-3">Processing Fee:</td>
                        <td class="value col-4">' . number_format($application->processing_fee, 2) . '%</td>
                    </tr>
                </table>
            </div>
            
            <div class="currency-note">
                All amounts in ' . $currencySymbol . '
            </div>';
        
        // Check if schedule data is available
        if (empty($scheduleData) || !is_array($scheduleData)) {
            $html .= '
            <div style="color: #856404; background-color: #fff3cd; padding: 4mm; border-radius: 2pt; margin: 6mm 0; border: 0.5pt solid #ffeaa7;">
                <strong>No schedule data available</strong>
            </div>';
        } else {
            $html .= '
            <h3 style="color: #0077C5; margin-top: 8mm; padding-bottom: 3mm; border-bottom: 0.5pt solid #0077C5; font-size: 10pt; text-align: left;">
                Payment Schedule
            </h3>
            <table class="schedule-table">
                <thead>
                    <tr>
                        <th class="col-1">#</th>
                        <th class="col-2">Date</th>
                        <th class="col-3">Type</th>
                        <th class="col-4">Total</th>
                        <th class="col-5">Principal</th>
                        <th class="col-6">Interest</th>
                        <th class="col-7">Fee</th>
                        <th class="col-8">Balance</th>
                        <th class="col-9">Status</th>
                    </tr>
                </thead>
                <tbody>';
            
            $totalDisbursement = 0;
            $totalPayments = 0;
            $totalPrincipal = 0;
            $totalInterest = 0;
            $totalFee = 0;
            $rowNumber = 1;
            
            foreach ($scheduleData as $index => $transaction) {
                $date = $transaction['date'] ?? 'N/A';
                $type = $transaction['type'] ?? 'Payment';
                $totalAmount = $transaction['total_due'] ?? 0;
                $principal = $transaction['principal'] ?? 0;
                $interest = $transaction['interest'] ?? 0;
                $fee = $transaction['fee'] ?? 0;
                $balance = $transaction['balance'] ?? 0;
                
                $rowClass = '';
                if (stripos($type, 'disbursement') !== false) {
                    $rowClass = 'disbursement-row';
                    $totalDisbursement += $principal;
                } else {
                    $totalPayments += $totalAmount;
                    $totalPrincipal += $principal;
                    $totalInterest += $interest;
                    $totalFee += $fee;
                }
                
                $totalAmountFormatted = number_format(abs($totalAmount), 2);
                $principalFormatted = number_format(abs($principal), 2);
                $interestFormatted = number_format(abs($interest), 2);
                $feeFormatted = number_format(abs($fee), 2);
                $balanceFormatted = number_format(abs($balance), 2);
                
                $totalAmountClass = $totalAmount < 0 ? 'negative-amount' : 'positive-amount';
                $principalClass = $principal < 0 ? 'negative-amount' : 'positive-amount';
                
                $html .= '
                    <tr class="' . $rowClass . '">
                        <td class="col-1">' . $rowNumber . '</td>
                        <td class="col-2">' . htmlspecialchars($date) . '</td>
                        <td class="col-3 text-left"><strong>' . htmlspecialchars($type) . '</strong></td>
                        <td class="col-4 amount ' . $totalAmountClass . '">' . $totalAmountFormatted . '</td>
                        <td class="col-5 amount ' . $principalClass . '">' . $principalFormatted . '</td>
                        <td class="col-6 amount">' . $interestFormatted . '</td>
                        <td class="col-7 amount">' . $feeFormatted . '</td>
                        <td class="col-8 amount"><strong>' . $balanceFormatted . '</strong></td>
                        <td class="col-9">' . ($type === 'Disbursement' ? 'Paid' : 'Due') . '</td>
                    </tr>';
                
                $rowNumber++;
            }
            
            $html .= '
                    <tr class="total-row">
                        <td class="total-label" colspan="2"><strong>TOTALS</strong></td>
                        <td class="col-3">-</td>
                        <td class="col-4 amount"><strong>' . number_format($totalPayments, 2) . '</strong></td>
                        <td class="col-5 amount"><strong>' . number_format($totalPrincipal, 2) . '</strong></td>
                        <td class="col-6 amount"><strong>' . number_format($totalInterest, 2) . '</strong></td>
                        <td class="col-7 amount"><strong>' . number_format($totalFee, 2) . '</strong></td>
                        <td class="col-8 amount"><strong>-</strong></td>
                        <td class="col-9"><strong>-</strong></td>
                    </tr>';
            
            $html .= '
                </tbody>
            </table>';
            
            $totalRepayment = $totalPrincipal + $totalInterest + $totalFee;
            $totalCost = $totalInterest + $totalFee;
            $effectiveInterestRate = $application->loan_amount > 0 ? ($totalCost / $application->loan_amount) * 100 : 0;
            
            $html .= '
            <div class="summary-box">
                <h3>Financial Summary (' . $currencySymbol . ')</h3>
                <table style="width: 100%; font-size: 9pt; table-layout: fixed;">
                    <tr>
                        <td style="width: 55%; padding: 2pt; text-align: left;"><strong>Loan Amount:</strong></td>
                        <td style="width: 45%; text-align: right; padding: 2pt;">' . number_format($application->loan_amount, 2) . '</td>
                    </tr>
                    <tr>
                        <td style="padding: 2pt; text-align: left;"><strong>Total Interest:</strong></td>
                        <td style="text-align: right; padding: 2pt; color: #dc3545;">' . number_format($totalInterest, 2) . '</td>
                    </tr>
                    <tr>
                        <td style="padding: 2pt; text-align: left;"><strong>Total Fees:</strong></td>
                        <td style="text-align: right; padding: 2pt; color: #dc3545;">' . number_format($totalFee, 2) . '</td>
                    </tr>
                    <tr style="border-top: 0.5pt solid #0077C5;">
                        <td style="padding: 2pt; font-weight: bold; text-align: left;"><strong>Total Cost:</strong></td>
                        <td style="text-align: right; padding: 2pt; font-weight: bold; color: #dc3545;">' . number_format($totalCost, 2) . '</td>
                    </tr>
                    <tr>
                        <td style="padding: 2pt; text-align: left;"><strong>Total Repayment:</strong></td>
                        <td style="text-align: right; padding: 2pt; font-weight: bold;">' . number_format($totalRepayment, 2) . '</td>
                    </tr>
                    <tr>
                        <td style="padding: 2pt; text-align: left;"><strong>Effective Rate:</strong></td>
                        <td style="text-align: right; padding: 2pt; font-weight: bold;">' . number_format($effectiveInterestRate, 2) . '%</td>
                    </tr>
                </table>
            </div>';
        }
        
        $html .= '
            <div class="footer">
                <p style="margin-bottom: 1pt; font-size: 7pt;"><strong>IMMIA FINANCE LTD</strong></p>
                <p style="margin-bottom: 1pt; font-size: 6pt;">Computer generated document - No signature required</p>
                <p style="margin-top: 3mm; font-size: 6pt; color: #999;">
                    Generated: ' . date('d/m/Y H:i:s') . ' | Page 1 of 1
                </p>
            </div>
        </body>
        </html>';
        
        return $html;
    }
    
    /**
     * Calculate first payment date
     */
    private function calculateFirstPaymentDate($date, $freq, $tenure) {
        $d = $date->copy();
        
        $tenure = (int) $tenure;
        
        return match($freq) {
            'one_off' => $d->addDays($tenure),
            'daily' => $d->addDay(),
            'weekly' => $d->addWeek(),
            'monthly' => $d->addMonth(),
            'quarterly' => $d->addMonths(3),
            default => $d->addMonth(),
        };
    }
    
    /**
     * Send application request notifications
     */
    private function sendRequestNotifications($application)
    {
        try {
            // Load customer relationship if not already loaded
            if (!$application->relationLoaded('customer')) {
                $application->load('customer');
            }
            
            $customer = $application->customer;
            if (!$customer) {
                Log::warning('Customer not found for application', [
                    'application_id' => $application->id
                ]);
                return;
            }

            $eventKey = 'app_request';
            
            // Send SMS if enabled
            $this->sendSmsNotification($eventKey, $customer, $application);
            
            // Send Email if enabled
            $this->sendEmailNotification($eventKey, $customer, $application);

        } catch (\Exception $e) {
            Log::error('❌ Exception in sendRequestNotifications', [
                'application_id' => $application->id,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
        }
    }

    /**
     * Send application approval notifications
     */
    private function sendApprovalNotifications($application)
    {
        try {
            if (!$application->relationLoaded('customer')) {
                $application->load('customer');
            }
            
            $customer = $application->customer;
            if (!$customer) {
                Log::warning('Customer not found for approved application', [
                    'application_id' => $application->id
                ]);
                return;
            }

            $eventKey = 'app_approve';
            
            // Send SMS if enabled
            $this->sendSmsNotification($eventKey, $customer, $application);
            
            // Send Email if enabled
            $this->sendEmailNotification($eventKey, $customer, $application);

        } catch (\Exception $e) {
            Log::error('❌ Exception in sendApprovalNotifications', [
                'application_id' => $application->id,
                'error' => $e->getMessage()
            ]);
        }
    }

    /**
     * Send application rejection notifications
     */
    private function sendRejectionNotifications($application)
    {
        try {
            if (!$application->relationLoaded('customer')) {
                $application->load('customer');
            }
            
            $customer = $application->customer;
            if (!$customer) {
                Log::warning('Customer not found for rejected application', [
                    'application_id' => $application->id
                ]);
                return;
            }

            $eventKey = 'app_rejection';
            
            // Send SMS if enabled
            $this->sendSmsNotification($eventKey, $customer, $application);
            
            // Send Email if enabled
            $this->sendEmailNotification($eventKey, $customer, $application);

        } catch (\Exception $e) {
            Log::error('❌ Exception in sendRejectionNotifications', [
                'application_id' => $application->id,
                'error' => $e->getMessage()
            ]);
        }
    }

    /**
     * Send disbursement notifications
     */
    private function sendDisbursementNotifications($application, $disbursedLoan)
    {
        try {
            if (!$application->relationLoaded('customer')) {
                $application->load('customer');
            }
            
            $customer = $application->customer;
            if (!$customer) return;

            $eventKey = 'loan_disburse';
            
            // Send SMS if enabled
            $smsSent = $this->sendSmsNotification($eventKey, $customer, $application, [
                'LoanNo' => $disbursedLoan->loannumber,
                'Date' => $disbursedLoan->firstpaymentdate ? $disbursedLoan->firstpaymentdate->format('d/m/Y') : 'N/A'
            ]);
            
            // Send Email if enabled
            if ($this->isNotificationEnabled($eventKey, 'email') && !empty($customer->email)) {
                try {
                    Mail::to($customer->email)->send(new LoanDisbursed($application, $disbursedLoan));
                    Log::info('✅ Loan disbursement email sent successfully', [
                        'loan_number' => $disbursedLoan->loannumber,
                        'customer_email' => $customer->email
                    ]);
                } catch (\Exception $e) {
                    Log::error('Failed to send disbursement email', [
                        'customer_email' => $customer->email,
                        'error' => $e->getMessage()
                    ]);
                }
            }

        } catch (\Exception $e) {
            Log::error('❌ Exception in sendDisbursementNotifications', [
                'application_id' => $application->id,
                'error' => $e->getMessage()
            ]);
        }
    }

    /**
     * Mobile disbursement notification
     */
    private function sendMobileDisbursementNotifications($application, $disbursedLoan, $lipilaData, $accountName)
    {
        try {
            if (!$application->relationLoaded('customer')) {
                $application->load('customer');
            }
            
            $customer = $application->customer;
            if (!$customer) return;

            $eventKey = 'loan_disburse';
            
            // Prepare additional variables for SMS
            $additionalVariables = [
                'LoanNo' => $disbursedLoan->loannumber,
                'Date' => $disbursedLoan->firstpaymentdate ? $disbursedLoan->firstpaymentdate->format('d/m/Y') : 'N/A',
                'Channel' => $accountName
            ];
            
            // Add Lipila reference if available
            $lipilaReference = $lipilaData['referenceId'] ?? '';
            if ($lipilaReference) {
                $additionalVariables['Reference'] = $lipilaReference;
            }

            // Send SMS if enabled
            $smsSent = $this->sendSmsNotification($eventKey, $customer, $application, $additionalVariables);
            
            // Send Email if enabled
            if ($this->isNotificationEnabled($eventKey, 'email') && !empty($customer->email)) {
                try {
                    Mail::to($customer->email)->send(new LoanDisbursed($application, $disbursedLoan));
                    Log::info('✅ Mobile disbursement email sent successfully', [
                        'loan_number' => $disbursedLoan->loannumber,
                        'customer_email' => $customer->email
                    ]);
                } catch (\Exception $e) {
                    Log::error('Failed to send mobile disbursement email', [
                        'customer_email' => $customer->email,
                        'error' => $e->getMessage()
                    ]);
                }
            }

        } catch (\Exception $e) {
            Log::error('❌ Exception in sendMobileDisbursementNotifications', [
                'application_id' => $application->id,
                'error' => $e->getMessage()
            ]);
        }
    }

    /**
     * Send SMS notification with template
     */
    private function sendSmsNotification($eventKey, $customer, $application, $additionalVariables = [])
    {
        try {
            // Check if SMS is enabled
            $smsEnabled = $this->isNotificationEnabled($eventKey, 'sms');
            
            if (!$smsEnabled) {
                Log::info('SMS notification disabled', ['event_key' => $eventKey]);
                return false;
            }
            
            if (empty($customer->phone)) {
                Log::warning('Customer phone number not available', [
                    'customer_id' => $customer->id,
                    'event_key' => $eventKey
                ]);
                return false;
            }
            
            // Get SMS template from database
            $smsTemplate = $this->getSmsTemplate($eventKey);
            
            if (!$smsTemplate) {
                Log::warning('No SMS template found', ['event_key' => $eventKey]);
                return false;
            }
            
            // Prepare variables for template
            $variables = array_merge([
                'Name' => $customer->first_name,
                'AppNo' => $application->application_number,
                'Amount' => number_format($application->loan_amount, 2),
                'Product' => $application->product->product ?? 'Loan',
                'Date' => $application->submitted_at ? $application->submitted_at->format('d/m/Y') : date('d/m/Y')
            ], $additionalVariables);
            
            // Replace template variables
            $message = $this->replaceTemplateVariables($smsTemplate, $variables);
            
            Log::info('Attempting to send SMS', [
                'phone' => $customer->phone,
                'formatted_message' => $message,
                'event_key' => $eventKey
            ]);

            // Send SMS
            $smsSent = $this->smsService->sendSms($customer->phone, $message);
            
            if ($smsSent) {
                Log::info('✅ SMS sent successfully', [
                    'customer_id' => $customer->id,
                    'phone' => $customer->phone,
                    'event_key' => $eventKey
                ]);
            } else {
                Log::warning('❌ Failed to send SMS (SMS service returned false)', [
                    'customer_id' => $customer->id,
                    'phone' => $customer->phone,
                    'event_key' => $eventKey
                ]);
            }
            
            return $smsSent;
            
        } catch (\Exception $e) {
            Log::error('❌ Exception in sendSmsNotification', [
                'event_key' => $eventKey,
                'customer_id' => $customer->id ?? 'unknown',
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }

    /**
     * Send email notification
     */
    private function sendEmailNotification($eventKey, $customer, $application, $additionalData = null)
    {
        try {
            // Check if email is enabled
            $emailEnabled = $this->isNotificationEnabled($eventKey, 'email');
            
            if (!$emailEnabled) {
                Log::info('Email notification disabled', ['event_key' => $eventKey]);
                return false;
            }
            
            if (empty($customer->email)) {
                Log::warning('Customer email not available', [
                    'customer_id' => $customer->id,
                    'event_key' => $eventKey
                ]);
                return false;
            }
            
            Log::info('Attempting to send email', [
                'email' => $customer->email,
                'event_key' => $eventKey
            ]);
            
            // Send email based on event type
            $emailSent = false;
            switch ($eventKey) {
                case 'app_request':
                    Mail::to($customer->email)->send(new LoanApplicationSubmitted($application));
                    $emailSent = true;
                    break;
                    
                case 'app_approve':
                    Mail::to($customer->email)->send(new LoanApplicationApproved($application));
                    $emailSent = true;
                    break;
                    
                case 'app_rejection':
                    Mail::to($customer->email)->send(new LoanApplicationRejected($application));
                    $emailSent = true;
                    break;
                    
                default:
                    Log::warning('No email template for event', ['event_key' => $eventKey]);
                    return false;
            }
            
            if ($emailSent) {
                Log::info('✅ Email sent successfully', [
                    'customer_id' => $customer->id,
                    'email' => $customer->email,
                    'event_key' => $eventKey
                ]);
            }
            
            return $emailSent;
            
        } catch (\Exception $e) {
            Log::error('❌ Exception in sendEmailNotification', [
                'event_key' => $eventKey,
                'customer_email' => $customer->email ?? 'unknown',
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            return false;
        }
    }

    /**
     * Check if a notification is enabled for a specific channel
     */
    private function isNotificationEnabled($eventKey, $channel)
    {
        try {
            $setting = DB::table('notification_settings')
                ->where('event_key', $eventKey)
                ->where('channel', $channel)
                ->first();
                
            return $setting && $setting->is_enabled == 1;
        } catch (\Exception $e) {
            Log::error('❌ Failed to check notification setting', [
                'event_key' => $eventKey,
                'channel' => $channel,
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }

    /**
     * Get SMS template for an event
     */
    private function getSmsTemplate($eventKey)
    {
        try {
            $event = DB::table('notification_events')
                ->where('event_key', $eventKey)
                ->first();
                
            return $event ? $event->sms_template : null;
        } catch (\Exception $e) {
            Log::error('❌ Failed to get SMS template', [
                'event_key' => $eventKey,
                'error' => $e->getMessage()
            ]);
            return null;
        }
    }

    /**
     * Replace variables in template
     */
    private function replaceTemplateVariables($template, $variables)
    {
        if (empty($template)) {
            return '';
        }
        
        foreach ($variables as $key => $value) {
            $template = str_replace("[$key]", $value, $template);
        }
        return $template;
    }
    
    /**
     * Get document statistics for an application
     */
    private function getDocumentStatistics($application, $productRequirements)
    {
        $totalRequirements = $productRequirements->count();
        $mandatoryRequirements = $productRequirements->where('is_mandatory', true)->count();
        
        $documents = $application->documents;
        $verifiedDocuments = $documents->where('status', 'verified')->count();
        $pendingDocuments = $documents->where('status', 'pending')->count();
        $rejectedDocuments = $documents->where('status', 'rejected')->count();
        
        // Calculate completion based on mandatory requirements
        $mandatoryCompleted = 0;
        foreach ($productRequirements as $pr) {
            if ($pr->is_mandatory && $pr->requirement) {
                $hasVerifiedDocument = $documents->where('requirement_id', $pr->requirement_id)
                    ->where('status', 'verified')
                    ->count() > 0;
                if ($hasVerifiedDocument) {
                    $mandatoryCompleted++;
                }
            }
        }
        
        $mandatoryCompletion = $mandatoryRequirements > 0 
            ? round(($mandatoryCompleted / $mandatoryRequirements) * 100, 0) 
            : 0;
        
        $overallCompletion = $totalRequirements > 0 
            ? round(($verifiedDocuments / $totalRequirements) * 100, 0) 
            : 0;
        
        return [
            'total_requirements' => $totalRequirements,
            'mandatory_requirements' => $mandatoryRequirements,
            'mandatory_completed' => $mandatoryCompleted,
            'mandatory_completion' => $mandatoryCompletion,
            'total_documents' => $documents->count(),
            'verified' => $verifiedDocuments,
            'pending' => $pendingDocuments,
            'rejected' => $rejectedDocuments,
            'overall_completion' => $overallCompletion,
            'has_mandatory_requirements' => $mandatoryRequirements > 0,
            'all_mandatory_verified' => $mandatoryCompleted == $mandatoryRequirements,
        ];
    }

    /**
     * Show document upload form with product-specific requirements
     */
    public function uploadDocument($id)
    {
        if (!auth()->user()->hasAnyPermission(['applications.view', 'applications.approve', 'poweruser'])) { 
            abort(403); 
        }

        $application = Application::with(['customer', 'product'])->findOrFail($id);
        
        // Get all requirements for the specific loan product
        $productRequirements = ProductRequirement::where('product_id', $application->product_id)
            ->with('requirement')
            ->orderBy('requirement_id')
            ->get();
        
        // Check for existing documents
        $existingDocuments = LoanApplicationDocument::where('application_id', $application->id)
            ->pluck('requirement_id')
            ->toArray();
        
        // Format for dropdown - only show requirements not yet uploaded
        $requirements = [];
        foreach ($productRequirements as $pr) {
            if ($pr->requirement && !in_array($pr->requirement_id, $existingDocuments)) {
                $requirements[] = [
                    'id' => $pr->requirement_id,
                    'name' => $pr->requirement->documentname . ($pr->is_mandatory ? ' (Mandatory)' : ''),
                    'is_mandatory' => $pr->is_mandatory,
                    'description' => $pr->requirement->description ?? '',
                ];
            }
        }
        
        // If all requirements have documents, show empty list with message
        $allRequirementsUploaded = empty($requirements);
        
        return view('admin.applications.upload-document', compact(
            'application', 
            'requirements',
            'allRequirementsUploaded'
        ));
    }

    /**
     * Store uploaded document with validation against product requirements
     */
    /**
 * Store uploaded document with validation against product requirements
 */
public function storeDocument(Request $request, $id)
{
    if (!auth()->user()->hasAnyPermission(['applications.view', 'applications.approve', 'poweruser'])) { 
        abort(403); 
    }

    $application = Application::findOrFail($id);
    
    $validator = Validator::make($request->all(), [
        'requirement_id' => 'required|exists:applicationrequirements,id',
        'document' => 'required|file|max:10240|mimes:pdf,jpg,jpeg,png,doc,docx,xls,xlsx', // Max 10MB
        'notes' => 'nullable|string|max:500',
    ]);

    if ($validator->fails()) {
        // Return JSON response for AJAX
        return response()->json([
            'success' => false,
            'message' => 'Validation error',
            'errors' => $validator->errors()->toArray()
        ], 422);
    }

    try {
        // Check if requirement belongs to the application's product
        $isValidRequirement = ProductRequirement::where('product_id', $application->product_id)
            ->where('requirement_id', $request->requirement_id)
            ->exists();
        
        if (!$isValidRequirement) {
            return response()->json([
                'success' => false,
                'message' => 'This document requirement is not valid for the selected loan product.'
            ], 400);
        }

        // Get requirement details
        $requirement = ApplicationRequirement::findOrFail($request->requirement_id);
        
        // Check if document already exists for this requirement
        $existingDocument = LoanApplicationDocument::where('application_id', $application->id)
            ->where('requirement_id', $request->requirement_id)
            ->first();
        
        if ($existingDocument) {
            // Delete old file if exists
            $existingDocument->deleteFile();
            $action = 'updated';
        } else {
            $action = 'uploaded';
        }

        // Upload file with proper naming
        $file = $request->file('document');
        $originalName = pathinfo($file->getClientOriginalName(), PATHINFO_FILENAME);
        $extension = $file->getClientOriginalExtension();
        
        // Create safe filename
        $safeName = Str::slug($originalName);
        $fileName = 'doc_' . $application->id . '_' . $requirement->id . '_' . time() . '_' . $safeName . '.' . $extension;
        $filePath = $file->storeAs('loan_documents/app_' . $application->id, $fileName, 'public');

        // Create or update document record
        $documentData = [
            'application_id' => $application->id,
            'requirement_id' => $request->requirement_id,
            'file_path' => $filePath,
            'file_name' => $file->getClientOriginalName(),
            'file_size' => $file->getSize(),
            'file_type' => $file->getMimeType(),
            'uploaded_by' => auth()->id(),
            'status' => 'pending',
            'notes' => $request->notes,
            'updated_at' => now(),
        ];

        if ($existingDocument) {
            $existingDocument->update($documentData);
            $document = $existingDocument;
        } else {
            $documentData['created_at'] = now();
            $document = LoanApplicationDocument::create($documentData);
        }

        Log::info('Document ' . $action . ' for application', [
            'application_id' => $application->id,
            'application_number' => $application->application_number,
            'document_id' => $document->id,
            'requirement' => $requirement->documentname,
            'file_name' => $file->getClientOriginalName(),
            'file_size' => $file->getSize(),
            'uploaded_by' => auth()->id(),
            'customer' => $application->customer->first_name . ' ' . $application->customer->surname
        ]);

        // Return JSON response for AJAX
        return response()->json([
            'success' => true,
            'message' => 'Document ' . $action . ' successfully.',
            'document' => [
                'id' => $document->id,
                'file_name' => $document->file_name,
                'file_size' => $document->file_size,
                'formatted_file_size' => $this->formatFileSize($document->file_size),
                'uploaded_at' => $document->created_at->format('M d, Y H:i'),
                'status' => $document->status,
                'requirement_name' => $requirement->documentname,
            ]
        ]);

    } catch (\Exception $e) {
        Log::error('Error uploading document', [
            'application_id' => $id,
            'error' => $e->getMessage(),
            'trace' => $e->getTraceAsString()
        ]);
        
        return response()->json([
            'success' => false,
            'message' => 'Error uploading document: ' . $e->getMessage()
        ], 500);
    }
}

/**
 * Helper function to format file size
 */
private function formatFileSize($bytes)
{
    if ($bytes == 0) return '0 Bytes';
    $k = 1024;
    $sizes = ['Bytes', 'KB', 'MB', 'GB', 'TB'];
    $i = floor(log($bytes) / log($k));
    return number_format($bytes / pow($k, $i), 2) . ' ' . $sizes[$i];
}

    /**
     * View document
     */
    public function viewDocument($applicationId, $documentId)
    {
        if (!auth()->user()->hasAnyPermission(['applications.view', 'applications.approve', 'poweruser'])) { 
            abort(403); 
        }

        $document = LoanApplicationDocument::with(['application', 'requirement'])
            ->where('application_id', $applicationId)
            ->findOrFail($documentId);

        if (!Storage::disk('public')->exists($document->file_path)) {
            return redirect()->back()->with('error', 'Document file not found.');
        }

        $filePath = Storage::disk('public')->path($document->file_path);
        $mimeType = Storage::disk('public')->mimeType($document->file_path);

        return response()->file($filePath, [
            'Content-Type' => $mimeType,
            'Content-Disposition' => 'inline; filename="' . $document->file_name . '"'
        ]);
    }

    /**
     * Download document
     */
    public function downloadDocument($applicationId, $documentId)
    {
        if (!auth()->user()->hasAnyPermission(['applications.view', 'applications.approve', 'poweruser'])) { 
            abort(403); 
        }

        $document = LoanApplicationDocument::with(['application', 'requirement'])
            ->where('application_id', $applicationId)
            ->findOrFail($documentId);

        if (!Storage::disk('public')->exists($document->file_path)) {
            return redirect()->back()->with('error', 'Document file not found.');
        }

        return Storage::disk('public')->download($document->file_path, $document->file_name);
    }

    /**
     * Verify document
     */
        /**
 * Verify document
 */
public function verifyDocument(Request $request, $applicationId, $documentId)
{
    if (!auth()->user()->hasPermission('applications.approve')) { 
        abort(403); 
    }

    try {
        $document = LoanApplicationDocument::where('application_id', $applicationId)
            ->findOrFail($documentId);

        // Check if document is already verified
        if ($document->status === 'verified') {
            return response()->json([
                'success' => false,
                'message' => 'Document is already verified.'
            ], 400);
        }

        $notes = $request->input('notes', 'Document verified by admin');
        
        // Update document status
        $document->status = 'verified';
        $document->verified_by = auth()->id();
        $document->verified_at = now();
        $document->rejection_reason = null; // Clear rejection reason if it exists
        $document->rejected_by = null;
        $document->rejected_at = null;
        
        if ($request->filled('notes')) {
            $document->notes = $notes;
        }
        
        $document->save();

        Log::info('Document verified', [
            'document_id' => $documentId,
            'application_id' => $applicationId,
            'requirement' => $document->requirement->documentname ?? 'Unknown',
            'verified_by' => auth()->id(),
            'customer' => $document->application->customer->first_name . ' ' . $document->application->customer->surname ?? 'Unknown'
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Document verified successfully.',
            'document' => [
                'id' => $document->id,
                'status' => $document->status,
                'verified_by' => auth()->user()->name,
                'verified_at' => $document->verified_at->format('M d, Y H:i')
            ]
        ]);

    } catch (\Exception $e) {
        Log::error('Error verifying document', [
            'document_id' => $documentId,
            'error' => $e->getMessage(),
            'trace' => $e->getTraceAsString()
        ]);
        
        return response()->json([
            'success' => false,
            'message' => 'Error verifying document: ' . $e->getMessage()
        ], 500);
    }
}

    /**
     * Reject document with modal form
     */
    public function rejectDocument(Request $request, $applicationId, $documentId)
    {
        if (!auth()->user()->hasPermission('applications.approve')) { 
            abort(403); 
        }

        $validator = Validator::make($request->all(), [
            'rejection_reason' => 'required|string|max:500',
            'notes' => 'nullable|string|max:500',
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        try {
            $document = LoanApplicationDocument::where('application_id', $applicationId)
                ->findOrFail($documentId);

            $document->reject(auth()->id(), $request->rejection_reason, $request->notes);

            Log::info('Document rejected', [
                'document_id' => $documentId,
                'application_id' => $applicationId,
                'requirement' => $document->requirement->documentname ?? 'Unknown',
                'rejection_reason' => $request->rejection_reason,
                'rejected_by' => auth()->id(),
                'customer' => $document->application->customer->first_name . ' ' . $document->application->customer->surname ?? 'Unknown'
            ]);

            return redirect()->back()->with('success', 'Document rejected successfully.');

        } catch (\Exception $e) {
            Log::error('Error rejecting document', [
                'document_id' => $documentId,
                'error' => $e->getMessage()
            ]);
            
            return redirect()->back()->with('error', 'Error rejecting document: ' . $e->getMessage());
        }
    }

    /**
     * Delete document
     */
    public function deleteDocument(Request $request, $applicationId, $documentId)
    {
        if (!auth()->user()->hasPermission('applications.approve')) { 
            abort(403); 
        }

        $validator = Validator::make($request->all(), [
            'confirmation' => 'required|in:yes',
        ]);

        if ($validator->fails()) {
            return redirect()->back()->with('error', 'Please confirm deletion.');
        }

        try {
            $document = LoanApplicationDocument::with(['requirement', 'application.customer'])
                ->where('application_id', $applicationId)
                ->findOrFail($documentId);

            $requirementName = $document->requirement->documentname ?? 'Unknown';
            $customerName = $document->application->customer ? 
                $document->application->customer->first_name . ' ' . $document->application->customer->surname : 'Unknown';
            
            // Delete file from storage
            $deletedFile = $document->deleteFile();
            
            // Delete record from database
            $document->delete();

            Log::info('Document deleted', [
                'document_id' => $documentId,
                'application_id' => $applicationId,
                'requirement' => $requirementName,
                'file_deleted' => $deletedFile,
                'customer' => $customerName,
                'deleted_by' => auth()->id()
            ]);

            return redirect()->back()->with('success', 'Document deleted successfully.');

        } catch (\Exception $e) {
            Log::error('Error deleting document', [
                'document_id' => $documentId,
                'error' => $e->getMessage()
            ]);
            
            return redirect()->back()->with('error', 'Error deleting document: ' . $e->getMessage());
        }
    }

    /**
     * Get document requirements completion status for approval eligibility
     */
    public function checkDocumentCompletion($id)
    {
        $application = Application::with(['documents', 'product'])->findOrFail($id);
        
        $productRequirements = ProductRequirement::where('product_id', $application->product_id)
            ->with('requirement')
            ->get();
        
        $mandatoryRequirements = $productRequirements->where('is_mandatory', true);
        $documents = $application->documents->where('status', 'verified');
        
        $missingMandatory = [];
        $allMandatoryMet = true;
        
        foreach ($mandatoryRequirements as $mandatoryReq) {
            if (!$mandatoryReq->requirement) {
                continue;
            }
            
            $hasVerifiedDocument = $documents->where('requirement_id', $mandatoryReq->requirement_id)->count() > 0;
            
            if (!$hasVerifiedDocument) {
                $missingMandatory[] = $mandatoryReq->requirement->documentname;
                $allMandatoryMet = false;
            }
        }
        
        $completionPercentage = $mandatoryRequirements->count() > 0 
            ? round(($mandatoryRequirements->count() - count($missingMandatory)) / $mandatoryRequirements->count() * 100, 0)
            : 100;
        
        return response()->json([
            'eligible_for_approval' => $allMandatoryMet,
            'all_mandatory_met' => $allMandatoryMet,
            'missing_mandatory' => $missingMandatory,
            'missing_count' => count($missingMandatory),
            'total_mandatory' => $mandatoryRequirements->count(),
            'mandatory_completed' => $mandatoryRequirements->count() - count($missingMandatory),
            'completion_percentage' => $completionPercentage,
            'message' => $allMandatoryMet 
                ? '✅ All mandatory documents are verified. Application is eligible for approval.'
                : '⚠️ ' . count($missingMandatory) . ' mandatory document(s) missing or not verified.',
            'recommendation' => $allMandatoryMet
                ? 'You can proceed with application approval.'
                : 'Please upload and verify the missing mandatory documents before approving.'
        ]);
    }

    /**
     * Get quick document status for dashboard or listing
     */
    public function getDocumentQuickStatus($id)
    {
        $application = Application::withCount(['documents', 'documents as pending_documents_count' => function($query) {
            $query->where('status', 'pending');
        }, 'documents as verified_documents_count' => function($query) {
            $query->where('status', 'verified');
        }])->findOrFail($id);
        
        $productRequirements = ProductRequirement::where('product_id', $application->product_id)
            ->where('is_mandatory', true)
            ->count();
        
        $mandatoryCompleted = LoanApplicationDocument::where('application_id', $id)
            ->where('status', 'verified')
            ->whereIn('requirement_id', function($query) use ($application) {
                $query->select('requirement_id')
                    ->from('product_requirements')
                    ->where('product_id', $application->product_id)
                    ->where('is_mandatory', true);
            })
            ->count();
        
        $mandatoryCompletion = $productRequirements > 0 
            ? round(($mandatoryCompleted / $productRequirements) * 100, 0)
            : 100;
        
        return response()->json([
            'total_documents' => $application->documents_count,
            'pending' => $application->pending_documents_count,
            'verified' => $application->verified_documents_count,
            'mandatory_requirements' => $productRequirements,
            'mandatory_completed' => $mandatoryCompleted,
            'mandatory_completion' => $mandatoryCompletion,
            'status_badge' => $this->getDocumentStatusBadge($mandatoryCompletion),
        ]);
    }

    /**
     * Generate document status badge
     */
    private function getDocumentStatusBadge($completion)
    {
        if ($completion == 100) {
            return '<span class="badge bg-success"><i class="fas fa-check-circle"></i> Complete</span>';
        } elseif ($completion >= 50) {
            return '<span class="badge bg-warning"><i class="fas fa-exclamation-circle"></i> ' . $completion . '%</span>';
        } else {
            return '<span class="badge bg-danger"><i class="fas fa-times-circle"></i> ' . $completion . '%</span>';
        }
    }

    /**
     * Generate document compliance report
     */
    public function generateDocumentReport($id)
    {
        $application = Application::with([
            'customer',
            'product',
            'documents',
            'documents.requirement',
            'documents.verifiedBy'
        ])->findOrFail($id);
        
        $productRequirements = ProductRequirement::where('product_id', $application->product_id)
            ->with('requirement')
            ->get();
        
        $report = [
            'application' => [
                'number' => $application->application_number,
                'date' => $application->submitted_at ? $application->submitted_at->format('Y-m-d') : 'N/A',
                'customer' => $application->customer->first_name . ' ' . $application->customer->surname,
                'loan_product' => $application->product->product ?? 'N/A',
                'loan_amount' => number_format($application->loan_amount, 2),
            ],
            'requirements' => [],
            'summary' => [
                'total_requirements' => $productRequirements->count(),
                'mandatory_requirements' => $productRequirements->where('is_mandatory', true)->count(),
                'documents_uploaded' => $application->documents->count(),
                'documents_verified' => $application->documents->where('status', 'verified')->count(),
                'documents_pending' => $application->documents->where('status', 'pending')->count(),
                'documents_rejected' => $application->documents->where('status', 'rejected')->count(),
            ]
        ];
        
        foreach ($productRequirements as $pr) {
            if (!$pr->requirement) continue;
            
            $document = $application->documents->firstWhere('requirement_id', $pr->requirement_id);
            
            $report['requirements'][] = [
                'document_name' => $pr->requirement->documentname,
                'is_mandatory' => $pr->is_mandatory ? 'Yes' : 'No',
                'status' => $document ? $document->status : 'missing',
                'upload_date' => $document ? $document->created_at->format('Y-m-d') : 'N/A',
                'verified_by' => $document && $document->verified_by ? ($document->verifiedBy->name ?? 'N/A') : 'N/A',
                'verification_date' => $document && $document->verified_at ? $document->verified_at->format('Y-m-d') : 'N/A',
                'notes' => $document ? $document->notes : 'N/A',
            ];
        }
        
        return response()->json($report);
    }
    
    /**
 * Get document requirements for a specific product
 */
public function getProductRequirements(Request $request)
{
    $productId = $request->get('product_id');
    
    if (!$productId) {
        return response()->json([
            'success' => false,
            'message' => 'Product ID is required'
        ]);
    }
    
    // Get all product requirements for the specific loan product
    $productRequirements = ProductRequirement::where('product_id', $productId)
        ->with('requirement')
        ->orderBy('is_mandatory', 'desc') // Show mandatory first
        ->orderBy('created_at', 'desc')
        ->get();
    
    $requirements = [];
    $mandatoryCount = 0;
    $optionalCount = 0;
    
    foreach ($productRequirements as $productRequirement) {
        $requirement = $productRequirement->requirement;
        if (!$requirement) {
            continue;
        }
        
        $requirements[] = [
            'id' => $requirement->id,
            'name' => $requirement->documentname,
            'description' => $requirement->description ?? 'No description available',
            'is_mandatory' => (bool)$productRequirement->is_mandatory,
        ];
        
        if ($productRequirement->is_mandatory) {
            $mandatoryCount++;
        } else {
            $optionalCount++;
        }
    }
    
    return response()->json([
        'success' => true,
        'requirements' => $requirements,
        'mandatory_count' => $mandatoryCount,
        'optional_count' => $optionalCount,
        'total_count' => count($requirements)
    ]);
}
    
    
}