<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Branch;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class BranchController extends Controller
{
    /**
     * Display a listing of branches.
     */
    public function index()
    {
        $branches = Branch::with('user')->latest()->get();
        return view('admin.branches.index', compact('branches'));
    }

    /**
     * Store a newly created branch.
     */
    public function store(Request $request)
    {
        $request->validate([
            'branchcode' => 'required|string|max:10|unique:branches',
            'branchname' => 'required|string|max:255',
            'setupdate' => 'required|date',
        ]);

        try {
            Branch::create([
                'branchcode' => $request->branchcode,
                'branchname' => $request->branchname,
                'setupdate' => $request->setupdate,
                'status' => 1,
                'userid' => Auth::id(),
                'branchmanager' => null,
                'companyid' => 1
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Branch created successfully!'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to create branch: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Display the specified branch for editing.
     */
    public function show($id)
    {
        try {
            $branch = Branch::find($id);
            
            if (!$branch) {
                return response()->json([
                    'success' => false,
                    'message' => 'Branch not found'
                ], 404);
            }

            return response()->json([
                'success' => true,
                'data' => [
                    'id' => $branch->id,
                    'branchcode' => $branch->branchcode,
                    'branchname' => $branch->branchname,
                    'setupdate' => $branch->setupdate->format('Y-m-d'),
                    'status' => $branch->status,
                    'userid' => $branch->userid,
                    'branchmanager' => $branch->branchmanager,
                    'companyid' => $branch->companyid
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error loading branch: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Update the specified branch.
     */
    public function update(Request $request, $id)
    {
        $request->validate([
            'branchcode' => 'required|string|max:10|unique:branches,branchcode,' . $id,
            'branchname' => 'required|string|max:255',
            'setupdate' => 'required|date',
        ]);

        try {
            $branch = Branch::find($id);
            
            if (!$branch) {
                return response()->json([
                    'success' => false,
                    'message' => 'Branch not found'
                ], 404);
            }

            $branch->update([
                'branchcode' => $request->branchcode,
                'branchname' => $request->branchname,
                'setupdate' => $request->setupdate,
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Branch updated successfully!'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to update branch: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Toggle branch status.
     */
    public function toggleStatus($id)
    {
        try {
            $branch = Branch::find($id);
            
            if (!$branch) {
                return response()->json([
                    'success' => false,
                    'message' => 'Branch not found'
                ], 404);
            }

            $newStatus = !$branch->status;
            $branch->update(['status' => $newStatus]);

            return response()->json([
                'success' => true,
                'message' => 'Branch status updated successfully!',
                'new_status' => $newStatus,
                'status_text' => $newStatus ? 'Active' : 'Inactive'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to update branch status: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Remove the specified branch.
     */
    public function destroy($id)
    {
        try {
            $branch = Branch::find($id);
            
            if (!$branch) {
                return response()->json([
                    'success' => false,
                    'message' => 'Branch not found'
                ], 404);
            }

            $branch->delete();

            return response()->json([
                'success' => true,
                'message' => 'Branch deleted successfully!'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete branch: ' . $e->getMessage()
            ], 500);
        }
    }
}