<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Repayment;
use Illuminate\Http\Request;
use Carbon\Carbon;
use Elibyy\TCPDF\Facades\TCPDF;

class CollectionsController extends Controller
{
    /**
     * Display a listing of the collections with filtering and export capabilities.
     */
    public function index(Request $request)
    {
        // Initialize query with relationships
        $query = Repayment::with(['customer', 'loan']);

        // 1. DATE FILTERING LOGIC
        $period = $request->query('period');
        $endDate = Carbon::now()->format('Y-m-d');
        
        if ($period == 'today') {
            $startDate = $endDate;
        } elseif ($period == 'month') {
            $startDate = Carbon::now()->startOfMonth()->format('Y-m-d');
        } elseif ($period == 'quarter') {
            $startDate = Carbon::now()->startOfQuarter()->format('Y-m-d');
        } else {
            // Default to manual date inputs or fallback to the last 30 days
            $startDate = $request->query('start_date', Carbon::now()->subMonth()->format('Y-m-d'));
            $endDate = $request->query('end_date', $endDate);
        }

        // Apply Date Range Filter
        $query->whereBetween('rdate', [$startDate, $endDate]);

        // 2. SEARCH LOGIC (Reference, Customer Name, or Loan Number)
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('paymentreference', 'like', "%{$search}%")
                  ->orWhereHas('customer', function($cq) use ($search) {
                      $cq->where('first_name', 'like', "%{$search}%")
                         ->orWhere('surname', 'like', "%{$search}%");
                  })
                  ->orWhereHas('loan', function($lq) use ($search) {
                      $lq->where('loannumber', 'like', "%{$search}%");
                  });
            });
        }

        // 3. STATS CALCULATION (Filtered)
        // We calculate these before pagination so they reflect the entire filtered set
        $stats = [
            'principal' => (float) $query->sum('rprincipal'),
            'interest'  => (float) $query->sum('rinterest'),
            'fees'      => (float) $query->sum('processing_fees_amount_paid'),
            'penalties' => (float) $query->sum('penalties'),
            'total'     => (float) $query->sum('rtotal'),
        ];

        // 4. TRIGGER EXPORTS
        // Styled Excel Export
        if ($request->has('export') && $request->export == 'xls') {
            $dataToExport = $query->orderBy('rdate', 'desc')->get();
            return $this->exportToExcel($dataToExport, $startDate, $endDate);
        }

        // Styled PDF Export using TCPDF
        if ($request->has('pdf')) {
            $dataToExport = $query->orderBy('rdate', 'desc')->get();
            return $this->printToPdf($dataToExport, $stats, $startDate, $endDate);
        }

        // 5. VIEW PAGINATION
        $collections = $query->orderBy('rdate', 'desc')
                             ->paginate(15)
                             ->withQueryString();

        return view('admin.collections.index', compact('collections', 'stats', 'startDate', 'endDate'));
    }

    /**
     * PDF Generation using TCPDF
     */
   private function printToPdf($data, $stats, $start, $end)
{
    // Render the styled HTML view
    $html = view('admin.collections.pdf', compact('data', 'stats', 'start', 'end'))->render();

    $pdf = new TCPDF(PDF_PAGE_ORIENTATION, PDF_UNIT, PDF_PAGE_FORMAT, true, 'UTF-8', false);

    $pdf::SetCreator('ZedBankOS');
    $pdf::SetTitle('Collections Report');
    $pdf::setPrintHeader(false);
    $pdf::setPrintFooter(true);
    $pdf::SetMargins(10, 10, 10);
    $pdf::SetAutoPageBreak(TRUE, 15);

    // Landscape orientation to match Excel width
    $pdf::AddPage('L', 'A4');
    $pdf::writeHTML($html, true, false, true, false, '');

    return response($pdf::Output('ZedBankOS_Collections_Report.pdf', 'I'), 200)
            ->header('Content-Type', 'application/pdf');
}

    /**
     * Generates a styled Excel (.xls) file using HTML/CSS
     */
    private function exportToExcel($data, $start, $end)
    {
        $fileName = 'ZedBankOS_Collections_' . date('Ymd_His') . '.xls';
        
        $headers = [
            "Content-Type"        => "application/vnd.ms-excel",
            "Content-Disposition" => "attachment; filename=\"$fileName\"",
            "Pragma"              => "no-cache",
            "Expires"             => "0"
        ];

        $html = '
        <html xmlns:o="urn:schemas-microsoft-com:office:office" xmlns:x="urn:schemas-microsoft-com:office:excel" xmlns="http://www.w3.org/TR/REC-html40">
        <head>
            <style>
                table { border-collapse: collapse; width: 100%; }
                th, td { border: 0.5pt solid #000; padding: 5px; font-family: Arial; font-size: 10pt; }
                .title { font-size: 14pt; font-weight: bold; text-align: center; color: #0077C5; border:none; }
                .thead { background-color: #0077C5; color: #ffffff; font-weight: bold; }
                .text-right { text-align: right; }
                .summary-label { font-weight: bold; text-align: right; background-color: #f9f9f9; }
                .summary-value { font-weight: bold; text-align: right; background-color: #f1f1f1; }
            </style>
        </head>
        <body>
            <table>
                <tr><th colspan="10" class="title">ZEDBANKOS COLLECTIONS REPORT</th></tr>
                <tr><th colspan="10" style="text-align:center; border:none;">Period: ' . $start . ' to ' . $end . '</th></tr>
                <tr><td colspan="10" style="border:none;">&nbsp;</td></tr>
                <tr class="thead">
                    <th>Date</th><th>Customer</th><th>Loan No.</th><th>Reference</th><th>Method</th>
                    <th>Principal</th><th>Interest</th><th>Fees</th><th>Penalties</th><th>Total Paid</th>
                </tr>';

        $tp = 0; $ti = 0; $tf = 0; $tpen = 0; $tt = 0;

        foreach ($data as $row) {
            $tp += $row->rprincipal; $ti += $row->rinterest; 
            $tf += $row->processing_fees_amount_paid; $tpen += $row->penalties; $tt += $row->rtotal;

            $html .= '<tr>
                <td>' . $row->rdate->format('Y-m-d') . '</td>
                <td>' . ($row->customer->first_name ?? "") . " " . ($row->customer->surname ?? "") . '</td>
                <td>' . ($row->loan->loannumber ?? "N/A") . '</td>
                <td>' . $row->paymentreference . '</td>
                <td>' . $row->pmethod . '</td>
                <td class="text-right">' . number_format($row->rprincipal, 2) . '</td>
                <td class="text-right">' . number_format($row->rinterest, 2) . '</td>
                <td class="text-right">' . number_format($row->processing_fees_amount_paid, 2) . '</td>
                <td class="text-right">' . number_format($row->penalties, 2) . '</td>
                <td class="text-right"><b>' . number_format($row->rtotal, 2) . '</b></td>
            </tr>';
        }

        $html .= '
                <tr><td colspan="10" style="border:none;">&nbsp;</td></tr>
                <tr><td colspan="5" style="border:none;">&nbsp;</td><td colspan="4" class="summary-label">Total Principal:</td><td class="summary-value">' . number_format($tp, 2) . '</td></tr>
                <tr><td colspan="5" style="border:none;">&nbsp;</td><td colspan="4" class="summary-label">Total Interest:</td><td class="summary-value">' . number_format($ti, 2) . '</td></tr>
                <tr><td colspan="5" style="border:none;">&nbsp;</td><td colspan="4" class="summary-label">Total Fees:</td><td class="summary-value">' . number_format($tf, 2) . '</td></tr>
                <tr><td colspan="5" style="border:none;">&nbsp;</td><td colspan="4" class="summary-label">Total Penalties:</td><td class="summary-value">' . number_format($tpen, 2) . '</td></tr>
                <tr><td colspan="5" style="border:none;">&nbsp;</td><td colspan="4" class="summary-label" style="font-size:12pt;">GRAND TOTAL:</td><td class="summary-value" style="font-size:12pt;">' . number_format($tt, 2) . '</td></tr>
            </table>
        </body>
        </html>';

        return response($html)->withHeaders($headers);
    }
}