<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Loan;
use App\Models\DisbursedLoan;
use App\Models\Repayment;
use App\Models\PaymentSchedule;
use App\Models\CashBankAccount;
use App\Models\BusinessProcessMapping;
use App\Models\JournalEntry;
use App\Models\JournalEntryItem;
use App\Models\GeneralLedgerEntry;
use App\Models\InterestIncome;
use App\Models\ProcessingFees;
use  App\Models\Company;
use  App\Models\Currency;
use  App\Models\FinancialYear;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class LoanController extends Controller
{
    /**
     * Display active loans
     */
    public function active()
    {
        if (!auth()->user()->hasPermission('loans.view')) {
            abort(403, 'You do not have permission to view loans.');
        }

        $loans = DisbursedLoan::where('status', 'active')
            ->with(['customer', 'loanType'])
            ->orderBy('disburseddate', 'desc')
            ->paginate(20);

        return view('admin.loans.active', compact('loans'));
    }

    /**
     * Display closed loans
     */
    public function closed()
    {
        if (!auth()->user()->hasPermission('loans.view')) {
            abort(403, 'You do not have permission to view loans.');
        }

        $loans = DisbursedLoan::where('status', 'closed')
            ->with(['customer', 'loanType'])
            ->orderBy('closeddate', 'desc')
            ->paginate(20);

        return view('admin.loans.closed', compact('loans'));
    }

    /**
     * Display non-performing loans
     */
    public function nonPerforming()
    {
        if (!auth()->user()->hasPermission('loans.view')) {
            abort(403, 'You do not have permission to view loans.');
        }

        $loans = DisbursedLoan::where('status', 'default')
            ->with(['customer'])
            ->orderBy('disburseddate', 'desc')
            ->paginate(20);

        return view('admin.loans.non-performing', compact('loans'));
    }

    /**
     * Display loan details
     */
    public function show($id)
    {
        if (!auth()->user()->hasPermission('loans.view')) {
            abort(403, 'You do not have permission to view loan details.');
        }

        $loan = DisbursedLoan::with(['customer', 'loanType', 'repayments', 'paymentSchedules'])->findOrFail($id);
        
        // Also load additional relationships needed for the view
        $loan->load([
            'branch',
            'company',
            'disbursedBy',
            'creator'
        ]);
        
        return view('admin.loans.show', compact('loan'));
    }

    /**
     * Show payment form for a specific loan
     */
    public function showPaymentForm($id)
    {
        // Temporarily bypass permission check for testing
        // if (!auth()->user()->hasPermission('payments.create')) {
        //     abort(403, 'You do not have permission to record payments.');
        // }

        $loan = DisbursedLoan::with(['customer', 'paymentSchedules', 'repayments'])
            ->findOrFail($id);

        // Check if loan is active
        if ($loan->status !== 'active') {
            return redirect()->route('admin.loans.active')
                ->with('error', 'Payments can only be recorded for active loans.');
        }

        // Log balances for debugging
        Log::info('Payment form loaded', [
            'loan_id' => $loan->loanid,
            'principal' => $loan->principalbalance,
            'interest' => $loan->interestbalance,
            'processing_fee' => $loan->processing_fee_balance,
            'penalty' => $loan->penalty_balance,
            'total' => $loan->totalbalance,
        ]);

        return view('admin.loans.payment', compact('loan'));
    }

    /**
     * Reverse a loan status
     */
    public function reverse($id)
    {
        if (!auth()->user()->hasPermission('loans.reverse')) {
            abort(403, 'You do not have permission to reverse loans.');
        }

        $loan = DisbursedLoan::findOrFail($id);
        
        DB::beginTransaction();
        try {
            $originalStatus = $loan->status;
            
            // Determine new status based on current status
            if ($loan->status === 'closed') {
                $loan->status = 'active';
                $loan->closeddate = null;
                $loan->closure_reason = null;
                $loan->closure_notes = null;
                $loan->closure_settlement_amount = null;
                $message = 'Loan reopened successfully.';
            } elseif ($loan->status === 'default') {
                $loan->status = 'active';
                $loan->npl_classification_date = null;
                $loan->npl_reason = null;
                $loan->npl_category = null;
                $loan->provision_percentage = null;
                $loan->provision_amount = null;
                $loan->assessment_details = null;
                $message = 'Loan restored to active status successfully.';
            } else {
                return back()->with('error', 'Cannot reverse loan with current status.');
            }
            
            $loan->updatedby = auth()->id();
            $loan->updated_at = now();
            $loan->save();
            
            // Log the reversal
            Log::info('Loan status reversed', [
                'loan_id' => $loan->loanid,
                'loan_number' => $loan->loannumber,
                'original_status' => $originalStatus,
                'new_status' => $loan->status,
                'reversed_by' => auth()->id(),
            ]);
            
            DB::commit();
            
            return redirect()->route('admin.loans.show', $loan->loanid)
                ->with('success', $message);
                
        } catch (\Exception $e) {
            DB::rollBack();
            
            Log::error('Loan reversal failed', [
                'loan_id' => $id,
                'error' => $e->getMessage(),
            ]);
            
            return back()->with('error', 'Loan reversal failed: ' . $e->getMessage());
        }
    }
    
    /**
     * Show NPL classification form
     */
    public function showNPLForm($id)
    {
        if (!auth()->user()->hasPermission('loans.transfer-npl')) {
            abort(403, 'You do not have permission to transfer loans to non-performing status.');
        }

        $loan = DisbursedLoan::with(['customer', 'loanType'])->findOrFail($id);
        
        // Check if loan is already NPL
        if ($loan->status === 'default') {
            return redirect()->route('admin.loans.show', $loan->loanid)
                ->with('error', 'This loan is already classified as non-performing.');
        }
        
        // Check if loan is closed
        if ($loan->status === 'closed') {
            return redirect()->route('admin.loans.show', $loan->loanid)
                ->with('error', 'Cannot classify a closed loan as non-performing.');
        }

        return view('admin.loans.npl-form', compact('loan'));
    }

    /**
     * Close a loan (simple version)
     */
    public function closeLoan(Request $request, $id)
    {
        if (!auth()->user()->hasPermission('loans.close')) {
            abort(403, 'You do not have permission to close loans.');
        }

        $loan = DisbursedLoan::findOrFail($id);
        
        // Check if loan is already closed
        if ($loan->status === 'closed') {
            return redirect()->route('admin.loans.show', $loan->loanid)
                ->with('error', 'This loan is already closed.');
        }
        
        // Update loan status
        $loan->update([
            'status' => 'closed',
            'closeddate' => now(),
            'updated_at' => now(),
            'updatedby' => auth()->id(),
        ]);

        return redirect()->route('admin.loans.show', $loan->loanid)
            ->with('success', 'Loan has been successfully closed.');
    }
    
    /**
     * Transfer loan to non-performing status (simplified)
     */


    /**
     * Delete a loan
     */
    public function destroy($id)
    {
        if (!auth()->user()->hasPermission('loans.delete')) {
            abort(403, 'You do not have permission to delete loans.');
        }

        $loan = DisbursedLoan::findOrFail($id);
        $loan->delete();

        return redirect()->route('admin.loans.active')
            ->with('success', 'Loan deleted successfully.');
    }

    /**
     * Debug payment functionality
     */
    public function debugPayment($loanId)
    {
        $loan = DisbursedLoan::with(['customer', 'company', 'branch', 'repayments', 'paymentSchedules'])->find($loanId);
        
        if (!$loan) {
            return response()->json(['error' => 'Loan not found'], 404);
        }
        
        // Test auto-allocation calculation
        $testAmount = 5.00;
        $allocations = $this->simulateAutoAllocation($loan, $testAmount);
        
        return response()->json([
            'loan' => [
                'id' => $loan->loanid,
                'number' => $loan->loannumber,
                'customer' => $loan->customer ? $loan->customer->first_name . ' ' . $loan->customer->surname : null,
                'status' => $loan->status,
            ],
            'balances' => [
                'principal' => (float) $loan->principalbalance,
                'interest' => (float) $loan->interestbalance,
                'processing_fee' => (float) $loan->processing_fee_balance,
                'penalty' => (float) $loan->penalty_balance,
                'total' => (float) $loan->totalbalance,
            ],
            'auto_allocation_test' => [
                'test_amount' => $testAmount,
                'allocations' => $allocations,
                'total_allocated' => array_sum($allocations),
            ],
            'database_check' => [
                'has_processing_fee_balance' => isset($loan->processing_fee_balance),
                'has_penalty_balance' => isset($loan->penalty_balance),
                'has_last_interest_calculation_date' => isset($loan->last_interest_calculation_date),
                'has_interest_calculation_status' => isset($loan->interest_calculation_status),
                'repayments_count' => $loan->repayments()->count(),
                'payment_schedules_count' => $loan->paymentSchedules()->count(),
            ],
            'form_balance_calculation' => [
                'view_calculation' => [
                    'principal' => $loan->principalbalance,
                    'interest' => $loan->interestbalance,
                    'processing_fee' => $loan->processing_fee_balance,
                    'penalty' => $loan->penalty_balance,
                    'total' => bcadd(
                        bcadd($loan->principalbalance, $loan->interestbalance, 2),
                        bcadd($loan->processing_fee_balance, $loan->penalty_balance, 2),
                        2
                    ),
                ]
            ]
        ]);
    }

    /**
     * Simulate auto-allocation for testing
     */
    private function simulateAutoAllocation($loan, $paymentAmount)
    {
        $allocations = [
            'principal' => 0,
            'interest' => 0,
            'processing' => 0,
            'penalty' => 0,
        ];
        
        $remaining = $paymentAmount;
        
        // Auto-allocate in priority order
        $priorityOrder = [
            'penalty' => (float) $loan->penalty_balance,
            'processing' => (float) $loan->processing_fee_balance,
            'interest' => (float) $loan->interestbalance,
            'principal' => (float) $loan->principalbalance,
        ];
        
        foreach ($priorityOrder as $type => $balance) {
            if ($remaining <= 0) break;
            $allocations[$type] = min($balance, $remaining);
            $remaining -= $allocations[$type];
        }
        
        return $allocations;
    }

    /**
     * Show the close loan form
     */
    public function showCloseForm($id)
    {
        if (!auth()->user()->hasPermission('loans.close')) {
            abort(403, 'You do not have permission to close loans.');
        }

        $loan = DisbursedLoan::with(['customer', 'loanType'])->findOrFail($id);
        
        // Check if loan is already closed
        if ($loan->status === 'closed') {
            return redirect()->route('admin.loans.show', $loan->loanid)
                ->with('error', 'This loan is already closed.');
        }
        
        // Check if principal balance is zero (required for closure)
        if ($loan->principalbalance > 0) {
            return redirect()->route('admin.loans.show', $loan->loanid)
                ->with('error', 'Cannot close loan with outstanding principal balance. Please clear the principal balance first.');
        }

        return view('admin.loans.close', compact('loan'));
    }

    /**
     * Close a loan with additional details and accounting entries
     */
       /**
 * Close a loan with additional details and accounting entries
 */
    /**
 * Close a loan with additional details and accounting entries
 */
        /**
 * Close a loan with additional details and accounting entries
 */
public function close(Request $request, $id)
{
    if (!auth()->user()->hasPermission('loans.close')) {
        abort(403, 'You do not have permission to close loans.');
    }

    $loan = DisbursedLoan::findOrFail($id);
    
    // Check if loan is already closed
    if ($loan->status === 'closed') {
        return redirect()->route('admin.loans.show', $loan->loanid)
            ->with('error', 'This loan is already closed.');
    }
    
    // Prevent closing if principal balance is greater than zero
    if ($loan->principalbalance > 0) {
        return redirect()->route('admin.loans.show', $loan->loanid)
            ->with('error', 'Cannot close loan with outstanding principal balance. Principal balance: K ' . number_format($loan->principalbalance, 2));
    }
    
    // Validate the request
    $validated = $request->validate([
        'closing_reason' => 'required|in:fully_settled,loan_forgiveness,settlement_agreement,write_off_approved,customer_bankruptcy,deceased_customer,system_error,other',
        'closure_date' => 'required|date|before_or_equal:today',
        'closure_notes' => 'nullable|string|max:1000',
        'final_settlement_amount' => 'nullable|numeric|min:0|max:' . ($loan->totalbalance + 1000),
        'confirm_writeoff' => ($loan->interestbalance > 0 || $loan->processing_fee_balance > 0 || $loan->penalty_balance > 0) ? 'required|accepted' : 'nullable',
        'confirm_closure' => 'required|accepted',
        'confirm_audit' => 'required|accepted',
    ]);

    DB::beginTransaction();
    try {
        $closureDate = $validated['closure_date'];
        $closingReason = $validated['closing_reason'];
        $closureNotes = $validated['closure_notes'] ?? null;
        $finalSettlementAmount = $validated['final_settlement_amount'] ?? null;
        
        // Store original balances for logging
        $originalInterestBalance = $loan->interestbalance;
        $originalProcessingFeeBalance = $loan->processing_fee_balance;
        $originalPenaltyBalance = $loan->penalty_balance;
        
        // Handle interest balance write-off/reversal
        if ($loan->interestbalance > 0) {
            $this->writeOffInterest($loan, $loan->interestbalance, $closureDate, $closingReason, $closureNotes);
        } elseif ($loan->interestbalance < 0) {
            $this->reverseExcessInterest($loan, abs($loan->interestbalance), $closureDate, $closingReason, $closureNotes);
        }
        
        // Handle processing fee balance write-off/reversal
        if ($loan->processing_fee_balance > 0) {
            $this->writeOffProcessingFees($loan, $loan->processing_fee_balance, $closureDate, $closingReason, $closureNotes);
        } elseif ($loan->processing_fee_balance < 0) {
            $this->reverseExcessProcessingFees($loan, abs($loan->processing_fee_balance), $closureDate, $closingReason, $closureNotes);
        }
        
        // Handle penalty balance if exists
        if ($loan->penalty_balance > 0) {
            $this->writeOffPenalty($loan, $loan->penalty_balance, $closureDate, $closingReason, $closureNotes);
        }
        
        // Update the loan status and add closure details
        $loan->update([
            'status' => 'closed',
            'closeddate' => $closureDate,
            'lastpaymentdate' => now(),
            'closure_reason' => $closingReason,
            'closure_notes' => $closureNotes,
            'closure_settlement_amount' => $finalSettlementAmount,
            'closed_by' => auth()->id(),
            'updated_at' => now(),
            'updatedby' => auth()->id(),
            'interestbalance' => 0,
            'processing_fee_balance' => 0,
            'penalty_balance' => 0,
            'totalbalance' => $loan->principalbalance, // Only principal should remain (should be 0)
        ]);
        
        // MARK PAYMENT SCHEDULES AS CLOSED - UPDATED CODE
        if ($loan->paymentSchedules()->exists()) {
            $unpaidSchedules = $loan->paymentSchedules()
                ->where('status', '!=', 'paid')
                ->where(function($query) use ($closureDate) {
                    $query->whereNull('paid_date')
                          ->orWhere('paid_date', '>', $closureDate);
                })
                ->get();
            
            foreach ($unpaidSchedules as $schedule) {
                $schedule->markAsClosed($closureDate, [
                    'principal' => 0,
                    'interest' => $loan->interestbalance > 0 ? ($schedule->actualaccruedinterest > 0 ? $schedule->actualaccruedinterest : $schedule->interestamount) : 0,
                    'processing_fee' => $loan->processing_fee_balance > 0 ? ($schedule->actualaccruedprocessingfee > 0 ? $schedule->actualaccruedprocessingfee : $schedule->processingfeesamount) : 0,
                    'total' => 0,
                ]);
            }
            
            // Close accrual periods for paid schedules
            $loan->paymentSchedules()
                ->where('status', 'paid')
                ->where(function($query) use ($closureDate) {
                    $query->whereNull('accrual_end_date')
                          ->orWhere('accrual_end_date', '>', $closureDate);
                })
                ->update([
                    'accrual_end_date' => $closureDate,
                    'updatedby' => auth()->id(),
                ]);
        }

        DB::commit();

        // Log the closure
        Log::info('Loan closed with accounting entries', [
            'loan_id' => $loan->loanid,
            'loan_number' => $loan->loannumber,
            'closed_by' => auth()->id(),
            'closure_date' => $closureDate,
            'closure_reason' => $closingReason,
            'principal_balance_at_closure' => $loan->principalbalance,
            'interest_written_off' => $originalInterestBalance,
            'processing_fees_written_off' => $originalProcessingFeeBalance,
            'penalty_written_off' => $originalPenaltyBalance,
            'payment_schedules_closed' => isset($unpaidSchedules) ? $unpaidSchedules->count() : 0,
        ]);

        return redirect()->route('admin.loans.show', $loan->loanid)
            ->with('success', 'Loan has been successfully closed with all accounting entries posted.');

    } catch (\Exception $e) {
        DB::rollBack();
        
        Log::error('Loan closure failed', [
            'loan_id' => $id,
            'user_id' => auth()->id(),
            'error' => $e->getMessage(),
            'trace' => $e->getTraceAsString(),
        ]);
        
        return back()
            ->withInput()
            ->with('error', 'Loan closure failed: ' . $e->getMessage());
    }
}
    
    /**
     * Write off interest balance to interest income account
     * When interest balance is POSITIVE (customer owes): Record as NEGATIVE in interestincome table (loss)
     */
    private function writeOffInterest($loan, $amount, $closureDate, $reason, $notes = null)
    {
        try {
            // Get interest income account from business process mappings
            $interestIncomeMapping = BusinessProcessMapping::where('nameofthebusinessprocess', 'Interest Income')->first();
            
            if (!$interestIncomeMapping) {
                throw new \Exception('Interest Income account mapping not found.');
            }
            
            $interestIncomeAccountId = $interestIncomeMapping->accountid;
            
            // Get loan receivable account from business process mappings
            $loanReceivableMapping = BusinessProcessMapping::where('nameofthebusinessprocess', 'Loan Disbursement')->first();
            
            if (!$loanReceivableMapping) {
                throw new \Exception('Loan Receivable account mapping not found.');
            }
            
            $loanReceivableAccountId = $loanReceivableMapping->accountid;
            
            // Generate entry number
            $entryNumber = 'JNL-' . date('Ymd') . '-' . str_pad(JournalEntry::count() + 1, 4, '0', STR_PAD_LEFT);
            
            // Create journal entry using your actual model structure
            $journalEntry = JournalEntry::create([
                'company_id' => $loan->companyid,
                'branch_id' => $loan->branchid,
                'entry_number' => $entryNumber,
                'entry_date' => $closureDate,
                'reference' => "LOAN-CLOSE-{$loan->loanid}-INT-WOFF",
                'description' => "Interest write-off for loan closure: {$loan->loannumber} - {$reason}",
                'total_debit' => $amount,
                'total_credit' => $amount,
                'status' => 'posted',
                'posted_by' => auth()->id(),
                'posted_at' => now(),
                'created_by' => auth()->id(),
                'loanid' => $loan->loanid,
            ]);
            
            // Journal Entry Item 1: Debit Interest Income (decrease income - loss)
            JournalEntryItem::create([
                'company_id' => $loan->companyid,
                'branch_id' => $loan->branchid,
                'journal_entry_id' => $journalEntry->id,
                'account_id' => $interestIncomeAccountId,
                'description' => "Interest write-off (loss): {$loan->loannumber}",
                'debit' => $amount,
                'credit' => 0,
                'created_by' => auth()->id(),
            ]);
            
            // Journal Entry Item 2: Credit Loan Receivable (decrease asset)
            JournalEntryItem::create([
                'company_id' => $loan->companyid,
                'branch_id' => $loan->branchid,
                'journal_entry_id' => $journalEntry->id,
                'account_id' => $loanReceivableAccountId,
                'description' => "Interest write-off (loss): {$loan->loannumber}",
                'debit' => 0,
                'credit' => $amount,
                'created_by' => auth()->id(),
            ]);
            
            // Create general ledger entries
            // Debit Interest Income (decrease income - loss)
            GeneralLedgerEntry::create([
                'entrydate' => $closureDate,
                'description' => "Interest write-off (loss): Loan {$loan->loannumber} - {$reason}",
                'accountid' => $interestIncomeAccountId,
                'referencedocument' => 'LOAN_CLOSURE',
                'reference_type' => 'journal_entry',
                'reference_id' => $journalEntry->id,
                'documentno' => $entryNumber,
                'entrytype' => 'debit', // DEBIT for income accounts (decrease income)
                'amount' => $amount,
                'createdby' => auth()->id(),
                'companyid' => $loan->companyid,
                'branchid' => $loan->branchid,
                'loanid' => $loan->loanid,
            ]);
            
            // Credit Loan Receivable (decrease asset)
            GeneralLedgerEntry::create([
                'entrydate' => $closureDate,
                'description' => "Interest write-off (loss): Loan {$loan->loannumber} - {$reason}",
                'accountid' => $loanReceivableAccountId,
                'referencedocument' => 'LOAN_CLOSURE',
                'reference_type' => 'journal_entry',
                'reference_id' => $journalEntry->id,
                'documentno' => $entryNumber,
                'entrytype' => 'credit', // CREDIT for asset accounts (decrease asset)
                'amount' => $amount,
                'createdby' => auth()->id(),
                'companyid' => $loan->companyid,
                'branchid' => $loan->branchid,
                'loanid' => $loan->loanid,
            ]);
            
            // Record in interest income table as NEGATIVE (loss/reduction of income)
            InterestIncome::create([
                'iloanid' => $loan->loanid,
                'customerid' => $loan->customerid,
                'ipid' => $loan->loantypeid,
                'charge' => -$amount, // NEGATIVE amount (loss)
                'chargeddate' => $closureDate,
                'poster' => auth()->id(),
                'branchid' => $loan->branchid,
                'companyid' => $loan->companyid,
            ]);
            
            Log::info('Interest written off (loss)', [
                'loan_id' => $loan->loanid,
                'amount' => $amount,
                'closure_date' => $closureDate,
                'reason' => $reason,
                'journal_entry_id' => $journalEntry->id,
                'entry_number' => $entryNumber,
                'table_entry' => 'negative (loss)',
            ]);
            
        } catch (\Exception $e) {
            Log::error('Interest write-off failed', [
                'loan_id' => $loan->loanid,
                'amount' => $amount,
                'error' => $e->getMessage(),
            ]);
            throw $e;
        }
    }

    /**
     * Reverse excess interest (when interest balance is NEGATIVE)
     * When interest balance is NEGATIVE (customer overpaid): Record as POSITIVE in interestincome table (income earned)
     */
    private function reverseExcessInterest($loan, $amount, $closureDate, $reason, $notes = null)
    {
        try {
            // Get interest income account
            $interestIncomeMapping = BusinessProcessMapping::where('nameofthebusinessprocess', 'Interest Income')->first();
            
            if (!$interestIncomeMapping) {
                throw new \Exception('Interest Income account mapping not found.');
            }
            
            $interestIncomeAccountId = $interestIncomeMapping->accountid;
            
            // Get loan receivable account
            $loanReceivableMapping = BusinessProcessMapping::where('nameofthebusinessprocess', 'Loan Disbursement')->first();
            
            if (!$loanReceivableMapping) {
                throw new \Exception('Loan Receivable account mapping not found.');
            }
            
            $loanReceivableAccountId = $loanReceivableMapping->accountid;
            
            // Generate entry number
            $entryNumber = 'JNL-' . date('Ymd') . '-' . str_pad(JournalEntry::count() + 1, 4, '0', STR_PAD_LEFT);
            
            // Create journal entry
            $journalEntry = JournalEntry::create([
                'company_id' => $loan->companyid,
                'branch_id' => $loan->branchid,
                'entry_number' => $entryNumber,
                'entry_date' => $closureDate,
                'reference' => "LOAN-CLOSE-{$loan->loanid}-INT-INC",
                'description' => "Excess interest recognized for loan closure: {$loan->loannumber} - {$reason}",
                'total_debit' => $amount,
                'total_credit' => $amount,
                'status' => 'posted',
                'posted_by' => auth()->id(),
                'posted_at' => now(),
                'created_by' => auth()->id(),
                'loanid' => $loan->loanid,
            ]);
            
            // Journal Entry Item 1: Credit Interest Income (increase income)
            JournalEntryItem::create([
                'company_id' => $loan->companyid,
                'branch_id' => $loan->branchid,
                'journal_entry_id' => $journalEntry->id,
                'account_id' => $interestIncomeAccountId,
                'description' => "Excess interest recognized (income): {$loan->loannumber}",
                'debit' => 0,
                'credit' => $amount,
                'created_by' => auth()->id(),
            ]);
            
            // Journal Entry Item 2: Debit Loan Receivable (increase asset)
            JournalEntryItem::create([
                'company_id' => $loan->companyid,
                'branch_id' => $loan->branchid,
                'journal_entry_id' => $journalEntry->id,
                'account_id' => $loanReceivableAccountId,
                'description' => "Excess interest recognized (income): {$loan->loannumber}",
                'debit' => $amount,
                'credit' => 0,
                'created_by' => auth()->id(),
            ]);
            
            // Create general ledger entries
            // Credit Interest Income (increase income)
            GeneralLedgerEntry::create([
                'entrydate' => $closureDate,
                'description' => "Excess interest recognized (income): Loan {$loan->loannumber} - {$reason}",
                'accountid' => $interestIncomeAccountId,
                'referencedocument' => 'LOAN_CLOSURE',
                'reference_type' => 'journal_entry',
                'reference_id' => $journalEntry->id,
                'documentno' => $entryNumber,
                'entrytype' => 'credit', // CREDIT for income accounts (increase income)
                'amount' => $amount,
                'createdby' => auth()->id(),
                'companyid' => $loan->companyid,
                'branchid' => $loan->branchid,
                'loanid' => $loan->loanid,
            ]);
            
            // Debit Loan Receivable (increase asset)
            GeneralLedgerEntry::create([
                'entrydate' => $closureDate,
                'description' => "Excess interest recognized (income): Loan {$loan->loannumber} - {$reason}",
                'accountid' => $loanReceivableAccountId,
                'referencedocument' => 'LOAN_CLOSURE',
                'reference_type' => 'journal_entry',
                'reference_id' => $journalEntry->id,
                'documentno' => $entryNumber,
                'entrytype' => 'debit', // DEBIT for asset accounts (increase asset)
                'amount' => $amount,
                'createdby' => auth()->id(),
                'companyid' => $loan->companyid,
                'branchid' => $loan->branchid,
                'loanid' => $loan->loanid,
            ]);
            
            // Record in interest income table as POSITIVE (income earned)
            InterestIncome::create([
                'iloanid' => $loan->loanid,
                'customerid' => $loan->customerid,
                'ipid' => $loan->loantypeid,
                'charge' => $amount, // POSITIVE amount (income)
                'chargeddate' => $closureDate,
                'poster' => auth()->id(),
                'branchid' => $loan->branchid,
                'companyid' => $loan->companyid,
            ]);
            
            Log::info('Excess interest recognized as income', [
                'loan_id' => $loan->loanid,
                'amount' => $amount,
                'closure_date' => $closureDate,
                'reason' => $reason,
                'journal_entry_id' => $journalEntry->id,
                'entry_number' => $entryNumber,
                'table_entry' => 'positive (income)',
            ]);
            
        } catch (\Exception $e) {
            Log::error('Excess interest recognition failed', [
                'loan_id' => $loan->loanid,
                'amount' => $amount,
                'error' => $e->getMessage(),
            ]);
            throw $e;
        }
    }

    /**
     * Write off processing fees balance
     * When processing fee balance is POSITIVE (customer owes): Record as NEGATIVE in processingfees table (loss)
     */
    private function writeOffProcessingFees($loan, $amount, $closureDate, $reason, $notes = null)
    {
        try {
            // Get processing fees account
            $processingFeesMapping = BusinessProcessMapping::where('nameofthebusinessprocess', 'Processing Fees')->first();
            
            if (!$processingFeesMapping) {
                throw new \Exception('Processing Fees account mapping not found.');
            }
            
            $processingFeesAccountId = $processingFeesMapping->accountid;
            
            // Get loan receivable account
            $loanReceivableMapping = BusinessProcessMapping::where('nameofthebusinessprocess', 'Loan Disbursement')->first();
            
            if (!$loanReceivableMapping) {
                throw new \Exception('Loan Receivable account mapping not found.');
            }
            
            $loanReceivableAccountId = $loanReceivableMapping->accountid;
            
            // Generate entry number
            $entryNumber = 'JNL-' . date('Ymd') . '-' . str_pad(JournalEntry::count() + 1, 4, '0', STR_PAD_LEFT);
            
            // Create journal entry
            $journalEntry = JournalEntry::create([
                'company_id' => $loan->companyid,
                'branch_id' => $loan->branchid,
                'entry_number' => $entryNumber,
                'entry_date' => $closureDate,
                'reference' => "LOAN-CLOSE-{$loan->loanid}-FEE-WOFF",
                'description' => "Processing fees write-off for loan closure: {$loan->loannumber} - {$reason}",
                'total_debit' => $amount,
                'total_credit' => $amount,
                'status' => 'posted',
                'posted_by' => auth()->id(),
                'posted_at' => now(),
                'created_by' => auth()->id(),
                'loanid' => $loan->loanid,
            ]);
            
            // Journal Entry Item 1: Debit Processing Fees (decrease income - loss)
            JournalEntryItem::create([
                'company_id' => $loan->companyid,
                'branch_id' => $loan->branchid,
                'journal_entry_id' => $journalEntry->id,
                'account_id' => $processingFeesAccountId,
                'description' => "Processing fees write-off (loss): {$loan->loannumber}",
                'debit' => $amount,
                'credit' => 0,
                'created_by' => auth()->id(),
            ]);
            
            // Journal Entry Item 2: Credit Loan Receivable (decrease asset)
            JournalEntryItem::create([
                'company_id' => $loan->companyid,
                'branch_id' => $loan->branchid,
                'journal_entry_id' => $journalEntry->id,
                'account_id' => $loanReceivableAccountId,
                'description' => "Processing fees write-off (loss): {$loan->loannumber}",
                'debit' => 0,
                'credit' => $amount,
                'created_by' => auth()->id(),
            ]);
            
            // Create general ledger entries
            // Debit Processing Fees (decrease income - loss)
            GeneralLedgerEntry::create([
                'entrydate' => $closureDate,
                'description' => "Processing fees write-off (loss): Loan {$loan->loannumber} - {$reason}",
                'accountid' => $processingFeesAccountId,
                'referencedocument' => 'LOAN_CLOSURE',
                'reference_type' => 'journal_entry',
                'reference_id' => $journalEntry->id,
                'documentno' => $entryNumber,
                'entrytype' => 'debit', // DEBIT for income accounts (decrease income)
                'amount' => $amount,
                'createdby' => auth()->id(),
                'companyid' => $loan->companyid,
                'branchid' => $loan->branchid,
                'loanid' => $loan->loanid,
            ]);
            
            // Credit Loan Receivable (decrease asset)
            GeneralLedgerEntry::create([
                'entrydate' => $closureDate,
                'description' => "Processing fees write-off (loss): Loan {$loan->loannumber} - {$reason}",
                'accountid' => $loanReceivableAccountId,
                'referencedocument' => 'LOAN_CLOSURE',
                'reference_type' => 'journal_entry',
                'reference_id' => $journalEntry->id,
                'documentno' => $entryNumber,
                'entrytype' => 'credit', // CREDIT for asset accounts (decrease asset)
                'amount' => $amount,
                'createdby' => auth()->id(),
                'companyid' => $loan->companyid,
                'branchid' => $loan->branchid,
                'loanid' => $loan->loanid,
            ]);
            
            // Record in processing fees table as NEGATIVE (loss/reduction of income)
            ProcessingFees::create([
                'iloanid' => $loan->loanid,
                'customerid' => $loan->customerid,
                'ipid' => $loan->loantypeid,
                'charge' => -$amount, // NEGATIVE amount (loss)
                'chargeddate' => $closureDate,
                'poster' => auth()->id(),
                'branchid' => $loan->branchid,
                'companyid' => $loan->companyid,
            ]);
            
            Log::info('Processing fees written off (loss)', [
                'loan_id' => $loan->loanid,
                'amount' => $amount,
                'closure_date' => $closureDate,
                'reason' => $reason,
                'journal_entry_id' => $journalEntry->id,
                'entry_number' => $entryNumber,
                'table_entry' => 'negative (loss)',
            ]);
            
        } catch (\Exception $e) {
            Log::error('Processing fees write-off failed', [
                'loan_id' => $loan->loanid,
                'amount' => $amount,
                'error' => $e->getMessage(),
            ]);
            throw $e;
        }
    }

    /**
     * Reverse excess processing fees (when processing fee balance is NEGATIVE)
     * When processing fee balance is NEGATIVE (customer overpaid): Record as POSITIVE in processingfees table (income earned)
     */
    private function reverseExcessProcessingFees($loan, $amount, $closureDate, $reason, $notes = null)
    {
        try {
            // Get processing fees account
            $processingFeesMapping = BusinessProcessMapping::where('nameofthebusinessprocess', 'Processing Fees')->first();
            
            if (!$processingFeesMapping) {
                throw new \Exception('Processing Fees account mapping not found.');
            }
            
            $processingFeesAccountId = $processingFeesMapping->accountid;
            
            // Get loan receivable account
            $loanReceivableMapping = BusinessProcessMapping::where('nameofthebusinessprocess', 'Loan Disbursement')->first();
            
            if (!$loanReceivableMapping) {
                throw new \Exception('Loan Receivable account mapping not found.');
            }
            
            $loanReceivableAccountId = $loanReceivableMapping->accountid;
            
            // Generate entry number
            $entryNumber = 'JNL-' . date('Ymd') . '-' . str_pad(JournalEntry::count() + 1, 4, '0', STR_PAD_LEFT);
            
            // Create journal entry
            $journalEntry = JournalEntry::create([
                'company_id' => $loan->companyid,
                'branch_id' => $loan->branchid,
                'entry_number' => $entryNumber,
                'entry_date' => $closureDate,
                'reference' => "LOAN-CLOSE-{$loan->loanid}-FEE-INC",
                'description' => "Excess processing fees recognized for loan closure: {$loan->loannumber} - {$reason}",
                'total_debit' => $amount,
                'total_credit' => $amount,
                'status' => 'posted',
                'posted_by' => auth()->id(),
                'posted_at' => now(),
                'created_by' => auth()->id(),
                'loanid' => $loan->loanid,
            ]);
            
            // Journal Entry Item 1: Credit Processing Fees (increase income)
            JournalEntryItem::create([
                'company_id' => $loan->companyid,
                'branch_id' => $loan->branchid,
                'journal_entry_id' => $journalEntry->id,
                'account_id' => $processingFeesAccountId,
                'description' => "Excess processing fees recognized (income): {$loan->loannumber}",
                'debit' => 0,
                'credit' => $amount,
                'created_by' => auth()->id(),
            ]);
            
            // Journal Entry Item 2: Debit Loan Receivable (increase asset)
            JournalEntryItem::create([
                'company_id' => $loan->companyid,
                'branch_id' => $loan->branchid,
                'journal_entry_id' => $journalEntry->id,
                'account_id' => $loanReceivableAccountId,
                'description' => "Excess processing fees recognized (income): {$loan->loannumber}",
                'debit' => $amount,
                'credit' => 0,
                'created_by' => auth()->id(),
            ]);
            
            // Create general ledger entries
            // Credit Processing Fees (increase income)
            GeneralLedgerEntry::create([
                'entrydate' => $closureDate,
                'description' => "Excess processing fees recognized (income): Loan {$loan->loannumber} - {$reason}",
                'accountid' => $processingFeesAccountId,
                'referencedocument' => 'LOAN_CLOSURE',
                'reference_type' => 'journal_entry',
                'reference_id' => $journalEntry->id,
                'documentno' => $entryNumber,
                'entrytype' => 'credit', // CREDIT for income accounts (increase income)
                'amount' => $amount,
                'createdby' => auth()->id(),
                'companyid' => $loan->companyid,
                'branchid' => $loan->branchid,
                'loanid' => $loan->loanid,
            ]);
            
            // Debit Loan Receivable (increase asset)
            GeneralLedgerEntry::create([
                'entrydate' => $closureDate,
                'description' => "Excess processing fees recognized (income): Loan {$loan->loannumber} - {$reason}",
                'accountid' => $loanReceivableAccountId,
                'referencedocument' => 'LOAN_CLOSURE',
                'reference_type' => 'journal_entry',
                'reference_id' => $journalEntry->id,
                'documentno' => $entryNumber,
                'entrytype' => 'debit', // DEBIT for asset accounts (increase asset)
                'amount' => $amount,
                'createdby' => auth()->id(),
                'companyid' => $loan->companyid,
                'branchid' => $loan->branchid,
                'loanid' => $loan->loanid,
            ]);
            
            // Record in processing fees table as POSITIVE (income earned)
            ProcessingFees::create([
                'iloanid' => $loan->loanid,
                'customerid' => $loan->customerid,
                'ipid' => $loan->loantypeid,
                'charge' => $amount, // POSITIVE amount (income)
                'chargeddate' => $closureDate,
                'poster' => auth()->id(),
                'branchid' => $loan->branchid,
                'companyid' => $loan->companyid,
            ]);
            
            Log::info('Excess processing fees recognized as income', [
                'loan_id' => $loan->loanid,
                'amount' => $amount,
                'closure_date' => $closureDate,
                'reason' => $reason,
                'journal_entry_id' => $journalEntry->id,
                'entry_number' => $entryNumber,
                'table_entry' => 'positive (income)',
            ]);
            
        } catch (\Exception $e) {
            Log::error('Excess processing fees recognition failed', [
                'loan_id' => $loan->loanid,
                'amount' => $amount,
                'error' => $e->getMessage(),
            ]);
            throw $e;
        }
    }

    /**
     * Write off penalty balance (optional - implement as needed)
     */
    private function writeOffPenalty($loan, $amount, $closureDate, $reason, $notes = null)
    {
        try {
            // You need to create a penalty income account mapping first
            // For now, we'll just log it and clear the balance without accounting entries
            Log::info('Penalty cleared without accounting entries', [
                'loan_id' => $loan->loanid,
                'amount' => $amount,
                'closure_date' => $closureDate,
                'reason' => $reason,
                'note' => 'Penalty write-off not implemented. Penalty balance cleared without accounting entries.',
            ]);
            
        } catch (\Exception $e) {
            Log::error('Penalty write-off failed', [
                'loan_id' => $loan->loanid,
                'amount' => $amount,
                'error' => $e->getMessage(),
            ]);
            // Don't throw for penalty as it might not be critical
        }
    }
    
    /**
     * View general ledger entries for a specific loan
     */
    public function viewGeneralLedger($id)
    {
        if (!auth()->user()->hasPermission('loans.view')) {
            abort(403, 'You do not have permission to view loan details.');
        }

        $loan = DisbursedLoan::findOrFail($id);
        
        $generalLedgerEntries = GeneralLedgerEntry::where('loanid', $loan->loanid)
            ->with(['account', 'creator'])
            ->orderBy('entrydate', 'desc')
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        return view('admin.loans.general-ledger', compact('loan', 'generalLedgerEntries'));
    }
    
    /**
     * View repayment accounting details
     */
    public function viewRepaymentAccounting($loanId, $repaymentId)
    {
        if (!auth()->user()->hasPermission('loans.view')) {
            abort(403, 'You do not have permission to view loan details.');
        }

        $loan = DisbursedLoan::findOrFail($loanId);
        $repayment = Repayment::findOrFail($repaymentId);
        
        $generalLedgerEntries = GeneralLedgerEntry::where('loanid', $loan->loanid)
            ->where('reference_type', 'loan_repayment')
            ->where('reference_id', $repaymentId)
            ->with(['account'])
            ->orderBy('entrydate', 'desc')
            ->get();

        $journalEntries = JournalEntry::whereHas('items', function($query) use ($repaymentId) {
            $query->where('reference', 'like', '%REPAY-' . $repaymentId . '%');
        })->with(['items.account'])->get();

        return view('admin.loans.repayment-accounting', compact('loan', 'repayment', 'generalLedgerEntries', 'journalEntries'));
    }
    
    /**
     * View repayment schedule for a loan
     */
    public function viewRepaymentSchedule($id)
    {
        if (!auth()->user()->hasPermission('loans.view')) {
            abort(403, 'You do not have permission to view loan details.');
        }

        $loan = DisbursedLoan::with(['customer'])->findOrFail($id);
        
        $paymentSchedules = PaymentSchedule::where('loanid', $loan->loanid)
            ->orderBy('paymentdate', 'asc')
            ->paginate(20);

        return view('admin.loans.repayment-schedule', compact('loan', 'paymentSchedules'));
    }
    
    /**
     * View loan repayment history
     */
    public function viewRepaymentHistory($id)
    {
        if (!auth()->user()->hasPermission('loans.view')) {
            abort(403, 'You do not have permission to view loan details.');
        }

        $loan = DisbursedLoan::with(['customer'])->findOrFail($id);
        
        $repayments = Repayment::where('rloanid', $loan->loanid)
            ->with(['creator'])
            ->orderBy('rdate', 'desc')
            ->paginate(20);

        return view('admin.loans.repayment-history', compact('loan', 'repayments'));
    }
    
    /**
     * View loan statement
     */
    public function viewStatement($id)
    {
        if (!auth()->user()->hasPermission('loans.view')) {
            abort(403, 'You do not have permission to view loan details.');
        }

        $loan = DisbursedLoan::with(['customer', 'loanType', 'repayments', 'paymentSchedules'])->findOrFail($id);
        
        // Calculate totals
        $totalRepayments = $loan->repayments()->sum('rtotal');
        $totalPrincipalPaid = $loan->repayments()->sum('rprincipal');
        $totalInterestPaid = $loan->repayments()->sum('rinterest');
        $totalProcessingPaid = $loan->repayments()->sum('processing_fees_amount_paid');
        $totalPenaltyPaid = $loan->repayments()->sum('penalties');
        
        return view('admin.loans.statement', compact(
            'loan', 
            'totalRepayments',
            'totalPrincipalPaid',
            'totalInterestPaid',
            'totalProcessingPaid',
            'totalPenaltyPaid'
        ));
    }
    
    /**
     * Export loan statement as PDF
     */
    public function exportStatement($id)
    {
        if (!auth()->user()->hasPermission('loans.export')) {
            abort(403, 'You do not have permission to export loan statements.');
        }

        $loan = DisbursedLoan::with(['customer', 'loanType', 'repayments'])->findOrFail($id);
        
        // You can implement PDF generation here using DomPDF, TCPDF, or Laravel Excel
        
        return redirect()->route('admin.loans.show', $loan->loanid)
            ->with('success', 'Statement export functionality to be implemented.');
    }
    
    /**
     * View loan audit trail
     */
    public function viewAuditTrail($id)
    {
        if (!auth()->user()->hasPermission('loans.view')) {
            abort(403, 'You do not have permission to view loan details.');
        }

        $loan = DisbursedLoan::with(['customer'])->findOrFail($id);
        
        // This would typically query an audit trail table
        // For now, we'll just show a placeholder
        
        return view('admin.loans.audit-trail', compact('loan'));
    }
    
    /**
     * Export payment schedule as PDF
     */
    public function exportSchedule($id)
    {
        try {
            if (!auth()->user()->hasPermission('loans.export')) {
                abort(403, 'You do not have permission to export loan schedules.');
            }

            $loan = DisbursedLoan::with(['customer'])->findOrFail($id);
            
            $paymentSchedules = PaymentSchedule::where('loanid', $loan->loanid)
                ->orderBy('paymentdate', 'asc')
                ->get();

            // Create PDF with A4 size
            $pdf = new \TCPDF('P', 'mm', 'A4', true, 'UTF-8', false);
            
            // Set document information
            $pdf->SetCreator('IMMIA Finance');
            $pdf->SetAuthor('IMMIA Finance');
            $pdf->SetTitle('Loan Payment Schedule - ' . $loan->loannumber);
            $pdf->SetSubject('Loan Payment Schedule');
            
            // Remove default header/footer
            $pdf->setPrintHeader(false);
            $pdf->setPrintFooter(false);
            
            // Set margins
            $pdf->SetMargins(15, 15, 15);
            $pdf->SetAutoPageBreak(true, 15);
            
            // Add a page
            $pdf->AddPage();
            
            // Get the HTML content
            $html = $this->renderSchedulePdfHtml($loan, $paymentSchedules);
            
            // Write HTML content
            $pdf->writeHTML($html, true, false, true, false, '');
            
            // Output the PDF
            $filename = 'Loan_Schedule_' . $loan->loannumber . '.pdf';
            
            return response()->make($pdf->Output($filename, 'I'), 200, [
                'Content-Type' => 'application/pdf',
                'Content-Disposition' => 'inline; filename="' . $filename . '"'
            ]);
            
        } catch (\Exception $e) {
            Log::error('PDF Generation Error: ' . $e->getMessage(), [
                'loan_id' => $id,
                'user_id' => auth()->id(),
                'trace' => $e->getTraceAsString()
            ]);
            
            // Return a simple error PDF
            $pdf = new \TCPDF('P', 'mm', 'A4', true, 'UTF-8', false);
            $pdf->setPrintHeader(false);
            $pdf->setPrintFooter(false);
            $pdf->AddPage();
            
            $errorHtml = '<h1>Error Generating PDF</h1>
                         <p>There was an error generating the payment schedule PDF.</p>
                         <p>Error: ' . htmlspecialchars($e->getMessage()) . '</p>
                         <p>Please contact support.</p>';
            
            $pdf->writeHTML($errorHtml, true, false, true, false, '');
            
            return response()->make($pdf->Output('error.pdf', 'I'), 200, [
                'Content-Type' => 'application/pdf'
            ]);
        }
    }

    /**
     * Render HTML for PDF schedule
     */
    private function renderSchedulePdfHtml($loan, $paymentSchedules)
    {
        $customer = $loan->customer;
        $totalPrincipal = $paymentSchedules->sum('principalamount');
        $totalInterest = $paymentSchedules->sum('interestamount');
        $totalProcessingFee = $paymentSchedules->sum('processingfeesamount');
        $totalPenaltyPaid = $paymentSchedules->sum('paid_penalty');
        $totalAmount = $paymentSchedules->sum('totalamount');
        $totalPaid = $paymentSchedules->sum('total_paid');
        
        // Calculate running principal for outstanding balance
        $runningPrincipal = $loan->amount;
        $principalPaid = 0;
        
        $paidCount = $paymentSchedules->where('status', 'paid')->count();
        $scheduledCount = $paymentSchedules->where('status', 'scheduled')->count();
        $overdueCount = $paymentSchedules->where('status', 'overdue')->count();
        $totalCount = $paymentSchedules->count();
        
        // Calculate payment ratio
        $paymentRatio = $totalCount > 0 ? round(($paidCount / $totalCount) * 100, 1) : 0;
        
        // Get processing fee and penalty details
        $processingFeeRate = $loan->processing_fee ?? 0;
        $processingFeeBasis = $loan->processing_fee_basis ?? 'initial_amount';
        $penaltyRate = $loan->penalty_rate ?? 0;
        $penaltyBasis = $loan->penalty_basis ?? 'percentage_of_due';
        
        // Format basis for display
        $formatBasis = function($basis) {
            $mapping = [
                'initial_amount' => 'Initial Loan Amount',
                'outstanding_balance' => 'Outstanding Balance',
                'percentage_of_due' => 'Percentage of Due Amount',
                'fixed_amount' => 'Fixed Amount',
                'daily_rate' => 'Daily Rate'
            ];
            return $mapping[$basis] ?? ucfirst(str_replace('_', ' ', $basis));
        };
        
        $html = '<!DOCTYPE html>
        <html>
        <head>
            <meta charset="UTF-8">
            <title>Loan Payment Schedule - ' . $loan->loannumber . '</title>
            <style>
                /* A4 size: 210mm x 297mm - MINIMAL MARGINS */
                @page { size: A4; margin: 3mm; }
                body { 
                    font-family: Arial, sans-serif; 
                    margin: 0; 
                    padding: 0;
                    font-size: 8pt;
                    width: 204mm; /* 210mm - 6mm (3mm left + 3mm right) */
                    max-width: 204mm;
                }
                .header { 
                    text-align: center; 
                    margin-bottom: 4mm; 
                    border-bottom: 1pt solid #0077C5; 
                    padding-bottom: 2mm; 
                }
                .company-name { 
                    font-size: 11pt; 
                    font-weight: bold; 
                    color: #0077C5; 
                    margin-bottom: 1pt; 
                }
                .document-title { 
                    font-size: 10pt; 
                    margin: 2pt 0; 
                    color: #333; 
                }
                .loan-info { 
                    margin-bottom: 3mm; 
                    background-color: #f8f9fa; 
                    padding: 2mm; 
                    border-radius: 1pt; 
                    border: 0.5pt solid #dee2e6; 
                    font-size: 7pt;
                }
                .loan-info h3 { 
                    margin-top: 0; 
                    color: #0077C5; 
                    font-size: 8pt; 
                    margin-bottom: 1mm; 
                    text-align: left;
                }
                .loan-info table { 
                    width: 100%; 
                    border-collapse: collapse; 
                    font-size: 7pt; 
                    table-layout: fixed;
                }
                .loan-info td { 
                    padding: 1pt 2pt; 
                    border-bottom: 0.5pt solid #ddd; 
                    word-wrap: break-word;
                    overflow-wrap: break-word;
                    vertical-align: top;
                }
                .loan-info .label { 
                    font-weight: bold; 
                    background-color: #e9ecef; 
                    text-align: left;
                }
                .loan-info .value { 
                    text-align: left;
                }
                .loan-info .amount-value { 
                    text-align: left;
                    font-family: "Courier New", monospace;
                }
                /* Specific column widths for loan-info table - 4 equal columns */
                .loan-info .col-1 { width: 25%; }
                .loan-info .col-2 { width: 25%; }
                .loan-info .col-3 { width: 25%; }
                .loan-info .col-4 { width: 25%; }
                
                .schedule-table { 
                    width: 100% !important; 
                    border-collapse: collapse; 
                    margin-top: 3mm; 
                    font-size: 6pt; 
                    table-layout: fixed;
                }
                .schedule-table th, 
                .schedule-table td { 
                    border: 0.5pt solid #ddd; 
                    padding: 2pt 1pt; 
                    text-align: center; 
                    font-size: 6pt;
                    white-space: nowrap;
                    height: 10pt;
                    vertical-align: middle;
                    overflow: hidden;
                    box-sizing: border-box;
                }
                .schedule-table th { 
                    background-color: #0077C5; 
                    color: white; 
                    font-weight: bold;
                    border: 0.5pt solid #005a94;
                    font-size: 6pt;
                    padding: 2pt 1pt;
                }
                /* ADJUSTED COLUMN WIDTHS TO USE 100% OF AVAILABLE SPACE */
                .schedule-table .col-1 { width: 4%; }
                .schedule-table .col-2 { width: 9%; }
                .schedule-table .col-3 { width: 8%; }
                .schedule-table .col-4 { width: 8%; }
                .schedule-table .col-5 { width: 8%; }
                .schedule-table .col-6 { width: 8%; }
                .schedule-table .col-7 { width: 12%; }
                .schedule-table .col-8 { width: 10%; }
                .schedule-table .col-9 { width: 8%; }
                .schedule-table .col-10 { width: 12%; }
                .schedule-table .col-11 { width: 13%; } /* Added for balance column */
                
                .schedule-table tr:nth-child(even) { 
                    background-color: #f9f9f9; 
                }
                .schedule-table .paid-row { 
                    background-color: #d4edda; 
                    font-weight: bold;
                    color: #155724;
                }
                .schedule-table .overdue-row { 
                    background-color: #f8d7da; 
                    font-weight: bold;
                    color: #721c24;
                }
                .schedule-table .total-row { 
                    font-weight: bold; 
                    background-color: #cce5ff; 
                    color: #004085;
                    font-size: 6pt;
                }
                .schedule-table .total-label { 
                    text-align: left !important;
                    padding-left: 3pt !important;
                }
                .footer { 
                    margin-top: 4mm; 
                    text-align: center; 
                    font-size: 5pt; 
                    color: #666;
                    border-top: 0.5pt solid #ddd;
                    padding-top: 2mm;
                }
                .amount { 
                    text-align: right; 
                    font-family: "Courier New", monospace; 
                    font-size: 6pt;
                    padding-right: 3pt !important;
                }
                .negative-amount { color: #dc3545; }
                .positive-amount { color: #28a745; }
                .summary-box {
                    background-color: #f8f9fa;
                    padding: 3mm;
                    border-radius: 1pt;
                    border: 0.5pt solid #dee2e6;
                    margin-top: 4mm;
                    font-size: 7pt;
                }
                .summary-box h3 { 
                    color: #0077C5; 
                    margin-top: 0; 
                    border-bottom: 0.5pt solid #0077C5;
                    padding-bottom: 1mm;
                    font-size: 8pt;
                    margin-bottom: 2mm;
                    text-align: left;
                }
                .no-wrap { white-space: nowrap; }
                .text-left { text-align: left; }
                .text-right { text-align: right; }
                .text-center { text-align: center; }
                .currency-note {
                    font-size: 6pt;
                    color: #666;
                    text-align: center;
                    margin-top: 1mm;
                    font-style: italic;
                }
                .status-badge {
                    display: inline-block;
                    padding: 0.5pt 1.5pt;
                    border-radius: 0.5pt;
                    font-size: 5pt;
                    font-weight: bold;
                    white-space: nowrap;
                }
                .status-paid { background-color: #d4edda; color: #155724; }
                .status-overdue { background-color: #f8d7da; color: #721c24; }
                .status-scheduled { background-color: #cce5ff; color: #004085; }
                .status-closed { background-color: #e9ecef; color: #495057; }
                .principal-reduction {
                    font-size: 5pt;
                    color: #6c757d;
                    display: block;
                    margin-top: 0.5pt;
                    white-space: nowrap;
                }
                /* Force no text wrapping in table cells */
                .no-wrap-cell {
                    white-space: nowrap !important;
                    overflow: hidden !important;
                }
                /* Ensure table takes full width */
                .full-width {
                    width: 100% !important;
                    max-width: 100% !important;
                }
            </style>
        </head>
        <body>
            <div class="header">
                <div class="company-name">IMMIA FINANCE</div>
                <div class="document-title">LOAN PAYMENT SCHEDULE</div>
                <div style="font-size: 7pt; color: #666; margin-top: 1pt;">
                    Loan: ' . $loan->loannumber . ' | Date: ' . date('d/m/Y H:i') . '
                </div>
            </div>
            
            <div class="loan-info">
                <h3>Loan Details</h3>
                <table>
                    <tr>
                        <td class="label col-1">Customer Name:</td>
                        <td class="value col-2">' . htmlspecialchars($customer ? $customer->first_name . ' ' . $customer->surname : 'N/A') . '</td>
                        <td class="label col-3">Customer No:</td>
                        <td class="value col-4">' . htmlspecialchars($customer ? $customer->customer_number : 'N/A') . '</td>
                    </tr>
                    <tr>
                        <td class="label col-1">Loan Number:</td>
                        <td class="value col-2">' . htmlspecialchars($loan->loannumber) . '</td>
                        <td class="label col-3">Loan Amount:</td>
                        <td class="amount-value col-4"><strong>' . number_format($loan->amount, 2) . '</strong></td>
                    </tr>
                    <tr>
                        <td class="label col-1">Interest Rate:</td>
                        <td class="value col-2">' . number_format($loan->interestrate, 2) . '%</td>
                        <td class="label col-3">Term:</td>
                        <td class="value col-4">' . $loan->loanterm . ' ' . ($loan->paymentfrequency === 'monthly' ? 'Months' : ucfirst($loan->paymentfrequency)) . '</td>
                    </tr>
                    <tr>
                        <td class="label col-1">Payment Frequency:</td>
                        <td class="value col-2">' . ucfirst(str_replace('_', ' ', $loan->paymentfrequency)) . '</td>
                        <td class="label col-3">Disbursement Date:</td>
                        <td class="value col-4">' . ($loan->disburseddate ? \Carbon\Carbon::parse($loan->disburseddate)->format('d/m/Y') : 'N/A') . '</td>
                    </tr>
                    <tr>
                        <td class="label col-1">Maturity Date:</td>
                        <td class="value col-2">' . ($loan->maturitydate ? \Carbon\Carbon::parse($loan->maturitydate)->format('d/m/Y') : 'N/A') . '</td>
                        <td class="label col-3">Current Status:</td>
                        <td class="value col-4"><strong>' . ucfirst($loan->status) . '</strong></td>
                    </tr>
                </table>
            </div>
            
            <div class="currency-note">
                All amounts in Zambian Kwacha
            </div>';
            
            if ($paymentSchedules->isEmpty()) {
                $html .= '
                <div style="color: #856404; background-color: #fff3cd; padding: 2mm; border-radius: 1pt; margin: 3mm 0; border: 0.5pt solid #ffeaa7;">
                    <strong>No payment schedule data available</strong>
                </div>';
            } else {
                $html .= '
                <h3 style="color: #0077C5; margin-top: 4mm; padding-bottom: 1mm; border-bottom: 0.5pt solid #0077C5; font-size: 8pt; text-align: left;">
                    Payment Schedule (' . $paymentSchedules->count() . ' Installments)
                </h3>
                <table class="schedule-table full-width">
                    <thead>
                        <tr>
                            <th class="col-1 no-wrap-cell">#</th>
                            <th class="col-2 no-wrap-cell">Due Date</th>
                            <th class="col-3 no-wrap-cell">Principal</th>
                            <th class="col-4 no-wrap-cell">Interest</th>
                            <th class="col-5 no-wrap-cell">Fee</th>
                            <th class="col-6 no-wrap-cell">Penalty</th>
                            <th class="col-7 no-wrap-cell">Outstanding Principal</th>
                            <th class="col-8 no-wrap-cell">Total Due</th>
                            <th class="col-9 no-wrap-cell">Status</th>
                            <th class="col-10 no-wrap-cell">Amount Paid</th>
                            <th class="col-11 no-wrap-cell">Balance</th>
                        </tr>
                    </thead>
                    <tbody>';
                
                $rowNumber = 1;
                
                foreach ($paymentSchedules as $schedule) {
                    $outstandingPrincipal = $runningPrincipal;
                    
                    // Reduce the running principal for next iteration
                    if ($schedule->status === 'paid') {
                        $runningPrincipal -= $schedule->principalamount;
                        $principalPaid += $schedule->principalamount;
                    } else {
                        $runningPrincipal -= $schedule->principalamount;
                    }
                    
                    $runningPrincipal = max(0, $runningPrincipal);
                    
                    $rowClass = '';
                    if ($schedule->status === 'paid') {
                        $rowClass = 'paid-row';
                    } elseif ($schedule->status === 'overdue') {
                        $rowClass = 'overdue-row';
                    }
                    
                    $statusClass = 'status-' . $schedule->status;
                    $statusBadge = '<span class="status-badge ' . $statusClass . '">' . ucfirst($schedule->status) . '</span>';
                    
                    $balance = $schedule->totalamount - $schedule->total_paid;
                    
                    $html .= '
                        <tr class="' . $rowClass . '">
                            <td class="col-1 no-wrap-cell">' . $rowNumber . '</td>
                            <td class="col-2 no-wrap-cell">' . ($schedule->paymentdate ? \Carbon\Carbon::parse($schedule->paymentdate)->format('d/m/Y') : 'N/A') . '</td>
                            <td class="col-3 amount no-wrap-cell">' . number_format($schedule->principalamount, 2) . '</td>
                            <td class="col-4 amount no-wrap-cell">' . number_format($schedule->interestamount, 2) . '</td>
                            <td class="col-5 amount no-wrap-cell">' . number_format($schedule->processingfeesamount, 2) . '</td>
                            <td class="col-6 amount no-wrap-cell">' . number_format($schedule->paid_penalty, 2) . '</td>
                            <td class="col-7 amount no-wrap-cell">
                                ' . number_format($outstandingPrincipal, 2);
                    
                    if ($schedule->status === 'paid' && $schedule->principalamount > 0) {
                        $html .= '<span class="principal-reduction">→ ' . number_format($runningPrincipal, 2) . '</span>';
                    }
                    
                    $html .= '
                            </td>
                            <td class="col-8 amount no-wrap-cell"><strong>' . number_format($schedule->totalamount, 2) . '</strong></td>
                            <td class="col-9 no-wrap-cell">' . $statusBadge . '</td>
                            <td class="col-10 amount no-wrap-cell">' . number_format($schedule->total_paid, 2) . '</td>
                            <td class="col-11 amount no-wrap-cell ' . ($balance > 0 ? 'negative-amount' : 'positive-amount') . '">' . number_format($balance, 2) . '</td>
                        </tr>';
                    
                    $rowNumber++;
                }
                
                $remainingBalanceTotal = $totalAmount - $totalPaid;
                
                $html .= '
                        <tr class="total-row">
                            <td class="total-label" colspan="2"><strong>TOTALS</strong></td>
                            <td class="col-3 amount no-wrap-cell"><strong>' . number_format($totalPrincipal, 2) . '</strong></td>
                            <td class="col-4 amount no-wrap-cell"><strong>' . number_format($totalInterest, 2) . '</strong></td>
                            <td class="col-5 amount no-wrap-cell"><strong>' . number_format($totalProcessingFee, 2) . '</strong></td>
                            <td class="col-6 amount no-wrap-cell"><strong>' . number_format($totalPenaltyPaid, 2) . '</strong></td>
                            <td class="col-7 amount no-wrap-cell"><strong>' . number_format($runningPrincipal, 2) . '</strong></td>
                            <td class="col-8 amount no-wrap-cell"><strong>' . number_format($totalAmount, 2) . '</strong></td>
                            <td class="col-9 no-wrap-cell">' . $paidCount . ' paid / ' . $totalCount . ' total</td>
                            <td class="col-10 amount no-wrap-cell"><strong>' . number_format($totalPaid, 2) . '</strong></td>
                            <td class="col-11 amount no-wrap-cell ' . ($remainingBalanceTotal > 0 ? 'negative-amount' : 'positive-amount') . '"><strong>' . number_format($remainingBalanceTotal, 2) . '</strong></td>
                        </tr>';
                
                $html .= '
                    </tbody>
                </table>';
                
                $totalCost = $totalInterest + $totalProcessingFee + $totalPenaltyPaid;
                $effectiveInterestRate = $loan->amount > 0 ? ($totalCost / $loan->amount) * 100 : 0;
                $remainingBalance = $totalAmount - $totalPaid;
                
                $html .= '
                <div class="summary-box">
                    <h3>Financial Summary</h3>
                    <table style="width: 100%; font-size: 7pt; table-layout: fixed;">
                        <tr>
                            <td style="width: 60%; padding: 1pt; text-align: left;"><strong>Installment Statistics:</strong></td>
                            <td style="width: 40%; text-align: right; padding: 1pt;"></td>
                        </tr>
                        <tr>
                            <td style="padding: 1pt 1pt 1pt 8pt; text-align: left;">Total Installments:</td>
                            <td style="text-align: right; padding: 1pt;">' . $paymentSchedules->count() . '</td>
                        </tr>
                        <tr>
                            <td style="padding: 1pt 1pt 1pt 8pt; text-align: left;">Paid:</td>
                            <td style="text-align: right; padding: 1pt; color: #28a745;">' . $paidCount . ' (' . $paymentRatio . '%)</td>
                        </tr>
                        <tr>
                            <td style="padding: 1pt 1pt 1pt 8pt; text-align: left;">Scheduled:</td>
                            <td style="text-align: right; padding: 1pt; color: #007bff;">' . $scheduledCount . '</td>
                        </tr>
                        <tr>
                            <td style="padding: 1pt 1pt 1pt 8pt; text-align: left;">Overdue:</td>
                            <td style="text-align: right; padding: 1pt; color: #dc3545;">' . $overdueCount . '</td>
                        </tr>
                        <tr>
                            <td style="width: 60%; padding: 1pt; text-align: left;"><strong>Financial Breakdown:</strong></td>
                            <td style="width: 40%; text-align: right; padding: 1pt;"></td>
                        </tr>
                        <tr>
                            <td style="padding: 1pt 1pt 1pt 8pt; text-align: left;">Loan Amount:</td>
                            <td style="text-align: right; padding: 1pt;">' . number_format($loan->amount, 2) . '</td>
                        </tr>
                        <tr>
                            <td style="padding: 1pt 1pt 1pt 8pt; text-align: left;">Principal Paid:</td>
                            <td style="text-align: right; padding: 1pt; color: #28a745;">' . number_format($principalPaid, 2) . '</td>
                        </tr>
                        <tr>
                            <td style="padding: 1pt 1pt 1pt 8pt; text-align: left;">Principal Outstanding:</td>
                            <td style="text-align: right; padding: 1pt; color: #ffc107;">' . number_format($runningPrincipal, 2) . '</td>
                        </tr>
                        <tr>
                            <td style="padding: 1pt 1pt 1pt 8pt; text-align: left;">Total Interest:</td>
                            <td style="text-align: right; padding: 1pt; color: #dc3545;">' . number_format($totalInterest, 2) . '</td>
                        </tr>
                        <tr>
                            <td style="padding: 1pt 1pt 1pt 8pt; text-align: left;">Total Processing Fees:</td>
                            <td style="text-align: right; padding: 1pt; color: #dc3545;">' . number_format($totalProcessingFee, 2) . '</td>
                        </tr>
                        <tr>
                            <td style="padding: 1pt 1pt 1pt 8pt; text-align: left;">Total Penalties:</td>
                            <td style="text-align: right; padding: 1pt; color: #dc3545;">' . number_format($totalPenaltyPaid, 2) . '</td>
                        </tr>
                        <tr style="border-top: 0.5pt solid #0077C5;">
                            <td style="padding: 1pt; font-weight: bold; text-align: left;"><strong>Total Cost:</strong></td>
                            <td style="text-align: right; padding: 1pt; font-weight: bold; color: #dc3545;">' . number_format($totalCost, 2) . '</td>
                        </tr>
                        <tr>
                            <td style="padding: 1pt; font-weight: bold; text-align: left;"><strong>Total Repayment:</strong></td>
                            <td style="text-align: right; padding: 1pt; font-weight: bold;">' . number_format($totalAmount, 2) . '</td>
                        </tr>
                        <tr>
                            <td style="padding: 1pt; font-weight: bold; text-align: left;"><strong>Amount Paid:</strong></td>
                            <td style="text-align: right; padding: 1pt; font-weight: bold; color: #28a745;">' . number_format($totalPaid, 2) . '</td>
                        </tr>
                        <tr>
                            <td style="padding: 1pt; font-weight: bold; text-align: left;"><strong>Outstanding Balance:</strong></td>
                            <td style="text-align: right; padding: 1pt; font-weight: bold; color: #ffc107;">' . number_format($remainingBalance, 2) . '</td>
                        </tr>
                        <tr>
                            <td style="padding: 1pt; font-weight: bold; text-align: left;"><strong>Principal Repayment Rate:</strong></td>
                            <td style="text-align: right; padding: 1pt; font-weight: bold;">' . number_format(($principalPaid / $loan->amount) * 100, 2) . '%</td>
                        </tr>
                        <tr>
                            <td style="padding: 1pt; font-weight: bold; text-align: left;"><strong>Effective Interest Rate:</strong></td>
                            <td style="text-align: right; padding: 1pt; font-weight: bold;">' . number_format($effectiveInterestRate, 2) . '%</td>
                        </tr>
                        <tr>
                            <td style="padding: 1pt; font-weight: bold; text-align: left;"><strong>Completion Rate:</strong></td>
                            <td style="text-align: right; padding: 1pt; font-weight: bold; ' . ($paymentRatio >= 80 ? 'color: #28a745;' : ($paymentRatio >= 50 ? 'color: #ffc107;' : 'color: #dc3545;')) . '">' . $paymentRatio . '%</td>
                        </tr>
                    </table>
                </div>
                
                <div style="margin-top: 3mm; font-size: 6pt; color: #666; border: 0.5pt solid #dee2e6; padding: 2mm; border-radius: 1pt;">
                    <h4 style="margin-top: 0; margin-bottom: 1mm; color: #0077C5; font-size: 7pt;">Fee & Penalty Details:</h4>
                    <table style="width: 100%; border-collapse: collapse; font-size: 6pt;">
                        <tr>
                            <td style="width: 50%; padding: 1pt; vertical-align: top;">
                                <strong>Processing Fee:</strong><br>
                                Rate: ' . number_format($processingFeeRate, 2) . '%<br>
                                Basis: ' . $formatBasis($processingFeeBasis) . '<br>
                                Total: ' . number_format($totalProcessingFee, 2) . '
                            </td>
                            <td style="width: 50%; padding: 1pt; vertical-align: top;">
                                <strong>Penalty:</strong><br>
                                Rate: ' . number_format($penaltyRate, 2) . '%<br>
                                Basis: ' . $formatBasis($penaltyBasis) . '<br>
                                Paid: ' . number_format($totalPenaltyPaid, 2) . '
                            </td>
                        </tr>
                    </table>
                </div>
                
                <div style="margin-top: 2mm; font-size: 5pt; color: #666; padding: 1mm; background-color: #f8f9fa; border-radius: 1pt;">
                    <strong>Status Legend:</strong> 
                    <span style="background-color: #d4edda; color: #155724; padding: 0.5pt 1.5pt; border-radius: 0.5pt; margin-left: 1mm;">Paid</span>
                    <span style="background-color: #cce5ff; color: #004085; padding: 0.5pt 1.5pt; border-radius: 0.5pt; margin-left: 0.5mm;">Scheduled</span>
                    <span style="background-color: #f8d7da; color: #721c24; padding: 0.5pt 1.5pt; border-radius: 0.5pt; margin-left: 0.5mm;">Overdue</span>
                    <span style="background-color: #e9ecef; color: #495057; padding: 0.5pt 1.5pt; border-radius: 0.5pt; margin-left: 0.5mm;">Closed</span>
                    <span style="margin-left: 1mm;">Arrow (→) shows principal reduction after payment</span>
                </div>';
            }
            
            $html .= '
                <div class="footer">
                    <p style="margin-bottom: 1pt; font-size: 5pt;"><strong>IMMIA FINANCE LTD</strong></p>
                    <p style="margin-bottom: 1pt; font-size: 4pt;">Computer generated document - No signature required</p>
                    <p style="margin-top: 1mm; font-size: 4pt; color: #999;">
                        Generated: ' . date('d/m/Y H:i:s') . ' | Page 1 of 1
                    </p>
                </div>
            </body>
            </html>';
            
            return $html;
    }
    
    /**
     * View loan account statement (Detailed transaction history)
     * Route: GET /admin/loans/{id}/statement
     */
    public function viewLoanStatement($id, Request $request)
    {
        if (!auth()->user()->hasPermission('loans.view')) {
            abort(403, 'You do not have permission to view loan statements.');
        }

        $loan = DisbursedLoan::with(['customer', 'loanType', 'repayments'])->findOrFail($id);
        
        // Get date range from request or use defaults
        $startDate = $request->input('start_date', now()->subMonths(3)->format('Y-m-d'));
        $endDate = $request->input('end_date', now()->format('Y-m-d'));
        
        // Load additional data if needed
        $loan->load(['paymentSchedules' => function($query) {
            $query->orderBy('paymentdate', 'asc');
        }]);
        
        // Prepare transactions array
        $transactions = collect();
        
        // 1. Add loan disbursement as opening transaction
        if ($loan->disburseddate) {
            $transactions->push([
                'date' => $loan->disburseddate,
                'description' => 'Loan Disbursement',
                'reference' => 'DISB-' . $loan->loanid,
                'transaction_type' => 'disbursement',
                'type' => 'debit', // Money going out from company
                'amount' => (float) $loan->amount,
                'principal' => (float) $loan->amount,
                'interest' => 0,
                'penalty' => 0,
                'processing_fee' => 0,
            ]);
        }
        
        // 2. Add repayments within the selected date range
        foreach ($loan->repayments as $repayment) {
            if ($repayment->rdate >= $startDate && $repayment->rdate <= $endDate) {
                $transactions->push([
                    'date' => $repayment->rdate,
                    'description' => 'Loan Repayment',
                    'reference' => $repayment->paymentreference ?? 'REPAY-' . $repayment->repaymentid,
                    'transaction_type' => 'repayment',
                    'type' => 'credit', // Money coming in
                    'amount' => (float) $repayment->rtotal,
                    'principal' => (float) $repayment->rprincipal,
                    'interest' => (float) $repayment->rinterest,
                    'penalty' => (float) $repayment->penalties,
                    'processing_fee' => (float) $repayment->processing_fees_amount_paid,
                    'status' => $repayment->status,
                ]);
            }
        }
        
        // Sort transactions by date
        $transactions = $transactions->sortBy('date')->values();
        
        // Calculate summary
        $totalCredits = $transactions->where('type', 'credit')->sum('amount');
        $totalDebits = $transactions->where('type', 'debit')->sum('amount');
        
        // Calculate opening balance (loan amount minus principal paid before start date)
        $principalPaidBefore = $loan->repayments
            ->where('rdate', '<', $startDate)
            ->sum('rprincipal');
        
        $openingBalance = max(0, $loan->amount - $principalPaidBefore);
        
        // Calculate closing balance
        $closingBalance = $openingBalance + $totalCredits - $totalDebits;
        
        // Get upcoming payments
        $upcomingPayments = $loan->paymentSchedules
            ->where('status', '!=', 'paid')
            ->where('balance', '>', 0)
            ->sortBy('paymentdate')
            ->take(5);
        
        // Calculate schedule summary
        $scheduleSummary = [
            'total_installments' => $loan->paymentSchedules->count(),
            'paid_installments' => $loan->paymentSchedules->where('status', 'paid')->count(),
            'overdue_payments' => $loan->paymentSchedules->where('status', 'overdue')->count(),
            'scheduled_payments' => $loan->paymentSchedules->where('status', 'scheduled')->count(),
            'total_amount_due' => $loan->paymentSchedules->sum('totalamount'),
            'total_paid' => $loan->paymentSchedules->sum('total_paid'),
            'total_outstanding' => $loan->paymentSchedules->sum('balance'),
        ];
        
        // Calculate payment consistency
        $paymentConsistency = $this->calculatePaymentConsistency($loan);
        
        return view('admin.loans.statement', compact(
            'loan',
            'transactions',
            'startDate',
            'endDate',
            'openingBalance',
            'totalCredits',
            'totalDebits',
            'closingBalance',
            'upcomingPayments',
            'scheduleSummary',
            'paymentConsistency'
        ));
    }

    /**
     * Helper method to calculate payment consistency
     */
    private function calculatePaymentConsistency($loan)
    {
        if ($loan->repayments->isEmpty() || !$loan->paymentSchedules->count()) {
            return 0;
        }
        
        // Get scheduled payments that are due
        $duePayments = $loan->paymentSchedules
            ->where('paymentdate', '<=', now())
            ->where('status', '!=', 'paid')
            ->count();
        
        $paidPayments = $loan->repayments
            ->where('status', 'posted')
            ->count();
        
        $totalDuePayments = $loan->paymentSchedules
            ->where('paymentdate', '<=', now())
            ->count();
        
        if ($totalDuePayments === 0) {
            return 100; // No payments due yet
        }
        
        $consistency = ($paidPayments / $totalDuePayments) * 100;
        
        return min(100, max(0, $consistency));
    }
    
    /**
 * When NPL loan receives partial payment
 */
private function recordNPLRecovery($loan, $paymentAmount, $paymentDate)
{
    try {
        // Record the payment as normal (this would be in your repayment method)
        // Then check if we need to reverse excess provision
        
        $remainingBalanceAfterPayment = $loan->totalbalance - $paymentAmount;
        $requiredProvision = $remainingBalanceAfterPayment * ($loan->provision_percentage / 100);
        $currentProvision = $loan->provision_amount ?? 0;
        
        $excessProvision = max(0, $currentProvision - $requiredProvision);
        
        if ($excessProvision > 0) {
            // Reverse excess provision (income recognition)
            $this->reverseExcessProvision($loan, $excessProvision, $paymentDate);
        }
        
    } catch (\Exception $e) {
        Log::error('NPL recovery recording failed', [
            'loan_id' => $loan->loanid,
            'error' => $e->getMessage(),
        ]);
        throw $e;
    }
}

/**
 * Reverse excess provision when NPL loan recovers
 */
private function reverseExcessProvision($loan, $excessAmount, $reversalDate)
{
    try {
        // Get accounts from Business Process Mapping
        $provisionMapping = BusinessProcessMapping::getProvisionAccount();
        $lossExpenseMapping = BusinessProcessMapping::getLoanLossExpenseAccount();
        $recoveryIncomeMapping = BusinessProcessMapping::getRecoveredLoanLossAccount();
        
        if (!$provisionMapping || !$lossExpenseMapping || !$recoveryIncomeMapping) {
            throw new \Exception('Required account mappings not found for provision reversal.');
        }
        
        // Generate entry number
        $entryNumber = 'JNL-' . date('Ymd') . '-' . str_pad(JournalEntry::count() + 1, 4, '0', STR_PAD_LEFT);
        
        // Create journal entry for provision reversal
        $journalEntry = JournalEntry::create([
            'company_id' => $loan->companyid,
            'branch_id' => $loan->branchid,
            'entry_number' => $entryNumber,
            'entry_date' => $reversalDate,
            'reference' => "NPL-PROV-REV-{$loan->loanid}",
            'description' => "Excess provision reversal - {$loan->loannumber}",
            'total_debit' => $excessAmount,
            'total_credit' => $excessAmount,
            'status' => 'posted',
            'posted_by' => auth()->id(),
            'posted_at' => now(),
            'created_by' => auth()->id(),
            'loanid' => $loan->loanid,
        ]);
        
        // 1. Dr. Allowance for Loan Losses (reduce provision)
        JournalEntryItem::create([
            'company_id' => $loan->companyid,
            'branch_id' => $loan->branchid,
            'journal_entry_id' => $journalEntry->id,
            'account_id' => $provisionMapping->id,
            'description' => "Excess provision reversal - {$loan->loannumber}",
            'debit' => $excessAmount,
            'credit' => 0,
            'created_by' => auth()->id(),
        ]);
        
        // 2. Cr. Recovered Loan Loss Income (recognize income)
        JournalEntryItem::create([
            'company_id' => $loan->companyid,
            'branch_id' => $loan->branchid,
            'journal_entry_id' => $journalEntry->id,
            'account_id' => $recoveryIncomeMapping->id,
            'description' => "Recovery income - {$loan->loannumber}",
            'debit' => 0,
            'credit' => $excessAmount,
            'created_by' => auth()->id(),
        ]);
        
        Log::info('Excess provision reversed', [
            'loan_id' => $loan->loanid,
            'excess_amount' => $excessAmount,
            'journal_entry_id' => $journalEntry->id,
        ]);
        
    } catch (\Exception $e) {
        Log::error('Provision reversal failed', [
            'loan_id' => $loan->loanid,
            'error' => $e->getMessage(),
        ]);
        throw $e;
    }
}

/**
 * Create NPL provision entries (using BusinessProcessMapping)
 */
/**
 * Create NPL provision entries (using BusinessProcessMapping)
 */
private function createNPLProvisions($loan, $classificationDate, $category, $provisionPercentage)
{
    DB::beginTransaction();
    
    try {
        // Calculate total provision amount
        $totalProvision = $loan->totalbalance * ($provisionPercentage / 100);
        
        // Get accounts from Business Process Mapping
        $loanReceivableMapping = BusinessProcessMapping::where('nameofthebusinessprocess', 'Loan Disbursement')->first();
        if (!$loanReceivableMapping) {
            throw new \Exception('Loan Disbursement account mapping not found.');
        }
        $loanReceivableAccountId = $loanReceivableMapping->accountid;
        
        // Get provision account (contra asset)
        $provisionMapping = BusinessProcessMapping::where('nameofthebusinessprocess', 'LIKE', '%Provision%')
            ->orWhere('nameofthebusinessprocess', 'LIKE', '%Allowance%')
            ->orWhere('nameofthebusinessprocess', 'LIKE', '%Loan Loss%')
            ->first();
        
        if (!$provisionMapping) {
            throw new \Exception('Provision account mapping not found. Please set up "Loan Loss Provision" or "Allowance for Loan Losses" in Business Process Mapping.');
        }
        $provisionAccountId = $provisionMapping->accountid;
        
        // Get loan loss expense account
        $lossExpenseMapping = BusinessProcessMapping::where('nameofthebusinessprocess', 'LIKE', '%Loan Loss Expense%')
            ->orWhere('nameofthebusinessprocess', 'LIKE', '%Provision Expense%')
            ->orWhere('nameofthebusinessprocess', 'LIKE', '%Bad Debt Expense%')
            ->first();
        
        if (!$lossExpenseMapping) {
            throw new \Exception('Loan Loss Expense account mapping not found. Please set up "Loan Loss Expense" in Business Process Mapping.');
        }
        $lossExpenseAccountId = $lossExpenseMapping->accountid;
        
        // Generate entry number
        $entryCount = JournalEntry::whereDate('created_at', today())->count() + 1;
        $entryNumber = 'JNL-' . date('Ymd') . '-' . str_pad($entryCount, 4, '0', STR_PAD_LEFT);
        
        // Get the current financial year ID
        $company = Company::find($loan->companyid);
        $financialYearId = null;
        
        if ($company && $company->current_financial_year_id) {
            $financialYearId = $company->current_financial_year_id;
        } else {
            // Fallback: get the current financial year for the company
            $financialYear = FinancialYear::where('company_id', $loan->companyid)
                ->where('is_active', true)
                ->where('start_date', '<=', now())
                ->where('end_date', '>=', now())
                ->first();
            
            if (!$financialYear) {
                // Get any active financial year
                $financialYear = FinancialYear::where('company_id', $loan->companyid)
                    ->where('is_active', true)
                    ->orderBy('start_date', 'desc')
                    ->first();
            }
            
            if ($financialYear) {
                $financialYearId = $financialYear->id;
            } else {
                // Create a default financial year if none exists
                $financialYear = FinancialYear::create([
                    'company_id' => $loan->companyid,
                    'branch_id' => $loan->branchid,
                    'year' => date('Y'),
                    'start_date' => date('Y-01-01'),
                    'end_date' => date('Y-12-31'),
                    'is_active' => true,
                    'created_by' => auth()->id(),
                ]);
                $financialYearId = $financialYear->id;
                
                // Update company with this financial year
                $company->current_financial_year_id = $financialYearId;
                $company->save();
            }
        }
        
        // Get default currency (ZMW)
        $currency = Currency::where('code', 'ZMW')->first();
        if (!$currency) {
            $currency = Currency::where('company_id', $loan->companyid)->first();
        }
        $currencyId = $currency ? $currency->id : 1;
        
        // Create journal entry for NPL provision
        $journalEntryData = [
            'company_id' => $loan->companyid,
            'branch_id' => $loan->branchid,
            'financial_year_id' => $financialYearId,
            'entry_number' => $entryNumber,
            'entry_date' => $classificationDate,
            'reference' => "NPL-PROV-{$loan->loanid}",
            'description' => "NPL provision for {$loan->loannumber} ({$category}) - {$provisionPercentage}%",
            'total_debit' => $totalProvision,
            'total_credit' => $totalProvision,
            'status' => 'posted',
            'posted_by' => auth()->id(),
            'posted_at' => now(),
            'created_by' => auth()->id(),
            'loanid' => $loan->loanid,
            'currency_id' => $currencyId,
            'exchange_rate' => 1.0,
        ];
        
        // Remove null values to avoid issues
        $journalEntryData = array_filter($journalEntryData, function($value) {
            return $value !== null;
        });
        
        $journalEntry = JournalEntry::create($journalEntryData);
        
        // Create journal entry items
        // 1. Dr. Loan Loss Expense (Income Statement - Expense)
        $expenseItemData = [
            'company_id' => $loan->companyid,
            'branch_id' => $loan->branchid,
            'journal_entry_id' => $journalEntry->id,
            'account_id' => $lossExpenseAccountId,
            'description' => "NPL provision expense - {$loan->loannumber}",
            'debit' => $totalProvision,
            'credit' => 0,
            'created_by' => auth()->id(),
        ];
        
        // Add financial_year_id only if the column exists
        if (Schema::hasColumn('journal_entry_items', 'financial_year_id')) {
            $expenseItemData['financial_year_id'] = $financialYearId;
        }
        
        JournalEntryItem::create($expenseItemData);
        
        // 2. Cr. Allowance for Loan Losses (Balance Sheet - Contra Asset)
        $allowanceItemData = [
            'company_id' => $loan->companyid,
            'branch_id' => $loan->branchid,
            'journal_entry_id' => $journalEntry->id,
            'account_id' => $provisionAccountId,
            'description' => "NPL allowance - {$loan->loannumber}",
            'debit' => 0,
            'credit' => $totalProvision,
            'created_by' => auth()->id(),
        ];
        
        // Add financial_year_id only if the column exists
        if (Schema::hasColumn('journal_entry_items', 'financial_year_id')) {
            $allowanceItemData['financial_year_id'] = $financialYearId;
        }
        
        JournalEntryItem::create($allowanceItemData);
        
        // Create general ledger entries
        // Dr. Loan Loss Expense
        GeneralLedgerEntry::create([
            'entrydate' => $classificationDate,
            'description' => "NPL provision expense: {$loan->loannumber}",
            'accountid' => $lossExpenseAccountId,
            'referencedocument' => 'NPL_CLASSIFICATION',
            'reference_type' => 'journal_entry',
            'reference_id' => $journalEntry->id,
            'documentno' => $entryNumber,
            'entrytype' => 'debit',
            'amount' => $totalProvision,
            'createdby' => auth()->id(),
            'companyid' => $loan->companyid,
            'branchid' => $loan->branchid,
            'loanid' => $loan->loanid,
        ]);
        
        // Cr. Allowance for Loan Losses
        GeneralLedgerEntry::create([
            'entrydate' => $classificationDate,
            'description' => "NPL allowance: {$loan->loannumber}",
            'accountid' => $provisionAccountId,
            'referencedocument' => 'NPL_CLASSIFICATION',
            'reference_type' => 'journal_entry',
            'reference_id' => $journalEntry->id,
            'documentno' => $entryNumber,
            'entrytype' => 'credit',
            'amount' => $totalProvision,
            'createdby' => auth()->id(),
            'companyid' => $loan->companyid,
            'branchid' => $loan->branchid,
            'loanid' => $loan->loanid,
        ]);
        
        // Also update InterestIncome table for the provision amount (as negative - loss)
        InterestIncome::create([
            'iloanid' => $loan->loanid,
            'customerid' => $loan->customerid,
            'ipid' => $loan->loantypeid,
            'charge' => -$totalProvision, // Negative amount (loss)
            'chargeddate' => $classificationDate,
            'poster' => auth()->id(),
            'branchid' => $loan->branchid,
            'companyid' => $loan->companyid,
        ]);
        
        Log::info('NPL provision created successfully', [
            'loan_id' => $loan->loanid,
            'loan_number' => $loan->loannumber,
            'total_provision' => $totalProvision,
            'category' => $category,
            'percentage' => $provisionPercentage,
            'provision_account_id' => $provisionAccountId,
            'expense_account_id' => $lossExpenseAccountId,
            'journal_entry_id' => $journalEntry->id,
            'financial_year_id' => $financialYearId,
            'entry_number' => $entryNumber,
        ]);
        
        DB::commit();
        
        return [
            'success' => true,
            'total_provision' => $totalProvision,
            'journal_entry_id' => $journalEntry->id,
            'entry_number' => $entryNumber,
            'provision_account' => $provisionMapping->nameofthebusinessprocess,
            'expense_account' => $lossExpenseMapping->nameofthebusinessprocess,
            'financial_year_id' => $financialYearId,
        ];
        
    } catch (\Exception $e) {
        DB::rollBack();
        
        Log::error('NPL provision creation failed', [
            'loan_id' => $loan->loanid,
            'loan_number' => $loan->loannumber,
            'error' => $e->getMessage(),
            'trace' => $e->getTraceAsString(),
            'file' => $e->getFile(),
            'line' => $e->getLine(),
        ]);
        
        throw new \Exception('Failed to create NPL provision: ' . $e->getMessage());
    }
}

/**
 * Transfer loan to non-performing status (with correct accounting)
 */
/**
 * Transfer loan to non-performing status (with correct accounting)
 */
/**
 * Transfer loan to non-performing status (with correct accounting)
 */
public function transferToNPL(Request $request, DisbursedLoan $loan)
{
    // Debug logging
    \Log::info('NPL transfer started', [
        'loan_id' => $loan->loanid,
        'user_id' => auth()->id(),
        'request_data' => $request->all(),
    ]);
    
    // Check permissions
    if (!auth()->user()->hasPermission('loans.transfer-npl')) {
        abort(403, 'You do not have permission to transfer loans to non-performing status.');
    }
    
    // Check if loan is already NPL
    if ($loan->status === 'default') {
        return redirect()->route('admin.loans.show', $loan->loanid)
            ->with('error', 'This loan is already classified as non-performing.');
    }
    
    // Check if loan is closed
    if ($loan->status === 'closed') {
        return redirect()->route('admin.loans.show', $loan->loanid)
            ->with('error', 'Cannot classify a closed loan as non-performing.');
    }
    
    // Validate the request
    $validated = $request->validate([
        'npl_reason' => 'required|in:payment_default,financial_difficulties,bankruptcy,breach_of_covenants,significant_doubt,legal_issues,death_disability,other',
        'classification_date' => 'required|date|before_or_equal:today',
        'npl_category' => 'required|in:substandard,doubtful,loss,watch',
        'provision_percentage' => 'required|numeric|min:0|max:100',
        'assessment_details' => 'required|string|min:50',
        'recovery_plan' => 'nullable|string',
        'confirm_regulatory' => 'required|accepted',
        'confirm_approval' => 'required|accepted',
        'confirm_impact' => 'required|accepted',
        'confirm_accounting' => 'required|accepted',
    ]);
    
    DB::beginTransaction();
    
    try {
        $classificationDate = $validated['classification_date'];
        $nplCategory = $validated['npl_category'];
        $provisionPercentage = $validated['provision_percentage'];
        
        // Check if business process mappings exist
        $checkMappings = $this->checkRequiredAccountMappings();
        if (!$checkMappings['success']) {
            return back()->with('error', $checkMappings['message'])->withInput();
        }
        
        // Create NPL provisions
        $provisionResult = $this->createNPLProvisions($loan, $classificationDate, $nplCategory, $provisionPercentage);
        
        if (!$provisionResult['success']) {
            throw new \Exception('Failed to create provision entries');
        }
        
        // Calculate provision amount
        $provisionAmount = $loan->totalbalance * ($provisionPercentage / 100);
        
        // Prepare supporting docs
        $supportingDocs = null;
        if ($request->has('supporting_docs')) {
            $docs = $request->input('supporting_docs');
            if (is_array($docs) && !empty($docs)) {
                $supportingDocs = json_encode($docs);
            }
        }
        
        // === OPTION 4: Update fields individually ===
        $loan->status = 'default';
        $loan->npl_classification_date = $classificationDate;
        $loan->npl_reason = $validated['npl_reason'];
        $loan->npl_category = $nplCategory;
        $loan->provision_percentage = $provisionPercentage;
        $loan->provision_amount = $provisionAmount;
        $loan->assessment_details = $validated['assessment_details'];
        $loan->recovery_plan = $validated['recovery_plan'] ?? null;
        $loan->supporting_docs = $supportingDocs;
        $loan->npl_classified_by = auth()->id();
        $loan->npl_classified_at = now();
        $loan->updated_at = now();
        $loan->updatedby = auth()->id();
        
        // Save the loan
        $saved = $loan->save();
        
        if (!$saved) {
            throw new \Exception('Failed to save loan NPL details. Please check the logs.');
        }
        
        // Log the NPL classification
        Log::info('Loan classified as NPL successfully', [
            'loan_id' => $loan->loanid,
            'loan_number' => $loan->loannumber,
            'classified_by' => auth()->id(),
            'classification_date' => $classificationDate,
            'npl_reason' => $validated['npl_reason'],
            'npl_category' => $nplCategory,
            'outstanding_balance' => $loan->totalbalance,
            'provision_percentage' => $provisionPercentage,
            'provision_amount' => $provisionAmount,
            'journal_entry_number' => $provisionResult['entry_number'] ?? 'N/A',
            'supporting_docs' => $supportingDocs,
        ]);
        
        DB::commit();
        
        return redirect()->route('admin.loans.show', $loan->loanid)
            ->with('success', 'Loan has been successfully classified as non-performing. Provision of ' . $provisionPercentage . '% (K ' . number_format($provisionAmount, 2) . ') has been recorded. Journal Entry: ' . ($provisionResult['entry_number'] ?? 'N/A'));
        
    } catch (\Exception $e) {
        DB::rollBack();
        
        Log::error('NPL classification failed', [
            'loan_id' => $loan->loanid,
            'user_id' => auth()->id(),
            'error' => $e->getMessage(),
            'trace' => $e->getTraceAsString(),
            'file' => $e->getFile(),
            'line' => $e->getLine(),
        ]);
        
        return back()
            ->withInput()
            ->with('error', 'NPL classification failed: ' . $e->getMessage());
    }
}

/**
 * Check if required account mappings exist
 */
private function checkRequiredAccountMappings()
{
    try {
        // Check for Loan Disbursement account
        $loanDisbursement = BusinessProcessMapping::where('nameofthebusinessprocess', 'Loan Disbursement')->first();
        if (!$loanDisbursement) {
            return [
                'success' => false,
                'message' => 'Loan Disbursement account mapping not found. Please set it up in Accounting Settings.',
            ];
        }
        
        // Check for Provision/Allowance account
        $provisionAccount = BusinessProcessMapping::where('nameofthebusinessprocess', 'LIKE', '%Provision%')
            ->orWhere('nameofthebusinessprocess', 'LIKE', '%Allowance%')
            ->orWhere('nameofthebusinessprocess', 'LIKE', '%Loan Loss%')
            ->first();
        
        if (!$provisionAccount) {
            return [
                'success' => false,
                'message' => 'Provision/Allowance account mapping not found. Please set up "Loan Loss Provision" or similar in Business Process Mapping.',
            ];
        }
        
        // Check for Loan Loss Expense account
        $expenseAccount = BusinessProcessMapping::where('nameofthebusinessprocess', 'LIKE', '%Loan Loss Expense%')
            ->orWhere('nameofthebusinessprocess', 'LIKE', '%Provision Expense%')
            ->orWhere('nameofthebusinessprocess', 'LIKE', '%Bad Debt Expense%')
            ->first();
        
        if (!$expenseAccount) {
            return [
                'success' => false,
                'message' => 'Loan Loss Expense account mapping not found. Please set up "Loan Loss Expense" in Business Process Mapping.',
            ];
        }
        
        return ['success' => true, 'message' => 'All required accounts found.'];
        
    } catch (\Exception $e) {
        return [
            'success' => false,
            'message' => 'Error checking account mappings: ' . $e->getMessage(),
        ];
    }
}

/**
 * Get provision percentage based on NPL category
 */
private function getProvisionPercentageByCategory($category)
{
    switch ($category) {
        case 'substandard':
            return 20; // 20% provision (90-180 days overdue)
        case 'doubtful':
            return 50; // 50% provision (181-365 days overdue)
        case 'loss':
            return 100; // 100% provision (365+ days overdue)
        case 'watch':
            return 10; // 10% provision (early warning)
        default:
            return 20;
    }
}



}