<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Application;
use App\Models\LoanType;
use App\Models\ChartOfAccount;
use App\Models\DisbursedLoan;
use App\Models\Customer;
use App\Mail\LoanApplicationApproved;
use App\Mail\LoanApplicationRejected;
use App\Services\LoanDisbursementService;
use App\Services\CreditScoreService;
use App\Mail\LoanDisbursed;
use App\Services\SmsService;
use App\Services\PaymentScheduleService;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;

class ApplicationController extends Controller
{
    private $smsService;
    private $paymentScheduleService;
    private $loanDisbursementService;
    private $creditScoreService;

    public function __construct(
        SmsService $smsService, 
        PaymentScheduleService $paymentScheduleService,
        LoanDisbursementService $loanDisbursementService,
        CreditScoreService $creditScoreService
    ) {
        $this->smsService = $smsService;
        $this->paymentScheduleService = $paymentScheduleService;
        $this->loanDisbursementService = $loanDisbursementService;
        $this->creditScoreService = $creditScoreService;
    }

    /**
     * Display a listing of all applications.
     */
    public function index()
    {
        // Permission check (using Sanctum/API auth context)
        if (!auth()->user()->hasAnyPermission(['applications.view', 'applications.approve', 'poweruser'])) {
            return response()->json(['message' => 'You do not have permission to view applications.'], 403);
        }

        $applications = Application::with(['customer', 'product'])
            ->orderBy('created_at', 'desc')
            ->paginate(20);
            
        return response()->json($applications);
    }

    /**
     * Display approved applications ready for disbursement.
     */
    public function approved()
    {
        if (!auth()->user()->hasAnyPermission(['applications.view', 'applications.approve', 'poweruser'])) {
            return response()->json(['message' => 'You do not have permission to view applications.'], 403);
        }

        $applications = Application::with(['customer', 'product'])
            ->where('status', 'approved')
            ->orderBy('approved_at', 'desc')
            ->paginate(20);
            
        return response()->json($applications);
    }

    /**
     * Display the specified application.
     */
    public function show($id)
    {
        if (!auth()->user()->hasAnyPermission(['applications.view', 'applications.approve', 'applications.edit', 'poweruser'])) {
            return response()->json(['message' => 'You do not have permission to view application details.'], 403);
        }

        $application = Application::with(['customer', 'product'])->find($id);

        if (!$application) {
            return response()->json(['message' => 'Application not found'], 404);
        }

        return response()->json($application);
    }

    /**
     * Show the form for creating a new application.
     * (API Note: Typically creates usually provide data needed for a form, like loan types)
     */
    public function create()
    {
        if (!auth()->user()->hasPermission('applications.create')) {
             return response()->json(['message' => 'You do not have permission to create applications.'], 403);
        }

        $loanProducts = LoanType::all(); // Return available loan products
        return response()->json(['loan_products' => $loanProducts]);
    }

    /**
     * Store a newly created resource in storage.
     * (Added this method as it wasn't in original controller but is standard for API)
     */
    public function store(Request $request)
    {
         if (!auth()->user()->hasPermission('applications.create')) {
             return response()->json(['message' => 'You do not have permission to create applications.'], 403);
        }
        
        // Validation logic (similar to update logic)
        $validator = Validator::make($request->all(), [
            'customer_id' => 'required|exists:customers,id', // Assuming customer table name
            'product_id' => 'required|exists:loantype,id',
            'loan_amount' => 'required|numeric|min:0',
            'interest_rate' => 'required|numeric|min:0',
            'loan_tenure' => 'required|integer|min:1',
            'payment_frequency' => 'required|string|in:weekly,biweekly,monthly,quarterly',
            'source' => 'required|string|in:online,branch,agent,mobile',
            'remarks' => 'nullable|string|max:500',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        $application = Application::create(array_merge($validator->validated(), ['status' => 'submitted']));

        return response()->json(['message' => 'Application created successfully.', 'application' => $application], 201);
    }


    /**
     * Show the form for editing the specified application.
     * (API Note: Returns data needed to edit)
     */
    public function edit($id)
    {
        if (!auth()->user()->hasPermission('applications.edit')) {
            return response()->json(['message' => 'You do not have permission to edit applications.'], 403);
        }

        $application = Application::with(['customer', 'product'])->find($id);
        
        if (!$application) {
             return response()->json(['message' => 'Application not found'], 404);
        }

        $loanProducts = LoanType::all();
        
        return response()->json([
            'application' => $application,
            'loan_products' => $loanProducts
        ]);
    }

    /**
     * Show the disbursement form/data for an approved application.
     */
    public function disburse($id)
    {
        if (!auth()->user()->hasPermission('applications.approve')) {
             return response()->json(['message' => 'You do not have permission to disburse applications.'], 403);
        }

        $application = Application::with(['customer', 'product'])->find($id);
        
        if (!$application) {
             return response()->json(['message' => 'Application not found'], 404);
        }

        if ($application->status !== 'approved') {
             return response()->json(['message' => 'Only approved applications can be disbursed.'], 400);
        }

        if ($application->disbursed_at) {
             return response()->json(['message' => 'This application has already been disbursed.'], 400);
        }
        
        $cashAccounts = $this->getCashAccounts();

        return response()->json([
            'application' => $application,
            'cash_accounts' => $cashAccounts
        ]);
    }

    /**
     * Process loan disbursement.
     */
    public function processDisbursement(Request $request, $id)
    {
        if (!auth()->user()->hasPermission('applications.approve')) {
            return response()->json(['message' => 'You do not have permission to disburse applications.'], 403);
        }

        $validator = Validator::make($request->all(), [
            'disbursement_date' => 'required|date',
            'payment_account_id' => 'required|exists:chart_of_accounts,id',
            'disbursement_notes' => 'nullable|string|max:500',
        ]);

        if ($validator->fails()) {
             return response()->json(['errors' => $validator->errors()], 422);
        }

        try {
            $result = DB::transaction(function () use ($request, $id) {
                $application = Application::with('customer')->findOrFail($id);
                
                if ($application->status !== 'approved') {
                    throw new \Exception('Only approved applications can be disbursed.');
                }

                if ($application->disbursed_at) {
                    throw new \Exception('This application has already been disbursed.');
                }

                $paymentAccount = ChartOfAccount::where('id', $request->payment_account_id)
                    ->where('is_active', true)
                    ->first();

                if (!$paymentAccount) {
                    throw new \Exception('Selected payment account is not available or inactive.');
                }

                if ($paymentAccount->current_balance < $application->loan_amount) {
                    throw new \Exception('Insufficient balance in the selected payment account.');
                }

                $disbursementMapping = DB::table('business_process_mappings')
                    ->where('nameofthebusinessprocess', 'Loan Disbursement')
                    ->where('companyid', $application->company_id)
                    ->where('branchid', $application->branch_id)
                    ->first();

                if (!$disbursementMapping) {
                    throw new \Exception('Loan Disbursement business process mapping not found.');
                }

                $loanReceivableAccountId = $disbursementMapping->accountid;

                $loanReceivableAccount = ChartOfAccount::where('id', $loanReceivableAccountId)
                    ->where('is_active', true)
                    ->first();

                if (!$loanReceivableAccount) {
                    throw new \Exception('Loan Receivable account is not available or inactive.');
                }

                $this->ensureCustomerHasCreditScoreRecord($application->customer_id);

                $disbursementDate = \Carbon\Carbon::parse($request->disbursement_date);
                $firstPaymentDate = $this->calculateFirstPaymentDate($disbursementDate, $application->payment_frequency);
                $maturityDate = $disbursementDate->copy()->addMonths((int)$application->loan_tenure);

                $disbursedLoan = DisbursedLoan::create([
                    'application_id' => $application->id,
                    'loannumber' => DisbursedLoan::generateLoanNumber(),
                    'customerid' => $application->customer_id,
                    'loantypeid' => $application->product_id,
                    'amount' => $application->loan_amount,
                    'interestrate' => $application->interest_rate,
                    'loanterm' => (int)$application->loan_tenure,
                    'paymentfrequency' => $application->payment_frequency,
                    'installmentamount' => $application->installment_amount,
                    'disburseddate' => $disbursementDate,
                    'payment_account_id' => $request->payment_account_id,
                    'disbursement_notes' => $request->disbursement_notes,
                    'status' => 'active',
                    'firstpaymentdate' => $firstPaymentDate,
                    'maturitydate' => $maturityDate,
                    'principalbalance' => $application->loan_amount,
                    'interestbalance' => 0,
                    'totalbalance' => $application->loan_amount,
                    'branchid' => $application->branch_id,
                    'companyid' => $application->company_id,
                    'disbursedby' => auth()->id(),
                    'createdby' => auth()->id(),
                    'updatedby' => auth()->id(),
                ]);

                $this->paymentScheduleService->createDisbursementEntry($disbursedLoan);
                $this->paymentScheduleService->generatePaymentSchedule($disbursedLoan);

                $description = "Loan disbursement - {$disbursedLoan->loannumber} for {$application->customer->first_name} {$application->customer->surname}";
                
                DB::table('generalledgerentries')->insert([
                    'entrydate' => $disbursementDate->format('Y-m-d'),
                    'description' => $description,
                    'accountid' => $loanReceivableAccountId,
                    'referencedocument' => 'Loan Disbursement',
                    'documentno' => $disbursedLoan->loannumber,
                    'entrytype' => 'debit',
                    'amount' => $application->loan_amount,
                    'createdby' => auth()->id(),
                    'companyid' => $application->company_id,
                    'branchid' => $application->branch_id,
                    'created_at' => now(),
                    'updated_at' => now(),
                    'transtype' => 'loan_disbursement',
                    'loanid' => $disbursedLoan->loanid,
                ]);
                
                $creditamount = $application->loan_amount * -1;

                DB::table('generalledgerentries')->insert([
                    'entrydate' => $disbursementDate->format('Y-m-d'),
                    'description' => $description,
                    'accountid' => $request->payment_account_id,
                    'referencedocument' => 'Loan Disbursement',
                    'documentno' => $disbursedLoan->loannumber,
                    'entrytype' => 'credit',
                    'amount' => $creditamount,
                    'createdby' => auth()->id(),
                    'companyid' => $application->company_id,
                    'branchid' => $application->branch_id,
                    'created_at' => now(),
                    'updated_at' => now(),
                    'transtype' => 'loan_disbursement',
                    'loanid' => $disbursedLoan->loanid,
                ]);

                DB::table('chart_of_accounts')
                    ->where('id', $request->payment_account_id)
                    ->decrement('current_balance', $application->loan_amount);

                DB::table('chart_of_accounts')
                    ->where('id', $loanReceivableAccountId)
                    ->increment('current_balance', $application->loan_amount);

                $application->update([
                    'status' => 'disbursed',
                    'disbursed_at' => $disbursementDate,
                    'disbursed_by' => auth()->id(),
                    'disbursement_notes' => $request->disbursement_notes,
                    'loan_id' => $disbursedLoan->loanid,
                ]);

                $this->updateCreditScoreForDisbursement($application, $disbursedLoan);

                $notificationResults = $this->sendDisbursementNotifications($application, $disbursedLoan);
                
                return ['success' => true, 'message' => 'Loan disbursed successfully. ' . $notificationResults];
            });

            return response()->json($result);

        } catch (\Exception $e) {
            \Log::error('Loan disbursement failed', [
                'application_id' => $id,
                'error' => $e->getMessage(),
                'user_id' => auth()->id(),
                'payment_account_id' => $request->payment_account_id,
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Disbursement failed: ' . $e->getMessage()
            ], 500);
        }
    }
    
     private function ensureCustomerHasCreditScoreRecord($customerId)
    {
        $currentScore = \App\Models\CustomerCreditScore::byCustomer($customerId)->current()->first();
        
        if (!$currentScore) {
            $this->creditScoreService->initializeCreditScore($customerId);
        }
    }

    private function updateCreditScoreForDisbursement(Application $application, DisbursedLoan $disbursedLoan): void
    {
        try {
            $this->creditScoreService->updateCreditScoreForEvent(
                $application->customer_id,
                'loan_disbursed',
                [
                    'loan_id' => $disbursedLoan->loanid,
                    'application_id' => $application->id,
                    'loan_amount' => $application->loan_amount,
                    'interest_rate' => $application->interest_rate,
                    'loan_term' => $application->loan_tenure,
                    'disbursement_date' => $disbursedLoan->disburseddate,
                    'credit_utilization_impact' => $this->calculateUtilizationImpact($application->customer_id, $application->loan_amount),
                    'monthly_payment' => $application->installment_amount,
                ]
            );
        } catch (\Exception $e) {
            \Log::error('Failed to update credit score', ['error' => $e->getMessage()]);
        }
    }

    private function calculateUtilizationImpact($customerId, $newLoanAmount): float
    {
         try {
            $currentOutstanding = DisbursedLoan::where('customerid', $customerId)
                ->where('status', 'active')
                ->sum('totalbalance');

            $customer = Customer::find($customerId);
            $creditLimit = $customer->credit_limit ?? ($customer->income * 3);

            if ($creditLimit <= 0) return 0;

            $oldUtilization = ($currentOutstanding / $creditLimit) * 100;
            $newUtilization = (($currentOutstanding + $newLoanAmount) / $creditLimit) * 100;
            
            return $newUtilization - $oldUtilization;

        } catch (\Exception $e) {
            return 0;
        }
    }


    /**
     * Approve the specified application.
     */
    public function approve($id)
    {
        if (!auth()->user()->hasPermission('applications.approve')) {
             return response()->json(['message' => 'You do not have permission to approve applications.'], 403);
        }

        $application = Application::with('customer')->find($id);
        
        if (!$application) {
             return response()->json(['message' => 'Application not found'], 404);
        }
        
        $application->update([
            'status' => 'approved',
            'approved_at' => now(),
            'reviewed_by' => auth()->id()
        ]);

        $notificationResults = $this->sendApprovalNotifications($application);

        return response()->json(['message' => 'Application approved successfully. ' . $notificationResults]);
    }

    /**
     * Reject the specified application.
     */
    public function reject($id)
    {
        if (!auth()->user()->hasPermission('applications.approve')) {
             return response()->json(['message' => 'You do not have permission to reject applications.'], 403);
        }

        $application = Application::with('customer')->find($id);

         if (!$application) {
             return response()->json(['message' => 'Application not found'], 404);
        }

        $application->update([
            'status' => 'rejected',
            'reviewed_at' => now(),
            'reviewed_by' => auth()->id()
        ]);

        $notificationResults = $this->sendRejectionNotifications($application);

        return response()->json(['message' => 'Application rejected. ' . $notificationResults]);
    }

    /**
     * Update the specified application.
     */
    public function update(Request $request, $id)
    {
        if (!auth()->user()->hasPermission('applications.edit')) {
             return response()->json(['message' => 'You do not have permission to edit applications.'], 403);
        }

        $application = Application::find($id);
        
        if (!$application) {
             return response()->json(['message' => 'Application not found'], 404);
        }
        
        $validator = Validator::make($request->all(), [
            'product_id' => 'required|exists:loantype,id',
            'loan_amount' => 'required|numeric|min:0',
            'interest_rate' => 'required|numeric|min:0',
            'loan_tenure' => 'required|integer|min:1',
            'payment_frequency' => 'required|string|in:weekly,biweekly,monthly,quarterly',
            'status' => 'required|string|in:draft,submitted,under_review,approved,rejected,disbursed',
            'source' => 'required|string|in:online,branch,agent,mobile',
            'remarks' => 'nullable|string|max:500',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        $validated = $validator->validated();
        $oldStatus = $application->status;

        $application->update($validated);

        if ($validated['status'] === 'approved' && $oldStatus !== 'approved') {
            $application->load('customer');
            $notificationResults = $this->sendApprovalNotifications($application);
            return response()->json(['message' => 'Application updated and approved successfully. ' . $notificationResults]);
        }

        if ($validated['status'] === 'rejected' && $oldStatus !== 'rejected') {
            $application->load('customer');
            $notificationResults = $this->sendRejectionNotifications($application);
            return response()->json(['message' => 'Application updated and rejected. ' . $notificationResults]);
        }

        return response()->json(['message' => 'Application updated successfully.']);
    }

    /**
     * Remove the specified application.
     */
    public function destroy($id)
    {
        if (!auth()->user()->hasPermission('applications.delete')) {
             return response()->json(['message' => 'You do not have permission to delete applications.'], 403);
        }

        $application = Application::find($id);
        
        if (!$application) {
             return response()->json(['message' => 'Application not found'], 404);
        }
        
        $application->delete();

        return response()->json(['message' => 'Application deleted successfully.']);
    }

    /**
     * Resend notifications for an application.
     */
    public function resendNotifications($id)
    {
        if (!auth()->user()->hasPermission('applications.edit')) {
             return response()->json(['message' => 'You do not have permission to resend notifications.'], 403);
        }

        $application = Application::with('customer')->find($id);
        
        if (!$application) {
             return response()->json(['message' => 'Application not found'], 404);
        }
        
        $message = '';

        if ($application->status === 'approved') {
            $notificationResults = $this->sendApprovalNotifications($application);
            $message = 'Approval notifications resent. ' . $notificationResults;
        } elseif ($application->status === 'rejected') {
            $notificationResults = $this->sendRejectionNotifications($application);
            $message = 'Rejection notifications resent. ' . $notificationResults;
        } elseif ($application->status === 'disbursed') {
            $disbursedLoan = DisbursedLoan::where('application_id', $application->id)->first();
            if ($disbursedLoan) {
                $notificationResults = $this->sendDisbursementNotifications($application, $disbursedLoan);
                $message = 'Disbursement notifications resent. ' . $notificationResults;
            } else {
                return response()->json(['message' => 'No disbursed loan found for this application.'], 404);
            }
        } else {
             return response()->json(['message' => 'Notifications can only be resent for approved, rejected, or disbursed applications.'], 400);
        }

        return response()->json(['message' => $message]);
    }
    
    // ... (Keep existing Helper Methods: getCashAccounts, sendApprovalNotifications, sendRejectionNotifications, sendDisbursementNotifications, generateApprovalSms, generateRejectionSms, generateDisbursementSms, calculateFirstPaymentDate) ...

    /**
     * Get cash accounts for disbursement by account type.
     */
    private function getCashAccounts()
    {
         // Same logic as before but ensures we return data or empty collection correctly
        $cashTypeNames = [
            'Cash and Cash Equivalents', 'Cash & Cash Equivalents', 'Cash Equivalents', 'Cash',
            'Bank Accounts', 'Cash at Bank', 'Cash in Hand', 'Bank', 'Petty Cash'
        ];

        foreach ($cashTypeNames as $typeName) {
            $accounts = ChartOfAccount::whereHas('accountType', function($query) use ($typeName) {
                $query->where('name', $typeName)->where('is_active', true);
            })->where('is_active', true)->get();

            if ($accounts->isNotEmpty()) return $accounts;
        }

        $fallbackAccounts = ChartOfAccount::whereHas('accountType', function($query) {
            $query->where(function($q) {
                $q->where('name', 'like', '%cash%')->orWhere('name', 'like', '%bank%');
            })->where('is_active', true);
        })->where('is_active', true)->get();

        if ($fallbackAccounts->isNotEmpty()) return $fallbackAccounts;

        return ChartOfAccount::where('is_active', true)->get();
    }
    
     private function sendApprovalNotifications(Application $application)
    {
        $results = [];
        
        if ($application->customer->email) {
            try {
                Mail::to($application->customer->email)->send(new LoanApplicationApproved($application));
                $results[] = 'Approval email sent.';
            } catch (\Exception $e) {
                \Log::error('Email error: ' . $e->getMessage());
                $results[] = 'Email failed.';
            }
        }

        if ($application->customer->phone) {
            try {
                $smsMessage = $this->generateApprovalSms($application);
                $this->smsService->sendSms($application->customer->phone, $smsMessage);
                $results[] = 'SMS sent.';
            } catch (\Exception $e) {
                 \Log::error('SMS error: ' . $e->getMessage());
                $results[] = 'SMS failed.';
            }
        }

        return implode(' ', $results);
    }
    
    // ... [Other helper methods remain similar, just ensuring return types are compatible or used correctly] ...
     private function sendRejectionNotifications(Application $application)
    {
         $results = [];
        if ($application->customer->email) {
             try {
                Mail::to($application->customer->email)->send(new LoanApplicationRejected($application));
                 $results[] = 'Rejection email sent.';
            } catch (\Exception $e) {
                $results[] = 'Email failed.';
            }
        }
         if ($application->customer->phone) {
            try {
                $smsMessage = $this->generateRejectionSms($application);
                $this->smsService->sendSms($application->customer->phone, $smsMessage);
                 $results[] = 'SMS sent.';
            } catch (\Exception $e) {
                $results[] = 'SMS failed.';
            }
        }
        return implode(' ', $results);
    }
    
    private function sendDisbursementNotifications(Application $application, DisbursedLoan $disbursedLoan)
    {
         $results = [];
         if ($application->customer->email) {
            try {
                Mail::to($application->customer->email)->send(new LoanDisbursed($application, $disbursedLoan));
                 $results[] = 'Disbursement email sent.';
            } catch (\Exception $e) {
                 $results[] = 'Email failed.';
            }
        }
        if ($application->customer->phone) {
             try {
                $smsMessage = $this->generateDisbursementSms($application, $disbursedLoan);
                $this->smsService->sendSms($application->customer->phone, $smsMessage);
                 $results[] = 'SMS sent.';
            } catch (\Exception $e) {
                 $results[] = 'SMS failed.';
            }
        }
        return implode(' ', $results);
    }
    
    private function generateApprovalSms(Application $application)
    {
        return "Dear {$application->customer->first_name}, your loan application #{$application->application_number} has been APPROVED! Loan amount: ZMW " . number_format($application->loan_amount, 2) . ".";
    }

    private function generateRejectionSms(Application $application)
    {
        return "Dear {$application->customer->first_name}, your loan application #{$application->application_number} has been rejected.";
    }

    private function generateDisbursementSms(Application $application, DisbursedLoan $disbursedLoan)
    {
        return "Dear {$application->customer->first_name}, loan #{$disbursedLoan->loannumber} disbursed! Amount: ZMW " . number_format($disbursedLoan->amount, 2) . ".";
    }

    private function calculateFirstPaymentDate($disbursementDate, $paymentFrequency)
    {
        switch ($paymentFrequency) {
            case 'weekly': return $disbursementDate->copy()->addWeek();
            case 'biweekly': return $disbursementDate->copy()->addWeeks(2);
            case 'monthly': return $disbursementDate->copy()->addMonth();
            case 'quarterly': return $disbursementDate->copy()->addMonths(3);
            default: return $disbursementDate->copy()->addMonth();
        }
    }
}