<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Customer;
use App\Models\CustomerCreditScore;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use App\Services\SmsService;
use Illuminate\Support\Str;
use Carbon\Carbon;

class CustomerAuthController extends Controller
{
    private $smsService;

    public function __construct(SmsService $smsService)
    {
        $this->smsService = $smsService;
    }

    // ==================== API REGISTRATION METHODS (MULTI-STEP) ====================

    /**
     * Start registration - Step 1 (Email & Password) - API
     * POST /api/customer/register/step1
     */
    public function apiStep1(Request $request): JsonResponse
    {
        try {
            $validator = Validator::make($request->all(), [
                'email' => 'required|email|unique:customer,email',
                'password' => 'required|string|min:8|confirmed',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }

            // Generate registration token
            $registrationToken = Str::uuid()->toString();
            
            // Store step 1 data in cache (24-hour expiry)
            \Cache::put("api_registration.{$registrationToken}.step1", [
                'email' => $request->email,
                'password' => Hash::make($request->password),
                'expires_at' => now()->addHours(24),
                'created_at' => now(),
            ], 60 * 24);

            \Log::info('API Registration Step 1 completed', [
                'token' => $registrationToken,
                'email' => $request->email
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Account details saved successfully',
                'data' => [
                    'registration_token' => $registrationToken,
                    'next_step' => 'step2',
                    'next_step_name' => 'Profile Picture',
                    'expires_in_hours' => 24,
                    'progress' => 25,
                ]
            ]);

        } catch (\Exception $e) {
            \Log::error('API Registration Step 1 error: ' . $e->getMessage(), [
                'error' => $e->getTraceAsString()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to save account details. Please try again.'
            ], 500);
        }
    }

    /**
     * Step 2 - Profile Picture - API
     * POST /api/customer/register/step2
     */
    public function apiStep2(Request $request): JsonResponse
    {
        try {
            $validator = Validator::make($request->all(), [
                'registration_token' => 'required|string',
                'profile_picture' => 'nullable|string', // base64 image
                'webcam_picture' => 'nullable|string', // base64 webcam image
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }

            $token = $request->registration_token;
            
            // Check if step 1 exists
            $step1Data = \Cache::get("api_registration.{$token}.step1");
            if (!$step1Data) {
                return response()->json([
                    'success' => false,
                    'message' => 'Registration session expired or not found. Please start over.'
                ], 404);
            }

            $profilePicturePath = null;
            
            // Handle profile picture upload (file upload)
            if ($request->has('profile_picture') && !empty($request->profile_picture)) {
                $profilePicturePath = $this->handleBase64ImageUpload($request->profile_picture, 'profile-pictures', 'profile_');
            }
            
            // Handle webcam picture
            if ($request->has('webcam_picture') && !empty($request->webcam_picture)) {
                $profilePicturePath = $this->handleWebcamPicture($request->webcam_picture);
            }

            // Store step 2 data
            \Cache::put("api_registration.{$token}.step2", [
                'profile_picture_path' => $profilePicturePath,
                'has_picture' => !is_null($profilePicturePath),
                'updated_at' => now(),
            ], 60 * 24);

            \Log::info('API Registration Step 2 completed', [
                'token' => $token,
                'has_picture' => !is_null($profilePicturePath)
            ]);

            return response()->json([
                'success' => true,
                'message' => !is_null($profilePicturePath) 
                    ? 'Profile picture uploaded successfully' 
                    : 'Profile picture step completed',
                'data' => [
                    'registration_token' => $token,
                    'next_step' => 'step3',
                    'next_step_name' => 'Personal Information',
                    'has_profile_picture' => !is_null($profilePicturePath),
                    'progress' => 50,
                ]
            ]);

        } catch (\Exception $e) {
            \Log::error('API Registration Step 2 error: ' . $e->getMessage(), [
                'error' => $e->getTraceAsString()
            ]);
            return response()->json([
                'success' => false,
                'message' => 'Failed to upload profile picture. Please try again.'
            ], 500);
        }
    }

    /**
     * Step 3 - Personal Information - API
     * POST /api/customer/register/step3
     */
    public function apiStep3(Request $request): JsonResponse
    {
        try {
            $validator = Validator::make($request->all(), [
                'registration_token' => 'required|string',
                'title' => 'required|string|max:10',
                'surname' => 'required|string|max:255',
                'first_name' => 'required|string|max:255',
                'middle_name' => 'nullable|string|max:255',
                'customer' => 'required|string|max:255',
                'dob' => 'required|date',
                'phone' => 'required|string|max:25',
                'office_phone' => 'nullable|string|max:25',
                'cgroup' => 'required|in:Individual,Company',
                'idtype' => 'required|string',
                'number' => 'required|string',
                'address' => 'required|string',
                'town' => 'required|string',
                'province' => 'required|string',
                'income' => 'nullable|numeric|min:0',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }

            $token = $request->registration_token;
            
            // Check previous steps
            $step1Data = \Cache::get("api_registration.{$token}.step1");
            $step2Data = \Cache::get("api_registration.{$token}.step2");
            
            if (!$step1Data) {
                return response()->json([
                    'success' => false,
                    'message' => 'Please complete step 1 first.'
                ], 400);
            }
            
            if (!$step2Data) {
                return response()->json([
                    'success' => false,
                    'message' => 'Please complete step 2 first.'
                ], 400);
            }

            // Store step 3 data
            \Cache::put("api_registration.{$token}.step3", [
                'title' => $request->title,
                'surname' => $request->surname,
                'first_name' => $request->first_name,
                'middle_name' => $request->middle_name,
                'customer' => $request->customer,
                'dob' => $request->dob,
                'phone' => $request->phone,
                'office_phone' => $request->office_phone,
                'cgroup' => $request->cgroup,
                'idtype' => $request->idtype,
                'number' => $request->number,
                'address' => $request->address,
                'town' => $request->town,
                'province' => $request->province,
                'income' => $request->income ?? 5000,
                'updated_at' => now(),
            ], 60 * 24);

            \Log::info('API Registration Step 3 completed', [
                'token' => $token,
                'customer_name' => $request->customer
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Personal information saved successfully',
                'data' => [
                    'registration_token' => $token,
                    'next_step' => 'next-of-kin', // Update this metadata
                    'next_step_name' => 'Next of Kin',
                    'progress' =>60,
                ]
            ]);

        } catch (\Exception $e) {
            \Log::error('API Registration Step 3 error: ' . $e->getMessage(), [
                'error' => $e->getTraceAsString()
            ]);
            return response()->json([
                'success' => false,
                'message' => 'Failed to save personal information. Please try again.'
            ], 500);
        }
    }
    
    /**
 * Step 4 - Next of Kin Information - API
 * POST /api/customer/register/next-of-kin
 */
public function apiNextOfKin(Request $request): JsonResponse
{
    try {
        $validator = Validator::make($request->all(), [
            'registration_token' => 'required|string',
            'nok_first_name' => 'required|string|max:255',
            'nok_surname' => 'required|string|max:255',
            'nok_mobile_phone' => 'required|string|max:25',
            'nok_phone' => 'nullable|string|max:25',
            'nok_telephone' => 'nullable|string|max:25',
            'nok_physical_address' => 'nullable|string|max:255',
            'nok_postal_address' => 'nullable|string|max:255',
            'nok_town' => 'nullable|string|max:255',
            'nok_province' => 'nullable|string|max:255',
            'nok_country' => 'nullable|string|max:255',
            'nok_email' => 'nullable|email|max:255',
        ]);

        if ($validator->fails()) {
            return response()->json(['success' => false, 'errors' => $validator->errors()], 422);
        }

        $token = $request->registration_token;
        if (!\Cache::has("api_registration.{$token}.step1")) {
            return response()->json(['success' => false, 'message' => 'Session expired.'], 404);
        }

        \Cache::put("api_registration.{$token}.nok", $request->only([
            'nok_first_name', 'nok_surname', 'nok_mobile_phone', 'nok_phone', 'nok_telephone',
            'nok_physical_address', 'nok_postal_address', 'nok_town', 'nok_province', 'nok_country', 'nok_email'
        ]), 60 * 24);

        return response()->json([
            'success' => true,
            'message' => 'Next of kin saved',
            'data' => ['next_step' => 'employment', 'progress' => 80]
        ]);
    } catch (\Exception $e) {
        return response()->json(['success' => false, 'message' => $e->getMessage()], 500);
    }
}

/**
 * Step 5 - Employment Details - API
 * POST /api/customer/register/employment
 */
/**
 * Step 5 - Employment Details - API
 * POST /api/customer/register/employment
 */
public function apiEmployment(Request $request): JsonResponse
{
    try {
        $validator = Validator::make($request->all(), [
            'registration_token' => 'required|string',
            'job_title' => 'required|string|max:255',
            'employer_name' => 'required|string|max:255',
            // --- NEW FIELDS ADDED HERE ---
            'employee_no' => 'nullable|string|max:50',
            'years_in_employment' => 'required|integer|min:0',
            'retirement_year' => 'nullable|integer|min:'.date('Y'),
            // ------------------------------
            'employer_address' => 'required|string|max:255',
            'employer_postal_address' => 'nullable|string|max:255',
            'employer_town' => 'required|string|max:255',
            'employer_province' => 'required|string|max:255',
            'employment_type' => 'required|in:Permanent,Contract',
            'contract_expiry_date' => 'required_if:employment_type,Contract|nullable|date',
        ]);

        if ($validator->fails()) {
            return response()->json(['success' => false, 'errors' => $validator->errors()], 422);
        }

        $token = $request->registration_token;
        \Cache::put("api_registration.{$token}.employment", $request->only([
            'job_title', 'employer_name', 'employee_no', 'years_in_employment', 
            'retirement_year', 'employer_address', 'employer_postal_address',
            'employer_town', 'employer_province', 'employment_type', 'contract_expiry_date'
        ]), 60 * 24);

        return response()->json([
            'success' => true,
            'message' => 'Employment details saved',
            'data' => ['next_step' => 'step4', 'progress' => 90]
        ]);
    } catch (\Exception $e) {
        return response()->json(['success' => false, 'message' => $e->getMessage()], 500);
    }
}

    /**
     * Step 4 - NRC Verification & Final Submission - API
     * POST /api/customer/register/step4
     */
   /**
 * Step 4 (Final) - NRC Verification & Final Account Creation - API
 * POST /api/customer/register/step4
 */
/**
 * Step 4 (Final) - NRC Verification & Final Account Creation - API
 * POST /api/customer/register/step4
 * * This method merges Step 1, 2, 3, Next of Kin, and Employment caches
 * into a single record in the 'customer' table.
 */
public function apiStep4(Request $request): JsonResponse
{
    try {
        $validator = Validator::make($request->all(), [
            'registration_token' => 'required|string',
            'nrc_front' => 'nullable|string', // base64 string from Android
            'nrc_back' => 'nullable|string',  // base64 string from Android
            'verification_status' => 'required|in:pending,verified,failed,skipped',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        $token = $request->registration_token;
        
        // 1. Retrieve ALL cached steps from the multi-step flow
        $step1Data = \Cache::get("api_registration.{$token}.step1");
        $step2Data = \Cache::get("api_registration.{$token}.step2");
        $step3Data = \Cache::get("api_registration.{$token}.step3");
        $nokData   = \Cache::get("api_registration.{$token}.nok");
        $empData   = \Cache::get("api_registration.{$token}.employment");
        
        // 2. Comprehensive Session Check
        if (!$step1Data || !$step3Data) {
            return response()->json([
                'success' => false,
                'message' => 'Registration session expired or incomplete. Please restart registration.'
            ], 404);
        }

        DB::beginTransaction();

        // 3. Handle NRC uploads (Base64 to Physical Storage)
        $nrcFrontPath = null;
        $nrcBackPath = null;
        if ($request->filled('nrc_front')) {
            $nrcFrontPath = $this->handleBase64ImageUpload($request->nrc_front, 'nrc-documents', 'nrc_front_');
        }
        if ($request->filled('nrc_back')) {
            $nrcBackPath = $this->handleBase64ImageUpload($request->nrc_back, 'nrc-documents', 'nrc_back_');
        }

        // 4. Generate Unique Identifiers
        $customerNumber = Customer::generateCustomerNumber();
        $otp = $this->generateOTP();

        // 5. Create Customer Record with all seniority and tracking fields (Version 18)
        $customer = Customer::create([
            // Identity & Auth (Step 1 & 3)
            'email'      => $step1Data['email'],
            'pass'       => $step1Data['password'],
            'title'      => $step3Data['title'],
            'surname'    => $step3Data['surname'],
            'first_name' => $step3Data['first_name'],
            'middle_name'=> $step3Data['middle_name'],
            'customer'   => $step3Data['customer'], // Full Name String
            'dob'        => $step3Data['dob'],
            'phone'      => $step3Data['phone'],
            'office_phone' => $step3Data['office_phone'],
            'cgroup'     => $step3Data['cgroup'],
            'idtype'     => $step3Data['idtype'],
            'number'     => $step3Data['number'], // NRC Number
            'address'    => $step3Data['address'],
            'town'       => $step3Data['town'],
            'province'   => $step3Data['province'],
            'income'     => $step3Data['income'] ?? 0.00,
            
            // Image Paths (Step 2 & 4)
            'docpath'             => $step2Data['profile_picture_path'] ?? null,
            'nrc_front_path'      => $nrcFrontPath,
            'nrc_back_path'       => $nrcBackPath,
            'verification_status' => $request->verification_status,

            // Next of Kin Section
            'nok_first_name'       => $nokData['nok_first_name'] ?? null,
            'nok_surname'          => $nokData['nok_surname'] ?? null,
            'nok_mobile_phone'     => $nokData['nok_mobile_phone'] ?? null,
            'nok_phone'            => $nokData['nok_phone'] ?? null,
            'nok_telephone'        => $nokData['nok_telephone'] ?? null,
            'nok_physical_address' => $nokData['nok_physical_address'] ?? null,
            'nok_postal_address'   => $nokData['nok_postal_address'] ?? null,
            'nok_town'             => $nokData['nok_town'] ?? null,
            'nok_province'         => $nokData['nok_province'] ?? null,
            'nok_country'          => $nokData['nok_country'] ?? null,
            'nok_email'            => $nokData['nok_email'] ?? null,

            // Employment & Seniority Section (Updated for Version 18)
            'job_title'         => $empData['job_title'] ?? null,
            'employer_name'     => $empData['employer_name'] ?? null,
            'employee_no'       => $empData['employee_no'] ?? null,
            'years_in_employment' => $empData['years_in_employment'] ?? 0,
            'retirement_year'   => $empData['retirement_year'] ?? null,
            'employer_address'  => $empData['employer_address'] ?? null,
            'employer_postal_address' => $empData['employer_postal_address'] ?? null,
            'employer_town'     => $empData['employer_town'] ?? null,
            'employer_province' => $empData['employer_province'] ?? null,
            'employment_type'   => $empData['employment_type'] ?? null,
            'contract_expiry_date' => $empData['contract_expiry_date'] ?? null,
            'occupation'        => $empData['job_title'] ?? '', // Legacy column support
            'company'           => $empData['employer_name'] ?? '', // Legacy column support

            // System & Audit Metadata
            'customer_number'     => $customerNumber,
            'status'              => 'Active',
            'postdate'            => now()->toDateString(),
            'branchid'            => 1,
            'companyid'           => 1,
            'otp'                 => $otp,
            'otp_expires_at'      => now()->addMinutes(10),
            'otp_verified'        => false,
            'password_changed_at' => now(),
            'last_login_at'       => now(),
            'email_verified_at'   => now(),
        ]);

        // 6. Finalize Financial Profile
        $this->createDefaultCreditScore($customer);
        $smsSent = $this->sendOTP($customer, $otp, 'registration');

        // 7. Cleanup Cache - Free up server memory
        $cacheKeys = ['step1', 'step2', 'step3', 'nok', 'employment'];
        foreach($cacheKeys as $key) {
            \Cache::forget("api_registration.{$token}.{$key}");
        }

        DB::commit();

        // 8. Return response containing all data for Android Room database sync
return response()->json([
    'success' => true,
    'message' => 'Registration complete! Welcome to Immia Finance.',
    'data' => [
        // --- Core Account Details ---
        'customer_id'         => $customer->id,
        'customer_number'     => $customer->customer_number,
        'name'                => $customer->customer,
        'email'               => $customer->email,
        'phone'               => $customer->phone,
        'verification_status' => $customer->verification_status,
        'credit_score'        => 300,
        'score_band'          => 'Fair',
        'sms_sent'            => $smsSent,
        'requires_otp_verification' => true,

        // --- Personal Information (Syncs to Room Version 17) ---
        'title'        => $customer->title,
        'first_name'   => $customer->first_name,
        'surname'      => $customer->surname,
        'middle_name'  => $customer->middle_name,
        'dob'          => $customer->dob ? $customer->dob->toDateString() : null,
        'address'      => $customer->address,
        'town'         => $customer->town,
        'province'     => $customer->province,
        'cgroup'       => $customer->cgroup,
        'idtype'       => $customer->idtype,
        'number'       => $customer->number,
        'income'       => (double) $customer->income,

        // --- Next of Kin Details (Full Sync) ---
        'nok_first_name'       => $customer->nok_first_name,
        'nok_surname'          => $customer->nok_surname,
        'nok_mobile_phone'     => $customer->nok_mobile_phone,
        'nok_phone'            => $customer->nok_phone,
        'nok_telephone'        => $customer->nok_telephone,
        'nok_email'            => $customer->nok_email,
        'nok_physical_address' => $customer->nok_physical_address,
        'nok_postal_address'   => $customer->nok_postal_address,
        'nok_town'             => $customer->nok_town,
        'nok_province'         => $customer->nok_province,
        'nok_country'          => $customer->nok_country,

        // --- Employment Details (Syncs to Room Version 18) ---
        'job_title'               => $customer->job_title,
        'employer_name'           => $customer->employer_name,
        'employee_no'             => $customer->employee_no,
        'years_in_employment'     => (int) $customer->years_in_employment,
        'retirement_year'         => (int) $customer->retirement_year,
        'employer_address'        => $customer->employer_address,
        'employer_postal_address' => $customer->employer_postal_address,
        'employer_town'           => $customer->employer_town,
        'employer_province'       => $customer->employer_province,
        'employment_type'         => $customer->employment_type,
        'contract_expiry_date'    => $customer->contract_expiry_date ? $customer->contract_expiry_date->toDateString() : null,
    ]
], 201);

    } catch (\Exception $e) {
        DB::rollBack();
        \Log::error('API Registration Final Submission Error: ' . $e->getMessage());
        return response()->json([
            'success' => false,
            'message' => 'Registration failed: ' . $e->getMessage()
        ], 500);
    }
}

    /**
     * Get registration progress - API
     * GET /api/customer/register/progress/{token}
     */
    public function apiGetProgress($token): JsonResponse
    {
        try {
            $steps = [];
            $completedSteps = 0;
            $totalSteps = 4;
            
            for ($i = 1; $i <= $totalSteps; $i++) {
                $stepData = \Cache::get("api_registration.{$token}.step{$i}");
                $steps["step{$i}"] = !is_null($stepData);
                if (!is_null($stepData)) {
                    $completedSteps++;
                }
            }
            
            $progressPercentage = ($completedSteps / $totalSteps) * 100;
            
            // Determine next step
            $nextStep = null;
            $nextStepName = null;
            
            for ($i = 1; $i <= $totalSteps; $i++) {
                if (is_null(\Cache::get("api_registration.{$token}.step{$i}"))) {
                    $nextStep = $i;
                    break;
                }
            }
            
            // Map step numbers to names
            $stepNames = [
                1 => ['number' => 1, 'name' => 'Account Details', 'api_route' => 'step1'],
                2 => ['number' => 2, 'name' => 'Profile Picture', 'api_route' => 'step2'],
                3 => ['number' => 3, 'name' => 'Personal Information', 'api_route' => 'step3'],
                4 => ['number' => 4, 'name' => 'NRC Verification', 'api_route' => 'step4'],
            ];
            
            if ($nextStep && isset($stepNames[$nextStep])) {
                $nextStepName = $stepNames[$nextStep];
            }
            
            // Check if session expired
            $sessionActive = false;
            if ($completedSteps > 0) {
                $step1Data = \Cache::get("api_registration.{$token}.step1");
                if ($step1Data && isset($step1Data['expires_at'])) {
                    $expiresAt = Carbon::parse($step1Data['expires_at']);
                    $sessionActive = now()->lt($expiresAt);
                }
            }
            
            // Get step data without sensitive information
            $stepData = [];
            for ($i = 1; $i <= $totalSteps; $i++) {
                $data = \Cache::get("api_registration.{$token}.step{$i}");
                if ($data) {
                    // Remove sensitive data
                    unset($data['password']);
                    unset($data['pass']);
                    $stepData["step{$i}"] = $data;
                }
            }

            return response()->json([
                'success' => true,
                'data' => [
                    'registration_token' => $token,
                    'completed_steps' => $completedSteps,
                    'total_steps' => $totalSteps,
                    'progress_percentage' => $progressPercentage,
                    'next_step' => $nextStep,
                    'next_step_details' => $nextStepName,
                    'steps_completed' => $steps,
                    'session_active' => $sessionActive,
                    'step_data' => $stepData,
                    'all_steps' => $stepNames,
                ]
            ]);

        } catch (\Exception $e) {
            \Log::error('API Get registration progress error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to get registration progress'
            ], 500);
        }
    }

    /**
     * Resume registration - API
     * GET /api/customer/register/resume/{token}
     */
    public function apiResumeRegistration($token): JsonResponse
    {
        try {
            $progress = $this->apiGetProgress($token);
            $data = json_decode($progress->getContent(), true);
            
            if (!$data['success']) {
                return $progress;
            }
            
            if ($data['data']['completed_steps'] === 0) {
                return response()->json([
                    'success' => false,
                    'message' => 'No registration session found. Please start over.'
                ], 404);
            }
            
            if (!$data['data']['session_active']) {
                return response()->json([
                    'success' => false,
                    'message' => 'Registration session has expired. Please start over.'
                ], 410); // 410 Gone
            }
            
            return response()->json([
                'success' => true,
                'message' => 'Registration session resumed successfully',
                'data' => $data['data']
            ]);

        } catch (\Exception $e) {
            \Log::error('API Resume registration error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to resume registration'
            ], 500);
        }
    }

    /**
     * Cancel registration - API
     * DELETE /api/customer/register/cancel/{token}
     */
    public function apiCancelRegistration($token): JsonResponse
    {
        try {
            // Clear all registration data
            for ($i = 1; $i <= 4; $i++) {
                \Cache::forget("api_registration.{$token}.step{$i}");
            }
            
            \Log::info('API Registration cancelled', ['token' => $token]);
            
            return response()->json([
                'success' => true,
                'message' => 'Registration cancelled successfully'
            ]);

        } catch (\Exception $e) {
            \Log::error('API Cancel registration error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to cancel registration'
            ], 500);
        }
    }

    // ==================== AUTHENTICATION METHODS ====================

    /**
     * Customer login with OTP (API)
     * POST /api/customer/login
     */
 /**
 * Customer login with OTP (API)
 * POST /api/customer/login
 */
public function apiLogin(Request $request): JsonResponse
{
    try {
        $validator = Validator::make($request->all(), [
            'email' => 'required|email',
            'password' => 'required|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        $customer = Customer::where('email', $request->email)
            ->where('status', 'Active')
            ->first();

        if (!$customer) {
            return response()->json([
                'success' => false,
                'message' => 'Invalid credentials or account not active'
            ], 401);
        }

        // Check if account is locked
        if ($customer->locked_until && $customer->locked_until->isFuture()) {
            $minutesRemaining = now()->diffInMinutes($customer->locked_until);
            return response()->json([
                'success' => false,
                'message' => "Account is locked. Please try again in {$minutesRemaining} minutes."
            ], 423); // 423 Locked status code
        }

        // Verify password
        if (!Hash::check($request->password, $customer->pass)) {
            // Increment failed login attempts
            $attempts = $customer->login_attempts + 1;
            $customer->update(['login_attempts' => $attempts]);

            // Lock account after 5 failed attempts for 30 minutes
            if ($attempts >= 5) {
                $customer->update([
                    'locked_until' => now()->addMinutes(30),
                    'login_attempts' => 0
                ]);
                
                return response()->json([
                    'success' => false,
                    'message' => 'Too many failed attempts. Account locked for 30 minutes.'
                ], 423);
            }

            return response()->json([
                'success' => false,
                'message' => 'Invalid credentials',
                'attempts_remaining' => 5 - $attempts
            ], 401);
        }

        // Reset login attempts on successful password verification
        $customer->update(['login_attempts' => 0]);

        // Check if customer has credit score, create if not
        if (!$customer->creditScores()->exists()) {
            $this->createDefaultCreditScore($customer);
        }

        // Generate and send OTP
        $otp = $this->generateOTP();
        
        $customer->update([
            'otp' => $otp,
            'otp_expires_at' => Carbon::now()->addMinutes(10),
            'otp_verified' => false,
            // Note: We don't update verification_status here because 
            // it should only be updated when OTP is actually verified
            // in the apiVerifyOtp method
        ]);

        $smsSent = $this->sendOTP($customer, $otp, 'login');

        return response()->json([
            'success' => true,
            'message' => $smsSent 
                ? 'OTP has been sent to your phone' 
                : 'Please use OTP for verification',
            'data' => [
                'customer_id' => $customer->id,
                'requires_otp_verification' => true,
                'otp_expires_in' => 10,
                'sms_sent' => $smsSent,
                'current_verification_status' => $customer->verification_status, // Return current status
            ]
        ]);

    } catch (\Exception $e) {
        \Log::error('API Login error: ' . $e->getMessage(), [
            'error' => $e->getTraceAsString(),
            'email' => $request->email ?? 'N/A'
        ]);
        
        return response()->json([
            'success' => false,
            'message' => 'Login failed. Please try again.'
        ], 500);
    }
}
    /**
     * Verify OTP (API)
     * POST /api/customer/verify-otp
     */
       /**
 * Verify OTP (API)
 * POST /api/customer/verify-otp
 */

/**
 * Verify OTP (API)
 * POST /api/customer/verify-otp
 */
public function apiVerifyOtp(Request $request): JsonResponse
{
    try {
        // Log the incoming request
        \Log::info('OTP Verification Request Received', [
            'method' => $request->method(),
            'url' => $request->fullUrl(),
            'ip' => $request->ip(),
            'user_agent' => $request->userAgent(),
            'request_data' => $request->all(),
            'timestamp' => now()->toDateTimeString(),
        ]);

        $validator = Validator::make($request->all(), [
            'customer_id' => 'required|exists:customer,id',
            'otp' => 'required|string|size:6',
        ]);

        if ($validator->fails()) {
            \Log::warning('OTP Verification Validation Failed', [
                'errors' => $validator->errors()->toArray(),
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        $customer = Customer::where('id', $request->customer_id)
            ->where('status', 'Active')
            ->first();

        if (!$customer) {
            \Log::warning('OTP Verification - Customer Not Found', [
                'customer_id' => $request->customer_id,
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Customer not found or inactive'
            ], 404);
        }

        // Verify OTP
        if (!$customer->otp || !$customer->otp_expires_at) {
            \Log::warning('OTP Verification - No OTP Found', [
                'customer_id' => $customer->id,
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'No OTP found. Please request a new OTP.'
            ], 400);
        }

        // Check if OTP is expired
        if (Carbon::now()->gt($customer->otp_expires_at)) {
            \Log::warning('OTP Verification - OTP Expired', [
                'customer_id' => $customer->id,
                'otp_expires_at' => $customer->otp_expires_at,
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'OTP has expired. Please request a new OTP.'
            ], 400);
        }

        // Verify OTP
        if ($customer->otp !== $request->otp) {
            \Log::warning('OTP Verification - Invalid OTP', [
                'customer_id' => $customer->id,
                'attempt_ip' => $request->ip(),
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Invalid OTP'
            ], 400);
        }

        // OTP verified successfully - Update verification_status to 'verified'
        \Log::info('OTP Verification - Success', [
            'customer_id' => $customer->id,
            'customer_email' => $customer->email,
            'previous_verification_status' => $customer->verification_status,
            'new_verification_status' => 'verified',
        ]);

        // Update customer with verification_status and other fields
        $customer->update([
            'otp_verified' => true,
            'otp_verified_at' => now(),
            'last_login_at' => now(),
            'otp' => null, // Clear OTP after successful verification
            'otp_expires_at' => null,
            'verification_status' => 'verified', // Update verification_status to verified
        ]);

        // Generate API token (using Sanctum)
        $token = $customer->createToken('customer-api-token')->plainTextToken;

        // Create credit score if not exists
        if (!$customer->creditScores()->exists()) {
            $this->createDefaultCreditScore($customer);
        }

        return response()->json([
            'success' => true,
            'message' => 'OTP verified successfully',
            'data' => [
                'customer' => [
                    'id' => $customer->id,
                    'customer_number' => $customer->customer_number,
                    'name' => $customer->customer,
                    'email' => $customer->email,
                    'phone' => $customer->phone,
                    'profile_picture' => $customer->docpath ? asset('storage/' . $customer->docpath) : null,
                    'verification_status' => 'verified', // Return verified status
                    'credit_score' => $customer->current_credit_score ?? 300,
                    'score_band' => $customer->current_score_band,
                ],
                'access_token' => $token,
                'token_type' => 'Bearer',
                'expires_in' => config('sanctum.expiration', 525600), // in minutes
            ]
        ]);

    } catch (\Exception $e) {
        \Log::error('OTP Verification error: ' . $e->getMessage(), [
            'error' => $e->getTraceAsString(),
            'customer_id' => $request->customer_id ?? 'N/A',
        ]);
        
        return response()->json([
            'success' => false,
            'message' => 'OTP verification failed. Please try again.'
        ], 500);
    }
}
    /**
     * Resend OTP (API)
     * POST /api/customer/resend-otp
     */
    public function apiResendOtp(Request $request): JsonResponse
    {
        try {
            $validator = Validator::make($request->all(), [
                'customer_id' => 'required|exists:customer,id',
                'purpose' => 'nullable|in:login,registration,password_reset'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }

            $customer = Customer::where('id', $request->customer_id)
                ->where('status', 'Active')
                ->first();

            if (!$customer) {
                return response()->json([
                    'success' => false,
                    'message' => 'Customer not found or inactive'
                ], 404);
            }

            // Check if OTP was recently sent (prevent abuse)
            if ($customer->otp_expires_at && now()->lt($customer->otp_expires_at)) {
                $secondsRemaining = now()->diffInSeconds($customer->otp_expires_at);
                if ($secondsRemaining > 30) { // 30-second cooldown
                    return response()->json([
                        'success' => false,
                        'message' => 'Please wait before requesting a new OTP.'
                    ], 429); // Too Many Requests
                }
            }

            // Generate new OTP
            $otp = $this->generateOTP();
            
            $customer->update([
                'otp' => $otp,
                'otp_expires_at' => Carbon::now()->addMinutes(10),
                'otp_verified' => false,
            ]);

            $smsSent = $this->sendOTP($customer, $otp, $request->purpose ?? 'login');

            return response()->json([
                'success' => true,
                'message' => $smsSent 
                    ? 'New OTP has been sent to your phone' 
                    : 'OTP generated. Please check your messages.',
                'data' => [
                    'customer_id' => $customer->id,
                    'otp_expires_in' => 10,
                    'sms_sent' => $smsSent,
                ]
            ]);

        } catch (\Exception $e) {
            \Log::error('Resend OTP error: ' . $e->getMessage(), [
                'error' => $e->getTraceAsString(),
                'customer_id' => $request->customer_id ?? 'N/A'
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to resend OTP. Please try again.'
            ], 500);
        }
    }

    /**
     * Customer logout (API)
     * POST /api/customer/logout
     */
    public function apiLogout(Request $request): JsonResponse
    {
        try {
            $customer = $request->user();
            
            if (!$customer) {
                return response()->json([
                    'success' => false,
                    'message' => 'Not authenticated'
                ], 401);
            }
            
            // Revoke current token
            $request->user()->currentAccessToken()->delete();
            
            \Log::info('Customer logged out via API', [
                'customer_id' => $customer->id,
                'email' => $customer->email
            ]);
            
            return response()->json([
                'success' => true,
                'message' => 'Logged out successfully'
            ]);
            
        } catch (\Exception $e) {
            \Log::error('API Logout error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Logout failed'
            ], 500);
        }
    }

    /**
     * Check authentication (API)
     * GET /api/customer/check-auth
     */
    public function apiCheckAuth(Request $request): JsonResponse
    {
        try {
            $customer = $request->user();
            
            if (!$customer) {
                return response()->json([
                    'success' => false,
                    'message' => 'Not authenticated'
                ], 401);
            }
            
            // Refresh credit score from database
            $creditScore = CustomerCreditScore::where('customer_id', $customer->id)
                ->where('is_current', true)
                ->first();
            
            // Get active loans count
            $activeLoans = $customer->activeDisbursedLoans()->count();
            
            return response()->json([
                'success' => true,
                'data' => [
                    'customer' => [
                        'id' => $customer->id,
                        'customer_number' => $customer->customer_number,
                        'name' => $customer->customer,
                        'email' => $customer->email,
                        'phone' => $customer->phone,
                        'profile_picture' => $customer->docpath ? asset('storage/' . $customer->docpath) : null,
                        'verification_status' => $customer->verification_status,
                        'credit_score' => $customer->current_credit_score ?? 300,
                        'score_band' => $customer->current_score_band,
                        'credit_limit' => $customer->credit_limit,
                        'available_credit' => $creditScore->available_credit ?? $customer->credit_limit,
                    ],
                    'portfolio' => [
                        'active_loans' => $activeLoans,
                        'total_due_balance' => $customer->total_due_balance ?? 0,
                        'total_next_due_amount' => $customer->total_next_due_amount ?? 0,
                        'total_overdue_amount' => $customer->total_overdue_amount ?? 0,
                        'has_active_loans' => $activeLoans > 0,
                        'has_overdue' => ($customer->total_overdue_amount ?? 0) > 0,
                    ],
                    'authenticated' => true,
                    'last_login' => $customer->last_login_at
                ]
            ]);
            
        } catch (\Exception $e) {
            \Log::error('API Check auth error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Authentication check failed'
            ], 500);
        }
    }

    /**
     * Get customer profile (API)
     * GET /api/customer/profile
     */
    public function apiGetProfile(Request $request): JsonResponse
    {
        try {
            $customer = $request->user();
            
            if (!$customer) {
                return response()->json([
                    'success' => false,
                    'message' => 'Not authenticated'
                ], 401);
            }
            
            $creditScore = CustomerCreditScore::where('customer_id', $customer->id)
                ->where('is_current', true)
                ->first();
            
            // Safely load relationships
            $customer->load(['currentCreditScore', 'activeDisbursedLoans']);
            
            return response()->json([
                'success' => true,
                'data' => [
                    'personal_info' => [
                        'customer_number' => $customer->customer_number,
                        'title' => $customer->title,
                        'surname' => $customer->surname,
                        'first_name' => $customer->first_name,
                        'middle_name' => $customer->middle_name,
                        'full_name' => $customer->customer,
                        'email' => $customer->email,
                        'phone' => $customer->phone,
                        'office_phone' => $customer->office_phone,
                        'dob' => $customer->dob ? $customer->dob->format('Y-m-d') : null,
                        'age' => $customer->age,
                        'id_type' => $customer->idtype,
                        'id_number' => $customer->number,
                        'address' => $customer->address,
                        'town' => $customer->town,
                        'province' => $customer->province,
                        'complete_address' => $customer->complete_address,
                        'customer_group' => $customer->cgroup,
                        'profile_picture' => $customer->docpath ? asset('storage/' . $customer->docpath) : null,
                        'nrc_front' => $customer->nrc_front_path ? asset('storage/' . $customer->nrc_front_path) : null,
                        'nrc_back' => $customer->nrc_back_path ? asset('storage/' . $customer->nrc_back_path) : null,
                    ],
                    'employment_info' => [
                        'occupation' => $customer->occupation,
                        'company' => $customer->company,
                        'employment_info' => $customer->employment_info,
                        'income' => $customer->income,
                        'business_name' => $customer->bname,
                        'business_start_date' => $customer->bstartdate,
                        'incorporation_id' => $customer->incorporationid,
                        'tpin' => $customer->tpin,
                    ],
                    'account_info' => [
                        'status' => $customer->status,
                        'verification_status' => $customer->verification_status,
                        'email_verified_at' => $customer->email_verified_at,
                        'registered_since' => $customer->created_at->format('F d, Y'),
                        'last_login' => $customer->last_login_at,
                        'password_changed_at' => $customer->password_changed_at,
                    ],
                    'credit_info' => [
                        'credit_score' => $customer->current_credit_score ?? 300,
                        'score_band' => $customer->current_score_band,
                        'credit_limit' => $customer->credit_limit,
                        'available_credit' => $creditScore->available_credit ?? $customer->credit_limit,
                        'total_borrowed' => $creditScore->total_borrowed ?? 0,
                        'current_outstanding' => $creditScore->current_outstanding ?? 0,
                        'total_repaid' => $creditScore->total_repaid ?? 0,
                        'on_time_repayment_rate' => $creditScore->on_time_repayment_rate ?? 0,
                        'credit_utilization_ratio' => $creditScore->credit_utilization_ratio ?? 0,
                        'late_payments_count' => $creditScore->late_payments_count ?? 0,
                        'missed_payments_count' => $creditScore->missed_payments_count ?? 0,
                        'score_trend' => $customer->credit_score_trend,
                        'score_change' => $customer->credit_score_change,
                        'factors' => $customer->credit_score_factors,
                        'recommendations' => $customer->credit_improvement_recommendations,
                    ],
                    'loan_info' => [
                        'active_loans_count' => $customer->activeDisbursedLoans()->count(),
                        'total_next_due_amount' => $customer->total_next_due_amount,
                        'total_overdue_amount' => $customer->total_overdue_amount,
                        'total_due_balance' => $customer->total_due_balance,
                        'next_payment_due_date' => $customer->getNextPaymentDueDate(),
                        'can_apply_for_loan' => $customer->canApplyForLoan(),
                        'can_apply_with_credit_check' => $customer->canApplyForLoanWithCreditCheck(),
                        'recommended_max_loan_amount' => $customer->recommended_max_loan_amount,
                        'has_reached_application_limit' => $customer->hasReachedApplicationLimit(),
                    ],
                    'financial_overview' => $customer->getFinancialOverview(),
                ]
            ]);
            
        } catch (\Exception $e) {
            \Log::error('API Get profile error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch profile'
            ], 500);
        }
    }

    /**
     * Update customer profile (API)
     * PUT /api/customer/profile
     */
    public function apiUpdateProfile(Request $request): JsonResponse
    {
        try {
            $customer = $request->user();
            
            if (!$customer) {
                return response()->json([
                    'success' => false,
                    'message' => 'Not authenticated'
                ], 401);
            }
            
            $validator = Validator::make($request->all(), [
                'phone' => 'nullable|string|max:25',
                'office_phone' => 'nullable|string|max:25',
                'address' => 'nullable|string',
                'town' => 'nullable|string',
                'province' => 'nullable|string',
                'occupation' => 'nullable|string|max:255',
                'company' => 'nullable|string|max:255',
                'income' => 'nullable|numeric|min:0',
                'profile_picture' => 'nullable|string', // base64
            ]);
            
            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }
            
            DB::beginTransaction();
            
            $updateData = [];
            $needsPhoneVerification = false;
            
            // Update phone if provided (requires OTP verification)
            if ($request->has('phone') && $request->phone !== $customer->phone) {
                // Check if phone is already used by another customer
                $existingCustomer = Customer::where('phone', $request->phone)
                    ->where('id', '!=', $customer->id)
                    ->first();
                
                if ($existingCustomer) {
                    return response()->json([
                        'success' => false,
                        'message' => 'Phone number already registered to another customer'
                    ], 409);
                }
                
                $updateData['phone'] = $request->phone;
                $needsPhoneVerification = true;
            }
            
            // Update other fields
            $fields = ['office_phone', 'address', 'town', 'province', 'occupation', 'company', 'income'];
            foreach ($fields as $field) {
                if ($request->has($field) && $request->$field !== $customer->$field) {
                    $updateData[$field] = $request->$field;
                }
            }
            
            // Handle profile picture update
            if ($request->has('profile_picture') && !empty($request->profile_picture)) {
                // Delete old profile picture if exists
                if ($customer->docpath && Storage::disk('public')->exists($customer->docpath)) {
                    Storage::disk('public')->delete($customer->docpath);
                }
                
                // Upload new profile picture
                $profilePicturePath = $this->handleBase64ImageUpload(
                    $request->profile_picture, 
                    'profile-pictures', 
                    'profile_'
                );
                
                if ($profilePicturePath) {
                    $updateData['docpath'] = $profilePicturePath;
                }
            }
            
            // Update customer
            if (!empty($updateData)) {
                $customer->update($updateData);
            }
            
            DB::commit();
            
            $response = [
                'success' => true,
                'message' => 'Profile updated successfully',
                'data' => [
                    'updated_fields' => array_keys($updateData),
                    'profile_picture_url' => $customer->docpath ? asset('storage/' . $customer->docpath) : null,
                ]
            ];
            
            // If phone was changed, add verification requirement
            if ($needsPhoneVerification) {
                // Generate OTP for phone verification
                $otp = $this->generateOTP();
                $customer->update([
                    'otp' => $otp,
                    'otp_expires_at' => Carbon::now()->addMinutes(10),
                    'otp_verified' => false,
                ]);
                
                // Send OTP via SMS
                $smsSent = $this->sendOTP($customer, $otp, 'phone_verification');
                
                $response['requires_phone_verification'] = true;
                $response['phone_verification'] = [
                    'customer_id' => $customer->id,
                    'otp_expires_in' => 10,
                    'sms_sent' => $smsSent,
                ];
                $response['message'] .= ' Please verify your new phone number with OTP.';
            }
            
            return response()->json($response);
            
        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('API Update profile error: ' . $e->getMessage(), [
                'error' => $e->getTraceAsString(),
                'customer_id' => $customer->id ?? 'N/A'
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to update profile: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Change password (API)
     * POST /api/customer/change-password
     */
    public function apiChangePassword(Request $request): JsonResponse
    {
        try {
            $customer = $request->user();
            
            if (!$customer) {
                return response()->json([
                    'success' => false,
                    'message' => 'Not authenticated'
                ], 401);
            }
            
            $validator = Validator::make($request->all(), [
                'current_password' => 'required|string',
                'new_password' => 'required|string|min:8|confirmed',
            ]);
            
            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }
            
            // Verify current password
            if (!Hash::check($request->current_password, $customer->pass)) {
                return response()->json([
                    'success' => false,
                    'message' => 'Current password is incorrect'
                ], 401);
            }
            
            // Update password
            $customer->update([
                'pass' => Hash::make($request->new_password),
                'password_changed_at' => now(),
            ]);
            
            \Log::info('Customer changed password via API', [
                'customer_id' => $customer->id,
                'email' => $customer->email
            ]);
            
            return response()->json([
                'success' => true,
                'message' => 'Password changed successfully'
            ]);
            
        } catch (\Exception $e) {
            \Log::error('API Change password error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to change password'
            ], 500);
        }
    }

    /**
     * Forgot password (API)
     * POST /api/customer/forgot-password
     */
    public function apiForgotPassword(Request $request): JsonResponse
    {
        try {
            $validator = Validator::make($request->all(), [
                'email' => 'required|email|exists:customer,email',
            ]);
            
            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }
            
            $customer = Customer::where('email', $request->email)
                ->where('status', 'Active')
                ->first();
            
            if (!$customer) {
                return response()->json([
                    'success' => false,
                    'message' => 'Customer not found or account is inactive'
                ], 404);
            }
            
            // Check if OTP was recently sent (prevent abuse)
            if ($customer->otp_expires_at && now()->lt($customer->otp_expires_at)) {
                $secondsRemaining = now()->diffInSeconds($customer->otp_expires_at);
                if ($secondsRemaining > 30) { // 30-second cooldown
                    return response()->json([
                        'success' => false,
                        'message' => 'Please wait before requesting a new OTP.'
                    ], 429); // Too Many Requests
                }
            }
            
            // Generate OTP
            $otp = $this->generateOTP();
            
            $customer->update([
                'otp' => $otp,
                'otp_expires_at' => Carbon::now()->addMinutes(10),
                'otp_verified' => false,
            ]);
            
            // Send OTP via SMS
            $smsSent = $this->sendOTP($customer, $otp, 'password_reset');
            
            \Log::info('Password reset OTP sent', [
                'customer_id' => $customer->id,
                'email' => $customer->email,
                'sms_sent' => $smsSent
            ]);
            
            return response()->json([
                'success' => true,
                'message' => $smsSent 
                    ? 'OTP has been sent to your registered phone number for password reset' 
                    : 'Password reset initiated',
                'data' => [
                    'customer_id' => $customer->id,
                    'requires_otp_verification' => true,
                    'otp_expires_in' => 10,
                    'sms_sent' => $smsSent,
                ]
            ]);
            
        } catch (\Exception $e) {
            \Log::error('API Forgot password error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to process password reset request'
            ], 500);
        }
    }

    /**
     * Reset password (API)
     * POST /api/customer/reset-password
     */
    public function apiResetPassword(Request $request): JsonResponse
    {
        try {
            $validator = Validator::make($request->all(), [
                'customer_id' => 'required|exists:customer,id',
                'otp' => 'required|string|size:6',
                'new_password' => 'required|string|min:8|confirmed',
            ]);
            
            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }
            
            $customer = Customer::where('id', $request->customer_id)
                ->where('status', 'Active')
                ->first();
            
            if (!$customer) {
                return response()->json([
                    'success' => false,
                    'message' => 'Customer not found or inactive'
                ], 404);
            }
            
            // Verify OTP
            if (!$customer->otp || !$customer->otp_expires_at) {
                return response()->json([
                    'success' => false,
                    'message' => 'No OTP found. Please request a new OTP.'
                ], 400);
            }
            
            if (Carbon::now()->gt($customer->otp_expires_at)) {
                return response()->json([
                    'success' => false,
                    'message' => 'OTP has expired. Please request a new OTP.'
                ], 400);
            }
            
            if ($customer->otp !== $request->otp) {
                return response()->json([
                    'success' => false,
                    'message' => 'Invalid OTP'
                ], 400);
            }
            
            // Update password
            $customer->update([
                'pass' => Hash::make($request->new_password),
                'password_changed_at' => now(),
                'otp' => null,
                'otp_expires_at' => null,
                'otp_verified' => true,
                'otp_verified_at' => now(),
            ]);
            
            \Log::info('Customer reset password via API', [
                'customer_id' => $customer->id,
                'email' => $customer->email
            ]);
            
            return response()->json([
                'success' => true,
                'message' => 'Password reset successfully. You can now login with your new password.'
            ]);
            
        } catch (\Exception $e) {
            \Log::error('API Reset password error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to reset password'
            ], 500);
        }
    }

    /**
     * Verify phone number with OTP (API)
     * POST /api/customer/verify-phone
     */
    public function apiVerifyPhone(Request $request): JsonResponse
    {
        try {
            $customer = $request->user();
            
            if (!$customer) {
                return response()->json([
                    'success' => false,
                    'message' => 'Not authenticated'
                ], 401);
            }
            
            $validator = Validator::make($request->all(), [
                'otp' => 'required|string|size:6',
            ]);
            
            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }
            
            // Verify OTP
            if (!$customer->otp || !$customer->otp_expires_at) {
                return response()->json([
                    'success' => false,
                    'message' => 'No OTP found. Please request a new OTP.'
                ], 400);
            }
            
            if (Carbon::now()->gt($customer->otp_expires_at)) {
                return response()->json([
                    'success' => false,
                    'message' => 'OTP has expired. Please request a new OTP.'
                ], 400);
            }
            
            if ($customer->otp !== $request->otp) {
                return response()->json([
                    'success' => false,
                    'message' => 'Invalid OTP'
                ], 400);
            }
            
            // Mark phone as verified (you might want to add a phone_verified_at field)
            $customer->update([
                'otp_verified' => true,
                'otp_verified_at' => now(),
                'otp' => null,
                'otp_expires_at' => null,
            ]);
            
            \Log::info('Customer verified phone via API', [
                'customer_id' => $customer->id,
                'phone' => $customer->phone
            ]);
            
            return response()->json([
                'success' => true,
                'message' => 'Phone number verified successfully'
            ]);
            
        } catch (\Exception $e) {
            \Log::error('API Verify phone error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to verify phone number'
            ], 500);
        }
    }

    // ==================== WEB REGISTRATION METHODS ====================

    /**
     * Show customer registration step 1 form (email & password) - WEB
     */
    public function showRegistrationForm()
    {
        return view('auth.customer-register-step1');
    }

    /**
     * Handle registration step 1 (email & password) - WEB
     */
    public function step1(Request $request)
    {
        $request->validate([
            'email' => 'required|email|unique:customer,email',
            'password' => 'required|string|min:8|confirmed',
        ]);

        // Store step 1 data in session
        $request->session()->put('registration.step1', [
            'email' => $request->email,
            'password' => Hash::make($request->password),
        ]);

        return redirect()->route('customer.register.step2');
    }

    /**
     * Show registration step 2 form (profile picture only) - WEB
     */
    public function showStep2Form(Request $request)
    {
        // Check if step 1 is completed
        if (!$request->session()->has('registration.step1')) {
            return redirect()->route('customer.register')->with('error', 'Please complete step 1 first.');
        }

        return view('auth.customer-register-step2');
    }

    /**
     * Handle registration step 2 (profile picture only) - WEB
     */
    public function step2(Request $request)
    {
        // Check if step 1 is completed
        if (!$request->session()->has('registration.step1')) {
            return redirect()->route('customer.register')->with('error', 'Please complete step 1 first.');
        }

        $request->validate([
            'profile_picture' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
        ]);

        // Handle profile picture upload
        $profilePicturePath = null;
        if ($request->hasFile('profile_picture')) {
            $profilePicturePath = $this->handleProfilePictureUpload($request->file('profile_picture'));
        }

        // Handle webcam picture if provided
        if ($request->has('webcam_picture') && !empty($request->webcam_picture)) {
            $profilePicturePath = $this->handleWebcamPicture($request->webcam_picture);
        }

        // Store profile picture in session
        $request->session()->put('registration.step2', [
            'docpath' => $profilePicturePath,
        ]);

        return redirect()->route('customer.register.step3');
    }

    /**
     * Show registration step 3 form (personal information) - WEB
     */
    public function showStep3Form(Request $request)
    {
        // Check if step 1 and step 2 are completed
        if (!$request->session()->has('registration.step1')) {
            return redirect()->route('customer.register')->with('error', 'Please complete step 1 first.');
        }

        if (!$request->session()->has('registration.step2')) {
            return redirect()->route('customer.register.step2')->with('error', 'Please complete step 2 first.');
        }

        return view('auth.customer-register-step3');
    }

    /**
     * Handle registration step 3 (personal information) - WEB
     */
    public function step3(Request $request)
    {
        // Check if step 1 and step 2 are completed
        if (!$request->session()->has('registration.step1')) {
            return redirect()->route('customer.register')->with('error', 'Please complete step 1 first.');
        }

        if (!$request->session()->has('registration.step2')) {
            return redirect()->route('customer.register.step2')->with('error', 'Please complete step 2 first.');
        }

        $request->validate([
            'title' => 'required|string|max:10',
            'surname' => 'required|string|max:255',
            'first_name' => 'required|string|max:255',
            'middle_name' => 'nullable|string|max:255',
            'customer' => 'required|string|max:255',
            'dob' => 'required|date',
            'phone' => 'required|string|max:25',
            'office_phone' => 'nullable|string|max:25',
            'cgroup' => 'required|in:Individual,Company',
            'idtype' => 'required|string',
            'number' => 'required|string',
            'address' => 'required|string',
            'town' => 'required|string',
            'province' => 'required|string',
        ]);

        // Store step 3 data in session
        $step3Data = [
            'title' => $request->title,
            'surname' => $request->surname,
            'first_name' => $request->first_name,
            'middle_name' => $request->middle_name,
            'customer' => $request->customer,
            'dob' => $request->dob,
            'phone' => $request->phone,
            'office_phone' => $request->office_phone,
            'cgroup' => $request->cgroup,
            'idtype' => $request->idtype,
            'number' => $request->number,
            'address' => $request->address,
            'town' => $request->town,
            'province' => $request->province,
        ];

        $request->session()->put('registration.step3', $step3Data);

        return redirect()->route('customer.register.step4');
    }

    /**
     * Show registration step 4 form (NRC upload) - WEB
     */
    public function showStep4Form(Request $request)
    {
        // Check if previous steps are completed
        if (!$request->session()->has('registration.step1')) {
            return redirect()->route('customer.register')->with('error', 'Please complete step 1 first.');
        }

        if (!$request->session()->has('registration.step2')) {
            return redirect()->route('customer.register.step2')->with('error', 'Please complete step 2 first.');
        }

        if (!$request->session()->has('registration.step3')) {
            return redirect()->route('customer.register.step3')->with('error', 'Please complete step 3 first.');
        }

        return view('auth.customer-register-step4');
    }

    /**
     * Handle registration step 4 (NRC upload and verification) - WEB
     */
    public function step4(Request $request)
    {
        // Check if previous steps are completed
        if (!$request->session()->has('registration.step1')) {
            return redirect()->route('customer.register')->with('error', 'Please complete step 1 first.');
        }

        if (!$request->session()->has('registration.step2')) {
            return redirect()->route('customer.register.step2')->with('error', 'Please complete step 2 first.');
        }

        if (!$request->session()->has('registration.step3')) {
            return redirect()->route('customer.register.step3')->with('error', 'Please complete step 3 first.');
        }

        $request->validate([
            'nrc_front' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
            'nrc_back' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
            'verification_status' => 'required|in:pending,verified,failed,skipped',
        ]);

        // Get all previous step data
        $step1Data = $request->session()->get('registration.step1');
        $step2Data = $request->session()->get('registration.step2');
        $step3Data = $request->session()->get('registration.step3');

        // Handle NRC file uploads
        $nrcFrontPath = null;
        $nrcBackPath = null;

        if ($request->hasFile('nrc_front')) {
            $nrcFrontPath = $this->handleNrcUpload($request->file('nrc_front'), 'front');
        }

        if ($request->hasFile('nrc_back')) {
            $nrcBackPath = $this->handleNrcUpload($request->file('nrc_back'), 'back');
        }

        DB::beginTransaction();
        try {
            // Create customer with all data
            $customerData = array_merge($step3Data, [
                'email' => $step1Data['email'],
                'pass' => $step1Data['password'],
                'docpath' => $step2Data['docpath'] ?? null,
                'nrc_front_path' => $nrcFrontPath,
                'nrc_back_path' => $nrcBackPath,
                'verification_status' => $request->verification_status,
                'customer_number' => Customer::generateCustomerNumber(),
                'status' => 'Active',
                'postdate' => now()->toDateString(),
                'branchid' => 1,
                'companyid' => 1,
                'created_at' => now(),
                'updated_at' => now(),
                'income' => 5000, // Set default income for credit limit calculation
                // OTP fields
                'otp' => $this->generateOTP(),
                'otp_expires_at' => Carbon::now()->addMinutes(10),
                'otp_verified' => false,
                'password_changed_at' => now(),
                'last_login_at' => now(),
                'email_verified_at' => now(),
            ]);

            $customer = Customer::create($customerData);

            // Create default credit score
            $this->createDefaultCreditScore($customer);

            // Clear registration session
            $request->session()->forget('registration');

            // Send OTP via SMS
            $smsSent = $this->sendOTP($customer, $customer->otp, 'registration');

            DB::commit();

            // Build success message
            $message = 'Registration successful! Welcome to Immia Finance. Your customer number is: ' . $customer->customer_number;
            if ($request->verification_status === 'verified') {
                $message .= ' Your identity has been verified.';
            } elseif ($request->verification_status === 'failed') {
                $message .= ' Note: Identity verification failed. You may need to verify later for loan applications.';
            } elseif ($request->verification_status === 'skipped') {
                $message .= ' You can verify your identity later in your profile settings.';
            }

            if ($smsSent) {
                $message .= ' OTP has been sent to your phone for verification.';
            }

            // For web, we can either:
            // 1. Redirect to OTP verification page
            // 2. Or automatically log them in and redirect to dashboard
            // Let's go with option 2 for consistency with current web flow
            
            Auth::guard('customer')->login($customer);

            return redirect()->route('customer.dashboard')->with('success', $message);

        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('Customer registration failed: ' . $e->getMessage(), [
                'exception' => $e
            ]);

            return redirect()->back()
                ->withInput()
                ->with('error', 'Registration failed. Please try again. Error: ' . $e->getMessage());
        }
    }

    // ==================== HELPER METHODS ====================

    /**
     * Generate OTP
     */
    private function generateOTP(): string
    {
        return str_pad(rand(0, 999999), 6, '0', STR_PAD_LEFT);
    }

    /**
     * Send OTP via SMS
     */
    private function sendOTP(Customer $customer, string $otp, string $purpose = 'login'): bool
    {
        try {
            $messageTemplates = [
                'login' => "Your Immia Finance login code is: {$otp}. Valid for 10 minutes.",
                'registration' => "Your Immia Finance registration code is: {$otp}. Valid for 10 minutes.",
                'password_reset' => "Your Immia Finance password reset code is: {$otp}. Valid for 10 minutes.",
                'phone_verification' => "Your Immia Finance phone verification code is: {$otp}. Valid for 10 minutes.",
            ];

            $message = $messageTemplates[$purpose] ?? $messageTemplates['login'];
            
            $smsSent = $this->smsService->sendSms($customer->phone, $message);

            \Log::info('OTP sent', [
                'customer_id' => $customer->id,
                'phone' => $customer->phone,
                'purpose' => $purpose,
                'sms_sent' => $smsSent,
                'otp' => $otp,
            ]);

            return $smsSent;

        } catch (\Exception $e) {
            \Log::error('Failed to send OTP', [
                'customer_id' => $customer->id,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            return false;
        }
    }

    /**
     * Create default credit score for a customer
     */
    private function createDefaultCreditScore(Customer $customer): bool
    {
        try {
            $defaultIncome = $customer->income ?? 5000;
            $creditLimit = $defaultIncome * 3;

            $creditScore = CustomerCreditScore::create([
                'customer_id' => $customer->id,
                'credit_score' => 300,
                'score_band' => 'Very Poor',
                'previous_score' => null,
                'score_change' => 0,
                'change_reason' => 'Initial credit score upon registration',
                'change_type' => 'initialization',
                'days_delinquent' => 0,
                'total_loans' => 0,
                'active_loans' => 0,
                'closed_loans' => 0,
                'defaulted_loans' => 0,
                'total_borrowed' => 0.00,
                'current_outstanding' => 0.00,
                'total_repaid' => 0.00,
                'on_time_repayment_rate' => 0.00,
                'late_payments_count' => 0,
                'missed_payments_count' => 0,
                'credit_utilization_ratio' => 0.00,
                'available_credit' => $creditLimit,
                'credit_limit' => $creditLimit,
                'last_repayment_date' => null,
                'last_repayment_amount' => 0.00,
                'last_loan_date' => null,
                'last_credit_inquiry' => null,
                'has_active_default' => false,
                'has_recent_late_payment' => false,
                'has_high_utilization' => false,
                'has_multiple_active_loans' => false,
                'score_date' => now()->toDateString(),
                'is_current' => true,
                'created_by' => 1,
                'updated_by' => 1,
                'created_at' => now(),
                'updated_at' => now(),
            ]);

            \Log::info('Default credit score created', [
                'customer_id' => $customer->id,
                'credit_score_id' => $creditScore->id
            ]);

            return true;

        } catch (\Exception $e) {
            \Log::error('Error creating default credit score', [
                'customer_id' => $customer->id,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            return false;
        }
    }

    /**
     * Handle base64 image upload
     */
    private function handleBase64ImageUpload($base64Image, $folder, $prefix): ?string
    {
        try {
            if (preg_match('/^data:image\/(\w+);base64,/', $base64Image, $type)) {
                $image = substr($base64Image, strpos($base64Image, ',') + 1);
                $type = strtolower($type[1]);

                if (!in_array($type, ['jpg', 'jpeg', 'png', 'gif', 'webp'])) {
                    throw new \Exception('Invalid image type');
                }

                $image = str_replace(' ', '+', $image);
                $imageData = base64_decode($image);

                if ($imageData === false) {
                    throw new \Exception('Base64 decode failed');
                }

                $filename = $prefix . time() . '_' . uniqid() . '.' . $type;
                $path = $folder . '/' . $filename;

                Storage::disk('public')->put($path, $imageData);

                return $path;
            }

            return null;
        } catch (\Exception $e) {
            \Log::error('Base64 image upload failed: ' . $e->getMessage());
            return null;
        }
    }

    /**
     * Handle profile picture file upload (Web)
     */
    private function handleProfilePictureUpload($file)
    {
        $filename = 'profile_' . time() . '_' . uniqid() . '.' . $file->getClientOriginalExtension();
        $path = 'profile-pictures/' . $filename;
        
        // Store the file directly without resizing
        $file->storeAs('profile-pictures', $filename, 'public');
        
        return $path;
    }

    /**
     * Handle webcam picture (base64) - Web
     */
    private function handleWebcamPicture($base64Image)
    {
        // Remove the data:image/png;base64, part
        $image = str_replace('data:image/png;base64,', '', $base64Image);
        $image = str_replace(' ', '+', $image);
        $imageData = base64_decode($image);

        $filename = 'webcam_profile_' . time() . '_' . uniqid() . '.png';
        $path = 'profile-pictures/' . $filename;

        // Store the webcam image directly without processing
        Storage::disk('public')->put($path, $imageData);

        return $path;
    }

    /**
     * Handle NRC file upload (Web)
     */
    private function handleNrcUpload($file, $type)
    {
        $filename = 'nrc_' . $type . '_' . time() . '_' . uniqid() . '.' . $file->getClientOriginalExtension();
        $path = 'nrc-documents/' . $filename;
        
        $file->storeAs('nrc-documents', $filename, 'public');
        
        return $path;
    }
    
   

    /**
     * Sync Existing Customer - Step 1: Lookup by Phone & Send OTP
     * POST /api/customer/sync/lookup
     */
    public function apiSyncLookup(Request $request): JsonResponse
    {
        try {
            $validator = Validator::make($request->all(), [
                'phone' => 'required|string|max:25',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false, 
                    'message' => 'Valid phone number is required',
                    'errors' => $validator->errors()
                ], 422);
            }

            // 1. Find the customer - Ensure they are 'Active'
            $customer = Customer::where('phone', $request->phone)
                ->where('status', 'Active')
                ->first();

            if (!$customer) {
                return response()->json([
                    'success' => false,
                    'message' => 'No active profile found with this phone number. Please register as a new user.'
                ], 404);
            }

            // 2. Generate OTP specifically for Syncing
            $otp = $this->generateOTP();
            
            $customer->update([
                'otp' => $otp,
                'otp_expires_at' => Carbon::now()->addMinutes(10),
                'otp_verified' => false,
            ]);

            // 3. Send SMS using existing service
            $smsSent = $this->sendOTP($customer, $otp, 'phone_verification');

            \Log::info('Sync Lookup Success', ['customer_id' => $customer->id, 'phone' => $request->phone]);

            return response()->json([
                'success' => true,
                'message' => 'Account found! A verification code has been sent to your phone.',
                'data' => [
                    'customer_id' => $customer->id,
                    'phone_masked' => substr($customer->phone, 0, 4) . '****' . substr($customer->phone, -3),
                    'otp_expires_in' => 10 
                ]
            ]);

        } catch (\Exception $e) {
            \Log::error('API Sync Lookup Error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'An error occurred while looking up your profile.'
            ], 500);
        }
    }

    /**
     * Sync Existing Customer - Step 2: Verify OTP and Return FULL Profile
     * POST /api/customer/sync/verify
     */
    public function apiSyncVerify(Request $request): JsonResponse
    {
        try {
            $validator = Validator::make($request->all(), [
                'customer_id' => 'required|exists:customer,id',
                'otp' => 'required|string|size:6',
            ]);

            if ($validator->fails()) {
                return response()->json(['success' => false, 'errors' => $validator->errors()], 422);
            }

            $customer = Customer::find($request->customer_id);

            // 1. Verify OTP and Expiry
            if (!$customer->otp || $customer->otp !== $request->otp || Carbon::now()->gt($customer->otp_expires_at)) {
                return response()->json(['success' => false, 'message' => 'Invalid or expired code.'], 400);
            }

            // 2. Clear OTP and mark as verified
            $customer->update([
                'otp' => null, 
                'otp_verified' => true,
                'last_login_at' => now()
            ]);
            
            // 3. Generate a fresh API token for the Android App
            $token = $customer->createToken('sync-token')->plainTextToken;

            \Log::info('Account Sync Verified', ['customer_id' => $customer->id]);

            return response()->json([
                'success' => true,
                'message' => 'Profile synced successfully!',
                'data' => [
                    'access_token' => $token,
                    // Returning the FULL customer object so Android Room can save everything
                    'customer' => [
                        'id' => $customer->id,
                        'customer_number' => $customer->customer_number,
                        'name' => $customer->customer,
                        'email' => $customer->email,
                        'phone' => $customer->phone,
                        'title' => $customer->title,
                        'first_name' => $customer->first_name,
                        'surname' => $customer->surname,
                        'middle_name' => $customer->middle_name,
                        'dob' => $customer->dob ? $customer->dob->toDateString() : null,
                        'address' => $customer->address,
                        'town' => $customer->town,
                        'province' => $customer->province,
                        'idtype' => $customer->idtype,
                        'number' => $customer->number,
                        'income' => (double)$customer->income,
                        'verification_status' => $customer->verification_status,
                        'status' => $customer->status,
                        
                        // Next of Kin
                        'nok_first_name' => $customer->nok_first_name,
                        'nok_surname' => $customer->nok_surname,
                        'nok_mobile_phone' => $customer->nok_mobile_phone,
                        'nok_email' => $customer->nok_email,
                        'nok_town' => $customer->nok_town,
                        'nok_physical_address' => $customer->nok_physical_address,

                        // Employment (Version 18)
                        'job_title' => $customer->job_title,
                        'employer_name' => $customer->employer_name,
                        'employee_no' => $customer->employee_no,
                        'years_in_employment' => (int)$customer->years_in_employment,
                        'retirement_year' => (int)$customer->retirement_year,
                        'employer_town' => $customer->employer_town,
                        'employment_type' => $customer->employment_type,
                        
                        // Images
                        'profile_picture' => $customer->docpath ? asset('storage/' . $customer->docpath) : null,
                    ]
                ]
            ]);

        } catch (\Exception $e) {
            \Log::error('API Sync Verify Error: ' . $e->getMessage());
            return response()->json(['success' => false, 'message' => 'Sync failed.'], 500);
        }
    }
    
    /**
 * Update Profile Photo
 * POST /api/customer/profile/photo
 */
/**
 * Update Profile Photo
 * POST /api/customer/profile/photo
 */
/**
 * Update Profile Photo
 * POST /api/customer/profile/photo
 */
/**
 * Update Profile Photo
 * POST /api/customer/profile/photo
 */
public function uploadProfilePhoto(Request $request): JsonResponse
{
    try {
        $validator = Validator::make($request->all(), [
            'customer_id'     => 'required|exists:customer,id',
            'profile_picture' => 'required|image|mimes:jpeg,png,jpg|max:2048',
        ]);

        if ($validator->fails()) {
            return response()->json(['success' => false, 'errors' => $validator->errors()], 422);
        }

        $customer = Customer::findOrFail($request->customer_id);

        // Delete previous file if it exists
        if ($customer->docpath && Storage::disk('public')->exists($customer->docpath)) {
            Storage::disk('public')->delete($customer->docpath);
        }

        $file = $request->file('profile_picture');
        $filename = 'profile_' . $customer->id . '_' . time() . '.' . $file->getClientOriginalExtension();
        
        // Store in 'storage/app/public/profile-pictures'
        $path = $file->storeAs('profile-pictures', $filename, 'public');

        $customer->update(['docpath' => $path]);

        return response()->json([
            'success' => true,
            'message' => 'Photo updated',
            'data' => ['path' => $path] // Repository needs this to update Room
        ]);

    } catch (\Exception $e) {
        return response()->json(['success' => false, 'message' => 'Photo upload failed'], 500);
    }
}


/**
 * Update Profile Details (All Room Fields)
 * POST /api/customer/profile/update
 */
/**
 * Update Profile Details (All Room Fields)
 * POST /api/customer/profile/update
 */
public function updateProfile(Request $request): JsonResponse
{
    try {
        $validator = Validator::make($request->all(), [
            'customer_id'             => 'required|exists:customer,id',
            
            // Basic Info (Matching Android @SerializedName)
            'name'                    => 'nullable|string|max:255', 
            // CRITICAL FIX: Ignore the current customer_id for the unique email check
            'email'                   => 'nullable|email|unique:customer,email,' . $request->customer_id,
            'phone'                   => 'nullable|string|max:25',
            'office_phone'            => 'nullable|string|max:25',
            'dob'                     => 'nullable|date',
            'address'                 => 'nullable|string',
            'town'                    => 'nullable|string',
            'province'                => 'nullable|string',
            'income'                  => 'nullable|numeric',
            'number'                  => 'nullable|string', // NRC Number

            // Next of Kin Section
            'nok_first_name'          => 'nullable|string|max:255',
            'nok_surname'             => 'nullable|string|max:255',
            'nok_mobile_phone'        => 'nullable|string|max:25',
            'nok_email'               => 'nullable|email|max:255',
            'nok_town'                => 'nullable|string|max:255',
            'nok_physical_address'    => 'nullable|string|max:255',

            // Employment Section (Version 18)
            'job_title'               => 'nullable|string|max:255',
            'employer_name'           => 'nullable|string|max:255',
            'employee_no'             => 'nullable|string|max:50',
            'years_in_employment'     => 'nullable|integer',
            'retirement_year'         => 'nullable|integer',
            'employer_town'           => 'nullable|string|max:255',
            'employment_type'         => 'nullable|in:Permanent,Contract',
        ]);

        if ($validator->fails()) {
            return response()->json(['success' => false, 'errors' => $validator->errors()], 422);
        }

        $customer = Customer::findOrFail($request->customer_id);

        // Map Android Keys to Eloquent Model Keys
        $data = $request->all();

        if ($request->has('name')) {
            $data['customer'] = $request->name;
        }
        
        // Note: 'occupation' and 'company' are legacy fields in your model 
        // We sync them with job_title and employer_name for backwards compatibility
        if ($request->has('job_title')) $data['occupation'] = $request->job_title;
        if ($request->has('employer_name')) $data['company'] = $request->employer_name;

        $customer->update($data);

        return response()->json([
            'success' => true,
            'message' => 'Profile synchronized successfully',
            'data'    => $customer->refresh()
        ]);

    } catch (\Exception $e) {
        \Log::error('Profile Sync Error: ' . $e->getMessage());
        return response()->json(['success' => false, 'message' => 'Update failed'], 500);
    }
}
/**
 * Update Profile Details (All Sections)
 * POST /api/customer/profile/update
 */

    
    
}