<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\DisbursedLoan;
use App\Models\Repayment;
use App\Models\PaymentSchedule;
use App\Models\CashBankAccount;
use App\Models\BusinessProcessMapping;
use App\Models\JournalEntry;
use App\Models\JournalEntryItem;
use App\Models\ChartOfAccount;
use App\Models\FinancialYear;
use App\Models\Currency;
use App\Models\GeneralLedgerEntry;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Schema;

class LoanRepaymentController extends Controller
{
    /**
     * Display a listing of repayments for a specific loan
     */
    public function index($loanId)
    {
        try {
            $loan = DisbursedLoan::findOrFail($loanId);
            
            $repayments = Repayment::where('rloanid', $loan->loanid)
                ->with(['customer', 'loan', 'creator'])
                ->orderBy('rdate', 'desc')
                ->paginate(20);

            return response()->json([
                'success' => true,
                'data' => [
                    'loan' => [
                        'id' => $loan->loanid,
                        'number' => $loan->loannumber,
                        'customer' => $loan->customer,
                        'total_balance' => $loan->totalbalance,
                        'status' => $loan->status,
                    ],
                    'repayments' => $repayments,
                ]
            ], 200);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch repayments',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Store a new repayment (API endpoint)
     */
    public function store(Request $request, $loanId)
    {
        DB::beginTransaction();
        try {
            // Find the loan
            $loan = DisbursedLoan::with(['customer'])->findOrFail($loanId);

            // Get user ID from request or use authenticated user
            $userId = $this->resolveUserId($request);
            
            // Temporarily set the authenticated user if needed
            if ($userId && !Auth::check()) {
                // If your application supports it, you can temporarily authenticate
                // For now, we'll just use the user ID directly
            }

            // Validate the request
            $validator = Validator::make($request->all(), $this->getValidationRules($loan));

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }

            $validated = $validator->validated();

            // Additional validation for allocation limits
            $validationErrors = $this->validateAllocationLimits($loan, $validated);
            
            if (!empty($validationErrors)) {
                return response()->json([
                    'success' => false,
                    'message' => 'Allocation validation failed',
                    'errors' => $validationErrors
                ], 422);
            }

            // Prepare allocations
            $allocations = [
                'principal' => $validated['rprincipal'],
                'interest' => $validated['rinterest'],
                'processing' => $validated['processing_fees_amount_paid'],
                'penalty' => $validated['penalties'],
            ];

            Log::info('API Payment request received', [
                'loan_id' => $loan->loanid,
                'amount' => $validated['rtotal'],
                'user_id' => $userId,
                'source' => $validated['source'] ?? 'api'
            ]);

            // 1. Update cash/bank account
            $this->updateCashBankAccount($validated['paccount'], $validated['rtotal']);

            // 2. Create repayment record
            $repaymentData = [
                'customerid' => $loan->customerid,
                'rpid' => $loan->loantypeid,
                'rloanid' => $loan->loanid,
                'rtotal' => $validated['rtotal'],
                'rinterest' => $allocations['interest'],
                'processing_fees_amount_paid' => $allocations['processing'],
                'penalties' => $allocations['penalty'],
                'rprincipal' => $allocations['principal'],
                'rdate' => $validated['rdate'],
                'userid' => $userId,
                'branchid' => $loan->branchid,
                'companyid' => $loan->companyid,
                'pmethod' => $validated['pmethod'],
                'paccount' => $validated['paccount'],
                'transtype' => 'Repayment',
                'status' => 'Verified',
                'paymentreference' => $validated['paymentreference'] ?? null,
                'payment_notes' => $validated['notes'] ?? null,
                'allocation_mode' => $validated['allocation_mode'],
                'source' => $validated['source'] ?? 'api',
            ];

            // Add payment method specific fields
            if ($validated['pmethod'] === 'Bank') {
                $repaymentData['bank_name'] = $validated['bank_name'];
                $repaymentData['account_holder'] = $validated['account_holder'];
                $repaymentData['account_number'] = $validated['account_number'];
                $repaymentData['bank_branch'] = $validated['bank_branch'];
                $repaymentData['swift_code'] = $validated['swift_code'] ?? null;
            } elseif ($validated['pmethod'] === 'Mobile Money') {
                $repaymentData['mobile_channel'] = $validated['mobile_channel'];
                $repaymentData['mobile_number'] = $validated['mobile_number'];
            }

            $repayment = Repayment::create($repaymentData);

            // 3. Update loan balances
            $this->updateLoanBalances($loan, $allocations);

            // 4. Update payment schedules
            if ($allocations['principal'] > 0 || $allocations['interest'] > 0 || $allocations['processing'] > 0) {
                $this->updatePaymentSchedules($loan, $allocations, $validated['rdate']);
            }

            // 5. Create accounting entries (both journal and general ledger)
            $journalEntry = $this->createAccountingEntries($loan, $repayment, $allocations, $userId);

            // 6. Update loan status
            $loan->lastpaymentdate = $validated['rdate'];
            $loan->lastpaymentamount = $validated['rtotal'];
            
            if (bccomp($loan->totalbalance, '0.01', 2) <= 0) {
                $loan->status = 'closed';
                $loan->closeddate = now();
                $loan->paymentSchedules()->update(['status' => 'paid']);
            }
            
            $loan->save();

            DB::commit();

            Log::info('API Payment processed successfully', [
                'loan_id' => $loan->loanid,
                'repayment_id' => $repayment->id,
                'amount' => $validated['rtotal'],
                'new_balance' => $loan->totalbalance,
                'user_id' => $userId,
            ]);

            return response()->json([
                'success' => true,
                'message' => $this->getPaymentSuccessMessage($loan, $validated['rtotal'], $validated['pmethod']),
                'data' => [
                    'repayment' => $repayment,
                    'loan' => [
                        'id' => $loan->loanid,
                        'number' => $loan->loannumber,
                        'new_balance' => $loan->totalbalance,
                        'status' => $loan->status,
                    ],
                    'journal_entry' => [
                        'number' => $journalEntry->entry_number,
                        'date' => $journalEntry->entry_date,
                    ]
                ]
            ], 201);

        } catch (\Exception $e) {
            DB::rollBack();
            
            Log::error('API Payment processing failed', [
                'loan_id' => $loanId,
                'user_id' => $request->input('user_id') ?? 'null',
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Payment processing failed',
                'error' => $e->getMessage()
            ], 500);
        }
    }
    
  public function storefromweborapi(Request $request, $loanId)
{
    Log::info('--- Phase 3: Finalizing Accounting ---', [
        'url_loan_id' => $loanId,
        'payload' => $request->all()
    ]);

    DB::beginTransaction();
    try {
        // 1. Find the loan
        $loan = DisbursedLoan::with(['customer'])->findOrFail($loanId);

        // 2. Map Android SimpleAccountingRequest to Internal System Requirements
        if ($request->has('transaction_reference')) {
            $amount = $request->input('amount');
            
            // Auto-calculate the split for principal, interest, etc.
            $allocations = $this->calculateAutoAllocation($loan, $amount);

            $request->merge([
                'rtotal' => $amount,
                'rprincipal' => $allocations['principal'],
                'rinterest' => $allocations['interest'],
                'processing_fees_amount_paid' => $allocations['processing'],
                'penalties' => $allocations['penalty'],
                'rdate' => now()->format('Y-m-d'),
                'pmethod' => 'Mobile Money', 
                'paccount' => 6, // Verify your Mobile Money GL account ID
                'allocation_mode' => 'auto',
                'source' => 'api',
                'paymentreference' => $request->input('transaction_reference'),
            ]);
        }

        // 3. Set System User (Admin) since no session exists via API
        $userId = 1; 

        // 4. Validate the mapped request
        $validator = Validator::make($request->all(), $this->getValidationRules($loan));

        if ($validator->fails()) {
            Log::error('Accounting Validation Failed', ['errors' => $validator->errors()->all()]);
            return response()->json(['success' => false, 'errors' => $validator->errors()], 422);
        }

        $validated = $validator->validated();

        // 5. Update Financial Records
        $this->updateCashBankAccount($validated['paccount'], $validated['rtotal']);

        $repaymentData = [
            'customerid' => $loan->customerid,
            'rpid' => $loan->loantypeid,
            'rloanid' => $loan->loanid,
            'rtotal' => $validated['rtotal'],
            'rinterest' => $validated['rinterest'],
            'processing_fees_amount_paid' => $validated['processing_fees_amount_paid'],
            'penalties' => $validated['penalties'],
            'rprincipal' => $validated['rprincipal'],
            'rdate' => $validated['rdate'],
            'userid' => $userId,
            'branchid' => $loan->branchid,
            'companyid' => $loan->companyid,
            'pmethod' => $validated['pmethod'],
            'paccount' => $validated['paccount'],
            'transtype' => 'Repayment',
            'status' => 'Verified',
            'paymentreference' => $validated['paymentreference'],
            'allocation_mode' => $validated['allocation_mode'],
            'source' => $validated['source'],
            'mobile_number' => $request->input('phone_number'),
            'mobile_channel' => $request->input('payment_method'),
        ];

        $repayment = Repayment::create($repaymentData);

        // 6. Update Ledger and Schedules
        $this->updateLoanBalances($loan, $allocations);
        $this->updatePaymentSchedules($loan, $allocations, $validated['rdate']);
        $this->createAccountingEntries($loan, $repayment, $allocations, $userId);

        // 7. Update Loan and Close if finished
        $loan->lastpaymentdate = $validated['rdate'];
        $loan->lastpaymentamount = $validated['rtotal'];
        
        if (bccomp($loan->totalbalance, '0.01', 2) <= 0) {
            $loan->status = 'closed';
            $loan->closeddate = now();
        }
        $loan->save();

        DB::commit();
        Log::info('Accounting Success for Loan: ' . $loan->loannumber);

        return response()->json([
            'success' => true, 
            'message' => 'Accounting success',
            'new_balance' => $loan->totalbalance
        ], 201);

    } catch (\Exception $e) {
        DB::rollBack();
        Log::error('Accounting Exception: ' . $e->getMessage());
        return response()->json(['success' => false, 'error' => $e->getMessage()], 500);
    }
}
    /**
     * Display a specific repayment
     */
    public function show($loanId, $repaymentId)
    {
        try {
            $loan = DisbursedLoan::findOrFail($loanId);
            $repayment = Repayment::where('id', $repaymentId)
                ->where('rloanid', $loan->loanid)
                ->with(['customer', 'loan', 'creator'])
                ->firstOrFail();

            // Get associated general ledger entries
            $generalLedgerEntries = GeneralLedgerEntry::where('loanid', $loan->loanid)
                ->where('reference_type', 'loan_repayment')
                ->where('reference_id', $repaymentId)
                ->with(['account'])
                ->get();

            // Get journal entries
            $journalEntries = JournalEntry::whereHas('items', function($query) use ($repaymentId) {
                $query->where('reference', 'like', '%REPAY-' . $repaymentId . '%');
            })->with(['items.account'])->get();

            return response()->json([
                'success' => true,
                'data' => [
                    'repayment' => $repayment,
                    'general_ledger_entries' => $generalLedgerEntries,
                    'journal_entries' => $journalEntries,
                ]
            ], 200);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch repayment details',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get auto-allocation calculation for a payment amount
     */
    public function calculateAllocation(Request $request, $loanId)
    {
        try {
            $loan = DisbursedLoan::findOrFail($loanId);
            
            $validator = Validator::make($request->all(), [
                'payment_amount' => 'required|numeric|min:0.01|max:' . ($loan->totalbalance + 1000),
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }

            $paymentAmount = $request->input('payment_amount');
            $allocations = $this->calculateAutoAllocation($loan, $paymentAmount);

            return response()->json([
                'success' => true,
                'data' => [
                    'payment_amount' => $paymentAmount,
                    'allocations' => $allocations,
                    'total_allocated' => array_sum($allocations),
                    'loan_balances' => [
                        'principal' => $loan->principalbalance,
                        'interest' => $loan->interestbalance,
                        'processing_fee' => $loan->processing_fee_balance,
                        'penalty' => $loan->penalty_balance,
                        'total' => $loan->totalbalance,
                    ]
                ]
            ], 200);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to calculate allocation',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Reverse/void a repayment
     */
    public function reverse($loanId, $repaymentId)
    {
        DB::beginTransaction();
        try {
            $loan = DisbursedLoan::findOrFail($loanId);
            $repayment = Repayment::where('id', $repaymentId)
                ->where('rloanid', $loan->loanid)
                ->firstOrFail();

            // Check if repayment can be reversed
            if ($repayment->status === 'reversed') {
                return response()->json([
                    'success' => false,
                    'message' => 'This repayment has already been reversed'
                ], 400);
            }

            // Get current user ID
            $userId = Auth::id() ?? 1;

            // Reverse the repayment
            $this->reverseRepayment($loan, $repayment, $userId);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Repayment reversed successfully',
                'data' => [
                    'repayment' => $repayment->fresh(),
                    'loan' => $loan->fresh(),
                ]
            ], 200);

        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Failed to reverse repayment',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Resolve user ID from request or authentication
     */
    private function resolveUserId(Request $request)
    {
        // First check if user_id is provided in the request
        if ($request->has('user_id') && $request->input('user_id')) {
            return $request->input('user_id');
        }
        
        // Check if user is authenticated
        if (Auth::check()) {
            return Auth::id();
        }
        
        // Default to system user (admin)
        return 1;
    }

    /**
     * Get validation rules for repayment
     */
  
    
    private function getValidationRules($loan)
   {
    return [
        'rtotal' => 'required|numeric',
        'pmethod' => 'required|string',
        'paccount' => 'required|integer',
        'rdate' => 'required|date',
        'rprincipal' => 'required|numeric',
        'rinterest' => 'required|numeric',
        'processing_fees_amount_paid' => 'required|numeric',
        'penalties' => 'required|numeric',
        'allocation_mode' => 'required|in:auto,manual',
        'paymentreference' => 'nullable|string',
        'source' => 'nullable|string',
        // Change user_id to nullable because we hardcode it to 1 if missing
        'user_id' => 'nullable|integer', 
    ];
   }

    /**
     * Validate allocation limits
     */
    private function validateAllocationLimits($loan, $validated)
    {
        $errors = [];
        
        if ($validated['rprincipal'] > $loan->principalbalance) {
            $errors['rprincipal'] = 'Principal allocation exceeds principal balance.';
        }
        
        if ($validated['rinterest'] > $loan->interestbalance) {
            $errors['rinterest'] = 'Interest allocation exceeds interest balance.';
        }
        
        if ($validated['processing_fees_amount_paid'] > $loan->processing_fee_balance) {
            $errors['processing_fees_amount_paid'] = 'Processing fee allocation exceeds processing fee balance.';
        }
        
        if ($validated['penalties'] > $loan->penalty_balance) {
            $errors['penalties'] = 'Penalty allocation exceeds penalty balance.';
        }
        
        // Verify allocations match payment amount
        $totalAllocated = bcadd(
            bcadd($validated['rprincipal'], $validated['rinterest'], 2),
            bcadd($validated['processing_fees_amount_paid'], $validated['penalties'], 2),
            2
        );
        
        if (abs($totalAllocated - $validated['rtotal']) > 0.01) {
            $errors['allocations'] = 'Total allocation does not match payment amount.';
        }
        
        return $errors;
    }

    /**
     * Calculate auto-allocation
     */
    private function calculateAutoAllocation($loan, $paymentAmount)
    {
        $allocations = [
            'principal' => 0,
            'interest' => 0,
            'processing' => 0,
            'penalty' => 0,
        ];
        
        $remaining = $paymentAmount;
        
        // Auto-allocate in priority order
        $priorityOrder = [
            'penalty' => (float) $loan->penalty_balance,
            'processing' => (float) $loan->processing_fee_balance,
            'interest' => (float) $loan->interestbalance,
            'principal' => (float) $loan->principalbalance,
        ];
        
        foreach ($priorityOrder as $type => $balance) {
            if ($remaining <= 0) break;
            $allocations[$type] = min($balance, $remaining);
            $remaining -= $allocations[$type];
        }
        
        return $allocations;
    }

    /**
     * Update loan balances after payment
     */
    private function updateLoanBalances($loan, $allocations)
    {
        // Update principal balance
        $loan->principalbalance = max(0, bcsub($loan->principalbalance, $allocations['principal'], 2));
        
        // Update interest balance
        $loan->interestbalance = max(0, bcsub($loan->interestbalance, $allocations['interest'], 2));
        
        // Update processing fee balance
        if ($allocations['processing'] > 0) {
            $loan->processing_fee_balance = max(0, bcsub($loan->processing_fee_balance, $allocations['processing'], 2));
        }
        
        // Update penalty balance
        if ($allocations['penalty'] > 0) {
            $loan->penalty_balance = max(0, bcsub($loan->penalty_balance, $allocations['penalty'], 2));
        }
        
        // Update totals
        $loan->totalprincipalpaid = bcadd($loan->totalprincipalpaid, $allocations['principal'], 2);
        $loan->totalinterestpaid = bcadd($loan->totalinterestpaid, $allocations['interest'], 2);
        $loan->totalpenaltypaid = bcadd($loan->totalpenaltypaid, $allocations['penalty'], 2);
        $loan->totalpaid = bcadd($loan->totalpaid, array_sum($allocations), 2);
        
        // Recalculate total balance
        $loan->totalbalance = bcadd(
            bcadd($loan->principalbalance, $loan->interestbalance, 2),
            bcadd($loan->processing_fee_balance, $loan->penalty_balance, 2),
            2
        );
        
        $loan->save();
    }

    /**
     * Update payment schedules
     */
    private function updatePaymentSchedules($loan, $allocations, $paymentDate)
    {
        $remainingPrincipal = $allocations['principal'];
        $remainingInterest = $allocations['interest'];
        $remainingProcessing = $allocations['processing'];
        
        if ($remainingPrincipal <= 0 && $remainingInterest <= 0 && $remainingProcessing <= 0) {
            return;
        }

        // Get schedules in the correct order for allocation
        $schedules = $this->getSchedulesInAllocationOrder($loan);

        foreach ($schedules as $schedule) {
            if ($remainingProcessing <= 0 && $remainingInterest <= 0 && $remainingPrincipal <= 0) {
                break;
            }

            // 1. FIRST: Allocate to PROCESSING FEES (highest priority)
            if ($remainingProcessing > 0) {
                $processingDue = bcsub($schedule->processingfeesamount, $schedule->paid_processingefees_amount ?? 0, 2);
                $processingPaid = min($remainingProcessing, $processingDue);
                
                if ($processingPaid > 0) {
                    $schedule->paid_processingefees_amount = bcadd($schedule->paid_processingefees_amount ?? 0, $processingPaid, 2);
                    $remainingProcessing -= $processingPaid;
                    
                    // Update totals immediately
                    $schedule->total_paid = bcadd($schedule->total_paid ?? 0, $processingPaid, 2);
                }
            }

            // 2. SECOND: Allocate to INTEREST (medium priority)
            if ($remainingInterest > 0) {
                $interestDue = bcsub($schedule->interestamount, $schedule->paid_interest ?? 0, 2);
                $interestPaid = min($remainingInterest, $interestDue);
                
                if ($interestPaid > 0) {
                    $schedule->paid_interest = bcadd($schedule->paid_interest ?? 0, $interestPaid, 2);
                    $remainingInterest -= $interestPaid;
                    
                    // Update totals
                    $schedule->total_paid = bcadd($schedule->total_paid ?? 0, $interestPaid, 2);
                }
            }

            // 3. THIRD: Allocate to PRINCIPAL (lowest priority)
            if ($remainingPrincipal > 0) {
                $principalDue = bcsub($schedule->principalamount, $schedule->paid_principal ?? 0, 2);
                $principalPaid = min($remainingPrincipal, $principalDue);
                
                if ($principalPaid > 0) {
                    $schedule->paid_principal = bcadd($schedule->paid_principal ?? 0, $principalPaid, 2);
                    $remainingPrincipal -= $principalPaid;
                    
                    // Update totals
                    $schedule->total_paid = bcadd($schedule->total_paid ?? 0, $principalPaid, 2);
                }
            }

            // Update schedule status and save if any payment was made
            if ($schedule->isDirty()) {
                $this->updateScheduleStatus($schedule, $paymentDate);
                $schedule->save();
            }
        }

        // If there are remaining amounts after allocating to all schedules,
        // they should be applied as advance payments to the next schedule
        if ($remainingProcessing > 0 || $remainingInterest > 0 || $remainingPrincipal > 0) {
            $this->handleRemainingAllocations($loan, $remainingProcessing, $remainingInterest, $remainingPrincipal, $paymentDate);
        }
    }

    /**
     * Get schedules in the correct order for allocation
     * Priority: Overdue → Partial → Scheduled (all sorted by payment date asc)
     */
    private function getSchedulesInAllocationOrder($loan)
    {
        // Get all schedules that are not fully paid
        return $loan->paymentSchedules()
            ->where(function($query) {
                // Check if the schedule is not fully paid
                $query->where('status', '!=', 'paid')
                      ->orWhere(function($q) {
                          // Include schedules marked as 'paid' but might have remaining balances due to rounding
                          // Check if total_paid is less than totalamount
                          $q->where('status', 'paid')
                            ->whereColumn('total_paid', '<', 'totalamount');
                      });
            })
            ->orderBy('paymentdate', 'asc') // Oldest first
            ->orderByRaw("FIELD(status, 'overdue', 'partial', 'scheduled')") // Status priority
            ->get();
    }

    /**
     * Update a schedule's status based on current payment state
     */
    private function updateScheduleStatus($schedule, $paymentDate)
    {
        // Calculate if schedule is fully paid based on total_paid and totalamount
        $isFullyPaid = bccomp($schedule->total_paid ?? 0, $schedule->totalamount, 2) >= 0;
        
        if ($isFullyPaid) {
            $schedule->status = 'paid';
            $schedule->paid_date = $paymentDate;
            return;
        }

        // Check if schedule has any payments
        $hasPayments = bccomp($schedule->total_paid ?? 0, 0, 2) > 0;
        
        if ($hasPayments) {
            // Check if overdue
            $isOverdue = $schedule->paymentdate < now();
            
            if ($isOverdue) {
                $schedule->status = 'overdue';
            } else {
                $schedule->status = 'partial';
            }
        } else {
            // No payments yet
            $isOverdue = $schedule->paymentdate < now();
            
            if ($isOverdue) {
                $schedule->status = 'overdue';
            } else {
                $schedule->status = 'scheduled';
            }
        }
    }

    /**
     * Handle any remaining allocations after processing all schedules
     * This applies advance payments to the next upcoming schedule
     */
    private function handleRemainingAllocations($loan, $remainingProcessing, $remainingInterest, $remainingPrincipal, $paymentDate)
    {
        // Find the next scheduled installment that's not fully paid
        $nextSchedule = $loan->paymentSchedules()
            ->where('status', '!=', 'paid')
            ->where('paymentdate', '>=', now())
            ->orderBy('paymentdate', 'asc')
            ->first();

        if (!$nextSchedule) {
            // If no future schedules, apply to the oldest overdue/partial
            $nextSchedule = $loan->paymentSchedules()
                ->where('status', '!=', 'paid')
                ->orderBy('paymentdate', 'asc')
                ->first();
        }

        if ($nextSchedule) {
            // Apply remaining allocations as advance payments
            if ($remainingProcessing > 0) {
                $nextSchedule->paid_processingefees_amount = bcadd($nextSchedule->paid_processingefees_amount ?? 0, $remainingProcessing, 2);
                $nextSchedule->total_paid = bcadd($nextSchedule->total_paid ?? 0, $remainingProcessing, 2);
            }
            
            if ($remainingInterest > 0) {
                $nextSchedule->paid_interest = bcadd($nextSchedule->paid_interest ?? 0, $remainingInterest, 2);
                $nextSchedule->total_paid = bcadd($nextSchedule->total_paid ?? 0, $remainingInterest, 2);
            }
            
            if ($remainingPrincipal > 0) {
                $nextSchedule->paid_principal = bcadd($nextSchedule->paid_principal ?? 0, $remainingPrincipal, 2);
                $nextSchedule->total_paid = bcadd($nextSchedule->total_paid ?? 0, $remainingPrincipal, 2);
            }
            
            // Update status
            $this->updateScheduleStatus($nextSchedule, $paymentDate);
            $nextSchedule->save();
        }
    }

    /**
     * Update cash/bank account balance
     */
    private function updateCashBankAccount($accountId, $amount)
    {
        $account = CashBankAccount::find($accountId);
        if (!$account) {
            throw new \Exception("Cash/Bank account with ID {$accountId} not found");
        }

        $account->current_balance = bcadd($account->current_balance, $amount, 2);
        $account->save();
    }

    /**
     * Create accounting entries
     */
    private function createAccountingEntries($loan, $repayment, $allocations, $userId)
    {
        $accounts = $this->getAllAccountMappings($loan, $repayment);
        $totalAmount = array_sum($allocations);
        $entryNumber = $this->generateJournalEntryNumber($loan->companyid);
        
        // Ensure we have a valid user ID
        if (!$userId) {
            $userId = 1; // Default system user
        }
        
        $journalEntry = JournalEntry::create([
            'company_id' => $loan->companyid,
            'branch_id' => $loan->branchid,
            'entry_number' => $entryNumber,
            'entry_date' => $repayment->rdate,
            'financial_year_id' => $this->getCurrentFinancialYearId($repayment->rdate),
            'reference' => $repayment->paymentreference ?? 'REPAY-' . $loan->loannumber . '-' . time(),
            'description' => 'Loan Repayment - ' . $loan->loannumber . ' - ' . ($loan->customer->first_name ?? 'Customer'),
            'currency_id' => $this->getDefaultCurrencyId(),
            'exchange_rate' => 1.0,
            'total_debit' => $totalAmount,
            'total_credit' => $totalAmount,
            'status' => 'posted',
            'posted_by' => $userId,
            'posted_at' => now(),
            'created_by' => $userId,
        ]);

        $items = $this->createJournalItems($journalEntry, $loan, $repayment, $totalAmount, $accounts, $userId);
        
        foreach ($items as $item) {
            JournalEntryItem::create($item);
        }

        $this->createGeneralLedgerEntries($loan, $repayment, $totalAmount, $journalEntry->entry_number, $accounts['loan_receivable_account']['id'], $userId);

        return $journalEntry;
    }

    /**
     * Create journal items
     */
    private function createJournalItems($journalEntry, $loan, $repayment, $totalAmount, $accounts, $userId)
    {
        $customerName = $loan->customer->first_name . ' ' . $loan->customer->surname;
        $loanNumber = $loan->loannumber;
        $paymentMethod = $repayment->pmethod;
        
        $cashBankAccount = $accounts['cash_account']['cash_bank_account'] ?? null;
        $accountName = $cashBankAccount ? $cashBankAccount->accountName : $paymentMethod;
        
        // Ensure user ID is valid
        if (!$userId) {
            $userId = 1; // Default system user
        }
        
        return [
            [
                'company_id' => $loan->companyid,
                'branch_id' => $loan->branchid,
                'journal_entry_id' => $journalEntry->id,
                'account_id' => $accounts['cash_account']['id'],
                'description' => "Receipt from {$customerName} - Loan {$loanNumber} via {$accountName}",
                'debit' => $totalAmount,
                'credit' => 0,
                'created_by' => $userId,
            ],
            [
                'company_id' => $loan->companyid,
                'branch_id' => $loan->branchid,
                'journal_entry_id' => $journalEntry->id,
                'account_id' => $accounts['loan_receivable_account']['id'],
                'description' => "Loan repayment - Loan {$loanNumber} - {$customerName}",
                'debit' => 0,
                'credit' => $totalAmount,
                'created_by' => $userId,
            ]
        ];
    }

    /**
     * Create general ledger entries
     */
    private function createGeneralLedgerEntries($loan, $repayment, $totalAmount, $journalEntryNumber, $loanReceivableAccountId, $userId)
    {
        $companyId = $loan->companyid;
        $branchId = $loan->branchid;
        $customerName = $loan->customer->first_name . ' ' . $loan->customer->surname;
        $loanNumber = $loan->loannumber;
        $cashBankAccountId = $this->getCashBankGLAccountId($repayment->paccount);
        
        // Ensure user ID is valid
        if (!$userId) {
            $userId = 1; // Default system user
        }
        
        $generalLedgerEntries = [
            [
                'entrydate' => $repayment->rdate,
                'description' => "Receipt from {$customerName} - Loan Repayment {$loanNumber}",
                'accountid' => $cashBankAccountId,
                'referencedocument' => $journalEntryNumber,
                'reference_type' => 'loan_repayment',
                'reference_id' => $repayment->id,
                'documentno' => $repayment->paymentreference ?? 'REPAY-' . $loanNumber,
                'entrytype' => 'debit',
                'amount' => $totalAmount,
                'createdby' => $userId,
                'companyid' => $companyId,
                'branchid' => $branchId,
                'transtype' => 'loan_repayment',
                'original_currency' => 'ZMW',
                'loanid' => $loan->loanid,
                'created_at' => now(),
                'updated_at' => now()
            ],
            [
                'entrydate' => $repayment->rdate,
                'description' => "Loan Repayment Received - {$customerName} - Loan {$loanNumber}",
                'accountid' => $loanReceivableAccountId,
                'referencedocument' => $journalEntryNumber,
                'reference_type' => 'loan_repayment',
                'reference_id' => $repayment->id,
                'documentno' => $repayment->paymentreference ?? 'REPAY-' . $loanNumber,
                'entrytype' => 'credit',
                'amount' => $totalAmount,
                'createdby' => $userId,
                'companyid' => $companyId,
                'branchid' => $branchId,
                'transtype' => 'loan_repayment',
                'original_currency' => 'ZMW',
                'loanid' => $loan->loanid,
                'created_at' => now(),
                'updated_at' => now()
            ]
        ];
        
        GeneralLedgerEntry::insert($generalLedgerEntries);
    }

    /**
     * Reverse a repayment
     */
    private function reverseRepayment($loan, $repayment, $userId)
    {
        // Mark repayment as reversed
        $repayment->status = 'reversed';
        $repayment->reversed_by = $userId;
        $repayment->reversed_at = now();
        $repayment->save();

        // Reverse allocations
        $allocations = [
            'principal' => $repayment->rprincipal,
            'interest' => $repayment->rinterest,
            'processing' => $repayment->processing_fees_amount_paid,
            'penalty' => $repayment->penalties,
        ];

        // Restore loan balances
        $loan->principalbalance = bcadd($loan->principalbalance, $allocations['principal'], 2);
        $loan->interestbalance = bcadd($loan->interestbalance, $allocations['interest'], 2);
        $loan->processing_fee_balance = bcadd($loan->processing_fee_balance, $allocations['processing'], 2);
        $loan->penalty_balance = bcadd($loan->penalty_balance, $allocations['penalty'], 2);
        $loan->totalbalance = bcadd(
            bcadd($loan->principalbalance, $loan->interestbalance, 2),
            bcadd($loan->processing_fee_balance, $loan->penalty_balance, 2),
            2
        );
        
        // Update payment totals
        $loan->totalprincipalpaid = max(0, bcsub($loan->totalprincipalpaid, $allocations['principal'], 2));
        $loan->totalinterestpaid = max(0, bcsub($loan->totalinterestpaid, $allocations['interest'], 2));
        $loan->totalpenaltypaid = max(0, bcsub($loan->totalpenaltypaid, $allocations['penalty'], 2));
        $loan->totalpaid = max(0, bcsub($loan->totalpaid, array_sum($allocations), 2));
        
        // Reverse cash/bank account
        $this->updateCashBankAccount($repayment->paccount, -$repayment->rtotal);
        
        // Update payment schedules
        $this->reversePaymentSchedules($loan, $allocations);
        
        // Update loan status if needed
        if ($loan->status === 'closed' && $loan->totalbalance > 0) {
            $loan->status = 'active';
            $loan->closeddate = null;
        }
        
        $loan->save();

        // Create reversal journal entry
        $this->createReversalJournalEntry($loan, $repayment, $allocations, $userId);
    }

    /**
     * Reverse payment schedules
     */
    private function reversePaymentSchedules($loan, $allocations)
    {
        $remainingPrincipal = $allocations['principal'];
        $remainingInterest = $allocations['interest'];
        $remainingProcessing = $allocations['processing'];
        
        if ($remainingPrincipal <= 0 && $remainingInterest <= 0 && $remainingProcessing <= 0) {
            return;
        }

        // Reverse in reverse order (newest schedules first)
        $schedules = $loan->paymentSchedules()
            ->where('status', 'paid')
            ->orderBy('paymentdate', 'desc')
            ->get();

        foreach ($schedules as $schedule) {
            if ($remainingPrincipal <= 0 && $remainingInterest <= 0 && $remainingProcessing <= 0) {
                break;
            }

            // Reverse in reverse priority order: Principal → Interest → Processing
            // (Last in, first out - LIFO)
            
            // 1. Reverse Principal first
            if ($remainingPrincipal > 0) {
                $principalPaid = $schedule->paid_principal ?? 0;
                $principalReverse = min($remainingPrincipal, $principalPaid);
                
                if ($principalReverse > 0) {
                    $schedule->paid_principal = bcsub($schedule->paid_principal, $principalReverse, 2);
                    $remainingPrincipal -= $principalReverse;
                }
            }
            
            // 2. Reverse Interest
            if ($remainingInterest > 0) {
                $interestPaid = $schedule->paid_interest ?? 0;
                $interestReverse = min($remainingInterest, $interestPaid);
                
                if ($interestReverse > 0) {
                    $schedule->paid_interest = bcsub($schedule->paid_interest, $interestReverse, 2);
                    $remainingInterest -= $interestReverse;
                }
            }
            
            // 3. Reverse Processing Fees
            if ($remainingProcessing > 0) {
                $processingPaid = $schedule->paid_processingefees_amount ?? 0;
                $processingReverse = min($remainingProcessing, $processingPaid);
                
                if ($processingReverse > 0) {
                    $schedule->paid_processingefees_amount = bcsub($schedule->paid_processingefees_amount, $processingReverse, 2);
                    $remainingProcessing -= $processingReverse;
                }
            }

            // Update totals and status if any reversal occurred
            if ($schedule->isDirty()) {
                $totalReverse = bcadd(
                    bcadd($principalReverse ?? 0, $interestReverse ?? 0, 2),
                    $processingReverse ?? 0,
                    2
                );
                
                $schedule->total_paid = bcsub($schedule->total_paid, $totalReverse, 2);
                
                $this->updateScheduleStatus($schedule, null);
                $schedule->save();
            }
        }
    }

    /**
     * Create reversal journal entry
     */
    private function createReversalJournalEntry($loan, $repayment, $allocations, $userId)
    {
        $accounts = $this->getAllAccountMappings($loan, $repayment);
        $totalAmount = array_sum($allocations);
        $entryNumber = 'REV-' . $this->generateJournalEntryNumber($loan->companyid);
        
        // Ensure user ID is valid
        if (!$userId) {
            $userId = 1; // Default system user
        }
        
        $journalEntry = JournalEntry::create([
            'company_id' => $loan->companyid,
            'branch_id' => $loan->branchid,
            'entry_number' => $entryNumber,
            'entry_date' => now(),
            'financial_year_id' => $this->getCurrentFinancialYearId(),
            'reference' => 'REVERSAL-' . ($repayment->paymentreference ?? 'REPAY-' . $loan->loannumber),
            'description' => 'Reversal of Loan Repayment - ' . $loan->loannumber,
            'currency_id' => $this->getDefaultCurrencyId(),
            'exchange_rate' => 1.0,
            'total_debit' => $totalAmount,
            'total_credit' => $totalAmount,
            'status' => 'posted',
            'posted_by' => $userId,
            'posted_at' => now(),
            'created_by' => $userId,
        ]);

        // Reverse journal items
        $items = [
            [
                'company_id' => $loan->companyid,
                'branch_id' => $loan->branchid,
                'journal_entry_id' => $journalEntry->id,
                'account_id' => $accounts['loan_receivable_account']['id'],
                'description' => "Reversal of repayment - Loan {$loan->loannumber}",
                'debit' => $totalAmount,
                'credit' => 0,
                'created_by' => $userId,
            ],
            [
                'company_id' => $loan->companyid,
                'branch_id' => $loan->branchid,
                'journal_entry_id' => $journalEntry->id,
                'account_id' => $accounts['cash_account']['id'],
                'description' => "Reversal of receipt - Loan {$loan->loannumber}",
                'debit' => 0,
                'credit' => $totalAmount,
                'created_by' => $userId,
            ]
        ];
        
        foreach ($items as $item) {
            JournalEntryItem::create($item);
        }

        return $journalEntry;
    }

    /**
     * Helper methods (copied from LoanController)
     */
    private function getAllAccountMappings($loan, $repayment)
    {
        $accounts = [];
        
        try {
            $accounts['cash_account'] = $this->getCashAccountId($repayment->pmethod, $repayment->paccount);
        } catch (\Exception $e) {
            throw new \Exception('Cash account mapping error: ' . $e->getMessage());
        }
        
        try {
            $accounts['loan_receivable_account'] = $this->getLoanReceivableAccountId();
        } catch (\Exception $e) {
            throw new \Exception('Loan receivable account mapping error: ' . $e->getMessage());
        }
        
        return $accounts;
    }

    private function getCashAccountId($paymentMethod, $cashBankAccountId)
    {
        $cashBankAccount = CashBankAccount::with('glAccount')->find($cashBankAccountId);
        
        if (!$cashBankAccount) {
            throw new \Exception('Cash/Bank account not found');
        }
        
        if ($cashBankAccount->glAccountId && $cashBankAccount->glAccount) {
            return [
                'id' => $cashBankAccount->glAccountId,
                'cash_bank_account' => $cashBankAccount,
                'account' => $cashBankAccount->glAccount,
            ];
        }
        
        $defaultAccounts = [
            'Cash' => 7,
            'Bank' => 5,
            'Mobile Money' => 6,
            'Card' => 8,
            'Cheque' => 9,
            'Other' => 10,
        ];
        
        $accountId = $defaultAccounts[$paymentMethod] ?? 7;
        $account = ChartOfAccount::find($accountId);
        
        if (!$account) {
            throw new \Exception("No valid account found for payment method '{$paymentMethod}'.");
        }
        
        return [
            'id' => $accountId,
            'account' => $account,
            'cash_bank_account' => $cashBankAccount,
        ];
    }

    private function getLoanReceivableAccountId()
    {
        $companyId = auth()->user()->company_id ?? 1;
        $branchId = auth()->user()->branch_id ?? 1;
        
        $mapping = BusinessProcessMapping::where('nameofthebusinessprocess', 'Loan Disbursement')
            ->where('companyid', $companyId)
            ->where('branchid', $branchId)
            ->first();
        
        if (!$mapping) {
            $mapping = BusinessProcessMapping::where('nameofthebusinessprocess', 'Loan Disbursement')
                ->where('companyid', $companyId)
                ->first();
        }
        
        if (!$mapping || !$mapping->accountid) {
            $accountId = 4;
            $account = ChartOfAccount::find($accountId);
            
            if (!$account) {
                throw new \Exception("Loan Disbursement account mapping not found.");
            }
            
            return [
                'id' => $accountId,
                'account' => $account,
                'mapping' => null,
            ];
        }
        
        $account = ChartOfAccount::find($mapping->accountid);
        if (!$account) {
            throw new \Exception("Account ID {$mapping->accountid} for Loan Disbursement does not exist.");
        }
        
        return [
            'id' => $mapping->accountid,
            'mapping' => $mapping,
            'account' => $account,
        ];
    }

    private function getCashBankGLAccountId($cashBankAccountId)
    {
        $cashBankAccount = CashBankAccount::with('glAccount')->find($cashBankAccountId);
        
        if ($cashBankAccount && $cashBankAccount->glAccount) {
            return $cashBankAccount->glAccount->id;
        }
        
        return 101;
    }

    private function generateJournalEntryNumber($companyId)
    {
        $prefix = 'JRN';
        $year = date('Y');
        $month = date('m');
        
        $lastEntry = JournalEntry::where('company_id', $companyId)
            ->where('entry_number', 'like', $prefix . '-' . $year . $month . '%')
            ->orderBy('entry_number', 'desc')
            ->first();
        
        if ($lastEntry) {
            $lastNumber = intval(substr($lastEntry->entry_number, -4));
            $nextNumber = str_pad($lastNumber + 1, 4, '0', STR_PAD_LEFT);
        } else {
            $nextNumber = '0001';
        }
        
        return $prefix . '-' . $year . $month . '-' . $nextNumber;
    }

   private function getCurrentFinancialYearId($date = null)
{
    // 1. First, check the Company table for the explicitly set current year
    // We use company ID 1 based on your dump
    $company = DB::table('companies')->where('id', 1)->first();

    if ($company && $company->current_financial_year_id) {
        // Verify this ID actually exists in the financial_years table
        $exists = DB::table('financial_years')->where('id', $company->current_financial_year_id)->exists();
        if ($exists) {
            return $company->current_financial_year_id;
        }
    }

    // 2. Fallback: Try to find a year based on the provided date
    $date = $date ?: now();
    $financialYear = FinancialYear::where('start_date', '<=', $date)
        ->where('end_date', '>=', $date)
        ->where('company_id', 1)
        ->first();

    if ($financialYear) {
        return $financialYear->id;
    }

    // 3. Last Resort: Create the year (fixing the missing 'year' field error)
    $yearValue = date('Y', strtotime($date));
    
    $newYear = FinancialYear::create([
        'company_id' => 1,
        'year'       => $yearValue, // Fixed the 1364 General error
        'name'       => 'FY' . $yearValue,
        'start_date' => $yearValue . '-01-01 00:00:00',
        'end_date'   => $yearValue . '-12-31 23:59:59',
        'is_active'  => true,
        'created_by' => 1,
    ]);

    // Optional: Update company table so future requests find it immediately
    DB::table('companies')->where('id', 1)->update([
        'current_financial_year_id' => $newYear->id
    ]);

    return $newYear->id;
}

    private function getDefaultCurrencyId()
    {
        $currency = Currency::where('code', 'ZMW')->first();
        
        if ($currency) {
            return $currency->id;
        }
        
        // Create default currency if not exists
        return Currency::create([
            'code' => 'ZMW',
            'name' => 'Zambian Kwacha',
            'symbol' => 'K',
            'exchange_rate' => 1.0,
            'is_default' => true,
            'created_by' => 1, // System user
        ])->id;
    }

    private function getPaymentSuccessMessage($loan, $paymentAmount, $paymentMethod)
    {
        $methodText = match($paymentMethod) {
            'Cash' => 'cash payment',
            'Bank' => 'bank transfer',
            'Mobile Money' => 'mobile money payment',
            'Card' => 'card payment',
            'Cheque' => 'cheque payment',
            default => 'payment'
        };

        if ($loan->status === 'closed') {
            return "{$methodText} of " . number_format($paymentAmount, 2) . " recorded successfully. Loan fully settled.";
        }
        
        return "{$methodText} of " . number_format($paymentAmount, 2) . " recorded successfully. Remaining balance: " . number_format($loan->totalbalance, 2);
    }
}