<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Customer;
use App\Models\DisbursedLoan;
use App\Models\Repayment;
use App\Models\InterestIncome;
use App\Models\ProcessingFee;
use App\Models\PaymentSchedule;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;
use Illuminate\Support\Facades\Validator;

class StatementController extends Controller
{
    /**
     * Get loan statement for a specific loan
     * Route: GET /api/loans/{loan}/statement
     * 
     * @param Request $request
     * @param int $loan
     * @return \Illuminate\Http\JsonResponse
     */
    public function getLoanStatement(Request $request, $loan)
    {
        $validator = Validator::make(array_merge($request->all(), ['loan_id' => $loan]), [
            'loan_id' => 'required|integer|exists:disbursedloans,loanid',
            'customer_id' => 'nullable|integer|exists:customer,id',
            'start_date' => 'nullable|date|before_or_equal:end_date',
            'end_date' => 'nullable|date|after_or_equal:start_date',
            'format' => 'nullable|in:json,pdf,csv',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors(),
            ], 422);
        }

        try {
            $customerId = $request->input('customer_id');
            $startDate = $request->input('start_date', Carbon::now()->subMonths(3)->format('Y-m-d'));
            $endDate = $request->input('end_date', Carbon::now()->format('Y-m-d'));
            $format = $request->input('format', 'json');

            // Build query with customer validation if provided
            $query = DisbursedLoan::where('loanid', $loan);
            
            if ($customerId) {
                $query->where('customerid', $customerId);
            }

            $loan = $query->with([
                'customer',
                'loanType',
                'paymentSchedules' => function($query) {
                    $query->orderBy('installmentnumber', 'asc');
                },
                'repayments' => function($query) use ($startDate, $endDate) {
                    $query->whereBetween('transactiondate', [$startDate, $endDate])
                          ->orderBy('transactiondate', 'desc');
                }
            ])->firstOrFail();

            $customer = $loan->customer;

            // Get interest charges for this loan within period
            $interestCharges = InterestIncome::where('iloanid', $loan->loanid)
                ->whereBetween('chargeddate', [$startDate, $endDate])
                ->orderBy('chargeddate', 'desc')
                ->get();

            // Get processing fees for this loan within period
            $processingFees = ProcessingFee::where('iloanid', $loan->loanid)
                ->whereBetween('chargeddate', [$startDate, $endDate])
                ->orderBy('chargeddate', 'desc')
                ->get();

            // Calculate loan summary
            $summary = [
                'loan_details' => [
                    'loan_number' => $loan->loannumber,
                    'loan_type' => $loan->loanType->product ?? 'N/A',
                    'disbursement_date' => $loan->disburseddate,
                    'maturity_date' => $loan->maturitydate,
                    'original_amount' => (float) $loan->amount,
                    'interest_rate' => (float) $loan->interestrate,
                    'loan_term' => $loan->loanterm,
                    'payment_frequency' => $loan->paymentfrequency,
                    'installment_amount' => (float) $loan->installmentamount,
                    'status' => $loan->status,
                ],
                'current_balance' => [
                    'principal_balance' => (float) $loan->principalbalance,
                    'interest_balance' => (float) $loan->interestbalance,
                    'processing_fee_balance' => (float) ($loan->processing_fee_balance ?? 0),
                    'penalty_balance' => (float) ($loan->penalty_balance ?? 0),
                    'total_balance' => (float) $loan->totalbalance,
                ],
                'payment_summary' => [
                    'total_principal_paid' => (float) $loan->totalprincipalpaid,
                    'total_interest_paid' => (float) $loan->totalinterestpaid,
                    'total_penalty_paid' => (float) $loan->totalpenaltypaid,
                    'total_fees_paid' => (float) $loan->repayments()->sum('processing_fees_amount_paid'),
                    'total_paid' => (float) $loan->totalpaid,
                    'last_payment_date' => $loan->lastpaymentdate,
                    'last_payment_amount' => (float) $loan->lastpaymentamount,
                ],
                'period_summary' => [
                    'period_start' => $startDate,
                    'period_end' => $endDate,
                    'period_days' => Carbon::parse($startDate)->diffInDays(Carbon::parse($endDate)),
                    'repayments_in_period' => $loan->repayments->sum('amount'),
                    'interest_in_period' => $interestCharges->sum('charge'),
                    'fees_in_period' => $processingFees->sum('charge'),
                    'net_flow' => $loan->repayments->sum('amount') - 
                                  ($interestCharges->sum('charge') + $processingFees->sum('charge')),
                ],
            ];

            // Combine all transactions
            $transactions = collect();
            
            // Add repayments
            foreach ($loan->repayments as $repayment) {
                $transactions->push([
                    'transaction_id' => $repayment->repaymentid,
                    'transaction_type' => 'repayment',
                    'transaction_date' => $repayment->transactiondate,
                    'amount' => (float) $repayment->amount,
                    'principal_paid' => (float) $repayment->principalamount,
                    'interest_paid' => (float) $repayment->interestamount,
                    'penalty_paid' => (float) $repayment->penalties,
                    'fees_paid' => (float) $repayment->processing_fees_amount_paid,
                    'description' => 'Loan Repayment',
                    'payment_method' => $repayment->paymentmethod,
                    'reference_number' => $repayment->referencenumber,
                    'created_at' => $repayment->created_at,
                ]);
            }

            // Add interest charges
            foreach ($interestCharges as $interest) {
                $transactions->push([
                    'transaction_id' => $interest->id,
                    'transaction_type' => 'interest',
                    'transaction_date' => $interest->chargeddate,
                    'amount' => (float) $interest->charge,
                    'principal_paid' => 0,
                    'interest_paid' => 0,
                    'penalty_paid' => 0,
                    'fees_paid' => 0,
                    'description' => 'Interest Charge',
                    'payment_method' => null,
                    'reference_number' => 'INT-' . $interest->id,
                    'created_at' => $interest->created_at,
                ]);
            }

            // Add processing fees
            foreach ($processingFees as $fee) {
                $transactions->push([
                    'transaction_id' => $fee->id,
                    'transaction_type' => 'fee',
                    'transaction_date' => $fee->chargeddate,
                    'amount' => (float) $fee->charge,
                    'principal_paid' => 0,
                    'interest_paid' => 0,
                    'penalty_paid' => 0,
                    'fees_paid' => 0,
                    'description' => 'Processing Fee',
                    'payment_method' => null,
                    'reference_number' => 'FEE-' . $fee->id,
                    'created_at' => $fee->created_at,
                ]);
            }

            // Sort transactions by date
            $transactions = $transactions->sortByDesc('transaction_date')->values();

            // Format response based on requested format
            if ($format === 'pdf') {
                return $this->generateLoanStatementPdf($customer, $loan, $summary, $transactions);
            } elseif ($format === 'csv') {
                return $this->generateLoanStatementCsv($customer, $loan, $summary, $transactions);
            }

            return response()->json([
                'success' => true,
                'data' => [
                    'customer' => [
                        'id' => $customer->id,
                        'name' => $customer->customer,
                        'customer_number' => $customer->customer_number,
                        'email' => $customer->email,
                        'phone' => $customer->phone,
                    ],
                    'loan_summary' => $summary,
                    'transactions' => $transactions,
                    'payment_schedule' => $loan->paymentSchedules->map(function ($schedule) {
                        return [
                            'installment_number' => $schedule->installmentnumber,
                            'due_date' => $schedule->paymentdate,
                            'principal_due' => (float) $schedule->principaldue,
                            'interest_due' => (float) $schedule->interestdue,
                            'total_due' => (float) $schedule->totaldue,
                            'status' => $schedule->status,
                            'amount_paid' => (float) $schedule->amountpaid,
                            'balance' => (float) $schedule->balance,
                            'is_overdue' => $schedule->status === 'overdue',
                        ];
                    }),
                ],
                'message' => 'Loan statement retrieved successfully',
            ]);

        } catch (\Exception $e) {
            \Log::error('Error fetching loan statement: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve loan statement',
                'error' => config('app.debug') ? $e->getMessage() : 'Internal server error',
            ], 500);
        }
    }

    /**
     * Get payment schedule for a loan
     * Route: GET /api/loans/{loan}/payment-schedule
     * 
     * @param Request $request
     * @param int $loan
     * @return \Illuminate\Http\JsonResponse
     */
    public function getPaymentSchedule(Request $request, $loan)
    {
        $validator = Validator::make(array_merge($request->all(), ['loan_id' => $loan]), [
            'loan_id' => 'required|integer|exists:disbursedloans,loanid',
            'customer_id' => 'nullable|integer|exists:customer,id',
            'include_past' => 'nullable|boolean',
            'include_future' => 'nullable|boolean',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors(),
            ], 422);
        }

        try {
            $customerId = $request->input('customer_id');
            $includePast = $request->input('include_past', true);
            $includeFuture = $request->input('include_future', true);

            // Build query
            $query = DisbursedLoan::where('loanid', $loan);
            
            if ($customerId) {
                $query->where('customerid', $customerId);
            }

            $loan = $query->with(['paymentSchedules' => function($query) use ($includePast, $includeFuture) {
                $query->orderBy('installmentnumber', 'asc');
                
                // Apply filters based on include_past and include_future
                $now = Carbon::now();
                if (!$includePast && !$includeFuture) {
                    // Show only current month
                    $query->whereMonth('paymentdate', $now->month)
                          ->whereYear('paymentdate', $now->year);
                } elseif (!$includePast) {
                    // Show only future payments
                    $query->where('paymentdate', '>=', $now);
                } elseif (!$includeFuture) {
                    // Show only past payments
                    $query->where('paymentdate', '<', $now);
                }
            }])->firstOrFail();

            $schedule = $loan->paymentSchedules;

            // Calculate schedule summary
            $now = Carbon::now();
            $summary = [
                'total_installments' => $schedule->count(),
                'total_amount_due' => $schedule->sum('totaldue'),
                'total_paid' => $schedule->sum('amountpaid'),
                'total_outstanding' => $schedule->sum('balance'),
                'next_payment' => $schedule->where('status', 'scheduled')
                    ->where('balance', '>', 0)
                    ->sortBy('paymentdate')
                    ->first(),
                'overdue_payments' => $schedule->where('status', 'overdue')->count(),
                'paid_installments' => $schedule->where('status', 'paid')->count(),
                'scheduled_payments' => $schedule->where('status', 'scheduled')->count(),
                'total_overdue_amount' => $schedule->where('status', 'overdue')->sum('balance'),
                'upcoming_payments_next_30_days' => $schedule->where('status', 'scheduled')
                    ->where('paymentdate', '<=', $now->copy()->addDays(30))
                    ->where('balance', '>', 0)
                    ->sum('balance'),
            ];

            return response()->json([
                'success' => true,
                'data' => [
                    'loan' => [
                        'loan_number' => $loan->loannumber,
                        'original_amount' => (float) $loan->amount,
                        'interest_rate' => (float) $loan->interestrate,
                        'disbursement_date' => $loan->disburseddate,
                        'maturity_date' => $loan->maturitydate,
                        'installment_amount' => (float) $loan->installmentamount,
                        'payment_frequency' => $loan->paymentfrequency,
                        'status' => $loan->status,
                    ],
                    'schedule_summary' => $summary,
                    'payment_schedule' => $schedule->map(function ($item) {
                        $paymentDate = Carbon::parse($item->paymentdate);
                        $now = Carbon::now();
                        
                        return [
                            'installment_number' => $item->installmentnumber,
                            'due_date' => $item->paymentdate,
                            'principal_due' => (float) $item->principaldue,
                            'interest_due' => (float) $item->interestdue,
                            'total_due' => (float) $item->totaldue,
                            'status' => $item->status,
                            'amount_paid' => (float) $item->amountpaid,
                            'balance' => (float) $item->balance,
                            'days_overdue' => $item->status === 'overdue' 
                                ? $paymentDate->diffInDays($now)
                                : null,
                            'is_past_due' => $paymentDate->lt($now) && $item->balance > 0,
                            'is_upcoming' => $paymentDate->gt($now) && $item->balance > 0,
                            'is_current' => $paymentDate->isSameMonth($now) && $item->balance > 0,
                        ];
                    }),
                ],
                'message' => 'Payment schedule retrieved successfully',
            ]);

        } catch (\Exception $e) {
            \Log::error('Error fetching payment schedule: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve payment schedule',
                'error' => config('app.debug') ? $e->getMessage() : 'Internal server error',
            ], 500);
        }
    }

    /**
     * Get loan transactions (simplified version for loan prefix)
     * Route: GET /api/loans/{loan}/transactions
     * 
     * @param Request $request
     * @param int $loan
     * @return \Illuminate\Http\JsonResponse
     */
    public function getLoanTransactions(Request $request, $loan)
    {
        $validator = Validator::make(array_merge($request->all(), ['loan_id' => $loan]), [
            'loan_id' => 'required|integer|exists:disbursedloans,loanid',
            'customer_id' => 'nullable|integer|exists:customer,id',
            'start_date' => 'nullable|date|before_or_equal:end_date',
            'end_date' => 'nullable|date|after_or_equal:start_date',
            'limit' => 'nullable|integer|min:1|max:200',
            'transaction_type' => 'nullable|in:all,repayment,interest,fee',
            'sort_by' => 'nullable|in:date_desc,date_asc,amount_desc,amount_asc',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors(),
            ], 422);
        }

        try {
            $customerId = $request->input('customer_id');
            $startDate = $request->input('start_date');
            $endDate = $request->input('end_date');
            $limit = $request->input('limit', 50);
            $transactionType = $request->input('transaction_type', 'all');
            $sortBy = $request->input('sort_by', 'date_desc');

            // Build query
            $query = DisbursedLoan::where('loanid', $loan);
            
            if ($customerId) {
                $query->where('customerid', $customerId);
            }

            $loan = $query->firstOrFail();

            // Get transactions based on type
            $transactions = collect();
            
            // Add repayments if needed
            if (in_array($transactionType, ['all', 'repayment'])) {
                $repaymentsQuery = Repayment::where('rloanid', $loan->loanid);
                
                if ($startDate && $endDate) {
                    $repaymentsQuery->whereBetween('transactiondate', [$startDate, $endDate]);
                }
                
                $repayments = $repaymentsQuery->orderBy('transactiondate', 'desc')
                    ->limit($limit)
                    ->get();
                
                foreach ($repayments as $repayment) {
                    $transactions->push([
                        'transaction_id' => $repayment->repaymentid,
                        'transaction_type' => 'repayment',
                        'transaction_date' => $repayment->transactiondate,
                        'amount' => (float) $repayment->amount,
                        'principal_amount' => (float) $repayment->principalamount,
                        'interest_amount' => (float) $repayment->interestamount,
                        'penalty_amount' => (float) $repayment->penalties,
                        'fee_amount' => (float) $repayment->processing_fees_amount_paid,
                        'description' => 'Loan Repayment',
                        'payment_method' => $repayment->paymentmethod,
                        'reference_number' => $repayment->referencenumber,
                        'created_at' => $repayment->created_at,
                    ]);
                }
            }

            // Add interest charges if needed
            if (in_array($transactionType, ['all', 'interest'])) {
                $interestQuery = InterestIncome::where('iloanid', $loan->loanid);
                
                if ($startDate && $endDate) {
                    $interestQuery->whereBetween('chargeddate', [$startDate, $endDate]);
                }
                
                $interest = $interestQuery->orderBy('chargeddate', 'desc')
                    ->limit($limit)
                    ->get();
                
                foreach ($interest as $item) {
                    $transactions->push([
                        'transaction_id' => $item->id,
                        'transaction_type' => 'interest',
                        'transaction_date' => $item->chargeddate,
                        'amount' => (float) $item->charge,
                        'principal_amount' => 0,
                        'interest_amount' => (float) $item->charge,
                        'penalty_amount' => 0,
                        'fee_amount' => 0,
                        'description' => 'Interest Charge',
                        'payment_method' => null,
                        'reference_number' => 'INT-' . $item->id,
                        'created_at' => $item->created_at,
                    ]);
                }
            }

            // Add processing fees if needed
            if (in_array($transactionType, ['all', 'fee'])) {
                $feesQuery = ProcessingFee::where('iloanid', $loan->loanid);
                
                if ($startDate && $endDate) {
                    $feesQuery->whereBetween('chargeddate', [$startDate, $endDate]);
                }
                
                $fees = $feesQuery->orderBy('chargeddate', 'desc')
                    ->limit($limit)
                    ->get();
                
                foreach ($fees as $fee) {
                    $transactions->push([
                        'transaction_id' => $fee->id,
                        'transaction_type' => 'fee',
                        'transaction_date' => $fee->chargeddate,
                        'amount' => (float) $fee->charge,
                        'principal_amount' => 0,
                        'interest_amount' => 0,
                        'penalty_amount' => 0,
                        'fee_amount' => (float) $fee->charge,
                        'description' => 'Processing Fee',
                        'payment_method' => null,
                        'reference_number' => 'FEE-' . $fee->id,
                        'created_at' => $fee->created_at,
                    ]);
                }
            }

            // Apply sorting
            switch ($sortBy) {
                case 'date_asc':
                    $transactions = $transactions->sortBy('transaction_date')->values();
                    break;
                case 'amount_desc':
                    $transactions = $transactions->sortByDesc('amount')->values();
                    break;
                case 'amount_asc':
                    $transactions = $transactions->sortBy('amount')->values();
                    break;
                default: // date_desc
                    $transactions = $transactions->sortByDesc('transaction_date')->values();
                    break;
            }

            // Limit results
            $transactions = $transactions->take($limit);

            // Calculate summary
            $summary = [
                'total_transactions' => $transactions->count(),
                'total_amount' => $transactions->sum('amount'),
                'repayment_total' => $transactions->where('transaction_type', 'repayment')->sum('amount'),
                'interest_total' => $transactions->where('transaction_type', 'interest')->sum('amount'),
                'fees_total' => $transactions->where('transaction_type', 'fee')->sum('amount'),
                'period' => $startDate && $endDate 
                    ? ['start_date' => $startDate, 'end_date' => $endDate]
                    : null,
            ];

            return response()->json([
                'success' => true,
                'data' => [
                    'loan' => [
                        'loan_id' => $loan->loanid,
                        'loan_number' => $loan->loannumber,
                        'customer_id' => $loan->customerid,
                    ],
                    'summary' => $summary,
                    'transactions' => $transactions->values(),
                ],
                'message' => 'Loan transactions retrieved successfully',
            ]);

        } catch (\Exception $e) {
            \Log::error('Error fetching loan transactions: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve loan transactions',
                'error' => config('app.debug') ? $e->getMessage() : 'Internal server error',
            ], 500);
        }
    }

    /**
     * Get loan balance summary
     * Route: GET /api/loans/{loan}/balance-summary
     * 
     * @param Request $request
     * @param int $loan
     * @return \Illuminate\Http\JsonResponse
     */
    public function getLoanBalanceSummary(Request $request, $loan)
    {
        $validator = Validator::make(array_merge($request->all(), ['loan_id' => $loan]), [
            'loan_id' => 'required|integer|exists:disbursedloans,loanid',
            'customer_id' => 'nullable|integer|exists:customer,id',
            'as_of_date' => 'nullable|date',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors(),
            ], 422);
        }

        try {
            $customerId = $request->input('customer_id');
            $asOfDate = $request->input('as_of_date', Carbon::now()->format('Y-m-d'));

            // Build query
            $query = DisbursedLoan::where('loanid', $loan);
            
            if ($customerId) {
                $query->where('customerid', $customerId);
            }

            $loan = $query->firstOrFail();

            // Calculate balances as of specific date if provided
            if ($asOfDate && $asOfDate != Carbon::now()->format('Y-m-d')) {
                $balanceSummary = $this->calculateBalanceAsOfDate($loan, $asOfDate);
            } else {
                // Use current balances
                $balanceSummary = [
                    'principal_balance' => (float) $loan->principalbalance,
                    'interest_balance' => (float) $loan->interestbalance,
                    'processing_fee_balance' => (float) ($loan->processing_fee_balance ?? 0),
                    'penalty_balance' => (float) ($loan->penalty_balance ?? 0),
                    'total_balance' => (float) $loan->totalbalance,
                    'as_of_date' => $asOfDate,
                    'is_current' => true,
                ];
            }

            // Calculate percentages
            $originalAmount = (float) $loan->amount;
            $totalPaid = (float) $loan->totalpaid;
            
            $percentagePaid = $originalAmount > 0 ? ($totalPaid / $originalAmount) * 100 : 0;
            $percentageOutstanding = 100 - $percentagePaid;

            $response = [
                'success' => true,
                'data' => [
                    'loan' => [
                        'loan_id' => $loan->loanid,
                        'loan_number' => $loan->loannumber,
                        'customer_id' => $loan->customerid,
                        'original_amount' => $originalAmount,
                        'interest_rate' => (float) $loan->interestrate,
                        'disbursement_date' => $loan->disburseddate,
                        'maturity_date' => $loan->maturitydate,
                        'status' => $loan->status,
                    ],
                    'balance_summary' => $balanceSummary,
                    'payment_summary' => [
                        'total_principal_paid' => (float) $loan->totalprincipalpaid,
                        'total_interest_paid' => (float) $loan->totalinterestpaid,
                        'total_penalty_paid' => (float) $loan->totalpenaltypaid,
                        'total_paid' => $totalPaid,
                        'percentage_paid' => round($percentagePaid, 2),
                        'percentage_outstanding' => round($percentageOutstanding, 2),
                        'last_payment_date' => $loan->lastpaymentdate,
                        'last_payment_amount' => (float) $loan->lastpaymentamount,
                    ],
                    'delinquency_info' => [
                        'days_delinquent' => $loan->daysdelinquent,
                        'is_overdue' => $loan->isOverdue(),
                        'overdue_amount' => $this->calculateOverdueAmount($loan),
                    ],
                ],
                'message' => 'Loan balance summary retrieved successfully',
            ];

            return response()->json($response);

        } catch (\Exception $e) {
            \Log::error('Error fetching loan balance summary: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve loan balance summary',
                'error' => config('app.debug') ? $e->getMessage() : 'Internal server error',
            ], 500);
        }
    }

    /**
     * Get customer transaction history (all loans)
     * Route: GET /api/customer/transactions/history
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function getCustomerTransactionHistory(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'customer_id' => 'required|integer|exists:customer,id',
            'start_date' => 'nullable|date|before_or_equal:end_date',
            'end_date' => 'nullable|date|after_or_equal:start_date',
            'limit' => 'nullable|integer|min:1|max:100',
            'transaction_type' => 'nullable|in:all,repayment,interest,fee',
            'loan_id' => 'nullable|integer|exists:disbursedloans,loanid',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors(),
            ], 422);
        }

        try {
            $customerId = $request->input('customer_id');
            $startDate = $request->input('start_date', Carbon::now()->subMonths(3)->format('Y-m-d'));
            $endDate = $request->input('end_date', Carbon::now()->format('Y-m-d'));
            $limit = $request->input('limit', 50);
            $transactionType = $request->input('transaction_type', 'all');
            $loanId = $request->input('loan_id');

            $customer = Customer::findOrFail($customerId);

            // Get all customer loans
            $loansQuery = $customer->disbursedLoans();
            
            if ($loanId) {
                $loansQuery->where('loanid', $loanId);
            }
            
            $loans = $loansQuery->get();

            // Collect transactions from all loans
            $allTransactions = collect();
            
            foreach ($loans as $loan) {
                $transactions = $this->getLoanTransactionsForPeriod(
                    $loan->loanid, 
                    $startDate, 
                    $endDate, 
                    $transactionType
                );
                $allTransactions = $allTransactions->concat($transactions);
            }

            // Sort by date and limit
            $allTransactions = $allTransactions->sortByDesc('transaction_date')
                ->take($limit)
                ->values();

            // Calculate summary
            $summary = [
                'total_transactions' => $allTransactions->count(),
                'total_amount' => $allTransactions->sum('amount'),
                'repayment_total' => $allTransactions->where('transaction_type', 'repayment')->sum('amount'),
                'interest_total' => $allTransactions->where('transaction_type', 'interest')->sum('amount'),
                'fees_total' => $allTransactions->where('transaction_type', 'fee')->sum('amount'),
                'loans_count' => $loans->count(),
                'period' => [
                    'start_date' => $startDate,
                    'end_date' => $endDate,
                ],
            ];

            // Group by loan for breakdown
            $loanBreakdown = [];
            foreach ($loans as $loan) {
                $loanTransactions = $allTransactions->where('loan_id', $loan->loanid);
                if ($loanTransactions->count() > 0) {
                    $loanBreakdown[] = [
                        'loan_number' => $loan->loannumber,
                        'transaction_count' => $loanTransactions->count(),
                        'total_amount' => $loanTransactions->sum('amount'),
                    ];
                }
            }

            return response()->json([
                'success' => true,
                'data' => [
                    'customer' => [
                        'id' => $customer->id,
                        'name' => $customer->customer,
                        'customer_number' => $customer->customer_number,
                    ],
                    'summary' => $summary,
                    'loan_breakdown' => $loanBreakdown,
                    'transactions' => $allTransactions,
                ],
                'message' => 'Customer transaction history retrieved successfully',
            ]);

        } catch (\Exception $e) {
            \Log::error('Error fetching customer transaction history: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve transaction history',
                'error' => config('app.debug') ? $e->getMessage() : 'Internal server error',
            ], 500);
        }
    }

    /**
     * Get customer comprehensive statement
     * Route: GET /api/customer/statement
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function getCustomerStatement(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'customer_id' => 'required|integer|exists:customer,id',
            'start_date' => 'nullable|date|before_or_equal:end_date',
            'end_date' => 'nullable|date|after_or_equal:start_date',
            'format' => 'nullable|in:json,pdf,csv',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors(),
            ], 422);
        }

        try {
            $customerId = $request->input('customer_id');
            $startDate = $request->input('start_date', Carbon::now()->subMonths(6)->format('Y-m-d'));
            $endDate = $request->input('end_date', Carbon::now()->format('Y-m-d'));
            $format = $request->input('format', 'json');

            $customer = Customer::with([
                'disbursedLoans' => function($query) use ($startDate, $endDate) {
                    $query->where(function($q) use ($startDate, $endDate) {
                        $q->whereBetween('disburseddate', [$startDate, $endDate])
                          ->orWhereBetween('closeddate', [$startDate, $endDate])
                          ->orWhere(function($q2) use ($startDate, $endDate) {
                              $q2->where('disburseddate', '<=', $endDate)
                                 ->where(function($q3) use ($startDate) {
                                     $q3->whereNull('closeddate')
                                        ->orWhere('closeddate', '>=', $startDate);
                                 });
                          });
                    })
                    ->with(['loanType', 'paymentSchedules']);
                },
                'currentCreditScore'
            ])->findOrFail($customerId);

            // Get all transactions for the period
            $transactions = $this->getAllCustomerTransactions($customerId, $startDate, $endDate);

            // Calculate comprehensive summary
            $summary = $this->calculateCustomerFinancialSummary($customer, $startDate, $endDate);

            // Format response
            if ($format === 'pdf') {
                return $this->generateCustomerStatementPdf($customer, $summary, $transactions, $startDate, $endDate);
            } elseif ($format === 'csv') {
                return $this->generateCustomerStatementCsv($customer, $summary, $transactions, $startDate, $endDate);
            }

            return response()->json([
                'success' => true,
                'data' => [
                    'customer' => [
                        'id' => $customer->id,
                        'name' => $customer->customer,
                        'customer_number' => $customer->customer_number,
                        'email' => $customer->email,
                        'phone' => $customer->phone,
                        'address' => $customer->address,
                        'income' => (float) $customer->income,
                        'credit_score' => $customer->currentCreditScore?->credit_score ?? 'N/A',
                        'score_band' => $customer->currentCreditScore?->score_band ?? 'N/A',
                    ],
                    'statement_period' => [
                        'start_date' => $startDate,
                        'end_date' => $endDate,
                        'period_days' => Carbon::parse($startDate)->diffInDays(Carbon::parse($endDate)),
                    ],
                    'financial_summary' => $summary,
                    'loan_portfolio' => $customer->disbursedLoans->map(function ($loan) {
                        return [
                            'loan_number' => $loan->loannumber,
                            'loan_type' => $loan->loanType->product ?? 'N/A',
                            'disbursement_date' => $loan->disburseddate,
                            'original_amount' => (float) $loan->amount,
                            'current_balance' => (float) $loan->totalbalance,
                            'status' => $loan->status,
                            'is_overdue' => $loan->isOverdue(),
                            'overdue_amount' => $this->calculateOverdueAmount($loan),
                        ];
                    }),
                    'recent_transactions' => $transactions->take(20), // Show last 20 transactions
                ],
                'message' => 'Customer statement retrieved successfully',
            ]);

        } catch (\Exception $e) {
            \Log::error('Error fetching customer statement: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve customer statement',
                'error' => config('app.debug') ? $e->getMessage() : 'Internal server error',
            ], 500);
        }
    }

    // ===================================================================
    // HELPER METHODS
    // ===================================================================

    /**
     * Get loan transactions for a specific period
     */
    private function getLoanTransactionsForPeriod($loanId, $startDate, $endDate, $transactionType = 'all')
    {
        $transactions = collect();
        
        // Get repayments
        if (in_array($transactionType, ['all', 'repayment'])) {
            $repayments = Repayment::where('rloanid', $loanId)
                ->whereBetween('transactiondate', [$startDate, $endDate])
                ->get()
                ->map(function ($repayment) use ($loanId) {
                    return [
                        'loan_id' => $loanId,
                        'transaction_id' => $repayment->repaymentid,
                        'transaction_type' => 'repayment',
                        'transaction_date' => $repayment->transactiondate,
                        'amount' => (float) $repayment->amount,
                        'description' => 'Loan Repayment',
                        'payment_method' => $repayment->paymentmethod,
                        'reference_number' => $repayment->referencenumber,
                    ];
                });
            $transactions = $transactions->concat($repayments);
        }

        // Get interest charges
        if (in_array($transactionType, ['all', 'interest'])) {
            $interest = InterestIncome::where('iloanid', $loanId)
                ->whereBetween('chargeddate', [$startDate, $endDate])
                ->get()
                ->map(function ($interest) use ($loanId) {
                    return [
                        'loan_id' => $loanId,
                        'transaction_id' => $interest->id,
                        'transaction_type' => 'interest',
                        'transaction_date' => $interest->chargeddate,
                        'amount' => (float) $interest->charge,
                        'description' => 'Interest Charge',
                        'payment_method' => null,
                        'reference_number' => 'INT-' . $interest->id,
                    ];
                });
            $transactions = $transactions->concat($interest);
        }

        // Get processing fees
        if (in_array($transactionType, ['all', 'fee'])) {
            $fees = ProcessingFee::where('iloanid', $loanId)
                ->whereBetween('chargeddate', [$startDate, $endDate])
                ->get()
                ->map(function ($fee) use ($loanId) {
                    return [
                        'loan_id' => $loanId,
                        'transaction_id' => $fee->id,
                        'transaction_type' => 'fee',
                        'transaction_date' => $fee->chargeddate,
                        'amount' => (float) $fee->charge,
                        'description' => 'Processing Fee',
                        'payment_method' => null,
                        'reference_number' => 'FEE-' . $fee->id,
                    ];
                });
            $transactions = $transactions->concat($fees);
        }

        return $transactions->sortByDesc('transaction_date')->values();
    }

    /**
     * Get all customer transactions within a period
     */
    private function getAllCustomerTransactions($customerId, $startDate, $endDate)
    {
        // Get all customer loans
        $loans = DisbursedLoan::where('customerid', $customerId)->pluck('loanid');

        $allTransactions = collect();
        
        foreach ($loans as $loanId) {
            $transactions = $this->getLoanTransactionsForPeriod($loanId, $startDate, $endDate, 'all');
            $allTransactions = $allTransactions->concat($transactions);
        }

        return $allTransactions->sortByDesc('transaction_date')->values();
    }

    /**
     * Calculate balance as of specific date
     */
    private function calculateBalanceAsOfDate($loan, $asOfDate)
    {
        // Get original loan amount
        $originalAmount = (float) $loan->amount;
        
        // Get repayments before as_of_date
        $repaymentsBefore = $loan->repayments()
            ->where('transactiondate', '<=', $asOfDate)
            ->sum('amount');
            
        // Get principal paid before as_of_date
        $principalPaidBefore = $loan->repayments()
            ->where('transactiondate', '<=', $asOfDate)
            ->sum('principalamount');
            
        // Get interest charged before as_of_date
        $interestChargedBefore = InterestIncome::where('iloanid', $loan->loanid)
            ->where('chargeddate', '<=', $asOfDate)
            ->sum('charge');
            
        // Get fees charged before as_of_date
        $feesChargedBefore = ProcessingFee::where('iloanid', $loan->loanid)
            ->where('chargeddate', '<=', $asOfDate)
            ->sum('charge');
            
        // Get interest paid before as_of_date
        $interestPaidBefore = $loan->repayments()
            ->where('transactiondate', '<=', $asOfDate)
            ->sum('interestamount');
            
        // Get fees paid before as_of_date
        $feesPaidBefore = $loan->repayments()
            ->where('transactiondate', '<=', $asOfDate)
            ->sum('processing_fees_amount_paid');

        // Calculate balances
        $principalBalance = max(0, $originalAmount - $principalPaidBefore);
        $interestBalance = max(0, $interestChargedBefore - $interestPaidBefore);
        $feeBalance = max(0, $feesChargedBefore - $feesPaidBefore);
        $totalBalance = $principalBalance + $interestBalance + $feeBalance;

        return [
            'principal_balance' => $principalBalance,
            'interest_balance' => $interestBalance,
            'processing_fee_balance' => $feeBalance,
            'penalty_balance' => 0, // Would need penalty model to calculate this
            'total_balance' => $totalBalance,
            'as_of_date' => $asOfDate,
            'is_current' => $asOfDate == Carbon::now()->format('Y-m-d'),
        ];
    }

    /**
     * Calculate overdue amount for a loan
     */
    private function calculateOverdueAmount($loan)
    {
        if (!$loan->isOverdue()) {
            return 0;
        }

        // Sum overdue payment schedules
        return $loan->paymentSchedules()
            ->where('status', 'overdue')
            ->sum('balance');
    }

    /**
     * Calculate customer financial summary
     */
    private function calculateCustomerFinancialSummary($customer, $startDate, $endDate)
    {
        $loans = $customer->disbursedLoans;
        
        $totalDisbursed = $loans->sum('amount');
        $totalOutstanding = $loans->sum('totalbalance');
        $totalPaid = $loans->sum('totalpaid');
        
        $activeLoans = $loans->where('status', 'active');
        $overdueLoans = $activeLoans->filter(fn($loan) => $loan->isOverdue());
        
        $totalOverdueAmount = $overdueLoans->sum(function($loan) {
            return $this->calculateOverdueAmount($loan);
        });

        // Get transactions in period
        $transactions = $this->getAllCustomerTransactions($customer->id, $startDate, $endDate);
        
        $periodRepayments = $transactions->where('transaction_type', 'repayment')->sum('amount');
        $periodInterest = $transactions->where('transaction_type', 'interest')->sum('amount');
        $periodFees = $transactions->where('transaction_type', 'fee')->sum('amount');

        return [
            'total_loans' => $loans->count(),
            'active_loans' => $activeLoans->count(),
            'closed_loans' => $loans->where('status', 'closed')->count(),
            'overdue_loans' => $overdueLoans->count(),
            'total_disbursed' => $totalDisbursed,
            'total_outstanding' => $totalOutstanding,
            'total_paid' => $totalPaid,
            'total_overdue_amount' => $totalOverdueAmount,
            'period_activity' => [
                'repayments' => $periodRepayments,
                'interest_charged' => $periodInterest,
                'fees_charged' => $periodFees,
                'net_cash_flow' => $periodRepayments - ($periodInterest + $periodFees),
            ],
            'credit_score' => [
                'current_score' => $customer->currentCreditScore?->credit_score ?? 'N/A',
                'score_band' => $customer->currentCreditScore?->score_band ?? 'N/A',
                'has_history' => !is_null($customer->currentCreditScore?->credit_score),
            ],
        ];
    }

    /**
     * Generate PDF for loan statement (placeholder)
     */
    private function generateLoanStatementPdf($customer, $loan, $summary, $transactions)
    {
        // TODO: Implement PDF generation using DomPDF, TCPDF, or similar
        return response()->json([
            'success' => true,
            'message' => 'PDF generation not yet implemented',
            'data' => [
                'customer' => $customer->only(['id', 'name', 'customer_number']),
                'loan' => $loan->only(['loanid', 'loannumber']),
                'summary' => $summary,
                'transaction_count' => $transactions->count(),
            ],
        ]);
    }

    /**
     * Generate CSV for loan statement (placeholder)
     */
    private function generateLoanStatementCsv($customer, $loan, $summary, $transactions)
    {
        // TODO: Implement CSV generation
        return response()->json([
            'success' => true,
            'message' => 'CSV generation not yet implemented',
            'data' => [
                'customer' => $customer->only(['id', 'name', 'customer_number']),
                'loan' => $loan->only(['loanid', 'loannumber']),
                'summary' => $summary,
                'transaction_count' => $transactions->count(),
            ],
        ]);
    }

    /**
     * Generate PDF for customer statement (placeholder)
     */
    private function generateCustomerStatementPdf($customer, $summary, $transactions, $startDate, $endDate)
    {
        // TODO: Implement PDF generation
        return response()->json([
            'success' => true,
            'message' => 'Customer statement PDF generation not yet implemented',
            'data' => [
                'customer' => $customer->only(['id', 'name', 'customer_number']),
                'summary' => $summary,
                'transaction_count' => $transactions->count(),
                'period' => ['start_date' => $startDate, 'end_date' => $endDate],
            ],
        ]);
    }

    /**
     * Generate CSV for customer statement (placeholder)
     */
    private function generateCustomerStatementCsv($customer, $summary, $transactions, $startDate, $endDate)
    {
        // TODO: Implement CSV generation
        return response()->json([
            'success' => true,
            'message' => 'Customer statement CSV generation not yet implemented',
            'data' => [
                'customer' => $customer->only(['id', 'name', 'customer_number']),
                'summary' => $summary,
                'transaction_count' => $transactions->count(),
                'period' => ['start_date' => $startDate, 'end_date' => $endDate],
            ],
        ]);
    }
}