<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use App\Models\PaymentCollection;
use App\Models\PaymentDisbursement;

class WebhookController extends Controller
{
    /**
     * Log complete webhook payload and headers
     */
    private function logWebhookPayload(Request $request, string $type)
    {
        $payload = $request->all();
        $headers = $request->headers->all();
        $clientIp = $request->ip();
        
        // Log full webhook details
        Log::info("📥 {$type} Webhook Received", [
            'timestamp' => now()->toDateTimeString(),
            'client_ip' => $clientIp,
            'request_method' => $request->method(),
            'request_url' => $request->fullUrl(),
            'user_agent' => $request->userAgent(),
            'content_type' => $request->header('Content-Type'),
            'content_length' => $request->header('Content-Length'),
            'headers' => $this->sanitizeHeaders($headers),
            'raw_payload_size' => strlen($request->getContent()),
            'parsed_payload' => $payload,
            'raw_payload_preview' => substr($request->getContent(), 0, 1000) . (strlen($request->getContent()) > 1000 ? '...' : ''),
        ]);
        
        // Log specific payment gateway response fields
        if (!empty($payload)) {
            Log::info("🔍 Payment Gateway Response Details", [
                'referenceId' => $payload['referenceId'] ?? 'NOT_PROVIDED',  // Primary identifier
                'identifier' => $payload['identifier'] ?? 'NOT_PROVIDED',    // Secondary identifier
                'externalId' => $payload['externalId'] ?? 'NOT_PROVIDED',
                'status' => $payload['status'] ?? 'NOT_PROVIDED',
                'amount' => $payload['amount'] ?? 'NOT_PROVIDED',
                'accountNumber' => $payload['accountNumber'] ?? 'NOT_PROVIDED',
                'message' => $payload['message'] ?? 'NOT_PROVIDED',
                'narration' => $payload['narration'] ?? 'NOT_PROVIDED',
                'paymentType' => $payload['paymentType'] ?? $payload['paymentMethod'] ?? 'NOT_PROVIDED',
                'type' => $payload['type'] ?? 'NOT_PROVIDED',
                'ipAddress' => $payload['ipAddress'] ?? 'NOT_PROVIDED',
                'currency' => $payload['currency'] ?? 'NOT_PROVIDED',
                'timestamp' => $payload['timestamp'] ?? 'NOT_PROVIDED',
                'transactionId' => $payload['transactionId'] ?? 'NOT_PROVIDED',
                'providerReference' => $payload['providerReference'] ?? 'NOT_PROVIDED',
            ]);
        }
        
        return $payload;
    }

    /**
     * Sanitize headers for logging (remove sensitive information)
     */
    private function sanitizeHeaders(array $headers): array
    {
        $sensitiveHeaders = ['authorization', 'x-api-key', 'x-api-secret', 'api-key', 'token'];
        $sanitized = [];
        
        foreach ($headers as $key => $value) {
            $lowerKey = strtolower($key);
            if (in_array($lowerKey, $sensitiveHeaders)) {
                $sanitized[$key] = ['***REDACTED***'];
            } else {
                $sanitized[$key] = $value;
            }
        }
        
        return $sanitized;
    }

    /**
     * Handle incoming collection (Pay-In) webhooks.
     */
    public function handleCollectionCallback(Request $request)
    {
        // 1. Log the complete incoming webhook
        $payload = $this->logWebhookPayload($request, 'Collection');
        
        // 2. Extract and validate the referenceId (primary identifier)
        $referenceId = $payload['referenceId'] ?? null;
        $identifier = $payload['identifier'] ?? null; // Optional field
        $status = $payload['status'] ?? 'UNKNOWN';
        $message = $payload['message'] ?? 'No message';
        
        // 3. Validate that we have at least one identifier
        if (!$referenceId && !$identifier) {
            Log::error("❌ Collection Webhook Error: Missing 'referenceId' and 'identifier' fields", [
                'complete_payload' => $payload,
                'available_fields' => array_keys($payload)
            ]);
            return response('Missing referenceId/identifier', 400);
        }
        
        // 4. Use referenceId as primary identifier, fall back to identifier if needed
        $primaryId = $referenceId ?? $identifier;

        Log::info("🔄 Processing Collection Transaction", [
            'primary_id' => $primaryId,
            'reference_id' => $referenceId,
            'identifier' => $identifier,
            'status' => $status,
            'message' => $message,
            'step' => 'starting_processing'
        ]);

        // 5. Update Database using Eloquent Model
        // Search by reference_id first (priority), then fall back to identifier
        $collection = PaymentCollection::where('reference_id', $referenceId)
            ->orWhere('identifier', $identifier)
            ->first();

        if ($collection) {
            $previousStatus = $collection->status;
            
            Log::info("📝 Found Collection Record", [
                'payment_id' => $collection->id,
                'previous_status' => $previousStatus,
                'new_status' => $status,
                'amount' => $collection->amount,
                'account_number' => $collection->account_number,
                'payment_type' => $collection->payment_type,
                'found_by_reference_id' => $collection->reference_id === $referenceId,
                'found_by_identifier' => $collection->identifier === $identifier
            ]);

            // Prepare update data
            $updateData = [
                'status' => $status,
                'message' => $message,
                'webhook_received_at' => now(),
                'webhook_payload' => $payload, // Store full payload (will be JSON encoded)
                'updated_at' => now(),
            ];
            
            // Update ip_address if provided in webhook
            if (isset($payload['ipAddress'])) {
                $updateData['ip_address'] = $payload['ipAddress'];
            }
            
            // Update payment_type if provided in webhook
            if (isset($payload['paymentType'])) {
                $updateData['payment_type'] = $payload['paymentType'];
            } elseif (isset($payload['paymentMethod'])) {
                $updateData['payment_type'] = $payload['paymentMethod'];
            }
            
            // Update narration if provided
            if (isset($payload['narration'])) {
                $updateData['narration'] = $payload['narration'];
            }
            
            $collection->update($updateData);

            // Log status change
            if ($previousStatus !== $status) {
                Log::info("🔄 Collection Status Updated", [
                    'reference_id' => $referenceId,
                    'identifier' => $identifier,
                    'previous_status' => $previousStatus,
                    'new_status' => $status,
                    'payment_id' => $collection->id,
                    'change_timestamp' => now()->toDateTimeString()
                ]);
            }

            // Handle specific statuses
            $normalizedStatus = strtolower($status);
            if (in_array($normalizedStatus, ['successful', 'success', 'paid', 'completed'])) {
                Log::info("✅ SUCCESS: Collection {$primaryId} marked as paid", [
                    'payment_id' => $collection->id,
                    'amount' => $collection->amount,
                    'company_id' => $collection->company_id,
                    'loan_id' => $collection->loan_id,
                    'webhook_amount' => $payload['amount'] ?? null,
                    'webhook_currency' => $payload['currency'] ?? null
                ]);
                
                // Add any additional logic here (e.g., updating loan balances)
                $this->handleSuccessfulCollection($collection, $payload);
                
            } elseif (in_array($normalizedStatus, ['failed', 'failure', 'rejected', 'declined'])) {
                Log::warning("❌ FAILURE: Collection {$primaryId} marked as failed", [
                    'payment_id' => $collection->id,
                    'failure_reason' => $message,
                    'company_id' => $collection->company_id,
                    'webhook_message' => $payload['message'] ?? null
                ]);
                
                $this->handleFailedCollection($collection, $payload);
                
            } else {
                Log::info("📊 Collection Status Update", [
                    'reference_id' => $referenceId,
                    'identifier' => $identifier,
                    'status' => $status,
                    'message' => $message,
                    'payment_id' => $collection->id,
                    'status_normalized' => $normalizedStatus
                ]);
            }
            
        } else {
            Log::warning("⚠️ Collection Record Not Found", [
                'reference_id' => $referenceId,
                'identifier' => $identifier,
                'status' => $status,
                'amount' => $payload['amount'] ?? null,
                'account_number' => $payload['accountNumber'] ?? null,
                'searched_fields' => ['reference_id', 'identifier'],
                'payload_received' => $payload
            ]);
            
            // Create a new record if not found (for tracking)
            $this->createMissingCollectionRecord($payload);
        }

        // 6. Log webhook processing completion
        Log::info("🏁 Collection Webhook Processing Complete", [
            'reference_id' => $referenceId,
            'identifier' => $identifier,
            'status' => $status,
            'processing_time' => round(microtime(true) - LARAVEL_START, 3) . 's',
            'record_found' => !is_null($collection),
            'http_response' => 'ACK'
        ]);

        // 7. Respond to Lipila to stop retries
        return response('ACK', 200);
    }

    /**
     * Handle incoming disbursement (Payout) webhooks.
     */
    public function handleDisbursementCallback(Request $request)
    {
        // 1. Log the complete incoming webhook
        $payload = $this->logWebhookPayload($request, 'Disbursement');
        
        $referenceId = $payload['referenceId'] ?? null;
        $identifier = $payload['identifier'] ?? null;
        $status = $payload['status'] ?? 'UNKNOWN';
        $amount = $payload['amount'] ?? 0;
        $recipient = $payload['accountNumber'] ?? 'Unknown';
        $message = $payload['message'] ?? 'No message';

        if (!$referenceId && !$identifier) {
            Log::error("❌ Disbursement Webhook Error: Missing 'referenceId' and 'identifier' fields", [
                'complete_payload' => $payload,
                'available_fields' => array_keys($payload)
            ]);
            return response('Missing referenceId/identifier', 400);
        }

        $primaryId = $referenceId ?? $identifier;

        Log::info("🔄 Processing Disbursement Transaction", [
            'primary_id' => $primaryId,
            'reference_id' => $referenceId,
            'identifier' => $identifier,
            'status' => $status,
            'amount' => $amount,
            'recipient' => $recipient,
            'message' => $message,
            'step' => 'starting_processing'
        ]);

        // 2. Update Database using Eloquent Model
        $disbursement = PaymentDisbursement::where('reference_id', $referenceId)
            ->orWhere('identifier', $identifier)
            ->first();

        if ($disbursement) {
            $previousStatus = $disbursement->status;
            
            Log::info("📝 Found Disbursement Record", [
                'disbursement_id' => $disbursement->id,
                'previous_status' => $previousStatus,
                'new_status' => $status,
                'amount' => $disbursement->amount,
                'account_number' => $disbursement->account_number,
                'payment_type' => $disbursement->payment_type,
                'loan_id' => $disbursement->loan_id,
                'customer_id' => $disbursement->customer_id,
                'found_by_reference_id' => $disbursement->reference_id === $referenceId,
                'found_by_identifier' => $disbursement->identifier === $identifier
            ]);

            $updateData = [
                'status' => $status,
                'narration' => $message,
                'webhook_received_at' => now(),
                'webhook_payload' => $payload,
                'updated_at' => now(),
            ];
            
            if (isset($payload['paymentType'])) {
                $updateData['payment_type'] = $payload['paymentType'];
            } elseif (isset($payload['paymentMethod'])) {
                $updateData['payment_type'] = $payload['paymentMethod'];
            }

            $disbursement->update($updateData);

            // Log status change
            if ($previousStatus !== $status) {
                Log::info("🔄 Disbursement Status Updated", [
                    'reference_id' => $referenceId,
                    'identifier' => $identifier,
                    'previous_status' => $previousStatus,
                    'new_status' => $status,
                    'disbursement_id' => $disbursement->id,
                    'change_timestamp' => now()->toDateTimeString()
                ]);
            }

            $normalizedStatus = strtolower($status);
            if (in_array($normalizedStatus, ['successful', 'success', 'paid', 'completed', 'sent'])) {
                Log::info("✅ SUCCESS: Disbursement {$primaryId} successfully sent to {$recipient}", [
                    'disbursement_id' => $disbursement->id,
                    'amount' => $amount,
                    'recipient' => $recipient,
                    'loan_id' => $disbursement->loan_id,
                    'customer_id' => $disbursement->customer_id
                ]);
                
                $this->handleSuccessfulDisbursement($disbursement, $payload);
                
            } elseif (in_array($normalizedStatus, ['failed', 'failure', 'rejected', 'declined'])) {
                Log::warning("❌ FAILURE: Disbursement {$primaryId} failed", [
                    'disbursement_id' => $disbursement->id,
                    'failure_reason' => $message,
                    'amount' => $amount,
                    'recipient' => $recipient,
                    'loan_id' => $disbursement->loan_id
                ]);
                
                $this->handleFailedDisbursement($disbursement, $payload);
                
            } else {
                Log::info("📊 Disbursement Status Update", [
                    'reference_id' => $referenceId,
                    'identifier' => $identifier,
                    'status' => $status,
                    'message' => $message,
                    'disbursement_id' => $disbursement->id
                ]);
            }
            
        } else {
            Log::warning("⚠️ Disbursement Record Not Found", [
                'reference_id' => $referenceId,
                'identifier' => $identifier,
                'status' => $status,
                'amount' => $amount,
                'recipient' => $recipient,
                'searched_fields' => ['reference_id', 'identifier'],
                'payload_received' => $payload
            ]);
            
            $this->createMissingDisbursementRecord($payload);
        }

        // 3. Log webhook processing completion
        Log::info("🏁 Disbursement Webhook Processing Complete", [
            'reference_id' => $referenceId,
            'identifier' => $identifier,
            'status' => $status,
            'processing_time' => round(microtime(true) - LARAVEL_START, 3) . 's',
            'record_found' => !is_null($disbursement),
            'http_response' => 'ACK'
        ]);

        // 4. Respond to Lipila to stop retries
        return response('ACK', 200);
    }

    /**
     * Handle successful collection
     */
    private function handleSuccessfulCollection(PaymentCollection $collection, array $payload)
    {
        try {
            // Update related loan if loan_id exists
            if ($collection->loan_id) {
                Log::info("💳 Updating loan for successful collection", [
                    'payment_id' => $collection->id,
                    'loan_id' => $collection->loan_id,
                    'amount_paid' => $collection->amount,
                    'payload_reference' => $payload['referenceId'] ?? 'N/A'
                ]);
                
                // Example: Update loan repayment
                // $loan = DisbursedLoan::find($collection->loan_id);
                // if ($loan) {
                //     // Update loan repayment logic here
                // }
            }
            
            // Trigger any business logic (email notifications, etc.)
            Log::info("📧 Collection successful - business logic can be triggered", [
                'payment_id' => $collection->id,
                'company_id' => $collection->company_id,
                'user_id' => $collection->created_by
            ]);
            
        } catch (\Exception $e) {
            Log::error("⚠️ Error handling successful collection", [
                'payment_id' => $collection->id,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
        }
    }

    /**
     * Handle failed collection
     */
    private function handleFailedCollection(PaymentCollection $collection, array $payload)
    {
        try {
            Log::warning("💸 Collection failed - business logic can be triggered", [
                'payment_id' => $collection->id,
                'failure_reason' => $payload['message'] ?? 'Unknown',
                'company_id' => $collection->company_id,
                'user_id' => $collection->created_by
            ]);
            
            // Trigger failed payment notifications, retry logic, etc.
            
        } catch (\Exception $e) {
            Log::error("⚠️ Error handling failed collection", [
                'payment_id' => $collection->id,
                'error' => $e->getMessage()
            ]);
        }
    }

    /**
     * Handle successful disbursement
     */
    private function handleSuccessfulDisbursement(PaymentDisbursement $disbursement, array $payload)
    {
        try {
            // Update loan record if loan_id exists
            if ($disbursement->loan_id) {
                Log::info("💰 Updating loan for successful disbursement", [
                    'disbursement_id' => $disbursement->id,
                    'loan_id' => $disbursement->loan_id,
                    'amount_disbursed' => $disbursement->amount,
                    'customer_id' => $disbursement->customer_id,
                    'payload_reference' => $payload['referenceId'] ?? 'N/A'
                ]);
                
                // Example: Update loan disbursement status
                // $loan = DisbursedLoan::find($disbursement->loan_id);
                // if ($loan) {
                //     $loan->update([
                //         'lipila_status' => 'SUCCESS',
                //         'lipila_callback_status' => 'SUCCESS',
                //         'lipila_callback_message' => $payload['message'] ?? null,
                //         'updated_at' => now(),
                //     ]);
                // }
            }
            
            // Trigger notifications, etc.
            Log::info("📧 Disbursement successful - business logic can be triggered", [
                'disbursement_id' => $disbursement->id,
                'company_id' => $disbursement->company_id,
                'customer_id' => $disbursement->customer_id
            ]);
            
        } catch (\Exception $e) {
            Log::error("⚠️ Error handling successful disbursement", [
                'disbursement_id' => $disbursement->id,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
        }
    }

    /**
     * Handle failed disbursement
     */
    private function handleFailedDisbursement(PaymentDisbursement $disbursement, array $payload)
    {
        try {
            Log::warning("💸 Disbursement failed - business logic can be triggered", [
                'disbursement_id' => $disbursement->id,
                'failure_reason' => $payload['message'] ?? 'Unknown',
                'loan_id' => $disbursement->loan_id,
                'customer_id' => $disbursement->customer_id
            ]);
            
            // Trigger failed disbursement notifications, retry logic, etc.
            // Update loan status to failed disbursement
            
        } catch (\Exception $e) {
            Log::error("⚠️ Error handling failed disbursement", [
                'disbursement_id' => $disbursement->id,
                'error' => $e->getMessage()
            ]);
        }
    }

    /**
     * Create missing collection record for tracking
     */
    private function createMissingCollectionRecord(array $payload)
    {
        try {
            $referenceId = $payload['referenceId'] ?? null;
            $identifier = $payload['identifier'] ?? null;
            
            if (!$referenceId && !$identifier) {
                Log::error("Cannot create missing collection record: No referenceId or identifier in payload");
                return;
            }
            
            // Use referenceId as primary, fall back to identifier
            $primaryReference = $referenceId ?? $identifier;
            
            $collection = PaymentCollection::create([
                'reference_id' => $primaryReference,
                'identifier' => $identifier ?? $primaryReference,
                'amount' => $payload['amount'] ?? 0,
                'account_number' => $payload['accountNumber'] ?? 'UNKNOWN',
                'status' => $payload['status'] ?? 'UNKNOWN',
                'payment_type' => $payload['paymentType'] ?? $payload['paymentMethod'] ?? 'UNKNOWN',
                'message' => $payload['message'] ?? 'Created from webhook',
                'narration' => $payload['narration'] ?? 'Auto-created from webhook',
                'currency' => $payload['currency'] ?? 'ZMW',
                'ip_address' => $payload['ipAddress'] ?? null,
                'webhook_received_at' => now(),
                'webhook_payload' => $payload,
                'api_created_at' => now(),
                'company_id' => 1, // Default company ID
                'created_by' => 1, // Default user ID
                'metadata' => [
                    'auto_created' => true,
                    'source' => 'webhook_missing_record',
                    'webhook_timestamp' => now()->toDateTimeString(),
                    'created_from' => 'handleCollectionCallback',
                    'original_payload_summary' => [
                        'referenceId' => $referenceId,
                        'identifier' => $identifier,
                        'amount' => $payload['amount'] ?? null,
                        'accountNumber' => $payload['accountNumber'] ?? null
                    ]
                ],
            ]);
            
            Log::info("📝 Created missing collection record from webhook", [
                'payment_id' => $collection->id,
                'reference_id' => $collection->reference_id,
                'identifier' => $collection->identifier,
                'status' => $collection->status,
                'amount' => $collection->amount,
                'account_number' => $collection->account_number
            ]);
            
        } catch (\Exception $e) {
            Log::error("❌ Failed to create missing collection record", [
                'reference_id' => $payload['referenceId'] ?? 'UNKNOWN',
                'identifier' => $payload['identifier'] ?? 'UNKNOWN',
                'error' => $e->getMessage(),
                'payload' => $payload
            ]);
        }
    }

    /**
     * Create missing disbursement record for tracking
     */
    private function createMissingDisbursementRecord(array $payload)
    {
        try {
            $referenceId = $payload['referenceId'] ?? null;
            $identifier = $payload['identifier'] ?? null;
            
            if (!$referenceId && !$identifier) {
                Log::error("Cannot create missing disbursement record: No referenceId or identifier in payload");
                return;
            }
            
            $primaryReference = $referenceId ?? $identifier;
            
            $disbursement = PaymentDisbursement::create([
                'reference_id' => $primaryReference,
                'identifier' => $identifier ?? $primaryReference,
                'amount' => $payload['amount'] ?? 0,
                'account_number' => $payload['accountNumber'] ?? 'UNKNOWN',
                'status' => $payload['status'] ?? 'UNKNOWN',
                'payment_type' => $payload['paymentType'] ?? $payload['paymentMethod'] ?? 'UNKNOWN',
                'narration' => $payload['message'] ?? 'Created from webhook',
                'currency' => $payload['currency'] ?? 'ZMW',
                'webhook_received_at' => now(),
                'webhook_payload' => $payload,
                'api_created_at' => now(),
                'company_id' => 1,
                'created_by' => 1,
                'metadata' => [
                    'auto_created' => true,
                    'source' => 'webhook_missing_record',
                    'webhook_timestamp' => now()->toDateTimeString(),
                    'created_from' => 'handleDisbursementCallback',
                    'original_payload_summary' => [
                        'referenceId' => $referenceId,
                        'identifier' => $identifier,
                        'amount' => $payload['amount'] ?? null,
                        'accountNumber' => $payload['accountNumber'] ?? null
                    ]
                ],
            ]);
            
            Log::info("📝 Created missing disbursement record from webhook", [
                'disbursement_id' => $disbursement->id,
                'reference_id' => $disbursement->reference_id,
                'identifier' => $disbursement->identifier,
                'status' => $disbursement->status,
                'amount' => $disbursement->amount,
                'recipient' => $disbursement->account_number
            ]);
            
        } catch (\Exception $e) {
            Log::error("❌ Failed to create missing disbursement record", [
                'reference_id' => $payload['referenceId'] ?? 'UNKNOWN',
                'identifier' => $payload['identifier'] ?? 'UNKNOWN',
                'error' => $e->getMessage(),
                'payload' => $payload
            ]);
        }
    }

    /**
     * Test endpoint to verify webhook is working
     */
    public function testWebhook(Request $request)
    {
        $type = $request->get('type', 'collection');
        
        Log::info("🧪 Test Webhook Called", [
            'type' => $type,
            'client_ip' => $request->ip(),
            'headers' => $request->headers->all(),
            'payload' => $request->all()
        ]);
        
        $testPayload = [
            'referenceId' => 'TEST-REF-' . time(),
            'identifier' => 'TEST-ID-' . time(),
            'status' => 'Successful',
            'amount' => 100.00,
            'accountNumber' => '260976379025',
            'message' => 'Test webhook successful',
            'narration' => 'Test payment',
            'paymentType' => 'Card',
            'ipAddress' => '102.89.34.10',
            'currency' => 'ZMW',
            'type' => 'Collection'
        ];
        
        if ($type === 'disbursement') {
            $testPayload['type'] = 'Disbursement';
            $this->handleDisbursementCallback(new Request($testPayload));
        } else {
            $this->handleCollectionCallback(new Request($testPayload));
        }
        
        return response()->json([
            'success' => true,
            'message' => 'Test webhook processed',
            'test_payload' => $testPayload,
            'logs_generated' => true
        ]);
    }
    
    /**
     * View webhook logs (for debugging)
     */
    public function viewLogs(Request $request)
    {
        $limit = $request->get('limit', 50);
        $type = $request->get('type', 'all');
        
        // In production, you would read from Laravel log files
        // This is a simplified version for demonstration
        
        $logFile = storage_path('logs/laravel.log');
        $logs = [];
        
        if (file_exists($logFile)) {
            $logContent = file($logFile, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
            $logContent = array_reverse($logContent);
            
            foreach ($logContent as $line) {
                if ($type === 'all' || 
                    ($type === 'collection' && strpos($line, 'Collection') !== false) ||
                    ($type === 'disbursement' && strpos($line, 'Disbursement') !== false)) {
                    $logs[] = $line;
                    if (count($logs) >= $limit) break;
                }
            }
        }
        
        return response()->json([
            'success' => true,
            'log_file' => $logFile,
            'log_count' => count($logs),
            'logs' => $logs
        ]);
    }

    /**
     * Health check endpoint for webhook
     */
    public function healthCheck()
    {
        return response()->json([
            'status' => 'healthy',
            'timestamp' => now()->toDateTimeString(),
            'service' => 'Webhook Controller',
            'models_available' => [
                'PaymentCollection' => class_exists(PaymentCollection::class),
                'PaymentDisbursement' => class_exists(PaymentDisbursement::class)
            ]
        ]);
    }
}