<?php

namespace App\Http\Controllers;

use App\Models\ApplicationRequirement;
use App\Models\Employee;
use App\Models\Company;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Log;

class ApplicationRequirementController extends Controller
{
    /**
     * Get the company ID from authenticated user.
     * This method looks for company_id in different locations.
     */
    protected function getUserCompanyId()
    {
        $user = Auth::user();
        
        if (!$user) {
            Log::warning('No authenticated user found');
            return null;
        }
        
        // Debug: Log all user attributes
        Log::info('User details for company lookup:', [
            'user_id' => $user->id,
            'user_email' => $user->email,
            'user_name' => $user->name,
            'user_attributes' => $user->toArray(), // This will show ALL user fields
        ]);
        
        // Method 1: Check if user has direct company_id attribute
        if (isset($user->company_id) && !empty($user->company_id)) {
            Log::info('Found company_id on user model', ['company_id' => $user->company_id]);
            return $user->company_id;
        }
        
        // Method 2: Get company_id from employee record using email (Most likely)
        Log::info('Looking for employee with email: ' . $user->email);
        $employee = Employee::where('email', $user->email)->first();
        
        if ($employee) {
            Log::info('Employee record found:', [
                'employee_id' => $employee->id,
                'employee_email' => $employee->email,
                'employee_company_id' => $employee->company_id,
                'employee_full_name' => $employee->first_name . ' ' . $employee->last_name,
            ]);
            
            if (!empty($employee->company_id)) {
                return $employee->company_id;
            } else {
                Log::warning('Employee found but company_id is empty or null', [
                    'employee_id' => $employee->id
                ]);
            }
        } else {
            Log::warning('NO employee record found for email: ' . $user->email);
            
            // Try to find any employee (for debugging)
            $allEmployees = Employee::limit(5)->get(['id', 'email', 'company_id', 'first_name', 'last_name']);
            Log::info('Sample employees in database:', $allEmployees->toArray());
        }
        
        // Method 3: Check if user has an employee relationship method
        if (method_exists($user, 'employee')) {
            Log::info('User has employee() method, checking...');
            if ($user->employee && $user->employee->company_id) {
                Log::info('Found company_id through employee() relationship', [
                    'company_id' => $user->employee->company_id
                ]);
                return $user->employee->company_id;
            }
        }
        
        // Method 4: Try to get first company as fallback (for testing)
        $firstCompany = Company::first();
        if ($firstCompany) {
            Log::warning('Using first company as fallback for user', [
                'user_id' => $user->id,
                'company_id' => $firstCompany->id,
                'company_name' => $firstCompany->name
            ]);
            return $firstCompany->id;
        }
        
        Log::error('Could not find company ID for user', [
            'user_id' => $user->id,
            'email' => $user->email
        ]);
        
        return null;
    }

    /**
     * Display a listing of application requirements for the user's company.
     */
    public function index()
    {
        $user = Auth::user();
        $companyId = $this->getUserCompanyId();
        
        if (!$companyId) {
            Log::error('No company ID found for user in index method', [
                'user_id' => $user->id,
                'user_email' => $user->email
            ]);
            
            return redirect()->back()
                ->with('error', 'You are not associated with any company. Please contact administrator.');
        }
        
        // Get requirements for user's company
        $requirements = ApplicationRequirement::where('companyid', $companyId)
            ->orderBy('createdat', 'desc')
            ->get();
    
        // Return the view
        return view('admin.applications.applicationrequirements', compact('requirements'));
    }

    /**
     * Store a newly created application requirement.
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'documentname' => 'required|string|max:255',
            'description' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        // Get company ID from authenticated user
        $companyId = $this->getUserCompanyId();
        
        if (!$companyId) {
            // Try one more time with a different approach
            Log::warning('Company ID not found, trying alternative methods...');
            
            // Alternative 1: Get any company
            $anyCompany = Company::first();
            if ($anyCompany) {
                $companyId = $anyCompany->id;
                Log::warning('Using any company as fallback', [
                    'company_id' => $companyId,
                    'company_name' => $anyCompany->name
                ]);
            } else {
                // Create a test company if none exists
                Log::warning('No companies exist, creating test company...');
                $testCompany = Company::create([
                    'name' => 'Test Company',
                    'email' => 'test@example.com',
                    'phone' => '1234567890',
                    'address' => 'Test Address',
                    'status' => 'active',
                ]);
                $companyId = $testCompany->id;
                Log::info('Created test company', ['company_id' => $companyId]);
            }
            
            if (!$companyId) {
                return redirect()->route('applications.applicationrequirements')
                    ->with('error', 'You are not associated with any company. Please contact administrator.')
                    ->withInput();
            }
        }

        // Check if document name already exists for this company
        $exists = ApplicationRequirement::where('companyid', $companyId)
            ->where('documentname', $request->documentname)
            ->exists();
            
        if ($exists) {
            return redirect()->route('applications.applicationrequirements')
                ->with('error', 'A requirement with this document name already exists.')
                ->withInput();
        }
        
        try {
            // Get user details
            $user = Auth::user();
            
            // Debug: Log the data being saved
            Log::info('Attempting to save application requirement', [
                'documentname' => $request->documentname,
                'description' => $request->description,
                'companyid' => $companyId,
                'user' => $user->name,
                'user_email' => $user->email
            ]);
            
            // Add user's company ID automatically
            $data = [
                'documentname' => $request->documentname,
                'description' => $request->description,
                'companyid' => $companyId,
                'status' => 'active',
                'createdby' => $user->name ?? $user->email ?? 'system',
                'createdat' => now(),
                'updatedat' => now(),
            ];
            
            Log::debug('Data being saved to database:', $data);
            
            $requirement = ApplicationRequirement::create($data);
            
            Log::info('Application requirement saved successfully', [
                'id' => $requirement->id,
                'documentname' => $requirement->documentname
            ]);
            
            return redirect()->route('applications.applicationrequirements')
                ->with('success', 'Application requirement added successfully.');
                
        } catch (\Exception $e) {
            // Log detailed error
            Log::error('Failed to add application requirement: ' . $e->getMessage(), [
                'trace' => $e->getTraceAsString(),
                'request_data' => $request->all(),
                'user_id' => Auth::id(),
                'company_id' => $companyId ?? 'null'
            ]);
            
            // Return with actual error message for debugging
            return redirect()->route('applications.applicationrequirements')
                ->with('error', 'Failed to add requirement: ' . $e->getMessage())
                ->withInput();
        }
    }

    // ... [rest of your controller methods]
}