<?php

namespace App\Http\Controllers;

use App\Models\CreditScoreTier;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class CreditScoreTierController extends Controller
{
    /**
     * Display a listing of the resource.
     */
  
    public function index()
    {
        $tiers = CreditScoreTier::ordered()->get();
    
        return view('accounting.settings.credit-tiers.index', [
            'title' => 'Credit Score Tiers',
            'subtitle' => 'Manage credit score tiers and loan parameters',
            'tiers' => $tiers
        ]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255|unique:credit_score_tiers,name',
            'min_score' => 'required|integer|min:300|max:849',
            'max_score' => 'required|integer|min:301|max:850|gt:min_score',
            'base_interest_rate' => 'required|numeric|min:0|max:100',
            'max_loan_amount' => 'required|numeric|min:0',
            'max_loan_term' => 'required|integer|min:1|max:120',
            'description' => 'nullable|string|max:500',
        ]);

        try {
            DB::beginTransaction();

            $tier = CreditScoreTier::create($validated);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Credit score tier created successfully!',
                'tier' => $tier
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to create credit score tier: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(CreditScoreTier $creditScoreTier)
    {
        return response()->json([
            'success' => true,
            'tier' => $creditScoreTier
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, CreditScoreTier $creditScoreTier)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255|unique:credit_score_tiers,name,' . $creditScoreTier->id,
            'min_score' => 'required|integer|min:300|max:849',
            'max_score' => 'required|integer|min:301|max:850|gt:min_score',
            'base_interest_rate' => 'required|numeric|min:0|max:100',
            'max_loan_amount' => 'required|numeric|min:0',
            'max_loan_term' => 'required|integer|min:1|max:120',
            'description' => 'nullable|string|max:500',
        ]);

        try {
            DB::beginTransaction();

            $creditScoreTier->update($validated);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Credit score tier updated successfully!',
                'tier' => $creditScoreTier
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to update credit score tier: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(CreditScoreTier $creditScoreTier)
    {
        try {
            DB::beginTransaction();

            // Check if tier is being used in loan applications
            if ($creditScoreTier->loanApplications()->exists()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Cannot delete tier because it is being used in loan applications.'
                ], 422);
            }

            $creditScoreTier->delete();

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Credit score tier deleted successfully!'
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete credit score tier: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Toggle tier status
     */
    public function toggleStatus(CreditScoreTier $creditScoreTier)
    {
        try {
            $creditScoreTier->update([
                'is_active' => !$creditScoreTier->is_active
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Tier status updated successfully!',
                'tier' => $creditScoreTier
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to update tier status: ' . $e->getMessage()
            ], 500);
        }
    }
}