<?php
// app/Models/Branch.php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Branch extends Model
{
    use HasFactory;

    protected $fillable = [
        'branchcode',
        'branchname',
        'setupdate',
        'status',
        'userid',
        'branchmanager',
        'companyid'
    ];

    protected $attributes = [
        'companyid' => 1,
        'status' => 1,
    ];

    protected $casts = [
        'setupdate' => 'date',
        'status' => 'boolean',
        'companyid' => 'integer', // Cast to integer
    ];

    /**
     * Get the company that owns the branch
     */
    public function company()
    {
        return $this->belongsTo(Company::class, 'companyid', 'id');
    }

    /**
     * Get the user who created the branch
     */
    public function user()
    {
        return $this->belongsTo(User::class, 'userid');
    }

    /**
     * Get the status as a human-readable string
     */
    public function getStatusTextAttribute()
    {
        return $this->status ? 'Active' : 'Inactive';
    }

    /**
     * Get formatted set update date
     */
    public function getFormattedSetupdateAttribute()
    {
        return $this->setupdate->format('M j, Y');
    }

    /**
     * Scope a query to only include active branches.
     */
    public function scopeActive($query)
    {
        return $query->where('status', 1);
    }

    /**
     * Scope a query to only include inactive branches.
     */
    public function scopeInactive($query)
    {
        return $query->where('status', 0);
    }

    /**
     * Scope a query to filter by company
     */
    public function scopeForCompany($query, $companyId)
    {
        return $query->where('companyid', $companyId);
    }
}