<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class BusinessProcessMapping extends Model
{
    use HasFactory;

    protected $table = 'business_process_mappings';
    protected $primaryKey = 'businessmappingid';
    public $incrementing = true;
    protected $keyType = 'int';

    protected $fillable = [
        'nameofthebusinessprocess',
        'businessprocessdescription',
        'accountid',
        'createdby',
        'companyid',
        'branchid'
    ];

    protected $casts = [
        'created_at' => 'datetime',
        'updated_at' => 'datetime'
    ];

    // Relationships
    public function account()
    {
        return $this->belongsTo(ChartOfAccount::class, 'accountid');
    }

    public function creator()
    {
        return $this->belongsTo(User::class, 'createdby');
    }

    public function company()
    {
        return $this->belongsTo(Company::class, 'companyid');
    }

    public function branch()
    {
        return $this->belongsTo(Branch::class, 'branchid');
    }

    // Scopes
    public function scopeActive($query)
    {
        return $query->whereHas('account', function($q) {
            $q->where('is_active', true);
        });
    }

    // Helper methods
    public static function getProcessNames()
    {
        return [
            // Original processes
            'Loan Disbursement' => 'Loan Disbursement',
            'Interest Income' => 'Interest Income',
            'Opening Balances' => 'Opening Balances',
            'Fixed Assets' => 'Fixed Assets',
            'Foreign Currency Valuation' => 'Foreign Currency Valuation',
            'Loan Repayment' => 'Loan Repayment',
            'Fee Income' => 'Fee Income',
            'Commission Income' => 'Commission Income',
            'Salary Expense' => 'Salary Expense',
            'Rent Expense' => 'Rent Expense',
            'Utilities Expense' => 'Utilities Expense',
            'Depreciation Expense' => 'Depreciation Expense',
            'Bad Debt Expense' => 'Bad Debt Expense',
            'Tax Expense' => 'Tax Expense',
            
            // New loan-related processes
            'Loan Insurance' => 'Loan Insurance',
            'Processing Fees' => 'Processing Fees',
            'Administration Fees' => 'Administration Fees',
            'Penalties on Overdue Payments' => 'Penalties on Overdue Payments',
            
            // Payroll processes
            'Payroll - Gross Salary' => 'Payroll - Gross Salary',
            'Payroll - PAYE' => 'Payroll - PAYE',
            'Payroll - NAPSA Employer' => 'Payroll - NAPSA Employer',
            'Payroll - NAPSA Employee' => 'Payroll - NAPSA Employee',
            'Payroll - NHIMA Employer' => 'Payroll - NHIMA Employer',
            'Payroll - NHIMA Employee' => 'Payroll - NHIMA Employee',
            'Payroll - Net Pay' => 'Payroll - Net Pay',
            
            // =========== NEW: PROVISIONING PROCESSES ===========
            'Loan Loss Provision' => 'Loan Loss Provision',
            'Loan Loss Expense' => 'Loan Loss Expense',
            'Allowance for Doubtful Accounts' => 'Allowance for Doubtful Accounts',
            'Provision for Loan Losses' => 'Provision for Loan Losses',
            'NPL Interest Reversal' => 'NPL Interest Reversal',
            'NPL Fee Reversal' => 'NPL Fee Reversal',
            'Loan Write-off' => 'Loan Write-off',
            'Recovered Loan Loss' => 'Recovered Loan Loss',
            // ==================================================
        ];
    }

    // Simple helper for payroll check
    public function isPayrollRelated()
    {
        return str_starts_with($this->nameofthebusinessprocess, 'Payroll');
    }

    // Helper for provisioning check
    public function isProvisioningRelated()
    {
        $provisioningKeywords = [
            'provision', 'allowance', 'doubtful', 'write-off', 'recovery', 'loss', 'npl'
        ];
        
        $processName = strtolower($this->nameofthebusinessprocess);
        foreach ($provisioningKeywords as $keyword) {
            if (str_contains($processName, $keyword)) {
                return true;
            }
        }
        
        return false;
    }

    // Original simple methods
    public static function getAccountForProcess($processName)
    {
        return self::where('nameofthebusinessprocess', $processName)
            ->with('account')
            ->first();
    }

    public function getFormattedCreatedAtAttribute()
    {
        return $this->created_at->format('M j, Y g:i A');
    }

    public function getAccountDetailsAttribute()
    {
        return $this->account ? "{$this->account->code} - {$this->account->name}" : 'N/A';
    }

    // Simple default mappings - with new loan-related mappings
    public static function createDefaultMappings()
    {
        $defaultMappings = [
            // Original mappings
            [
                'nameofthebusinessprocess' => 'Loan Disbursement',
                'businessprocessdescription' => 'Mapping for loan disbursement transactions',
                'accountid' => ChartOfAccount::where('code', '1101')->first()->id ?? null,
            ],
            [
                'nameofthebusinessprocess' => 'Interest Income',
                'businessprocessdescription' => 'Mapping for interest income recognition',
                'accountid' => ChartOfAccount::where('code', '4101')->first()->id ?? null,
            ],
            
            // New loan-related mappings
            [
                'nameofthebusinessprocess' => 'Loan Insurance',
                'businessprocessdescription' => 'Mapping for loan insurance income',
                'accountid' => ChartOfAccount::where('code', '4102')->first()->id ?? null,
            ],
            [
                'nameofthebusinessprocess' => 'Processing Fees',
                'businessprocessdescription' => 'Mapping for loan processing fees income',
                'accountid' => ChartOfAccount::where('code', '4103')->first()->id ?? null,
            ],
            [
                'nameofthebusinessprocess' => 'Administration Fees',
                'businessprocessdescription' => 'Mapping for loan administration fees income',
                'accountid' => ChartOfAccount::where('code', '4104')->first()->id ?? null,
            ],
            [
                'nameofthebusinessprocess' => 'Penalties on Overdue Payments',
                'businessprocessdescription' => 'Mapping for penalties on overdue loan payments',
                'accountid' => ChartOfAccount::where('code', '4105')->first()->id ?? null,
            ],
            
            // Payroll mappings
            [
                'nameofthebusinessprocess' => 'Payroll - Gross Salary',
                'businessprocessdescription' => 'Gross salary expense',
                'accountid' => ChartOfAccount::where('code', '6201')->first()->id ?? null,
            ],
            [
                'nameofthebusinessprocess' => 'Payroll - Net Pay',
                'businessprocessdescription' => 'Net salary payable',
                'accountid' => ChartOfAccount::where('code', '2204')->first()->id ?? null,
            ],
            
            // =========== NEW: PROVISIONING MAPPINGS ===========
            [
                'nameofthebusinessprocess' => 'Loan Loss Provision',
                'businessprocessdescription' => 'Allowance/Provision for loan losses (Contra Asset)',
                'accountid' => ChartOfAccount::where('code', '1102')->first()->id ?? null, // Example: 1102 - Allowance for Loan Losses
            ],
            [
                'nameofthebusinessprocess' => 'Loan Loss Expense',
                'businessprocessdescription' => 'Expense for loan loss provisions',
                'accountid' => ChartOfAccount::where('code', '6202')->first()->id ?? null, // Example: 6202 - Loan Loss Expense
            ],
            [
                'nameofthebusinessprocess' => 'Allowance for Doubtful Accounts',
                'businessprocessdescription' => 'Allowance for doubtful accounts receivable',
                'accountid' => ChartOfAccount::where('code', '1103')->first()->id ?? null, // Example: 1103 - Allowance for Doubtful Accounts
            ],
            [
                'nameofthebusinessprocess' => 'Provision for Loan Losses',
                'businessprocessdescription' => 'General provision for loan losses',
                'accountid' => ChartOfAccount::where('code', '2205')->first()->id ?? null, // Example: 2205 - Provision for Loan Losses (Liability)
            ],
            [
                'nameofthebusinessprocess' => 'Loan Write-off',
                'businessprocessdescription' => 'Loan write-off when provision is used',
                'accountid' => ChartOfAccount::where('code', '6203')->first()->id ?? null, // Example: 6203 - Loan Write-off Expense
            ],
            [
                'nameofthebusinessprocess' => 'Recovered Loan Loss',
                'businessprocessdescription' => 'Income from recovered loan losses',
                'accountid' => ChartOfAccount::where('code', '4106')->first()->id ?? null, // Example: 4106 - Recovered Loan Loss Income
            ],
            // ==================================================
        ];

        $createdCount = 0;
        foreach ($defaultMappings as $mappingData) {
            if ($mappingData['accountid']) {
                $existing = self::where('nameofthebusinessprocess', $mappingData['nameofthebusinessprocess'])->first();
                if (!$existing) {
                    self::create(array_merge($mappingData, [
                        'createdby' => 1,
                        'companyid' => 1,
                        'branchid' => 1,
                    ]));
                    $createdCount++;
                }
            }
        }

        return $createdCount;
    }

    // Keep event handlers simple
    protected static function boot()
    {
        parent::boot();

        static::creating(function ($mapping) {
            if (empty($mapping->createdby)) {
                $mapping->createdby = auth()->id() ?? 1;
            }
        });
    }

    // Simple array conversion
    public function toArray()
    {
        return [
            'businessmappingid' => $this->businessmappingid,
            'nameofthebusinessprocess' => $this->nameofthebusinessprocess,
            'businessprocessdescription' => $this->businessprocessdescription,
            'accountid' => $this->accountid,
            'account_code' => $this->account->code ?? null,
            'account_name' => $this->account->name ?? null,
            'createdby' => $this->createdby,
            'created_at' => $this->created_at->format('Y-m-d H:i'),
            'is_payroll_related' => $this->isPayrollRelated(),
            'is_provisioning_related' => $this->isProvisioningRelated(),
        ];
    }

    // =========== NEW: PROVISIONING HELPER METHODS ===========

    /**
     * Get provisioning-related mapping
     */
    public static function getProvisioningMapping($processName)
    {
        return self::where('nameofthebusinessprocess', $processName)
            ->with('account')
            ->first();
    }

    /**
     * Get all provisioning-related mappings
     */
    public static function getAllProvisioningMappings()
    {
        return self::where(function($query) {
            $query->where('nameofthebusinessprocess', 'like', '%provision%')
                  ->orWhere('nameofthebusinessprocess', 'like', '%allowance%')
                  ->orWhere('nameofthebusinessprocess', 'like', '%loss%')
                  ->orWhere('nameofthebusinessprocess', 'like', '%write%')
                  ->orWhere('nameofthebusinessprocess', 'like', '%recover%');
        })->with('account')->get();
    }

    /**
     * Get specific account for provisioning
     */
    public static function getProvisionAccount()
    {
        // Try multiple possible names
        $provisionNames = [
            'Loan Loss Provision',
            'Allowance for Doubtful Accounts',
            'Provision for Loan Losses'
        ];
        
        foreach ($provisionNames as $name) {
            $mapping = self::getAccountForProcess($name);
            if ($mapping) {
                return $mapping->account;
            }
        }
        
        return null;
    }

    /**
     * Get loan loss expense account
     */
    public static function getLoanLossExpenseAccount()
    {
        $mapping = self::getAccountForProcess('Loan Loss Expense');
        if (!$mapping) {
            // Fallback to Bad Debt Expense
            $mapping = self::getAccountForProcess('Bad Debt Expense');
        }
        
        return $mapping ? $mapping->account : null;
    }

    /**
     * Get loan write-off account
     */
    public static function getLoanWriteOffAccount()
    {
        $mapping = self::getAccountForProcess('Loan Write-off');
        if (!$mapping) {
            // Fallback to Bad Debt Expense
            $mapping = self::getAccountForProcess('Bad Debt Expense');
        }
        
        return $mapping ? $mapping->account : null;
    }

    /**
     * Get recovered loan loss account
     */
    public static function getRecoveredLoanLossAccount()
    {
        $mapping = self::getAccountForProcess('Recovered Loan Loss');
        if (!$mapping) {
            // Fallback to Other Income
            $mapping = self::getAccountForProcess('Fee Income');
        }
        
        return $mapping ? $mapping->account : null;
    }
}