<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

class CashBankAccount extends Model
{
    use HasFactory;

    protected $primaryKey = 'cashBankId';
    
    // Specify the table name explicitly
    protected $table = 'cash_bank_accounts';
    
    protected $fillable = [
        'accountType',
        'accountName',
        'currency',
        'bankAccountName',
        'category',
        'defaultPaymentMethod',
        'bankName',
        'accountNumber',
        'branchName',
        'description',
        'openingBalance',
        'openingBalanceDate',
        'glAccountId',
        'isActive',
        'isDefault',
        'current_balance',
        'userid',
        'companyid',
        'gl_account_id' // ADD THIS if it exists in your database
    ];

    protected $casts = [
        'openingBalance' => 'decimal:2',
        'current_balance' => 'decimal:2',
        'openingBalanceDate' => 'date',
        'isActive' => 'boolean',
        'isDefault' => 'boolean',
    ];

    /**
     * Get the GL account
     */
    public function glAccount(): BelongsTo
    {
        return $this->belongsTo(ChartOfAccount::class, 'glAccountId', 'id');
    }

    // ADD THIS RELATIONSHIP FOR gl_account_id
    public function glLinkedAccount()
    {
        return $this->belongsTo(ChartOfAccount::class, 'gl_account_id');
    }

    /**
     * Get the creator
     */
    public function creator(): BelongsTo
    {
        return $this->belongsTo(User::class, 'userid');
    }

    /**
     * Get the company
     */
    public function company(): BelongsTo
    {
        return $this->belongsTo(Company::class, 'companyid', 'id');
    }

    /**
     * Get outgoing transfers
     */
    public function outgoingTransfers(): HasMany
    {
        return $this->hasMany(InternalCashBankTransfer::class, 'accountfrom');
    }

    /**
     * Get incoming transfers
     */
    public function incomingTransfers(): HasMany
    {
        return $this->hasMany(InternalCashBankTransfer::class, 'accountto');
    }

    /**
     * Get petty cash transactions
     */
    public function pettyCashTransactions(): HasMany
    {
        return $this->hasMany(PettyCash::class, 'source_account_id');
    }

    // ADD THIS RELATIONSHIP
    public function cashbookEntries(): HasMany
    {
        return $this->hasMany(CashBook::class, 'account_id', 'cashBankId');
    }

    /**
     * Scope for active accounts
     */
    public function scopeActive($query)
    {
        return $query->where('isActive', true);
    }

    /**
     * Scope by account type
     */
    public function scopeType($query, $type)
    {
        return $query->where('accountType', $type);
    }

    /**
     * Update account balance
     */
    public function updateBalance($amount, $type = 'debit')
    {
        $oldBalance = $this->current_balance;
        
        if ($type === 'debit') {
            $this->current_balance += $amount;
        } else {
            $this->current_balance -= $amount;
        }
        
        $this->save();
        
        // Log balance change
        \Log::info('Account balance updated', [
            'account_id' => $this->cashBankId,
            'account_name' => $this->accountName,
            'old_balance' => $oldBalance,
            'new_balance' => $this->current_balance,
            'amount' => $amount,
            'type' => $type
        ]);
        
        return $this;
    }

    /**
     * Get account balance formatted
     */
    public function getFormattedBalanceAttribute()
    {
        return number_format($this->current_balance, 2);
    }

    /**
     * Check if account has sufficient balance
     */
    public function hasSufficientBalance($amount): bool
    {
        return $this->current_balance >= $amount;
    }

    /**
     * Get related chart of account name
     */
    public function getGlAccountNameAttribute()
    {
        return $this->glAccount ? $this->glAccount->name : 'Not Linked';
    }

    /**
     * Get related chart of account code
     */
    public function getGlAccountCodeAttribute()
    {
        return $this->glAccount ? $this->glAccount->code : 'N/A';
    }

    // ADD THIS HELPER METHOD TO GET GL ACCOUNT ID
    public function getGlAccountIdAttribute()
    {
        // First try gl_account_id, then glAccountId
        if ($this->attributes['gl_account_id'] ?? null) {
            return $this->attributes['gl_account_id'];
        }
        
        if ($this->attributes['glAccountId'] ?? null) {
            return $this->attributes['glAccountId'];
        }
        
        return null;
    }
}