<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class CashBook extends Model
{
    use HasFactory;

    protected $table = 'cash_book';
    protected $primaryKey = 'cashbook_id';

    protected $fillable = [
        'companyid',
        'branchid',
        'account_id',
        'transaction_date',
        'transaction_type',
        'reference_number',
        'transfer_id', // ADDED THIS
        'description',
        'amount',
        'balance_before',
        'balance_after',
        'currency',
        'exchange_rate',
        'status',
        'related_transaction_type',
        'related_transaction_id',
        'created_by',
        'updated_by',
        'reversed_at',
        'reversed_by'
    ];

    protected $casts = [
        'transaction_date' => 'date',
        'amount' => 'decimal:2',
        'balance_before' => 'decimal:2',
        'balance_after' => 'decimal:2',
        'exchange_rate' => 'decimal:4',
        'reversed_at' => 'datetime'
    ];

    // Constants for transaction types
    const TRANSACTION_OPENING_BALANCE = 'Opening Balance';
    const TRANSACTION_TRANSFER_IN = 'Transfer In';
    const TRANSACTION_TRANSFER_OUT = 'Transfer Out';
    const TRANSACTION_PETTY_CASH_EXPENSE = 'Petty Cash Expense';
    const TRANSACTION_PETTY_CASH_INCOME = 'Petty Cash Income';
    const TRANSACTION_EXPENSE = 'Expense';
    const TRANSACTION_INCOME = 'Income';
    const TRANSACTION_ADJUSTMENT = 'Adjustment';
    const TRANSACTION_REVERSAL = 'Reversal';
    const TRANSACTION_OTHER = 'Other';

    // Relationships
    public function account()
    {
        return $this->belongsTo(CashBankAccount::class, 'account_id', 'cashBankId');
    }

    // ADD THIS RELATIONSHIP
    public function transfer()
    {
        return $this->belongsTo(InternalCashBankTransfer::class, 'transfer_id', 'transferid');
    }

    public function company()
    {
        return $this->belongsTo(Company::class, 'companyid');
    }

    public function branch()
    {
        return $this->belongsTo(Branch::class, 'branchid');
    }

    public function creator()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    public function updater()
    {
        return $this->belongsTo(User::class, 'updated_by');
    }

    public function reverser()
    {
        return $this->belongsTo(User::class, 'reversed_by');
    }

    public function relatedTransaction()
    {
        switch ($this->related_transaction_type) {
            case 'Internal Transfer':
                return $this->belongsTo(InternalCashBankTransfer::class, 'related_transaction_id', 'transferid');
            case 'Petty Cash':
                return $this->belongsTo(PettyCashTransaction::class, 'related_transaction_id');
            case 'Journal Entry':
                return $this->belongsTo(GeneralLedgerEntry::class, 'related_transaction_id', 'entryid');
            default:
                return null;
        }
    }

    // Scopes
    public function scopePosted($query)
    {
        return $query->where('status', 'Posted');
    }

    public function scopePending($query)
    {
        return $query->where('status', 'Pending');
    }

    public function scopeReversed($query)
    {
        return $query->where('status', 'Reversed');
    }

    public function scopeByAccount($query, $accountId)
    {
        return $query->where('account_id', $accountId);
    }

    public function scopeByDateRange($query, $startDate, $endDate)
    {
        return $query->whereBetween('transaction_date', [$startDate, $endDate]);
    }

    public function scopeByTransactionType($query, $type)
    {
        return $query->where('transaction_type', $type);
    }

    public function scopeByCompany($query, $companyId)
    {
        return $query->where('companyid', $companyId);
    }

    // ADD THIS SCOPE
    public function scopeByTransfer($query, $transferId)
    {
        return $query->where('transfer_id', $transferId);
    }

    // Helper methods
    public function isDebit()
    {
        return $this->amount > 0;
    }

    public function isCredit()
    {
        return $this->amount < 0;
    }

    public function getFormattedAmountAttribute()
    {
        return number_format(abs($this->amount), 2);
    }

    public function getAmountWithSignAttribute()
    {
        return ($this->amount >= 0 ? '+' : '') . number_format($this->amount, 2);
    }

    public function getTransactionTypeBadgeAttribute()
    {
        $badges = [
            self::TRANSACTION_OPENING_BALANCE => 'bg-blue-100 text-blue-800',
            self::TRANSACTION_TRANSFER_IN => 'bg-green-100 text-green-800',
            self::TRANSACTION_TRANSFER_OUT => 'bg-red-100 text-red-800',
            self::TRANSACTION_PETTY_CASH_EXPENSE => 'bg-amber-100 text-amber-800',
            self::TRANSACTION_PETTY_CASH_INCOME => 'bg-emerald-100 text-emerald-800',
            self::TRANSACTION_EXPENSE => 'bg-red-100 text-red-800',
            self::TRANSACTION_INCOME => 'bg-green-100 text-green-800',
            self::TRANSACTION_ADJUSTMENT => 'bg-purple-100 text-purple-800',
            self::TRANSACTION_REVERSAL => 'bg-gray-100 text-gray-800',
            self::TRANSACTION_OTHER => 'bg-gray-100 text-gray-800',
        ];

        $color = $badges[$this->transaction_type] ?? 'bg-gray-100 text-gray-800';

        return "<span class='px-2 py-1 text-xs font-medium rounded-full {$color}'>{$this->transaction_type}</span>";
    }

    // Record opening balance
    public static function recordOpeningBalance(CashBankAccount $account, $amount, $description = null)
    {
        $user = auth()->user();
        
        return self::create([
            'companyid' => $account->companyid,
            'branchid' => 1, // Default branch
            'account_id' => $account->cashBankId,
            'transaction_date' => $account->openingBalanceDate,
            'transaction_type' => self::TRANSACTION_OPENING_BALANCE,
            'reference_number' => 'OB-' . $account->cashBankId . '-' . date('Ymd'),
            'description' => $description ?? "Opening balance for {$account->accountName}",
            'amount' => $amount,
            'balance_before' => 0,
            'balance_after' => $amount,
            'currency' => $account->currency,
            'exchange_rate' => 1,
            'status' => 'Posted',
            'related_transaction_type' => 'Opening Balance',
            'related_transaction_id' => null,
            'created_by' => $user->id,
        ]);
    }

    // ADD THIS METHOD FOR INTERNAL TRANSFERS
    public static function recordInternalTransfer($transfer, $account, $transactionType, $amount, $balanceBefore, $balanceAfter)
    {
        $user = auth()->user();
        
        return self::create([
            'companyid' => $account->companyid,
            'branchid' => $transfer->branchid,
            'account_id' => $account->cashBankId,
            'transaction_date' => $transfer->transactiondate,
            'transaction_type' => $transactionType,
            'reference_number' => $transfer->reference ?? 'TRF-' . $transfer->transferid,
            'transfer_id' => $transfer->transferid, // ADDED THIS
            'description' => $transfer->details ?? ($transactionType === self::TRANSACTION_TRANSFER_IN 
                ? 'Transfer from ' . $transfer->fromAccount->accountName 
                : 'Transfer to ' . $transfer->toAccount->accountName),
            'amount' => $amount,
            'balance_before' => $balanceBefore,
            'balance_after' => $balanceAfter,
            'currency' => $transfer->currency,
            'exchange_rate' => $transfer->exchangerate,
            'status' => 'Posted',
            'related_transaction_type' => 'Internal Transfer',
            'related_transaction_id' => $transfer->transferid,
            'created_by' => $user->id,
            'updated_by' => $user->id
        ]);
    }

    // Record transaction
   
    // Reverse a transaction
    public function reverse($reason = null)
    {
        $user = auth()->user();
        
        // Create reversal entry
        $reversal = self::create([
            'companyid' => $this->companyid,
            'branchid' => $this->branchid,
            'account_id' => $this->account_id,
            'transaction_date' => now()->format('Y-m-d'),
            'transaction_type' => self::TRANSACTION_REVERSAL,
            'reference_number' => 'REV-' . $this->reference_number,
            'transfer_id' => $this->transfer_id, // ADDED THIS
            'description' => $reason ?? "Reversal of: {$this->description}",
            'amount' => -$this->amount, // Reverse the amount
            'balance_before' => $this->account->current_balance,
            'balance_after' => $this->account->current_balance - $this->amount,
            'currency' => $this->currency,
            'exchange_rate' => $this->exchange_rate,
            'status' => 'Posted',
            'related_transaction_type' => $this->related_transaction_type,
            'related_transaction_id' => $this->related_transaction_id,
            'created_by' => $user->id,
        ]);
        
        // Update original transaction
        $this->update([
            'status' => 'Reversed',
            'reversed_at' => now(),
            'reversed_by' => $user->id,
            'updated_by' => $user->id
        ]);
        
        // Update account balance
        $account = $this->account;
        $account->update(['current_balance' => $account->current_balance - $this->amount]);
        
        return $reversal;
    }
    
    /**
 * Record a transaction in the cash book
 */
public static function recordTransaction(
    $accountId,
    $amount,
    $entryType,
    $transactionType,
    $description,
    $referenceNumber,
    $companyId,
    $branchId,
    $currency
) {
    // Get the account
    $account = CashBankAccount::find($accountId);
    if (!$account) {
        throw new \Exception('Account not found');
    }

    // Calculate new balance
    $balanceBefore = $account->current_balance;
    $balanceAfter = $balanceBefore + $amount;

    // Create the cash book entry
    return self::create([
        'account_id' => $accountId,
        'transaction_date' => now(),
        'amount' => $amount,
        'entry_type' => $entryType,
        'transaction_type' => $transactionType,
        'description' => $description,
        'reference_number' => $referenceNumber,
        'balance_before' => $balanceBefore,
        'balance_after' => $balanceAfter,
        'currency' => $currency,
        'companyid' => $companyId,
        'branchid' => $branchId,
        'created_by' => auth()->id(),
        'status' => 'completed'
    ]);
}
}