<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Company extends Model
{
    use HasFactory;

    protected $fillable = [
        'owner_id',
        'name',
        'email',
        'phone',
        'address',
        'town',
        'country',
        'postal_code',
        'timezone',
        'contact_person',
        'contact_person_phone',
        'contact_person_email',
        'company_logo',
        'website',
        'industry',
        'company_size',
        'subscription_plan',
        'subscription_status',
        'trial_ends_at',
        'is_active',
        'company_type',
        'support_email',
        'auto_assign_tickets',
        'ticket_auto_close_days',
        'supportid',
        'notification_email',
        'currency',
        'reporting_currency_id',
        'current_financial_year_id'
    ];

    protected $casts = [
        'is_active' => 'boolean',
        'auto_assign_tickets' => 'boolean',
        'trial_ends_at' => 'datetime',
        'ticket_auto_close_days' => 'integer'
    ];

    // Relationships
    public function owner()
    {
        return $this->belongsTo(User::class, 'owner_id');
    }

    public function reportingCurrency()
    {
        return $this->belongsTo(Currency::class, 'reporting_currency_id');
    }

    public function currentFinancialYear()
    {
        return $this->belongsTo(FinancialYear::class, 'current_financial_year_id');
    }

    public function settingsHistory()
    {
        return $this->hasMany(CompanySettingsHistory::class);
    }

    public function users()
    {
        return $this->hasMany(User::class, 'companyid', 'id');
    }

    public function branches()
    {
        return $this->hasMany(Branch::class, 'company_id');
    }

    public function currencies()
    {
        return $this->hasMany(Currency::class, 'company_id');
    }

    public function financialYears()
    {
        return $this->hasMany(FinancialYear::class, 'company_id');
    }

    // Helper methods
    public function getFormattedAddressAttribute()
    {
        $addressParts = [];
        if ($this->address) $addressParts[] = $this->address;
        if ($this->town) $addressParts[] = $this->town;
        if ($this->postal_code) $addressParts[] = $this->postal_code;
        if ($this->country) $addressParts[] = $this->country;
        
        return implode(', ', $addressParts);
    }

  
    public function getReportingCurrencyNameAttribute()
    {
        return $this->reportingCurrency ? $this->reportingCurrency->code . ' - ' . $this->reportingCurrency->name : 'Not Set';
    }

    public function getCurrentFinancialYearNameAttribute()
    {
        return $this->currentFinancialYear ? $this->currentFinancialYear->year . ' (' . $this->currentFinancialYear->start_date->format('M Y') . ' - ' . $this->currentFinancialYear->end_date->format('M Y') . ')' : 'Not Set';
    }

    // Update reporting currency with history tracking
    public function updateReportingCurrency($currencyId, $reason = null, $changedBy = null)
    {
        $oldCurrency = $this->reportingCurrency;
        $newCurrency = Currency::find($currencyId);

        $this->reporting_currency_id = $currencyId;
        $this->save();

        // Record the change
        CompanySettingsHistory::create([
            'company_id' => $this->id,
            'setting_type' => 'reporting_currency',
            'old_value_id' => $oldCurrency ? $oldCurrency->id : null,
            'old_value_name' => $oldCurrency ? $oldCurrency->code . ' - ' . $oldCurrency->name : null,
            'new_value_id' => $newCurrency->id,
            'new_value_name' => $newCurrency->code . ' - ' . $newCurrency->name,
            'changed_by' => $changedBy ?? auth()->id(),
            'reason' => $reason
        ]);
    }

    // Update financial year with history tracking
    public function updateFinancialYear($yearId, $reason = null, $changedBy = null)
    {
        $oldYear = $this->currentFinancialYear;
        $newYear = FinancialYear::find($yearId);

        $this->current_financial_year_id = $yearId;
        $this->save();

        // Record the change
        CompanySettingsHistory::create([
            'company_id' => $this->id,
            'setting_type' => 'financial_year',
            'old_value_id' => $oldYear ? $oldYear->id : null,
            'old_value_name' => $oldYear ? $oldYear->year . ' (' . $oldYear->start_date->format('M Y') . ' - ' . $oldYear->end_date->format('M Y') . ')' : null,
            'new_value_id' => $newYear->id,
            'new_value_name' => $newYear->year . ' (' . $newYear->start_date->format('M Y') . ' - ' . $newYear->end_date->format('M Y') . ')',
            'changed_by' => $changedBy ?? auth()->id(),
            'reason' => $reason
        ]);
    }
    
    public function uploadLogo($file)
    {
        // Delete old logo if exists
        $this->deleteLogo();
        
        // Generate unique filename
        $filename = 'company-' . $this->id . '-' . time() . '.' . $file->getClientOriginalExtension();
        
        // Store the file
        $path = $file->storeAs('company-logos', $filename, 'public');
        
        // Update company record
        $this->update(['company_logo' => $path]);
        
        return $path;
    }

    /**
     * Delete logo from storage
     */
    public function deleteLogo()
    {
        if ($this->company_logo && Storage::disk('public')->exists($this->company_logo)) {
            Storage::disk('public')->delete($this->company_logo);
        }
    }

    /**
     * Get logo URL
     */
    public function getLogoUrlAttribute()
    {
        if ($this->company_logo && Storage::disk('public')->exists($this->company_logo)) {
            return asset('storage/' . $this->company_logo);
        }
        
        // Default logo
        return asset('images/default-company-logo.png');
    }

    /**
     * Get logo path
     */
    public function getLogoPathAttribute()
    {
        if ($this->company_logo) {
            return storage_path('app/public/' . $this->company_logo);
        }
        
        return null;
    }

    /**
     * Check if company has logo
     */
    public function hasLogo()
    {
        return !empty($this->company_logo) && Storage::disk('public')->exists($this->company_logo);
    }
}