<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Laravel\Sanctum\HasApiTokens;

class Customer extends Authenticatable
{
    use HasFactory, Notifiable, HasApiTokens;

    protected $table = 'customer';
    protected $primaryKey = 'id';
    public $incrementing = true;
    protected $keyType = 'int';

    protected $fillable = [
        // System Fields
        'id', 'customer_number', 'cgroup',
        
        // Personal Information
        'title', 'surname', 'first_name', 'middle_name', 'customer', 'dob',
        
        // Contact Information
        'email', 'phone', 'office_phone',
        
        // Address Information
        'town', 'address', 'province',
        
        // Professional & Employment Information
        'occupation', 'income', 'bname', 'company', 'bstartdate',
        'job_title', 'employer_name', 'employer_address', 'employer_postal_address',
        'employer_town', 'employer_province', 'employment_type', 'contract_expiry_date',
        'employee_no', 'years_in_employment', 'retirement_year',
        
        // Identification
        'idtype', 'number', 'incorporationid', 'tpin',
        
        // Documents
        'docpath', 'nrc_front_path', 'nrc_back_path',
        
        // Next of Kin Fields
        'nok_first_name', 'nok_surname', 'nok_mobile_phone', 'nok_phone', 'nok_telephone',
        'nok_physical_address', 'nok_postal_address', 'nok_town', 'nok_province',
        'nok_country', 'nok_email',
        
        // Status & Verification
        'status', 'verification_status',
        
        // API & Auth Fields
        'api_key', 'api_key_expires_at', 'otp', 'otp_expires_at', 'otp_verified',
        'otp_verified_at', 'password_reset_token', 'password_reset_expires_at',
        'password_changed_at', 'last_login_at', 'email_verified_at',
        
        // Security Fields
        'login_attempts', 'locked_until',
        
        // Meta & System Fields
        'postdate', 'pass', 'userid', 'branchid', 'companyid', 'created_at', 'updated_at'
    ];

    protected $hidden = [
        'pass', 'api_key', 'otp', 'password_reset_token',
    ];

    protected $casts = [
        'income' => 'decimal:2',
        'postdate' => 'datetime',
        'bstartdate' => 'date',
        'dob' => 'date',
        'contract_expiry_date' => 'date',
        'retirement_year' => 'integer',
        'api_key_expires_at' => 'datetime',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
        'otp_expires_at' => 'datetime',
        'otp_verified_at' => 'datetime',
        'password_reset_expires_at' => 'datetime',
        'password_changed_at' => 'datetime',
        'last_login_at' => 'datetime',
        'email_verified_at' => 'datetime',
        'locked_until' => 'datetime',
        'incorporationid' => 'integer',
        'tpin' => 'integer',
        'userid' => 'integer',
        'branchid' => 'integer',
        'companyid' => 'integer',
        'otp_verified' => 'boolean',
        'years_in_employment' => 'integer',
    ];

    // ====================== MODEL EVENTS ======================
    protected static function boot()
    {
        parent::boot();

        static::saving(function ($customer) {
            // Auto-generate full customer name
            $customer->customer = collect([
                $customer->title,
                $customer->first_name,
                $customer->middle_name,
                $customer->surname
            ])->filter()->implode(' ');

            // Set default verification status
            if (empty($customer->verification_status)) {
                $customer->verification_status = 'pending';
            }

            // Auto-generate customer number if missing (moved to saving for migration support)
            if (empty($customer->customer_number)) {
                $customer->customer_number = static::generateCustomerNumber();
            }

            // Auto-calculate years in employment if bstartdate is set
            if ($customer->bstartdate && empty($customer->years_in_employment)) {
                $startDate = \Carbon\Carbon::parse($customer->bstartdate);
                $customer->years_in_employment = $startDate->diffInYears(now());
            }

            // Auto-calculate retirement year if dob is set
            if ($customer->dob && empty($customer->retirement_year)) {
                $dob = \Carbon\Carbon::parse($customer->dob);
                $retirementAge = 60; // Default retirement age
                $retirementYear = $dob->addYears($retirementAge)->year;
                $customer->retirement_year = $retirementYear;
            }
        });

        static::creating(function ($customer) {
            // Set default status
            if (empty($customer->status)) {
                $customer->status = 'Active';
            }
        });

        static::created(function ($customer) {
            try {
                if (class_exists(\App\Services\CreditScoreService::class)) {
                    app(\App\Services\CreditScoreService::class)->initializeCreditScore($customer->id);
                }
            } catch (\Exception $e) {
                \Log::error('Credit score init failed', ['customer_id' => $customer->id]);
            }
        });
    }

    public static function generateCustomerNumber()
    {
        // Use a lock-safe approach for rapid increments during migration
        $lastCustomer = static::whereNotNull('customer_number')
                            ->orderBy('customer_number', 'desc')
                            ->first();
                            
        $newNumber = 1;
        if ($lastCustomer && preg_match('/CU(\d+)/', $lastCustomer->customer_number, $matches)) {
            $newNumber = (int)$matches[1] + 1;
        }

        return 'CU' . str_pad($newNumber, 8, '0', STR_PAD_LEFT);
    }

    // ====================== RELATIONSHIPS ======================
    public function applications() { return $this->hasMany(Application::class, 'customer_id'); }
    public function loans() { return $this->hasMany(Loan::class, 'customerid', 'id'); }
    public function disbursedLoans() { return $this->hasMany(DisbursedLoan::class, 'customerid', 'id'); }
    public function repayments() { return $this->hasMany(Repayment::class, 'customerid', 'id'); }
    public function interestIncome() { return $this->hasMany(InterestIncome::class, 'customerid', 'id'); }
    public function creditScores() { return $this->hasMany(CustomerCreditScore::class, 'customer_id'); }

    public function currentCreditScore()
    {
        return $this->hasOne(CustomerCreditScore::class, 'customer_id')
            ->where('is_current', true)
            ->latest('updated_at')
            ->withDefault();
    }

    public function creditScoreHistory()
    {
        return $this->hasMany(CustomerCreditScore::class, 'customer_id')
            ->where('is_current', false)
            ->orderBy('score_date', 'desc');
    }

    // Scopes
    public function activeDisbursedLoans() { return $this->hasMany(DisbursedLoan::class, 'customerid', 'id')->where('status', 'active'); }
    public function activeLoans() { return $this->loans()->where('status', 'active'); }
    public function submittedApplications() { return $this->applications()->where('status', 'submitted'); }
    public function approvedApplications() { return $this->applications()->where('status', 'approved'); }
    public function rejectedApplications() { return $this->applications()->where('status', 'rejected'); }
    public function pendingApplications() { return $this->applications()->whereIn('status', ['draft', 'submitted', 'under_review']); }
    public function disbursedApplications() { return $this->applications()->where('status', 'disbursed'); }
    public function activeApplications() { return $this->applications()->whereNotIn('status', ['draft', 'rejected']); }
    public function latestApplication() { return $this->hasOne(Application::class, 'customer_id')->latest(); }

    // ====================== AUTH & SECURITY ======================
    public function getAuthPassword() { return $this->pass; }
    public function routeNotificationForMail($notification = null) { return $this->email; }

    public function isValidOTP($otp): bool {
        if (!$this->otp || !$this->otp_expires_at) return false;
        return !now()->gt($this->otp_expires_at) && $this->otp === $otp;
    }

    public function generateNewOTP(): string {
        $otp = str_pad(rand(0, 999999), 6, '0', STR_PAD_LEFT);
        $this->update(['otp' => $otp, 'otp_expires_at' => now()->addMinutes(10), 'otp_verified' => false]);
        return $otp;
    }

    public function clearOTP(): void { $this->update(['otp' => null, 'otp_expires_at' => null, 'otp_verified' => false]); }

    public function generatePasswordResetToken(): string {
        $token = Str::random(60);
        $this->update(['password_reset_token' => $token, 'password_reset_expires_at' => now()->addMinutes(30)]);
        return $token;
    }

    public function isValidResetToken($token): bool {
        if (!$this->password_reset_token || !$this->password_reset_expires_at) return false;
        return !now()->gt($this->password_reset_expires_at) && $this->password_reset_token === $token;
    }

    public function clearPasswordResetToken(): void { $this->update(['password_reset_token' => null, 'password_reset_expires_at' => null]); }

    public function generateApiKey(): string {
        $apiKey = bin2hex(random_bytes(32));
        $this->api_key = hash('sha256', $apiKey);
        $this->api_key_expires_at = now()->addDays(30);
        $this->save();
        return $apiKey;
    }

    public function validateApiKey($apiKey): bool {
        return $this->api_key && $this->api_key_expires_at &&
               hash_equals($this->api_key, hash('sha256', $apiKey)) &&
               $this->api_key_expires_at->isFuture();
    }

    public function revokeApiKey(): void { $this->api_key = null; $this->api_key_expires_at = null; $this->save(); }
    public function updateLastLogin(): void { $this->update(['last_login_at' => now()]); }
    public function updatePassword($password): void { $this->update(['pass' => Hash::make($password), 'password_changed_at' => now()]); }

    // Security Methods
    public function incrementLoginAttempts(): void {
        $this->login_attempts = $this->login_attempts + 1;
        if ($this->login_attempts >= 5) {
            $this->locked_until = now()->addMinutes(30);
        }
        $this->save();
    }

    public function resetLoginAttempts(): void {
        $this->login_attempts = 0;
        $this->locked_until = null;
        $this->save();
    }

    public function isLocked(): bool {
        return $this->locked_until && now()->lt($this->locked_until);
    }

    // ====================== ATTRIBUTES & ACCESSORS ======================
   
    
    public function getFullNameAttribute() { 
       // If the 'customer' field is empty, build it from parts, with fallbacks
       return $this->customer ?: collect([
          $this->title, 
          $this->first_name ?? 'Unknown', 
          $this->middle_name, 
          $this->surname ?? 'Customer'
       ])->filter()->implode(' ');
    }

    public function getAgeAttribute() { 
        return $this->dob ? \Carbon\Carbon::parse($this->dob)->age : null; 
    }

    public function getHasProfilePictureAttribute() { 
        return $this->docpath && Storage::disk('public')->exists($this->docpath); 
    }

    public function getProfilePictureUrlAttribute() {
        if ($this->has_profile_picture) {
            return Storage::disk('public')->url($this->docpath);
        }
        return 'https://images.unsplash.com/photo-1535713875002-d1d0cf377fde?auto=format&fit=crop&w=500&q=60';
    }

    public function getCompleteAddressAttribute() { 
        return implode(', ', array_filter([$this->address, $this->town, $this->province])); 
    }

    public function getEmploymentInfoAttribute() { 
        return implode(' at ', array_filter([$this->occupation, $this->company])) ?: 'Not specified'; 
    }

    public function getStatusColorAttribute() { 
        return match($this->status) { 
            'Active' => 'green', 
            'Inactive' => 'red', 
            'Suspended' => 'orange', 
            'Pending' => 'yellow', 
            default => 'gray' 
        }; 
    }

    public function getIsVerifiedAttribute() { 
        return $this->verification_status === 'verified'; 
    }

    public function getVerificationStatusColorAttribute() { 
        return match($this->verification_status) { 
            'verified' => 'green', 
            'failed' => 'red', 
            'skipped' => 'orange', 
            default => 'gray' 
        }; 
    }

    // ====================== FINANCIAL & CREDIT SCORE METHODS ======================
    private function currentScore() {
        if ($this->relationLoaded('currentCreditScore')) return $this->getRelation('currentCreditScore');
        try { 
            $this->load('currentCreditScore'); 
            return $this->getRelation('currentCreditScore'); 
        } catch (\Exception $e) { 
            return null; 
        }
    }

    public function getSafeCurrentCreditScoreAttribute() { 
        return $this->currentScore(); 
    }

    public function getCurrentCreditScoreAttribute() { 
        return $this->currentScore()?->credit_score; 
    }

    public function getCurrentScoreBandAttribute() { 
        $score = $this->currentScore(); 
        return $score && $score->credit_score !== null ? $score->score_band : 'No History'; 
    }

    public function getCreditScoreColorAttribute() { 
        return match($this->current_score_band) { 
            'Excellent' => 'green', 
            'Good' => 'blue', 
            'Fair' => 'yellow', 
            'Poor' => 'orange', 
            'Very Poor' => 'red', 
            'No History' => 'gray', 
            default => 'gray' 
        }; 
    }

    public function getCreditScoreChangeAttribute() { 
        return $this->currentScore()?->score_change ?? 0; 
    }

    public function getCurrentCreditUtilizationAttribute() { 
        return $this->currentScore()?->credit_utilization_ratio ?? 0; 
    }

    public function getCurrentRepaymentRateAttribute() { 
        return $this->currentScore()?->on_time_repayment_rate ?? 0; 
    }

    public function getHasCreditHistoryAttribute() { 
        $score = $this->currentScore(); 
        return $score && $score->credit_score !== null && $score->credit_score > 0; 
    }

    public function getHasNoCreditHistoryAttribute() { 
        return !$this->has_credit_history; 
    }

    public function getHasGoodCreditAttribute() { 
        return $this->current_credit_score !== null && $this->current_credit_score >= 600; 
    }

    public function getHasPoorCreditAttribute() { 
        return $this->current_credit_score !== null && $this->current_credit_score < 500; 
    }

    public function getNeedsCreditAttentionAttribute() {
        $score = $this->currentScore(); 
        if (!$score || $score->credit_score === null) return false;
        return $score->credit_score < 500 || 
               ($score->has_active_default ?? false) || 
               ($score->has_recent_late_payment ?? false) || 
               ($score->credit_utilization_ratio ?? 0) > 80;
    }

    public function getCreditScoreTrendAttribute() {
        $score = $this->currentScore(); 
        if (!$score || $score->credit_score === null) return 'no_history';
        return $score->score_change > 0 ? 'improving' : ($score->score_change < 0 ? 'declining' : 'stable');
    }

    public function getCreditLimitAttribute() { 
        return $this->attributes['credit_limit'] ?? ($this->income * 3); 
    }

    public function getRecommendedMaxLoanAmountAttribute() {
        $creditLimit = $this->credit_limit; 
        $score = $this->currentScore();
        if (!$score || $score->credit_score === null) return $creditLimit * 0.3;
        $currentScore = $score->credit_score;
        return match (true) { 
            $currentScore >= 700 => $creditLimit * 0.8, 
            $currentScore >= 600 => $creditLimit * 0.6, 
            $currentScore >= 500 => $creditLimit * 0.4, 
            $currentScore >= 400 => $creditLimit * 0.2, 
            default => $creditLimit * 0.1 
        };
    }

    public function qualifiesForLoanTier($tierName) {
        $score = $this->currentScore(); 
        if (!$score || $score->credit_score === null) return $tierName === 'No History';
        return match ($tierName) { 
            'Excellent' => $score->credit_score >= 700, 
            'Good' => $score->credit_score >= 600, 
            'Fair' => $score->credit_score >= 500, 
            'Poor' => $score->credit_score >= 400, 
            'Very Poor' => $score->credit_score < 400, 
            'No History' => false, 
            default => false 
        };
    }

    public function getCreditScoreFactorsAttribute() {
        $score = $this->currentScore(); 
        if (!$score || $score->credit_score === null) return ['Limited Credit History'];
        $factors = [];
        if (($score->on_time_repayment_rate ?? 0) >= 95) $factors[] = 'Excellent Payment History';
        elseif (($score->on_time_repayment_rate ?? 0) < 80) $factors[] = 'Late Payments';
        
        if (($score->credit_utilization_ratio ?? 0) <= 30) $factors[] = 'Low Credit Utilization';
        elseif (($score->credit_utilization_ratio ?? 0) > 75) $factors[] = 'High Credit Utilization';
        
        if ($score->has_active_default ?? false) $factors[] = 'Active Default';
        if ($score->has_recent_late_payment ?? false) $factors[] = 'Recent Late Payment';
        
        return $factors ?: ['Standard Credit Profile'];
    }

    public function getCreditImprovementRecommendationsAttribute() {
        $score = $this->currentScore(); 
        if (!$score || $score->credit_score === null) return ['Build credit history with small, well-managed loans'];
        
        $recs = [];
        if (($score->on_time_repayment_rate ?? 0) < 95) $recs[] = 'Make all future payments on time';
        if (($score->credit_utilization_ratio ?? 0) > 50) $recs[] = 'Reduce credit utilization below 50%';
        if ($score->has_active_default ?? false) $recs[] = 'Resolve any outstanding defaults';
        if (($score->late_payments_count ?? 0) > 0) $recs[] = 'Avoid late payments for 6 months';
        
        return $recs ?: ['Maintain current good financial habits'];
    }

    public function getTotalNextDueAmountAttribute() { 
        return $this->activeDisbursedLoans->sum('next_due_amount'); 
    }

    public function getTotalOverdueAmountAttribute() { 
        return $this->activeDisbursedLoans->sum('overdue_amount'); 
    }

    public function getTotalDueBalanceAttribute() { 
        return $this->activeDisbursedLoans->sum('total_due_balance'); 
    }

    // ====================== STATISTICS & OVERVIEW ======================
    public function getApplicationStats() {
        return [ 
            'total' => $this->applications()->count(), 
            'submitted' => $this->submittedApplications()->count(), 
            'approved' => $this->approvedApplications()->count(), 
            'rejected' => $this->rejectedApplications()->count(), 
            'pending' => $this->pendingApplications()->count() 
        ];
    }

    public function getLoanStats() {
        $activeLoans = $this->activeDisbursedLoans()->get();
        $totalActiveLoanAmount = $activeLoans->sum('amount');
        $totalActiveLoansCount = $activeLoans->count();
        $totalExpectedRepayment = 0; 
        $totalActualRepayment = 0;
        
        foreach ($activeLoans as $loan) {
            $repay = $loan->total_repayment ?? ($loan->amount + ($loan->amount * ($loan->interestrate ?? 0) / 100));
            $paid = $loan->totalpaid ?? 0;
            $totalExpectedRepayment += $repay; 
            $totalActualRepayment += $paid;
        }
        
        $averageProgress = $totalExpectedRepayment > 0 ? 
            round(($totalActualRepayment / $totalExpectedRepayment) * 100, 2) : 0;
        
        $overdueLoans = $activeLoans->filter(fn($loan) => $loan->is_overdue);
        
        return [ 
            'active_loans_count' => $totalActiveLoansCount, 
            'total_active_loan_amount' => $totalActiveLoanAmount, 
            'average_collection_progress' => $averageProgress, 
            'total_expected_repayment' => $totalExpectedRepayment, 
            'total_actual_repayment' => $totalActualRepayment, 
            'total_outstanding_balance' => $totalExpectedRepayment - $totalActualRepayment, 
            'overdue_loans_count' => $overdueLoans->count(), 
            'active_loans' => $activeLoans, 
            'total_next_due_amount' => $this->total_next_due_amount, 
            'total_overdue_amount' => $this->total_overdue_amount, 
            'total_due_balance' => $this->total_due_balance 
        ];
    }

    public function getFinancialOverview() {
        $loanStats = $this->getLoanStats();
        $creditOverview = [ 
            'has_history' => $this->has_credit_history, 
            'score_band' => $this->current_score_band, 
            'trend' => $this->credit_score_trend 
        ];
        
        if ($this->has_credit_history) { 
            $creditOverview += [ 
                'current_score' => $this->current_credit_score, 
                'score_change' => $this->credit_score_change, 
                'utilization' => $this->current_credit_utilization, 
                'repayment_rate' => $this->current_repayment_rate, 
                'factors' => $this->credit_score_factors, 
                'recommendations' => $this->credit_improvement_recommendations 
            ]; 
        } else { 
            $creditOverview += [ 
                'message' => 'No credit history available', 
                'recommendations' => ['Build credit history with your first loan'] 
            ]; 
        }
        
        return [ 
            'credit_score' => $creditOverview, 
            'loan_portfolio' => $loanStats, 
            'credit_health' => [ 
                'has_credit_history' => $this->has_credit_history, 
                'has_good_credit' => $this->has_good_credit, 
                'needs_attention' => $this->needs_credit_attention, 
                'qualifies_for_premium' => $this->qualifiesForLoanTier('Good'), 
                'recommended_max_loan' => $this->recommended_max_loan_amount 
            ], 
            'overall_risk' => $this->calculateOverallRiskLevel() 
        ];
    }

    public function calculateOverallRiskLevel() {
        $score = $this->currentScore(); 
        if (!$score || $score->credit_score === null) return 'Moderate Risk';
        
        $riskScore = 0;
        $riskScore += match (true) { 
            $score->credit_score >= 700 => 0.1, 
            $score->credit_score >= 600 => 0.3, 
            $score->credit_score >= 500 => 0.6, 
            $score->credit_score >= 400 => 0.8, 
            default => 1.0 
        } * 0.4;
        
        $riskScore += match (true) { 
            ($score->on_time_repayment_rate ?? 0) >= 98 => 0.1, 
            ($score->on_time_repayment_rate ?? 0) >= 95 => 0.3, 
            ($score->on_time_repayment_rate ?? 0) >= 90 => 0.6, 
            ($score->on_time_repayment_rate ?? 0) >= 80 => 0.8, 
            default => 1.0 
        } * 0.3;
        
        $riskScore += match (true) { 
            ($score->credit_utilization_ratio ?? 0) <= 30 => 0.1, 
            ($score->credit_utilization_ratio ?? 0) <= 50 => 0.3, 
            ($score->credit_utilization_ratio ?? 0) <= 75 => 0.6, 
            ($score->credit_utilization_ratio ?? 0) <= 90 => 0.8, 
            default => 1.0 
        } * 0.2;
        
        $riskScore += match (true) { 
            $this->activeDisbursedLoans()->count() === 0 => 0.1, 
            $this->activeDisbursedLoans()->count() === 1 => 0.3, 
            $this->activeDisbursedLoans()->count() <= 3 => 0.6, 
            default => 0.8 
        } * 0.1;
        
        return match (true) { 
            $riskScore <= 0.3 => 'Low Risk', 
            $riskScore <= 0.6 => 'Moderate Risk', 
            $riskScore <= 0.8 => 'High Risk', 
            default => 'Very High Risk' 
        };
    }

    // ====================== NEW EMPLOYMENT ATTRIBUTES ======================
    public function getYearsToRetirementAttribute() {
        if (!$this->retirement_year) return null;
        $currentYear = now()->year;
        return max(0, $this->retirement_year - $currentYear);
    }

    public function getEmploymentStabilityAttribute() {
        if (!$this->years_in_employment) return 'Unknown';
        
        return match (true) {
            $this->years_in_employment >= 10 => 'Very Stable',
            $this->years_in_employment >= 5 => 'Stable',
            $this->years_in_employment >= 2 => 'Moderate',
            $this->years_in_employment >= 1 => 'New',
            default => 'Probationary'
        };
    }

    public function getEmploymentStabilityScoreAttribute() {
        if (!$this->years_in_employment) return 0;
        
        return match (true) {
            $this->years_in_employment >= 10 => 100,
            $this->years_in_employment >= 5 => 80,
            $this->years_in_employment >= 3 => 60,
            $this->years_in_employment >= 1 => 40,
            default => 20
        };
    }

    public function getEmploymentDetailsAttribute() {
        return [
            'employee_no' => $this->employee_no,
            'years_in_employment' => $this->years_in_employment,
            'retirement_year' => $this->retirement_year,
            'years_to_retirement' => $this->years_to_retirement,
            'employment_stability' => $this->employment_stability,
            'stability_score' => $this->employment_stability_score,
            'job_title' => $this->job_title,
            'employer' => $this->employer_name,
            'employment_type' => $this->employment_type,
            'contract_expiry' => $this->contract_expiry_date,
            'business_start_date' => $this->bstartdate
        ];
    }
    
    // Add this inside App\Models\Customer.php
    public function messages(){
         return $this->hasMany(CustomerMessage::class, 'customer_id')->latest();
    }
    
    // Inside App\Models\Customer.php

// Replace the old disbursements method with this:
public function disbursements()
{
    // In your schema, a customer HAS MANY DisbursedLoan directly
    return $this->hasMany(DisbursedLoan::class, 'customerid', 'id');
}

}