<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Carbon\Carbon;

class DisbursedLoan extends Model
{
    use HasFactory, SoftDeletes;

    protected $primaryKey = 'loanid';
    protected $table = 'disbursedloans';
    
    protected $fillable = [
        'application_id',
        'loannumber',
        'customerid',
        'loantypeid',
        'amount',
        'interestrate',
        'interest_method',
        'loanterm',
        'paymentfrequency',
        'installmentamount',
        'processing_fee',
        'processing_fee_basis',
        'adminfee',
        'insurancefee',
        'penalty_rate',
        'penalty_basis',
        'disburseddate',
        'payment_account_id',
        'payment_account_type',
        'disbursement_method',
        'bank_name',
        'account_holder',
        'account_number',
        'bank_branch',
        'swift_code',
        'mobile_channel',
        'mobile_number',
        'disbursement_notes',
        
        // Lipila Integration Fields
        'disbursement_reference',
        'lipila_status',
        'lipila_identifier',
        'lipila_callback_status',
        'lipila_callback_message',
        
        // Status and closure fields
        'status',
        'firstpaymentdate',
        'maturitydate',
        'closeddate',
        'closure_reason',
        'closure_notes',
        'closure_settlement_amount',
        'closed_by',
        
        // NPL Classification Fields
        'npl_classification_date',
        'npl_reason',
        'npl_category',
        'provision_percentage',
        'provision_amount',
        'assessment_details',
        'recovery_plan',
        'supporting_docs',
        'npl_classified_by',
        'npl_classified_at',
        
        // Balance fields
        'principalbalance',
        'interestbalance',
        'cumulative_interest_charged',
        'totalbalance',
        'totalprincipalpaid',
        'totalinterestpaid',
        'totalpenaltypaid',
        'totalpaid',
        'daysdelinquent',
        'lastpaymentdate',
        'lastpaymentamount',
        
        // Organizational fields
        'branchid',
        'companyid',
        'disbursedby',
        'createdby',
        'updatedby',
        
        // Backdating support fields (kept for tracking)
        'last_interest_calculation_date',
        'interest_calculation_status',
        
        // Processing fee fields
        'processing_fee_balance',
        'cumulative_processing_fee_charged', 
        'last_processing_fee_calculation',
        'penalty_balance',
    ];

    protected $casts = [
        'amount' => 'decimal:2',
        'interestrate' => 'decimal:2',
        'processing_fee' => 'decimal:2',
        'adminfee' => 'decimal:2',
        'insurancefee' => 'decimal:2',
        'penalty_rate' => 'decimal:2',
        'loanterm' => 'integer',
        'installmentamount' => 'decimal:2',
        'principalbalance' => 'decimal:2',
        'interestbalance' => 'decimal:2',
        'cumulative_interest_charged' => 'decimal:2',
        'totalbalance' => 'decimal:2',
        'totalprincipalpaid' => 'decimal:2',
        'totalinterestpaid' => 'decimal:2',
        'totalpenaltypaid' => 'decimal:2',
        'totalpaid' => 'decimal:2',
        'lastpaymentamount' => 'decimal:2',
        
        // Date fields
        'disburseddate' => 'date',
        'firstpaymentdate' => 'date',
        'maturitydate' => 'date',
        'closeddate' => 'date',
        'lastpaymentdate' => 'date',
        'last_interest_calculation_date' => 'date',
        'last_processing_fee_calculation' => 'date',
        
        // NPL Date Fields
        'npl_classification_date' => 'date',
        'npl_classified_at' => 'datetime',
        
        // NPL Numeric Fields
        'provision_percentage' => 'decimal:2',
        'provision_amount' => 'decimal:2',
        'closure_settlement_amount' => 'decimal:2',
        
        // Other numeric fields
        'payment_account_id' => 'integer',
        'branchid' => 'integer',
        'companyid' => 'integer',
        'disbursedby' => 'integer',
        'createdby' => 'integer',
        'updatedby' => 'integer',
        'daysdelinquent' => 'integer',
        'closed_by' => 'integer',
        'npl_classified_by' => 'integer',
        
        // String fields
        'disbursement_reference' => 'string',
        'lipila_status' => 'string',
        'lipila_identifier' => 'string',
        'lipila_callback_status' => 'string',
        'lipila_callback_message' => 'string',
        
        // Status and reason fields
        'status' => 'string',
        'closure_reason' => 'string',
        'npl_reason' => 'string',
        'npl_category' => 'string',
        'interest_method' => 'string',
        'paymentfrequency' => 'string',
        'processing_fee_basis' => 'string',
        'penalty_basis' => 'string',
        'payment_account_type' => 'string',
        'disbursement_method' => 'string',
        'mobile_channel' => 'string',
        
        // Backdating casts
        'interest_calculation_status' => 'string',
        
        // JSON and text fields
        'supporting_docs' => 'array',
        'assessment_details' => 'string',
        'recovery_plan' => 'string',
        'closure_notes' => 'string',
        'disbursement_notes' => 'string',
        
        // Processing fee fields
        'processing_fee_balance' => 'decimal:2',
        'cumulative_processing_fee_charged' => 'decimal:2',
        'penalty_balance' => 'decimal:2',
    ];

    // ================= RELATIONSHIPS =================

    public function application()
    {
        return $this->belongsTo(Application::class, 'application_id', 'id');
    }

    public function customer()
    {
        return $this->belongsTo(Customer::class, 'customerid');
    }

    public function loanType()
    {
        return $this->belongsTo(LoanType::class, 'loantypeid');
    }
    
    public function paymentAccount()
    {
        return $this->belongsTo(ChartOfAccount::class, 'payment_account_id');
    }

    public function cashBankAccount()
    {
        return $this->belongsTo(CashBankAccount::class, 'payment_account_id', 'cashBankId');
    }

    public function branch()
    {
        return $this->belongsTo(Branch::class, 'branchid');
    }

    public function company()
    {
        return $this->belongsTo(Company::class, 'companyid');
    }

    public function disbursedBy()
    {
        return $this->belongsTo(User::class, 'disbursedby');
    }

    public function creator()
    {
        return $this->belongsTo(User::class, 'createdby');
    }

    public function updater()
    {
        return $this->belongsTo(User::class, 'updatedby');
    }

    public function closedByUser()
    {
        return $this->belongsTo(User::class, 'closed_by');
    }

    public function nplClassifiedBy()
    {
        return $this->belongsTo(User::class, 'npl_classified_by');
    }

    public function repayments()
    {
        return $this->hasMany(Repayment::class, 'rloanid', 'loanid');
    }

    public function paymentSchedules()
    {
        return $this->hasMany(PaymentSchedule::class, 'loanid', 'loanid');
    }

    public function generalLedgerEntries()
    {
        return $this->hasMany(GeneralLedgerEntry::class, 'loanid', 'loanid');
    }

    public function paymentDisbursements()
    {
        return $this->hasMany(PaymentDisbursement::class, 'reference_id', 'disbursement_reference');
    }

    public function interestIncomes()
    {
        return $this->hasMany(InterestIncome::class, 'iloanid', 'loanid');
    }

    public function processingFees()
    {
        return $this->hasMany(ProcessingFee::class, 'iloanid', 'loanid');
    }

    public function journalEntries()
    {
        return $this->hasMany(JournalEntry::class, 'loanid', 'loanid');
    }

    // ================= NPL RELATED METHODS =================

    /**
     * Check if loan is classified as NPL
     */
    public function isNPL()
    {
        return $this->status === 'default';
    }

    /**
     * Get NPL classification details
     */
    public function getNPLDetails()
    {
        if (!$this->isNPL()) {
            return null;
        }

        return [
            'classification_date' => $this->npl_classification_date,
            'reason' => $this->npl_reason,
            'category' => $this->npl_category,
            'provision_percentage' => $this->provision_percentage,
            'provision_amount' => $this->provision_amount,
            'assessment_details' => $this->assessment_details,
            'recovery_plan' => $this->recovery_plan,
            'supporting_docs' => $this->supporting_docs,
            'classified_by' => $this->nplClassifiedBy ? $this->nplClassifiedBy->name : null,
            'classified_at' => $this->npl_classified_at,
            'days_since_classification' => $this->npl_classification_date ? 
                Carbon::parse($this->npl_classification_date)->diffInDays(now()) : null,
        ];
    }

    /**
     * Get provision details
     */
    public function getProvisionDetails()
    {
        return [
            'percentage' => $this->provision_percentage ?? 0,
            'amount' => $this->provision_amount ?? 0,
            'loan_balance' => $this->totalbalance,
            'net_balance_after_provision' => $this->totalbalance - ($this->provision_amount ?? 0),
            'coverage_ratio' => $this->totalbalance > 0 ? 
                (($this->provision_amount ?? 0) / $this->totalbalance) * 100 : 0,
        ];
    }

    /**
     * Get NPL category label with color
     */
    public function getNPLCategoryLabelAttribute()
    {
        $categories = [
            'substandard' => ['label' => 'Substandard', 'color' => 'bg-yellow-100 text-yellow-800'],
            'doubtful' => ['label' => 'Doubtful', 'color' => 'bg-orange-100 text-orange-800'],
            'loss' => ['label' => 'Loss', 'color' => 'bg-red-100 text-red-800'],
            'watch' => ['label' => 'Watch', 'color' => 'bg-blue-100 text-blue-800'],
        ];

        return $categories[$this->npl_category] ?? ['label' => 'Unknown', 'color' => 'bg-gray-100 text-gray-800'];
    }

    /**
     * Get NPL reason label
     */
    public function getNPLReasonLabelAttribute()
    {
        $reasons = [
            'payment_default' => 'Payment Default',
            'financial_difficulties' => 'Financial Difficulties',
            'bankruptcy' => 'Bankruptcy',
            'breach_of_covenants' => 'Breach of Covenants',
            'significant_doubt' => 'Significant Doubt of Recovery',
            'legal_issues' => 'Legal Issues',
            'death_disability' => 'Death/Disability',
            'other' => 'Other',
        ];

        return $reasons[$this->npl_reason] ?? 'Unknown';
    }

    /**
     * Generate loan number
     */
    public static function generateLoanNumber()
    {
        $prefix = 'LN';
        $year = date('Y');
        $month = date('m');
        
        $lastLoan = self::where('loannumber', 'like', "{$prefix}{$year}{$month}%")
            ->orderBy('loannumber', 'desc')
            ->first();
            
        $sequence = 1;
        if ($lastLoan) {
            $lastSequence = intval(substr($lastLoan->loannumber, -4));
            $sequence = $lastSequence + 1;
        }
        
        return $prefix . $year . $month . str_pad($sequence, 4, '0', STR_PAD_LEFT);
    }

    /**
     * Check if the loan is overdue
     */
    public function isOverdue()
    {
        // If no first payment date, can't be overdue
        if (!$this->firstpaymentdate) {
            return false;
        }
        
        $now = now();
        $firstPaymentDate = Carbon::parse($this->firstpaymentdate);
        
        // If first payment date hasn't arrived yet, loan can't be overdue
        if ($now->lessThan($firstPaymentDate)) {
            return false;
        }
        
        // Check if there are any payment schedules
        if (!$this->paymentSchedules()->exists()) {
            return false;
        }
        
        // Check for any overdue payment schedules
        return $this->paymentSchedules()
            ->where(function($query) use ($now) {
                $query->where('status', 'overdue')
                      ->orWhere(function($q) use ($now) {
                          $q->where('status', 'scheduled')
                            ->where('paymentdate', '<', $now);
                      });
            })
            ->where(function($query) {
                // Calculate balance as totalamount - total_paid
                $query->whereRaw('(totalamount - COALESCE(total_paid, 0)) > 0')
                      ->orWhereNull('totalamount');
            })
            ->exists();
    }

    /**
     * Get days overdue
     */
    public function getDaysOverdueAttribute()
    {
        if (!$this->isOverdue()) {
            return 0;
        }
        
        // Get the earliest overdue payment date
        $earliestOverdue = $this->paymentSchedules()
            ->where(function($query) {
                $query->where('status', 'overdue')
                      ->orWhere('paymentdate', '<', now());
            })
            ->where(function($query) {
                // Calculate balance as totalamount - total_paid
                $query->whereRaw('(totalamount - COALESCE(total_paid, 0)) > 0')
                      ->orWhereNull('totalamount');
            })
            ->orderBy('paymentdate', 'asc')
            ->first();
        
        if (!$earliestOverdue) {
            return 0;
        }
        
        return now()->diffInDays(Carbon::parse($earliestOverdue->paymentdate));
    }

    /**
     * Get the penalty balance attribute with fallback
     */
    public function getPenaltyBalanceAttribute()
    {
        if (isset($this->attributes['penalty_balance'])) {
            return $this->attributes['penalty_balance'];
        }
        
        // Calculate from repayments if penalty balance field doesn't exist
        $penaltyCharged = $this->repayments()->sum('penalties');
        return max(0, $penaltyCharged);
    }

    /**
     * Get the processing fee balance attribute with fallback
     */
    public function getProcessingFeeBalanceAttribute()
    {
        if (isset($this->attributes['processing_fee_balance'])) {
            return $this->attributes['processing_fee_balance'];
        }
        
        // Calculate if field doesn't exist
        $processingFeePaid = $this->repayments()->sum('processing_fees_amount_paid');
        $processingFeeCharged = $this->processing_fee ?? 0;
        return max(0, $processingFeeCharged - $processingFeePaid);
    }

    /**
     * Get total processing fees charged to date
     */
    public function getTotalProcessingFeesChargedAttribute()
    {
        return $this->processingFees()->sum('charge');
    }

    /**
     * Get processing fees charged this month
     */
    public function getProcessingFeesThisMonthAttribute()
    {
        $startOfMonth = Carbon::now()->startOfMonth();
        $endOfMonth = Carbon::now()->endOfMonth();
        
        return $this->processingFees()
            ->whereBetween('chargeddate', [$startOfMonth, $endOfMonth])
            ->sum('charge');
    }

    /**
     * Calculate and update total balance correctly
     * This ensures processing fees are included in the total balance
     */
    public function recalculateTotalBalance()
    {
        $principalBalance = (float) $this->principalbalance;
        $interestBalance = (float) $this->interestbalance;
        $processingFeeBalance = (float) ($this->processing_fee_balance ?? 0);
        
        $principalPlusInterest = bcadd($principalBalance, $interestBalance, 2);
        $totalBalance = bcadd($principalPlusInterest, $processingFeeBalance, 2);
        
        $this->totalbalance = $totalBalance;
        
        return $totalBalance;
    }

    /**
     * Get total balance due attribute (including processing fees)
     */
    public function getTotalBalanceDueAttribute()
    {
        return $this->recalculateTotalBalance();
    }

    /**
     * Get balance breakdown
     */
    public function getBalanceBreakdownAttribute()
    {
        return [
            'principal' => (float) $this->principalbalance,
            'interest' => (float) $this->interestbalance,
            'processing_fees' => (float) ($this->processing_fee_balance ?? 0),
            'total' => (float) $this->recalculateTotalBalance(),
        ];
    }

    /**
     * Verify that total balance is correctly calculated
     */
    public function verifyTotalBalanceCalculation()
    {
        $principalBalance = (float) $this->principalbalance;
        $interestBalance = (float) $this->interestbalance;
        $processingFeeBalance = (float) ($this->processing_fee_balance ?? 0);
        $currentTotalBalance = (float) $this->totalbalance;
        
        $calculatedTotalBalance = bcadd(
            bcadd($principalBalance, $interestBalance, 2),
            $processingFeeBalance,
            2
        );
        
        $balanceMatch = abs($currentTotalBalance - $calculatedTotalBalance) < 0.01;
        
        return [
            'loan_number' => $this->loannumber,
            'field_values' => [
                'principal_balance' => $principalBalance,
                'interest_balance' => $interestBalance,
                'processing_fee_balance' => $processingFeeBalance,
                'current_total_balance' => $currentTotalBalance,
            ],
            'calculated_total_balance' => $calculatedTotalBalance,
            'balance_match' => $balanceMatch,
            'discrepancy' => $balanceMatch ? 0 : $calculatedTotalBalance - $currentTotalBalance,
            'verification_passed' => $balanceMatch,
        ];
    }

    /**
     * Fix total balance calculation for this loan
     */
    public function fixTotalBalance()
    {
        $verification = $this->verifyTotalBalanceCalculation();
        
        if (!$verification['verification_passed']) {
            $correctTotalBalance = $verification['calculated_total_balance'];
            $oldTotalBalance = $this->totalbalance;
            
            $this->totalbalance = $correctTotalBalance;
            $this->save();
            
            return [
                'success' => true,
                'message' => 'Total balance fixed',
                'old_total_balance' => $oldTotalBalance,
                'new_total_balance' => $correctTotalBalance,
                'difference' => $correctTotalBalance - $oldTotalBalance,
            ];
        }
        
        return [
            'success' => true,
            'message' => 'Total balance already correct',
            'total_balance' => $this->totalbalance,
        ];
    }

    // ================= BOOT METHOD =================

    /**
     * Boot method for model events
     */
    protected static function boot()
    {
        parent::boot();

        static::creating(function ($loan) {
            if (empty($loan->loannumber)) {
                $loan->loannumber = self::generateLoanNumber();
            }
            
            // Set initial balances if not set
            if (empty($loan->principalbalance)) {
                $loan->principalbalance = $loan->amount;
            }
            if (empty($loan->cumulative_interest_charged)) {
                $loan->cumulative_interest_charged = 0;
            }
            if (empty($loan->cumulative_processing_fee_charged)) {
                $loan->cumulative_processing_fee_charged = 0;
            }
            if (empty($loan->processing_fee_balance)) {
                $loan->processing_fee_balance = 0;
            }
            
            // Calculate initial total balance including processing fees
            $principalBalance = (float) $loan->principalbalance;
            $processingFeeBalance = (float) ($loan->processing_fee_balance ?? 0);
            $initialTotalBalance = bcadd($principalBalance, $processingFeeBalance, 2);
            
            if (empty($loan->totalbalance)) {
                $loan->totalbalance = $initialTotalBalance;
            }
            
            if (empty($loan->status)) {
                $loan->status = 'active';
            }
            if (empty($loan->interest_calculation_status)) {
                $loan->interest_calculation_status = 'pending';
            }
        });

        static::updating(function ($loan) {
            // Update total balance whenever principal, interest, or processing fee balance changes
            // Total Balance = Principal Balance + Interest Balance + Processing Fee Balance
            $principalBalance = (float) $loan->principalbalance;
            $interestBalance = (float) $loan->interestbalance;
            $processingFeeBalance = (float) ($loan->processing_fee_balance ?? 0);
            
            // Calculate total balance with precise decimal arithmetic
            $principalPlusInterest = bcadd($principalBalance, $interestBalance, 2);
            $totalBalance = bcadd($principalPlusInterest, $processingFeeBalance, 2);
            
            $loan->totalbalance = $totalBalance;
        });
    }

    // ================= ADDITIONAL HELPER METHODS =================

    /**
     * Get comprehensive loan summary
     */
    public function getLoanSummary()
    {
        return [
            'loan_info' => [
                'loan_number' => $this->loannumber,
                'customer_id' => $this->customerid,
                'loan_type' => $this->loanType ? $this->loanType->product : 'N/A',
                'amount' => $this->amount,
                'interest_rate' => $this->interestrate,
                'disbursement_date' => $this->disburseddate,
                'maturity_date' => $this->maturitydate,
                'status' => $this->status,
            ],
            'balance_info' => [
                'principal_balance' => $this->principalbalance,
                'interest_balance' => $this->interestbalance,
                'processing_fee_balance' => $this->processing_fee_balance ?? 0,
                'total_balance' => $this->recalculateTotalBalance(),
                'cumulative_interest_charged' => $this->cumulative_interest_charged,
                'cumulative_processing_fee_charged' => $this->cumulative_processing_fee_charged ?? 0,
            ],
            'payment_info' => [
                'total_principal_paid' => $this->totalprincipalpaid,
                'total_interest_paid' => $this->totalinterestpaid,
                'total_processing_fee_paid' => $this->repayments()->sum('processing_fees_amount_paid'),
                'total_paid' => $this->totalpaid,
                'last_payment_date' => $this->lastpaymentdate,
                'last_payment_amount' => $this->lastpaymentamount,
            ],
            'calculation_info' => [
                'last_interest_calculation_date' => $this->last_interest_calculation_date,
                'last_processing_fee_calculation' => $this->last_processing_fee_calculation,
                'interest_calculation_status' => $this->interest_calculation_status,
            ],
            'npl_info' => $this->isNPL() ? $this->getNPLDetails() : null,
            'verification' => $this->verifyTotalBalanceCalculation(),
        ];
    }

    /**
     * Get total outstanding balance including all components
     */
    public function getOutstandingBalance()
    {
        return $this->recalculateTotalBalance();
    }

    /**
     * Get amount due for next payment
     */
    public function getNextPaymentAmount()
    {
        // This would typically come from payment schedules
        // For now, return the installment amount if set
        return $this->installmentamount ?? 0;
    }

    /**
     * Check if loan is fully paid
     */
    public function isFullyPaid()
    {
        $outstandingBalance = $this->getOutstandingBalance();
        return $outstandingBalance <= 0.01; // Allow small rounding differences
    }

    /**
     * Get days since disbursement
     */
    public function getDaysSinceDisbursementAttribute()
    {
        $disbursedDate = Carbon::parse($this->disburseddate);
        $today = Carbon::today();
        
        return $disbursedDate->diffInDays($today);
    }

    /**
     * Get days to maturity
     */
    public function getDaysToMaturityAttribute()
    {
        if (!$this->maturitydate) {
            return null;
        }
        
        $maturityDate = Carbon::parse($this->maturitydate);
        $today = Carbon::today();
        
        if ($today->gt($maturityDate)) {
            return 0; // Already matured
        }
        
        return $today->diffInDays($maturityDate);
    }
}