<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class InternalCashBankTransfer extends Model
{
    use HasFactory;

    protected $primaryKey = 'transferid';
    
    // Specify the table name explicitly
    protected $table = 'internal_cash_bank_transfers';
    
    protected $fillable = [
        'transactiondate',
        'accountfrom',
        'accountto',
        'details',
        'currency',
        'exchangerate',
        'amountsent',
        'amountreceived',
        'reference',
        'referenceno',
        'createdby',
        'companyid',
        'branchid',
        'transferstatus',
        'sent_at',
        'sent_by'
    ];

    protected $casts = [
        'transactiondate' => 'date',
        'exchangerate' => 'decimal:4',
        'amountsent' => 'decimal:2',
        'amountreceived' => 'decimal:2',
        'sent_at' => 'datetime',
    ];

    /**
     * Get the source account
     */
    public function fromAccount(): BelongsTo
    {
        return $this->belongsTo(CashBankAccount::class, 'accountfrom', 'cashBankId');
    }

    /**
     * Get the destination account
     */
    public function toAccount(): BelongsTo
    {
        return $this->belongsTo(CashBankAccount::class, 'accountto', 'cashBankId');
    }

    /**
     * Get the creator
     */
    public function creator(): BelongsTo
    {
        return $this->belongsTo(User::class, 'createdby');
    }

    /**
     * Get the user who sent the transfer
     */
    public function sentByUser(): BelongsTo
    {
        return $this->belongsTo(User::class, 'sent_by');
    }

    /**
     * Get the company
     */
    public function company(): BelongsTo
    {
        return $this->belongsTo(Company::class, 'companyid', 'id');
    }

    /**
     * Get the branch
     */
    public function branch(): BelongsTo
    {
        return $this->belongsTo(Branch::class, 'branchid', 'id');
    }

    /**
     * Get cashbook entries for this transfer
     */
    public function cashbookEntries()
    {
        return $this->hasMany(CashBook::class, 'transfer_id', 'transferid');
    }

    /**
     * Get general ledger entries for this transfer
     */
    public function ledgerEntries()
    {
        return $this->hasMany(GeneralLedgerEntry::class, 'reference_id', 'transferid')
            ->where('reference_type', 'internal_transfer');
    }

    /**
     * Scope for pending transfers
     */
    public function scopePending($query)
    {
        return $query->where('transferstatus', 'Pending');
    }

    /**
     * Scope for completed transfers
     */
    public function scopeCompleted($query)
    {
        return $query->where('transferstatus', 'Sent');
    }

    /**
     * Get formatted amount sent
     */
    public function getFormattedAmountSentAttribute()
    {
        return number_format($this->amountsent, 2);
    }

    /**
     * Get formatted amount received
     */
    public function getFormattedAmountReceivedAttribute()
    {
        return number_format($this->amountreceived, 2);
    }

    /**
     * Get formatted exchange rate
     */
    public function getFormattedExchangeRateAttribute()
    {
        return number_format($this->exchangerate, 4);
    }

    /**
     * Check if transfer has been sent
     */
    public function isSent()
    {
        return $this->transferstatus === 'Sent';
    }

    /**
     * Check if transfer is pending
     */
    public function isPending()
    {
        return $this->transferstatus === 'Pending';
    }
}