<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Storage;

class LoanApplicationDocument extends Model
{
    use HasFactory;

    /**
     * The table associated with the model.
     *
     * @var string
     */
    protected $table = 'loanapplicationdocuments';

    /**
     * The primary key associated with the table.
     *
     * @var string
     */
    protected $primaryKey = 'id';

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'application_id',
        'requirement_id',
        'file_path',
        'file_name',
        'file_size',
        'file_type',
        'uploaded_by',
        'status',
        'notes',
        'verified_by',
        'verified_at',
        'rejected_by', // ADDED THIS
        'rejected_at', // ADDED THIS
        'rejection_reason',
        'created_at',
        'updated_at'
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'verified_at' => 'datetime',
        'rejected_at' => 'datetime', // ADDED THIS
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
        'file_size' => 'integer',
        'verified_by' => 'integer',
        'rejected_by' => 'integer', // ADDED THIS
        'uploaded_by' => 'integer',
        'application_id' => 'integer',
        'requirement_id' => 'integer',
    ];

    /**
     * The model's default values for attributes.
     *
     * @var array
     */
    protected $attributes = [
        'status' => 'pending',
    ];

    /**
     * Indicates if the model should be timestamped.
     *
     * @var bool
     */
    public $timestamps = true;

    /**
     * Get the application that owns the document.
     */
    public function application()
    {
        return $this->belongsTo(Application::class, 'application_id');
    }

    /**
     * Get the requirement that this document fulfills.
     */
    public function requirement()
    {
        return $this->belongsTo(ApplicationRequirement::class, 'requirement_id');
    }

    /**
     * Get the user who uploaded the document.
     */
    public function uploadedBy()
    {
        return $this->belongsTo(User::class, 'uploaded_by');
    }

    /**
     * Get the staff who verified the document.
     */
    public function verifiedBy()
    {
        return $this->belongsTo(User::class, 'verified_by');
    }

    /**
     * Get the staff who rejected the document.
     */
    public function rejectedBy()
    {
        return $this->belongsTo(User::class, 'rejected_by');
    }

    /**
     * Get the customer who owns the application (through application).
     */
    public function customer()
    {
        return $this->hasOneThrough(
            Customer::class,
            Application::class,
            'id', // Foreign key on applications table
            'id', // Foreign key on customers table
            'application_id', // Local key on loanapplicationdocuments table
            'customer_id' // Local key on applications table
        );
    }

    /**
     * Check if document is pending verification.
     *
     * @return bool
     */
    public function isPending(): bool
    {
        return $this->status === 'pending';
    }

    /**
     * Check if document is verified.
     *
     * @return bool
     */
    public function isVerified(): bool
    {
        return $this->status === 'verified';
    }

    /**
     * Check if document is rejected.
     *
     * @return bool
     */
    public function isRejected(): bool
    {
        return $this->status === 'rejected';
    }

    /**
     * Get the full URL to the document.
     *
     * @return string|null
     */
    public function getFileUrlAttribute(): ?string
    {
        if ($this->file_path && Storage::disk('public')->exists($this->file_path)) {
            return Storage::disk('public')->url($this->file_path);
        }
        return null;
    }

    /**
     * Get the full path to the document.
     *
     * @return string|null
     */
    public function getFullPathAttribute(): ?string
    {
        if ($this->file_path) {
            return Storage::disk('public')->path($this->file_path);
        }
        return null;
    }

    /**
     * Get formatted file size.
     *
     * @return string
     */
    public function getFormattedFileSizeAttribute(): string
    {
        $bytes = $this->file_size;
        
        if ($bytes >= 1073741824) {
            return number_format($bytes / 1073741824, 2) . ' GB';
        } elseif ($bytes >= 1048576) {
            return number_format($bytes / 1048576, 2) . ' MB';
        } elseif ($bytes >= 1024) {
            return number_format($bytes / 1024, 2) . ' KB';
        } elseif ($bytes > 1) {
            return $bytes . ' bytes';
        } elseif ($bytes == 1) {
            return $bytes . ' byte';
        } else {
            return '0 bytes';
        }
    }

    /**
     * Get the file extension.
     *
     * @return string|null
     */
    public function getFileExtensionAttribute(): ?string
    {
        if ($this->file_name) {
            return pathinfo($this->file_name, PATHINFO_EXTENSION);
        }
        return null;
    }

    /**
     * Get the file name without extension.
     *
     * @return string|null
     */
    public function getFileNameWithoutExtensionAttribute(): ?string
    {
        if ($this->file_name) {
            return pathinfo($this->file_name, PATHINFO_FILENAME);
        }
        return null;
    }

    /**
     * Get file icon based on file type.
     *
     * @return string
     */
    public function getFileIconAttribute(): string
    {
        $extension = strtolower($this->file_extension);
        
        $icons = [
            'pdf' => 'fas fa-file-pdf',
            'doc' => 'fas fa-file-word',
            'docx' => 'fas fa-file-word',
            'xls' => 'fas fa-file-excel',
            'xlsx' => 'fas fa-file-excel',
            'jpg' => 'fas fa-file-image',
            'jpeg' => 'fas fa-file-image',
            'png' => 'fas fa-file-image',
            'gif' => 'fas fa-file-image',
            'txt' => 'fas fa-file-alt',
            'csv' => 'fas fa-file-csv',
        ];
        
        return $icons[$extension] ?? 'fas fa-file';
    }

    /**
     * Get file icon color based on file type.
     *
     * @return string
     */
    public function getFileIconColorAttribute(): string
    {
        $extension = strtolower($this->file_extension);
        
        $colors = [
            'pdf' => 'text-red-500',
            'doc' => 'text-blue-500',
            'docx' => 'text-blue-500',
            'xls' => 'text-green-500',
            'xlsx' => 'text-green-500',
            'jpg' => 'text-purple-500',
            'jpeg' => 'text-purple-500',
            'png' => 'text-purple-500',
            'gif' => 'text-purple-500',
        ];
        
        return $colors[$extension] ?? 'text-gray-500';
    }

    /**
     * Scope a query to only include pending documents.
     *
     * @param  \Illuminate\Database\Eloquent\Builder  $query
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopePending($query)
    {
        return $query->where('status', 'pending');
    }

    /**
     * Scope a query to only include verified documents.
     *
     * @param  \Illuminate\Database\Eloquent\Builder  $query
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeVerified($query)
    {
        return $query->where('status', 'verified');
    }

    /**
     * Scope a query to only include rejected documents.
     *
     * @param  \Illuminate\Database\Eloquent\Builder  $query
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeRejected($query)
    {
        return $query->where('status', 'rejected');
    }

    /**
     * Scope a query to only include documents for a specific application.
     *
     * @param  \Illuminate\Database\Eloquent\Builder  $query
     * @param  int  $applicationId
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeForApplication($query, $applicationId)
    {
        return $query->where('application_id', $applicationId);
    }

    /**
     * Scope a query to only include documents for a specific requirement.
     *
     * @param  \Illuminate\Database\Eloquent\Builder  $query
     * @param  int  $requirementId
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeForRequirement($query, $requirementId)
    {
        return $query->where('requirement_id', $requirementId);
    }

    /**
     * Scope a query to only include documents uploaded by a specific user.
     *
     * @param  \Illuminate\Database\Eloquent\Builder  $query
     * @param  int  $userId
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeUploadedBy($query, $userId)
    {
        return $query->where('uploaded_by', $userId);
    }

    /**
     * Scope a query to only include documents verified by a specific user.
     *
     * @param  \Illuminate\Database\Eloquent\Builder  $query
     * @param  int  $userId
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeVerifiedBy($query, $userId)
    {
        return $query->where('verified_by', $userId);
    }

    /**
     * Scope a query to only include documents rejected by a specific user.
     *
     * @param  \Illuminate\Database\Eloquent\Builder  $query
     * @param  int  $userId
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeRejectedBy($query, $userId)
    {
        return $query->where('rejected_by', $userId);
    }

    /**
     * Scope a query to only include documents with notes.
     *
     * @param  \Illuminate\Database\Eloquent\Builder  $query
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeWithNotes($query)
    {
        return $query->whereNotNull('notes')->where('notes', '!=', '');
    }

    /**
     * Scope a query to only include documents without notes.
     *
     * @param  \Illuminate\Database\Eloquent\Builder  $query
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeWithoutNotes($query)
    {
        return $query->whereNull('notes')->orWhere('notes', '');
    }

    /**
     * Verify the document.
     *
     * @param  int  $verifiedBy
     * @param  string|null  $notes
     * @return bool
     */
    public function verify(int $verifiedBy, ?string $notes = null): bool
    {
        $this->status = 'verified';
        $this->verified_by = $verifiedBy;
        $this->verified_at = now();
        $this->rejection_reason = null; // Clear rejection reason
        $this->rejected_by = null; // Clear rejected by
        $this->rejected_at = null; // Clear rejected at
        
        if ($notes) {
            $this->notes = $notes;
        }
        
        return $this->save();
    }

    /**
     * Reject the document.
     *
     * @param  int  $rejectedBy
     * @param  string  $rejectionReason
     * @param  string|null  $notes
     * @return bool
     */
    public function reject(int $rejectedBy, string $rejectionReason, ?string $notes = null): bool
    {
        $this->status = 'rejected';
        $this->rejected_by = $rejectedBy;
        $this->rejected_at = now();
        $this->rejection_reason = $rejectionReason;
        $this->verified_by = null; // Clear verified by
        $this->verified_at = null; // Clear verified at
        
        if ($notes) {
            $this->notes = $notes;
        }
        
        return $this->save();
    }

    /**
     * Mark document as pending (reset verification).
     *
     * @return bool
     */
    public function markAsPending(): bool
    {
        $this->status = 'pending';
        $this->verified_by = null;
        $this->verified_at = null;
        $this->rejected_by = null;
        $this->rejected_at = null;
        $this->rejection_reason = null;
        
        return $this->save();
    }

    /**
     * Update document notes.
     *
     * @param  string  $notes
     * @return bool
     */
    public function updateNotes(string $notes): bool
    {
        $this->notes = $notes;
        return $this->save();
    }

    /**
     * Check if the file exists in storage.
     *
     * @return bool
     */
    public function fileExists(): bool
    {
        if ($this->file_path) {
            return Storage::disk('public')->exists($this->file_path);
        }
        return false;
    }

    /**
     * Delete the file from storage.
     *
     * @return bool
     */
    public function deleteFile(): bool
    {
        if ($this->file_path && Storage::disk('public')->exists($this->file_path)) {
            return Storage::disk('public')->delete($this->file_path);
        }
        return false;
    }

    /**
     * Delete the document and its file.
     *
     * @return bool|null
     */
    public function deleteWithFile(): ?bool
    {
        $this->deleteFile();
        return $this->delete();
    }

    /**
     * Get all document statuses.
     *
     * @return array
     */
    public static function getStatuses(): array
    {
        return [
            'pending' => 'Pending',
            'verified' => 'Verified',
            'rejected' => 'Rejected',
        ];
    }

    /**
     * Get status text.
     *
     * @return string
     */
    public function getStatusTextAttribute(): string
    {
        $statuses = self::getStatuses();
        return $statuses[$this->status] ?? ucfirst($this->status);
    }

    /**
     * Get status badge HTML.
     *
     * @return string
     */
    public function getStatusBadgeAttribute(): string
    {
        $badges = [
            'pending' => '<span class="badge bg-yellow-100 text-yellow-800">Pending</span>',
            'verified' => '<span class="badge bg-green-100 text-green-800">Verified</span>',
            'rejected' => '<span class="badge bg-red-100 text-red-800">Rejected</span>',
        ];
        
        return $badges[$this->status] ?? '<span class="badge bg-gray-100 text-gray-800">' . ucfirst($this->status) . '</span>';
    }

    /**
     * Get human readable time since uploaded.
     *
     * @return string
     */
    public function getUploadedAgoAttribute(): string
    {
        return $this->created_at ? $this->created_at->diffForHumans() : 'N/A';
    }

    /**
     * Get human readable time since verified.
     *
     * @return string
     */
    public function getVerifiedAgoAttribute(): string
    {
        return $this->verified_at ? $this->verified_at->diffForHumans() : 'N/A';
    }

    /**
     * Check if document can be verified.
     *
     * @return bool
     */
    public function canBeVerified(): bool
    {
        return $this->isPending();
    }

    /**
     * Check if document can be rejected.
     *
     * @return bool
     */
    public function canBeRejected(): bool
    {
        return $this->isPending();
    }

    /**
     * Check if document can be deleted.
     *
     * @return bool
     */
    public function canBeDeleted(): bool
    {
        return $this->isPending() || $this->isRejected();
    }

    /**
     * Get verification details.
     *
     * @return array
     */
    public function getVerificationDetailsAttribute(): array
    {
        return [
            'status' => $this->status,
            'verified_by' => $this->verified_by,
            'verified_by_name' => $this->verifiedBy ? $this->verifiedBy->name : 'Unknown',
            'verified_at' => $this->verified_at ? $this->verified_at->format('Y-m-d H:i:s') : null,
            'verified_at_formatted' => $this->verified_at ? $this->verified_at->format('M d, Y H:i') : 'Not Verified',
            'rejected_by' => $this->rejected_by,
            'rejected_by_name' => $this->rejectedBy ? $this->rejectedBy->name : null,
            'rejected_at' => $this->rejected_at ? $this->rejected_at->format('Y-m-d H:i:s') : null,
            'rejection_reason' => $this->rejection_reason,
            'notes' => $this->notes,
        ];
    }

    /**
     * Get document details for API response.
     *
     * @return array
     */
    public function toApiResponse(): array
    {
        return [
            'id' => $this->id,
            'application_id' => $this->application_id,
            'requirement_id' => $this->requirement_id,
            'requirement_name' => $this->requirement ? $this->requirement->documentname : 'Unknown',
            'file_name' => $this->file_name,
            'file_size' => $this->file_size,
            'file_size_formatted' => $this->formatted_file_size,
            'file_url' => $this->file_url,
            'status' => $this->status,
            'status_text' => $this->status_text,
            'notes' => $this->notes,
            'uploaded_by' => $this->uploaded_by,
            'uploaded_by_name' => $this->uploadedBy ? $this->uploadedBy->name : 'Unknown',
            'verified_by' => $this->verified_by,
            'verified_by_name' => $this->verifiedBy ? $this->verifiedBy->name : null,
            'verified_at' => $this->verified_at ? $this->verified_at->toISOString() : null,
            'rejected_by' => $this->rejected_by,
            'rejected_by_name' => $this->rejectedBy ? $this->rejectedBy->name : null,
            'rejected_at' => $this->rejected_at ? $this->rejected_at->toISOString() : null,
            'rejection_reason' => $this->rejection_reason,
            'created_at' => $this->created_at->toISOString(),
            'updated_at' => $this->updated_at->toISOString(),
            'file_extension' => $this->file_extension,
            'file_icon' => $this->file_icon,
            'file_icon_color' => $this->file_icon_color,
        ];
    }

    /**
     * Get statistics for documents.
     *
     * @param  int|null  $applicationId
     * @return array
     */
    public static function getStatistics($applicationId = null): array
    {
        $query = self::query();
        
        if ($applicationId) {
            $query->where('application_id', $applicationId);
        }
        
        $total = $query->count();
        $pending = $query->clone()->pending()->count();
        $verified = $query->clone()->verified()->count();
        $rejected = $query->clone()->rejected()->count();
        
        return [
            'total' => $total,
            'pending' => $pending,
            'verified' => $verified,
            'rejected' => $rejected,
            'pending_percentage' => $total > 0 ? round(($pending / $total) * 100, 2) : 0,
            'verified_percentage' => $total > 0 ? round(($verified / $total) * 100, 2) : 0,
            'rejected_percentage' => $total > 0 ? round(($rejected / $total) * 100, 2) : 0,
        ];
    }

    /**
     * Scope a query to only include documents uploaded today.
     *
     * @param  \Illuminate\Database\Eloquent\Builder  $query
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeUploadedToday($query)
    {
        return $query->whereDate('created_at', today());
    }

    /**
     * Scope a query to only include documents uploaded this week.
     *
     * @param  \Illuminate\Database\Eloquent\Builder  $query
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeUploadedThisWeek($query)
    {
        return $query->whereBetween('created_at', [
            now()->startOfWeek(),
            now()->endOfWeek()
        ]);
    }

    /**
     * Scope a query to only include documents uploaded this month.
     *
     * @param  \Illuminate\Database\Eloquent\Builder  $query
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeUploadedThisMonth($query)
    {
        return $query->whereMonth('created_at', now()->month)
                     ->whereYear('created_at', now()->year);
    }
}