<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class LoanType extends Model
{
    use HasFactory;

    /**
     * The table associated with the model.
     * @var string
     */
    protected $table = 'loantype';

    /**
     * The attributes that are mass assignable.
     * Includes interest rates, fees, and penalty configurations.
     * @var array
     */
    protected $fillable = [
        'product',
        'details',
        'status',
        'prate',
        'dailyrate',
        'weeklyrate',
        'monthlyrate',
        'yearlyrate',
        'interest_method',      // NEW: simple_interest or reducing_balance
        'adminfee',
        'insurancefee',
        'processing_fee',
        'processing_fee_basis',
        'penalty_rate',
        'penalty_basis',
        'mouquestion',
        'userid',
        'branchid',
        'companyid',
    ];

    /**
     * The attributes that should be cast to native types.
     * @var array
     */
    protected $casts = [
        'prate' => 'decimal:2',
        'dailyrate' => 'decimal:2',
        'weeklyrate' => 'decimal:2',
        'monthlyrate' => 'decimal:2',
        'yearlyrate' => 'decimal:2',
        'adminfee' => 'decimal:2',
        'insurancefee' => 'decimal:2',
        'processing_fee' => 'decimal:2',
        'penalty_rate' => 'decimal:2',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
    ];

    // ====================== RELATIONSHIPS ======================

    /**
     * Get the applications associated with this loan product.
     */
    public function applications()
    {
        return $this->hasMany(Application::class, 'product_id');
    }

    /**
     * Get the loans associated with this loan product.
     */
    public function loans()
    {
        return $this->hasMany(Loan::class, 'productid');
    }

    /**
     * Get the application requirements for the loan type.
     */
    public function requirements()
    {
        return $this->belongsToMany(
            ApplicationRequirement::class,
            'product_requirements',
            'product_id',
            'requirement_id'
        )->withPivot('is_mandatory');
    }

    /**
     * Get the credit score tiers for the loan type.
     */
    public function creditScoreTiers()
    {
        return $this->belongsToMany(
            CreditScoreTier::class,
            'product_credit_score_tier',
            'product_id',
            'credit_score_tier_id'
        )->withPivot('min_amount', 'max_amount', 'daily_rate', 'monthly_rate', 'annual_rate', 'is_active');
    }

    // ====================== ATTRIBUTE ACCESSORS ======================

    /**
     * Get the total count of assigned requirements.
     */
    public function getRequirementsCountAttribute()
    {
        return $this->requirements()->count();
    }

    /**
     * Get the count of mandatory requirements.
     */
    public function getMandatoryRequirementsCountAttribute()
    {
        return $this->requirements()->wherePivot('is_mandatory', 1)->count();
    }

    /**
     * Get the count of optional requirements.
     */
    public function getOptionalRequirementsCountAttribute()
    {
        return $this->requirements()->wherePivot('is_mandatory', 0)->count();
    }
}