<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class PayrollAdjustment extends Model
{
    use HasFactory;

    protected $fillable = [
        'company_id',
        'payroll_period_id',
        'employee_id',
        'earning_type_id',
        'description',
        'amount',
        'is_taxable',
        'is_one_time',
        'approved_by',
        'approved_at',
        'notes',
        'created_by',
        'updated_by'
    ];

    protected $casts = [
        'amount' => 'decimal:2',
        'is_taxable' => 'boolean',
        'is_one_time' => 'boolean',
        'approved_at' => 'datetime'
    ];

    // Relationships
    public function company()
    {
        return $this->belongsTo(Company::class, 'company_id');
    }

    public function payrollPeriod()
    {
        return $this->belongsTo(PayrollPeriod::class, 'payroll_period_id');
    }

    public function employee()
    {
        return $this->belongsTo(Employee::class, 'employee_id');
    }

    public function earningType()
    {
        return $this->belongsTo(EarningType::class, 'earning_type_id');
    }

    public function approver()
    {
        return $this->belongsTo(User::class, 'approved_by');
    }

    public function creator()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    public function updater()
    {
        return $this->belongsTo(User::class, 'updated_by');
    }

    // Scopes
    public function scopeForPeriod($query, $periodId)
    {
        return $query->where('payroll_period_id', $periodId);
    }

    public function scopeForEmployee($query, $employeeId)
    {
        return $query->where('employee_id', $employeeId);
    }

    public function scopeApproved($query)
    {
        return $query->whereNotNull('approved_by');
    }

    public function scopePending($query)
    {
        return $query->whereNull('approved_by');
    }

    public function scopeOneTime($query)
    {
        return $query->where('is_one_time', true);
    }

    public function scopeTaxable($query)
    {
        return $query->where('is_taxable', true);
    }

    public function scopeNonTaxable($query)
    {
        return $query->where('is_taxable', false);
    }

    // Helper methods
    public function isApproved()
    {
        return !is_null($this->approved_by);
    }

    public function approve($userId)
    {
        $this->update([
            'approved_by' => $userId,
            'approved_at' => now(),
            'updated_by' => $userId
        ]);
    }

    public function getStatusBadgeAttribute()
    {
        if ($this->isApproved()) {
            return '<span class="px-2 py-1 text-xs font-semibold rounded-full bg-green-100 text-green-800">Approved</span>';
        }
        return '<span class="px-2 py-1 text-xs font-semibold rounded-full bg-yellow-100 text-yellow-800">Pending</span>';
    }

    public function getTypeBadgeAttribute()
    {
        if ($this->is_one_time) {
            return '<span class="px-2 py-1 text-xs font-semibold rounded-full bg-blue-100 text-blue-800">One-time</span>';
        }
        return '<span class="px-2 py-1 text-xs font-semibold rounded-full bg-purple-100 text-purple-800">Recurring</span>';
    }

    public function getTaxableBadgeAttribute()
    {
        if ($this->is_taxable) {
            return '<span class="px-2 py-1 text-xs font-semibold rounded-full bg-red-100 text-red-800">Taxable</span>';
        }
        return '<span class="px-2 py-1 text-xs font-semibold rounded-full bg-green-100 text-green-800">Non-Taxable</span>';
    }
}