<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class PayrollEntry extends Model
{
    use HasFactory;

    protected $fillable = [
        'payroll_run_id',
        'payroll_period_id',
        'company_id',
        'employee_id',
        'period_start_date',
        'period_end_date',
        'entry_date',
        'earnings_breakdown',
        'total_recurring',
        'total_non_recurring',
        'total_taxable_earnings',
        'total_non_taxable_earnings',
        'gross_earnings',
        'basic_salary_for_napsa',
        'paye_tax',
        'napsa_employee',
        'napsa_employer',
        'nhima_employee',
        'nhima_employer',
        'total_deductions',
        'net_pay',
        'calculation_status',
        'payment_status',
        'payment_date',
        'payment_reference',
        'calculated_by',
        'calculated_at',
        'approved_by',
        'approved_at',
        'paid_by',
        'paid_at'
    ];

    protected $casts = [
        'period_start_date' => 'date',
        'period_end_date' => 'date',
        'entry_date' => 'date',
        'payment_date' => 'date',
        'earnings_breakdown' => 'array',
        'total_recurring' => 'decimal:2',
        'total_non_recurring' => 'decimal:2',
        'total_taxable_earnings' => 'decimal:2',
        'total_non_taxable_earnings' => 'decimal:2',
        'gross_earnings' => 'decimal:2',
        'basic_salary_for_napsa' => 'decimal:2',
        'paye_tax' => 'decimal:2',
        'napsa_employee' => 'decimal:2',
        'napsa_employer' => 'decimal:2',
        'nhima_employee' => 'decimal:2',
        'nhima_employer' => 'decimal:2',
        'total_deductions' => 'decimal:2',
        'net_pay' => 'decimal:2',
        'calculated_at' => 'datetime',
        'approved_at' => 'datetime',
        'paid_at' => 'datetime'
    ];

    // Relationships
    public function payrollPeriod()
    {
        return $this->belongsTo(PayrollPeriod::class, 'payroll_period_id');
    }

    public function company()
    {
        return $this->belongsTo(Company::class, 'company_id');
    }

    public function employee()
    {
        return $this->belongsTo(Employee::class, 'employee_id');
    }

    public function payrollRun()
    {
        return $this->belongsTo(PayrollRun::class, 'payroll_run_id');
    }

    public function calculator()
    {
        return $this->belongsTo(User::class, 'calculated_by');
    }

    public function approver()
    {
        return $this->belongsTo(User::class, 'approved_by');
    }

    public function payer()
    {
        return $this->belongsTo(User::class, 'paid_by');
    }

    // Scopes
    public function scopeForPeriod($query, $periodId)
    {
        return $query->where('payroll_period_id', $periodId);
    }

    public function scopeForEmployee($query, $employeeId)
    {
        return $query->where('employee_id', $employeeId);
    }

    public function scopeForCompany($query, $companyId)
    {
        return $query->where('company_id', $companyId);
    }

    public function scopeDraft($query)
    {
        return $query->where('calculation_status', 'draft');
    }

    public function scopeCalculated($query)
    {
        return $query->where('calculation_status', 'calculated');
    }

    public function scopeApproved($query)
    {
        return $query->where('calculation_status', 'approved');
    }

    public function scopePaid($query)
    {
        return $query->where('calculation_status', 'paid');
    }

    public function scopePendingPayment($query)
    {
        return $query->where('payment_status', 'pending');
    }

    // Helper methods
    public function getEarningsBreakdownFormatted()
    {
        $breakdown = $this->earnings_breakdown ?? [];
        $formatted = [];
        
        if (!empty($breakdown['recurring'])) {
            foreach ($breakdown['recurring'] as $earning) {
                $formatted[] = [
                    'type' => 'recurring',
                    'name' => $earning['name'] ?? '',
                    'amount' => number_format($earning['amount'], 2),
                    'taxable' => $earning['is_taxable'] ?? true,
                    'taxable_text' => ($earning['is_taxable'] ?? true) ? 'Taxable' : 'Non-Taxable'
                ];
            }
        }
        
        if (!empty($breakdown['non_recurring'])) {
            foreach ($breakdown['non_recurring'] as $earning) {
                $formatted[] = [
                    'type' => 'non_recurring',
                    'name' => $earning['name'] ?? '',
                    'amount' => number_format($earning['amount'], 2),
                    'taxable' => $earning['is_taxable'] ?? true,
                    'taxable_text' => ($earning['is_taxable'] ?? true) ? 'Taxable' : 'Non-Taxable'
                ];
            }
        }
        
        return $formatted;
    }

    public function getStatusBadgeAttribute()
    {
        $badges = [
            'draft' => '<span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-gray-100 text-gray-800">Draft</span>',
            'calculated' => '<span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-blue-100 text-blue-800">Calculated</span>',
            'approved' => '<span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-green-100 text-green-800">Approved</span>',
            'paid' => '<span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-purple-100 text-purple-800">Paid</span>'
        ];
        
        return $badges[$this->calculation_status] ?? $badges['draft'];
    }

    public function getPaymentStatusBadgeAttribute()
    {
        $badges = [
            'pending' => '<span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-yellow-100 text-yellow-800">Pending</span>',
            'paid' => '<span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-green-100 text-green-800">Paid</span>',
            'cancelled' => '<span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-red-100 text-red-800">Cancelled</span>'
        ];
        
        return $badges[$this->payment_status] ?? $badges['pending'];
    }

    public function markAsCalculated($userId)
    {
        $this->update([
            'calculation_status' => 'calculated',
            'calculated_by' => $userId,
            'calculated_at' => now()
        ]);
    }

    public function markAsApproved($userId)
    {
        $this->update([
            'calculation_status' => 'approved',
            'approved_by' => $userId,
            'approved_at' => now()
        ]);
    }

    public function markAsPaid($userId, $paymentDate, $reference = null)
    {
        $this->update([
            'calculation_status' => 'paid',
            'payment_status' => 'paid',
            'paid_by' => $userId,
            'paid_at' => now(),
            'payment_date' => $paymentDate,
            'payment_reference' => $reference
        ]);
    }

    public function canCalculate()
    {
        return $this->calculation_status === 'draft';
    }

    public function canApprove()
    {
        return $this->calculation_status === 'calculated';
    }

    public function canPay()
    {
        return $this->calculation_status === 'approved' && $this->payment_status === 'pending';
    }
}