<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class ProductRequirement extends Model
{
    use HasFactory;

    /**
     * The table associated with the model.
     *
     * @var string
     */
    protected $table = 'product_requirements';

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'product_id',
        'requirement_id',
        'is_mandatory',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'is_mandatory' => 'boolean',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
    ];

    /**
     * Get the loan product that owns the requirement mapping.
     */
    public function product()
    {
        return $this->belongsTo(LoanType::class, 'product_id');
    }

    /**
     * Get the requirement that is mapped to the product.
     */
    public function requirement()
    {
        return $this->belongsTo(ApplicationRequirement::class, 'requirement_id');
    }

    /**
     * Scope a query to only include mandatory requirements.
     */
    public function scopeMandatory($query)
    {
        return $query->where('is_mandatory', true);
    }

    /**
     * Scope a query to only include optional requirements.
     */
    public function scopeOptional($query)
    {
        return $query->where('is_mandatory', false);
    }

    /**
     * Scope a query to only include requirements for a specific product.
     */
    public function scopeForProduct($query, $productId)
    {
        return $query->where('product_id', $productId);
    }

    /**
     * Check if this requirement is mandatory.
     */
    public function isMandatory(): bool
    {
        return $this->is_mandatory === true;
    }

    /**
     * Check if this requirement is optional.
     */
    public function isOptional(): bool
    {
        return $this->is_mandatory === false;
    }

    /**
     * Get formatted mandatory status.
     */
    public function getMandatoryStatusAttribute(): string
    {
        return $this->is_mandatory ? 'Mandatory' : 'Optional';
    }

    /**
     * Get the status badge for mandatory status.
     */
    public function getMandatoryBadgeAttribute(): string
    {
        if ($this->is_mandatory) {
            return '<span class="badge bg-danger">Mandatory</span>';
        }
        return '<span class="badge bg-secondary">Optional</span>';
    }

    /**
     * Check if a requirement exists for a product.
     */
    public static function requirementExists($productId, $requirementId): bool
    {
        return self::where('product_id', $productId)
            ->where('requirement_id', $requirementId)
            ->exists();
    }

    /**
     * Get requirements count for a product.
     */
    public static function countForProduct($productId): int
    {
        return self::where('product_id', $productId)->count();
    }

    /**
     * Get mandatory requirements count for a product.
     */
    public static function mandatoryCountForProduct($productId): int
    {
        return self::where('product_id', $productId)
            ->where('is_mandatory', true)
            ->count();
    }

    /**
     * Get optional requirements count for a product.
     */
    public static function optionalCountForProduct($productId): int
    {
        return self::where('product_id', $productId)
            ->where('is_mandatory', false)
            ->count();
    }

    /**
     * Get all requirements for a product with requirement details.
     */
    public static function getRequirementsForProduct($productId)
    {
        return self::where('product_id', $productId)
            ->with('requirement')
            ->orderBy('is_mandatory', 'desc')
            ->orderBy('requirement_id')
            ->get();
    }

    /**
     * Create or update a product requirement.
     */
    public static function syncRequirement($productId, $requirementId, $isMandatory = true): bool
    {
        return self::updateOrCreate(
            [
                'product_id' => $productId,
                'requirement_id' => $requirementId,
            ],
            [
                'is_mandatory' => $isMandatory,
            ]
        );
    }

    /**
     * Remove a requirement from a product.
     */
    public static function removeRequirement($productId, $requirementId): bool
    {
        return self::where('product_id', $productId)
            ->where('requirement_id', $requirementId)
            ->delete();
    }

    /**
     * Get product IDs that require a specific document.
     */
    public static function getProductsForRequirement($requirementId)
    {
        return self::where('requirement_id', $requirementId)
            ->with('product')
            ->get()
            ->pluck('product');
    }

    /**
     * Check if a product has any requirements.
     */
    public static function productHasRequirements($productId): bool
    {
        return self::where('product_id', $productId)->exists();
    }
}