<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Receivable extends Model
{
    use HasFactory;

    protected $fillable = [
        'company_id',
        'branch_id',
        'loan_id',
        'customer_name',
        'invoice_number',
        'invoice_date',
        'due_date',
        'description',
        'account_id',
        'currency_id',
        'amount',
        'amount_paid',
        'balance_due',
        'status',
        'created_by',
        'updated_by'
    ];

    protected $casts = [
        'invoice_date' => 'date',
        'due_date' => 'date',
        'amount' => 'decimal:2',
        'amount_paid' => 'decimal:2',
        'balance_due' => 'decimal:2'
    ];

    // Relationships
    public function company()
    {
        return $this->belongsTo(Company::class);
    }

    public function branch()
    {
        return $this->belongsTo(Branch::class);
    }

    public function account()
    {
        return $this->belongsTo(ChartOfAccount::class);
    }

    public function currency()
    {
        return $this->belongsTo(Currency::class);
    }

    public function creator()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    public function updater()
    {
        return $this->belongsTo(User::class, 'updated_by');
    }

    // Scopes
    public function scopeOutstanding($query)
    {
        return $query->where('status', 'outstanding');
    }

    public function scopeOverdue($query)
    {
        return $query->where('status', 'overdue');
    }

    public function scopePaid($query)
    {
        return $query->where('status', 'paid');
    }

    // Methods
    public function isOverdue()
    {
        return $this->due_date < now() && $this->balance_due > 0;
    }

    public function updateBalance()
    {
        $this->balance_due = $this->amount - $this->amount_paid;
        
        if ($this->balance_due <= 0) {
            $this->status = 'paid';
        } elseif ($this->isOverdue()) {
            $this->status = 'overdue';
        } elseif ($this->balance_due > 0) {
            $this->status = 'outstanding';
        }
        
        $this->save();
    }
}
