<?php

namespace App\Services;

use App\Models\Customer;
use App\Models\CustomerCreditScore;
use App\Models\DisbursedLoan;
use App\Models\PaymentSchedule;
use App\Models\Repayment;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class CreditScoreService
{
    private $baseScore = 300;
    private $maxScore = 850;

    /**
     * Initialize credit score record for new customer (without actual score)
     */
    public function initializeCreditScore($customerId)
    {
        try {
            // Check if customer already has a credit score record
            $existingScore = CustomerCreditScore::byCustomer($customerId)->current()->first();
            
            if ($existingScore) {
                return $existingScore;
            }

            $customer = Customer::find($customerId);
            
            if (!$customer) {
                throw new \Exception("Customer not found");
            }

            // Create record with NULL score - indicates no credit history yet
            $creditScore = CustomerCreditScore::create([
                'customer_id' => $customerId,
                'credit_score' => null, // No numerical score initially
                'score_band' => 'No History',
                'previous_score' => null,
                'score_change' => 0,
                'change_reason' => 'New customer - no credit history',
                'change_type' => 'initialization',
                'days_delinquent' => 0,
                'total_loans' => 0,
                'active_loans' => 0,
                'closed_loans' => 0,
                'defaulted_loans' => 0,
                'total_borrowed' => 0,
                'current_outstanding' => 0,
                'total_repaid' => 0,
                'on_time_repayment_rate' => 0, // No history yet
                'late_payments_count' => 0,
                'missed_payments_count' => 0,
                'credit_utilization_ratio' => 0,
                'available_credit' => $this->calculateCreditLimit($customerId),
                'credit_limit' => $this->calculateCreditLimit($customerId),
                'last_repayment_date' => null,
                'last_repayment_amount' => 0,
                'last_loan_date' => null,
                'last_credit_inquiry' => now(),
                'has_active_default' => false,
                'has_recent_late_payment' => false,
                'has_high_utilization' => false,
                'has_multiple_active_loans' => false,
                'score_date' => now(),
                'is_current' => true,
                'created_by' => auth()->id() ?? 1,
            ]);

            \Log::info('Credit score record initialized for new customer', [
                'customer_id' => $customerId,
                'status' => 'No History'
            ]);

            return $creditScore;

        } catch (\Exception $e) {
            \Log::error('Failed to initialize credit score record for customer', [
                'customer_id' => $customerId,
                'error' => $e->getMessage()
            ]);
            return null;
        }
    }

    // Update credit score for a specific event
    public function updateCreditScoreForEvent($customerId, $eventType, $eventData = [])
    {
        DB::transaction(function () use ($customerId, $eventType, $eventData) {
            // Get current score or create new one
            $currentScore = CustomerCreditScore::current()
                ->byCustomer($customerId)
                ->first();

            if (!$currentScore) {
                $currentScore = $this->initializeCreditScore($customerId);
            }

            // For customers with no history, assign first real score on loan disbursement
            if ($currentScore->credit_score === null && $eventType === 'loan_disbursed') {
                $currentScore->credit_score = 600; // Fair starting point for first loan
                $currentScore->score_band = CustomerCreditScore::getScoreBand(600);
                $currentScore->change_reason = 'First credit score assignment';
            }

            // Create new score record based on event
            $newScore = $this->calculateNewScore($currentScore, $eventType, $eventData);
            
            // Save the new score
            $newScore->save();
            
            // Mark previous scores as historical
            $newScore->markPreviousScoresAsHistorical();
        });
    }

    // Daily batch update for all customers
    public function runDailyCreditScoreUpdate()
    {
        $customers = Customer::where('status', 'active')->get();
        
        foreach ($customers as $customer) {
            $this->updateCreditScoreForEvent($customer->id, 'daily_update');
        }
        
        return $customers->count();
    }

    // Calculate new score based on event
    private function calculateNewScore(CustomerCreditScore $currentScore, $eventType, $eventData = [])
    {
        $newScoreValue = $currentScore->credit_score ?? 600; // Use 600 if no score yet
        $changeReason = '';
        $changePoints = 0;

        switch ($eventType) {
            case 'repayment_made':
                $changePoints = $this->calculateRepaymentImpact($eventData);
                $changeReason = 'Loan repayment received';
                break;
                
            case 'interest_booked':
                $changePoints = $this->calculateInterestBookingImpact($eventData);
                $changeReason = 'Interest income booked';
                break;
                
            case 'loan_disbursed':
                $changePoints = $this->calculateLoanDisbursementImpact($eventData);
                $changeReason = 'New loan disbursed';
                break;
                
            case 'payment_missed':
                $changePoints = $this->calculateMissedPaymentImpact($eventData);
                $changeReason = 'Payment missed or late';
                break;
                
            case 'loan_defaulted':
                $changePoints = $this->calculateDefaultImpact($eventData);
                $changeReason = 'Loan defaulted';
                break;
                
            case 'daily_update':
                $changePoints = $this->calculateDailyUpdateImpact($currentScore);
                $changeReason = 'Daily credit assessment';
                break;
        }

        // Apply boundaries
        $newScoreValue = max($this->baseScore, min($this->maxScore, $newScoreValue + $changePoints));

        // Create new score record
        return new CustomerCreditScore([
            'customer_id' => $currentScore->customer_id,
            'credit_score' => $newScoreValue,
            'score_band' => CustomerCreditScore::getScoreBand($newScoreValue),
            'previous_score' => $currentScore->credit_score,
            'score_change' => $changePoints,
            'change_reason' => $changeReason,
            'change_type' => $eventType,
            'score_date' => now(),
            'is_current' => true,
            'created_by' => auth()->id() ?? 1,
            // Update all metrics
            'days_delinquent' => $this->calculateDaysDelinquent($currentScore->customer_id),
            'total_loans' => $this->calculateTotalLoans($currentScore->customer_id),
            'active_loans' => $this->calculateActiveLoans($currentScore->customer_id),
            'closed_loans' => $this->calculateClosedLoans($currentScore->customer_id),
            'defaulted_loans' => $this->calculateDefaultedLoans($currentScore->customer_id),
            'total_borrowed' => $this->calculateTotalBorrowed($currentScore->customer_id),
            'current_outstanding' => $this->calculateCurrentOutstanding($currentScore->customer_id),
            'total_repaid' => $this->calculateTotalRepaid($currentScore->customer_id),
            'on_time_repayment_rate' => $this->calculateOnTimeRepaymentRate($currentScore->customer_id),
            'late_payments_count' => $this->calculateLatePaymentsCount($currentScore->customer_id),
            'missed_payments_count' => $this->calculateMissedPaymentsCount($currentScore->customer_id),
            'credit_utilization_ratio' => $this->calculateCreditUtilizationRatio($currentScore->customer_id),
            'available_credit' => $this->calculateAvailableCredit($currentScore->customer_id),
            'credit_limit' => $this->calculateCreditLimit($currentScore->customer_id),
            'last_repayment_date' => $this->getLastRepaymentDate($currentScore->customer_id),
            'last_repayment_amount' => $this->getLastRepaymentAmount($currentScore->customer_id),
            'last_loan_date' => $this->getLastLoanDate($currentScore->customer_id),
            'has_active_default' => $this->hasActiveDefault($currentScore->customer_id),
            'has_recent_late_payment' => $this->hasRecentLatePayment($currentScore->customer_id),
            'has_high_utilization' => $this->hasHighUtilization($currentScore->customer_id),
            'has_multiple_active_loans' => $this->hasMultipleActiveLoans($currentScore->customer_id),
        ]);
    }

    // Scoring algorithms for different events
    private function calculateRepaymentImpact($repaymentData)
    {
        $points = 0;
        
        // On-time repayment
        if ($repaymentData['is_on_time'] ?? true) {
            $points += 5; // Reward for on-time payment
            
            // Larger payments get more points
            $amount = $repaymentData['amount'] ?? 0;
            if ($amount > 1000) $points += 2;
            if ($amount > 5000) $points += 3;
        } else {
            $points -= 10; // Penalty for late payment
        }
        
        return $points;
    }

    private function calculateInterestBookingImpact($interestData)
    {
        // Interest booking doesn't directly affect score, but reflects loan activity
        return 0;
    }

    private function calculateLoanDisbursementImpact($loanData)
    {
        $points = 0;
        
        // New credit inquiry - small temporary decrease
        $points -= 2;
        
        // But having active credit (if managed well) can be positive long-term
        return $points;
    }

    private function calculateMissedPaymentImpact($paymentData)
    {
        // Significant penalty for missed payments
        return -25;
    }

    private function calculateDefaultImpact($defaultData)
    {
        // Major penalty for defaults
        return -50;
    }

    private function calculateDailyUpdateImpact(CustomerCreditScore $currentScore)
    {
        $points = 0;
        
        // Positive factors
        if ($currentScore->on_time_repayment_rate > 95) $points += 3;
        if ($currentScore->credit_utilization_ratio < 30) $points += 2;
        if ($currentScore->days_delinquent == 0) $points += 2;
        if (!$currentScore->has_active_default) $points += 5;
        
        // Negative factors
        if ($currentScore->has_recent_late_payment) $points -= 5;
        if ($currentScore->has_high_utilization) $points -= 3;
        if ($currentScore->has_multiple_active_loans) $points -= 2;
        
        return $points;
    }

    // Metric calculation methods
    private function calculateDaysDelinquent($customerId)
    {
        return DisbursedLoan::where('customerid', $customerId)
            ->where('status', 'active')
            ->sum('daysdelinquent');
    }

    private function calculateTotalLoans($customerId)
    {
        return DisbursedLoan::where('customerid', $customerId)->count();
    }

    private function calculateActiveLoans($customerId)
    {
        return DisbursedLoan::where('customerid', $customerId)
            ->where('status', 'active')
            ->count();
    }

    private function calculateClosedLoans($customerId)
    {
        return DisbursedLoan::where('customerid', $customerId)
            ->where('status', 'closed')
            ->count();
    }

    private function calculateDefaultedLoans($customerId)
    {
        return DisbursedLoan::where('customerid', $customerId)
            ->whereIn('status', ['default', 'written_off'])
            ->count();
    }

    private function calculateTotalBorrowed($customerId)
    {
        return DisbursedLoan::where('customerid', $customerId)->sum('amount');
    }

    private function calculateCurrentOutstanding($customerId)
    {
        return DisbursedLoan::where('customerid', $customerId)
            ->where('status', 'active')
            ->sum('totalbalance');
    }

    private function calculateTotalRepaid($customerId)
    {
        return DisbursedLoan::where('customerid', $customerId)
            ->sum('totalpaid');
    }

    private function calculateOnTimeRepaymentRate($customerId)
    {
        $totalPayments = PaymentSchedule::whereHas('loan', function($query) use ($customerId) {
            $query->where('customerid', $customerId);
        })->where('transactiontype', 'Payment')->count();

        $onTimePayments = PaymentSchedule::whereHas('loan', function($query) use ($customerId) {
            $query->where('customerid', $customerId);
        })->where('transactiontype', 'Payment')
          ->where('status', 'paid')
          ->whereColumn('paid_date', '<=', 'paymentdate')
          ->count();

        if ($totalPayments === 0) return 100;
        return ($onTimePayments / $totalPayments) * 100;
    }

    private function calculateLatePaymentsCount($customerId)
    {
        return PaymentSchedule::whereHas('loan', function($query) use ($customerId) {
            $query->where('customerid', $customerId);
        })->where('transactiontype', 'Payment')
          ->where('status', 'paid')
          ->whereColumn('paid_date', '>', 'paymentdate')
          ->count();
    }

    private function calculateMissedPaymentsCount($customerId)
    {
        return PaymentSchedule::whereHas('loan', function($query) use ($customerId) {
            $query->where('customerid', $customerId);
        })->where('transactiontype', 'Payment')
          ->where('status', 'overdue')
          ->count();
    }

    private function calculateCreditUtilizationRatio($customerId)
    {
        $outstanding = $this->calculateCurrentOutstanding($customerId);
        $creditLimit = $this->calculateCreditLimit($customerId);
        
        return CustomerCreditScore::calculateUtilizationRatio($outstanding, $creditLimit);
    }

    private function calculateAvailableCredit($customerId)
    {
        $creditLimit = $this->calculateCreditLimit($customerId);
        $outstanding = $this->calculateCurrentOutstanding($customerId);
        
        return max(0, $creditLimit - $outstanding);
    }

    private function calculateCreditLimit($customerId)
    {
        // Based on customer's income and history
        $customer = Customer::find($customerId);
        $income = $customer->income ?? 0;
        
        // Simple algorithm: 3x monthly income as credit limit
        return $income * 3;
    }

    private function getLastRepaymentDate($customerId)
    {
        $lastRepayment = Repayment::where('customerid', $customerId)
            ->orderBy('rdate', 'desc')
            ->first();
            
        return $lastRepayment ? $lastRepayment->rdate : null;
    }

    private function getLastRepaymentAmount($customerId)
    {
        $lastRepayment = Repayment::where('customerid', $customerId)
            ->orderBy('rdate', 'desc')
            ->first();
            
        return $lastRepayment ? $lastRepayment->rtotal : 0;
    }

    private function getLastLoanDate($customerId)
    {
        $lastLoan = DisbursedLoan::where('customerid', $customerId)
            ->orderBy('disburseddate', 'desc')
            ->first();
            
        return $lastLoan ? $lastLoan->disburseddate : null;
    }

    private function hasActiveDefault($customerId)
    {
        return DisbursedLoan::where('customerid', $customerId)
            ->where('status', 'default')
            ->exists();
    }

    private function hasRecentLatePayment($customerId)
    {
        return PaymentSchedule::whereHas('loan', function($query) use ($customerId) {
            $query->where('customerid', $customerId);
        })->where('transactiontype', 'Payment')
          ->where('status', 'paid')
          ->whereColumn('paid_date', '>', 'paymentdate')
          ->where('paid_date', '>=', now()->subDays(30))
          ->exists();
    }

    private function hasHighUtilization($customerId)
    {
        return $this->calculateCreditUtilizationRatio($customerId) > 75;
    }

    private function hasMultipleActiveLoans($customerId)
    {
        return $this->calculateActiveLoans($customerId) > 2;
    }
}