<?php

namespace App\Services;

use App\Models\Application;
use App\Models\DisbursedLoan;
use App\Models\Customer;
use App\Models\PaymentSchedule;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Carbon\Carbon;

class LoanDisbursementService
{
    private $creditScoreService;
    private $smsService;

    public function __construct(CreditScoreService $creditScoreService, SmsService $smsService)
    {
        $this->creditScoreService = $creditScoreService;
        $this->smsService = $smsService;
    }

    /**
     * Process loan disbursement with credit score update
     */
    public function disburseLoan(Application $application, array $disbursementData = [])
    {
        return DB::transaction(function () use ($application, $disbursementData) {
            // Generate loan number
            $loanNumber = $this->generateLoanNumber();
            
            // Calculate payment dates
            $firstPaymentDate = $this->calculateFirstPaymentDate();
            $maturityDate = $this->calculateMaturityDate($application->loan_tenure);

            // Create disbursed loan record
            $disbursedLoan = DisbursedLoan::create([
                'application_id' => $application->id,
                'customerid' => $application->customer_id,
                'loannumber' => $loanNumber,
                'loantypeid' => $application->product_id,
                'amount' => $application->loan_amount,
                'interestrate' => $application->interest_rate,
                'loanterm' => $application->loan_tenure,
                'paymentfrequency' => 'monthly',
                'installmentamount' => $application->estimated_monthly_payment,
                'disburseddate' => now(),
                'status' => 'active',
                'firstpaymentdate' => $firstPaymentDate,
                'maturitydate' => $maturityDate,
                'principalbalance' => $application->loan_amount,
                'interestbalance' => 0,
                'totalbalance' => $application->loan_amount,
                'totalprincipalpaid' => 0,
                'totalinterestpaid' => 0,
                'totalpaid' => 0,
                'branchid' => $application->branch_id ?? 1,
                'companyid' => $application->company_id ?? 1,
                'disbursedby' => auth()->id() ?? 1,
                'createdby' => auth()->id() ?? 1,
                'updatedby' => auth()->id() ?? 1,
            ]);

            // Update application status
            $application->update([
                'status' => 'disbursed',
                'disbursed_at' => now(),
                'disbursed_loan_id' => $disbursedLoan->loanid,
            ]);

            // Generate payment schedule
            $this->generatePaymentSchedule($disbursedLoan);

            // Update credit score for loan disbursement
            $this->updateCreditScoreForDisbursement($application, $disbursedLoan);

            // Send notifications
            $this->sendDisbursementNotifications($application, $disbursedLoan);

            Log::info('Loan disbursement completed successfully', [
                'application_id' => $application->id,
                'loan_id' => $disbursedLoan->loanid,
                'loan_number' => $loanNumber,
                'customer_id' => $application->customer_id
            ]);

            return $disbursedLoan;
        });
    }

    /**
     * Update credit score for loan disbursement event
     */
    private function updateCreditScoreForDisbursement(Application $application, DisbursedLoan $disbursedLoan): void
    {
        try {
            $this->creditScoreService->updateScoreForEvent(
                $application->customer_id,
                'loan_disbursed',
                [
                    'loan_id' => $disbursedLoan->loanid,
                    'application_id' => $application->id,
                    'loan_amount' => $application->loan_amount,
                    'interest_rate' => $application->interest_rate,
                    'loan_term' => $application->loan_tenure,
                    'product_type' => $application->product->product ?? 'Personal Loan',
                    'disbursement_date' => now(),
                    'credit_utilization_impact' => $this->calculateUtilizationImpact($application->customer_id, $application->loan_amount),
                    'monthly_payment' => $application->estimated_monthly_payment,
                ]
            );

            Log::info('Credit score updated for loan disbursement', [
                'customer_id' => $application->customer_id,
                'loan_id' => $disbursedLoan->loanid,
                'loan_amount' => $application->loan_amount,
                'credit_impact' => 'New credit account opened'
            ]);

        } catch (\Exception $e) {
            Log::error('Failed to update credit score for loan disbursement', [
                'customer_id' => $application->customer_id,
                'loan_id' => $disbursedLoan->loanid,
                'error' => $e->getMessage()
            ]);
            // Don't throw exception - credit score update shouldn't block disbursement
        }
    }

    /**
     * Calculate credit utilization impact
     */
    private function calculateUtilizationImpact($customerId, $newLoanAmount): float
    {
        try {
            // Get customer's current total outstanding from active loans
            $currentOutstanding = DisbursedLoan::where('customerid', $customerId)
                ->where('status', 'active')
                ->sum('totalbalance');

            // Get customer's credit limit (you might need to adjust this based on your business logic)
            $customer = Customer::find($customerId);
            $creditLimit = $customer->credit_limit ?? ($customer->income * 3); // Default 3x monthly income

            if ($creditLimit <= 0) return 0;

            $oldUtilization = ($currentOutstanding / $creditLimit) * 100;
            $newUtilization = (($currentOutstanding + $newLoanAmount) / $creditLimit) * 100;
            
            return $newUtilization - $oldUtilization;

        } catch (\Exception $e) {
            Log::error('Error calculating utilization impact', [
                'customer_id' => $customerId,
                'error' => $e->getMessage()
            ]);
            return 0;
        }
    }

    /**
     * Generate payment schedule for the loan
     */
    private function generatePaymentSchedule(DisbursedLoan $loan): void
    {
        try {
            $monthlyPayment = $loan->installmentamount;
            $principalBalance = $loan->amount;
            $monthlyInterestRate = $loan->interestrate / 12 / 100;
            $paymentDate = Carbon::parse($loan->firstpaymentdate);

            for ($i = 1; $i <= $loan->loanterm; $i++) {
                $interestAmount = $principalBalance * $monthlyInterestRate;
                $principalAmount = $monthlyPayment - $interestAmount;
                
                // Ensure principal doesn't exceed remaining balance
                if ($principalAmount > $principalBalance) {
                    $principalAmount = $principalBalance;
                }

                PaymentSchedule::create([
                    'loannumber' => $loan->loannumber,
                    'loanid' => $loan->loanid,
                    'interestamount' => round($interestAmount, 2),
                    'principalamount' => round($principalAmount, 2),
                    'totalamount' => round($monthlyPayment, 2),
                    'paymentdate' => $paymentDate->format('Y-m-d'),
                    'transactiontype' => 'Payment',
                    'status' => 'scheduled',
                    'installment_number' => $i,
                    'outstanding_principal' => round($principalBalance - $principalAmount, 2),
                    'branchid' => $loan->branchid,
                    'companyid' => $loan->companyid,
                    'createdby' => auth()->id() ?? 1,
                ]);

                $principalBalance -= $principalAmount;
                $paymentDate->addMonth();

                // Stop if loan is fully paid
                if ($principalBalance <= 0) {
                    break;
                }
            }

            Log::info('Payment schedule generated', [
                'loan_id' => $loan->loanid,
                'installments_created' => $i
            ]);

        } catch (\Exception $e) {
            Log::error('Failed to generate payment schedule', [
                'loan_id' => $loan->loanid,
                'error' => $e->getMessage()
            ]);
            throw $e; // Re-throw to rollback transaction
        }
    }

    /**
     * Send disbursement notifications
     */
    private function sendDisbursementNotifications(Application $application, DisbursedLoan $disbursedLoan): void
    {
        try {
            $customer = $application->customer;
            
            // Send SMS notification
            $smsMessage = "Dear {$customer->customer}, your loan #{$disbursedLoan->loannumber} of ZMW {$application->loan_amount} has been disbursed. First payment: ZMW {$application->estimated_monthly_payment} on {$disbursedLoan->firstpaymentdate}. Login to portal for details.";
            
            $this->smsService->sendSms($customer->phone, $smsMessage);

            Log::info('Disbursement SMS sent', [
                'customer_id' => $customer->id,
                'loan_id' => $disbursedLoan->loanid
            ]);

        } catch (\Exception $e) {
            Log::error('Failed to send disbursement notifications', [
                'customer_id' => $application->customer_id,
                'loan_id' => $disbursedLoan->loanid,
                'error' => $e->getMessage()
            ]);
            // Don't throw - notification failure shouldn't block disbursement
        }
    }

    /**
     * Generate unique loan number
     */
    private function generateLoanNumber(): string
    {
        $prefix = 'LN';
        $year = date('Y');
        $month = date('m');
        
        $lastLoan = DisbursedLoan::where('loannumber', 'like', "{$prefix}{$year}{$month}%")
            ->orderBy('loannumber', 'desc')
            ->first();
            
        $sequence = 1;
        if ($lastLoan) {
            $lastSequence = intval(substr($lastLoan->loannumber, -4));
            $sequence = $lastSequence + 1;
        }
        
        return $prefix . $year . $month . str_pad($sequence, 4, '0', STR_PAD_LEFT);
    }

    /**
     * Calculate first payment date (1 month from disbursement)
     */
    private function calculateFirstPaymentDate(): string
    {
        return now()->addMonth()->format('Y-m-d');
    }

    /**
     * Calculate maturity date based on loan term
     */
    private function calculateMaturityDate($loanTerm): string
    {
        return now()->addMonths($loanTerm)->format('Y-m-d');
    }

    /**
     * Reverse disbursement (if needed for errors)
     */
    public function reverseDisbursement(DisbursedLoan $disbursedLoan): bool
    {
        return DB::transaction(function () use ($disbursedLoan) {
            try {
                // Update loan status to reversed
                $disbursedLoan->update([
                    'status' => 'reversed',
                    'updatedby' => auth()->id(),
                ]);

                // Update application status back to approved
                if ($disbursedLoan->application) {
                    $disbursedLoan->application->update([
                        'status' => 'approved',
                        'disbursed_at' => null,
                        'disbursed_loan_id' => null,
                    ]);
                }

                // Delete payment schedules
                PaymentSchedule::where('loanid', $disbursedLoan->loanid)->delete();

                Log::info('Disbursement reversed successfully', [
                    'loan_id' => $disbursedLoan->loanid
                ]);

                return true;

            } catch (\Exception $e) {
                Log::error('Failed to reverse disbursement', [
                    'loan_id' => $disbursedLoan->loanid,
                    'error' => $e->getMessage()
                ]);
                return false;
            }
        });
    }
}