<?php

namespace App\Services;

use App\Models\DisbursedLoan;
use App\Models\InterestIncome;
use App\Models\ProcessingFee;
use App\Models\BusinessProcessMapping;
use App\Models\JournalEntry;
use App\Models\JournalEntryItem;
use App\Models\Company;
use App\Models\Currency;
use App\Models\LoanType;
use App\Models\Repayment;
use Carbon\Carbon;
use Carbon\CarbonPeriod;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class LoanInterestService
{
    // Cache for frequently accessed data
    private $businessProcessMappings = [];
    private $companies = [];
    private $currencies = [];
    private $loanTypes = [];

    /**
     * Calculate daily interest AND processing fees for all active loans
     * Automatically handles any missing dates from last calculation date to today
     * 
     * @param string|null $date Optional specific date (Y-m-d format)
     * @param bool $forceRecalculate Force recalculation for the date
     * @param bool $skipBalanceUpdate Skip updating loan balances (for backdating)
     * @param bool $calculateMissingDates Automatically calculate missing dates from last calculation to today
     */
    public function calculateDailyInterest($date = null, $forceRecalculate = false, $skipBalanceUpdate = false, $calculateMissingDates = true)
    {
        $now = Carbon::now();
        
        // If specific date provided, calculate only for that date
        if ($date) {
            $calculationDate = Carbon::parse($date);
            return $this->calculateForSingleDate($calculationDate, $forceRecalculate, $skipBalanceUpdate);
        }
        
        // Calculate for today by default
        $today = Carbon::today();
        Log::info("DAILY Interest & Processing Fee Calculation started at {$now->format('Y-m-d H:i:s')}");
        
        // Get all active loans
        $activeLoans = DisbursedLoan::where('status', 'active')
            ->where('principalbalance', '>', 0)
            ->with(['loanType', 'paymentSchedules' => function($query) {
                $query->where('status', '!=', 'paid')
                    ->orderBy('paymentdate', 'asc');
            }])
            ->get();
        
        $results = [];
        $totalLoansProcessed = 0;
        $totalInterestAmount = 0;
        $totalProcessingFeeAmount = 0;
        $totalErrors = 0;
        
        foreach ($activeLoans as $loan) {
            try {
                $loanResult = $this->calculateMissingDatesForLoan($loan, $today, $forceRecalculate, $skipBalanceUpdate, $calculateMissingDates);
                
                if ($loanResult['processed'] > 0) {
                    $totalLoansProcessed++;
                    $totalInterestAmount += $loanResult['total_interest'];
                    $totalProcessingFeeAmount += $loanResult['total_processing_fees'];
                }
                
                if ($loanResult['errors'] > 0) {
                    $totalErrors++;
                }
                
                $results[$loan->loannumber] = $loanResult;
                
            } catch (\Exception $e) {
                Log::error("Error calculating for loan {$loan->loannumber}: " . $e->getMessage());
                $totalErrors++;
                $results[$loan->loannumber] = [
                    'success' => false,
                    'error' => $e->getMessage(),
                    'processed' => 0,
                ];
            }
        }
        
        Log::info(sprintf(
            "DAILY Calculation completed at %s. Loans: %d, Interest: $%.2f, Processing Fees: $%.2f, Errors: %d",
            $now->format('Y-m-d H:i:s'),
            $totalLoansProcessed,
            $totalInterestAmount,
            $totalProcessingFeeAmount,
            $totalErrors
        ));
        
        return [
            'timestamp' => $now->format('Y-m-d H:i:s'),
            'total_loans' => count($activeLoans),
            'loans_processed' => $totalLoansProcessed,
            'total_interest_amount' => $totalInterestAmount,
            'total_processing_fee_amount' => $totalProcessingFeeAmount,
            'errors' => $totalErrors,
            'results' => $results,
            'status' => 'completed'
        ];
    }

    /**
     * Calculate missing dates for a single loan from last calculation date to target date
     */
    private function calculateMissingDatesForLoan($loan, $targetDate, $forceRecalculate, $skipBalanceUpdate, $calculateMissingDates)
    {
        $result = [
            'loan' => $loan->loannumber,
            'processed' => 0,
            'dates_calculated' => [],
            'total_interest' => 0,
            'total_processing_fees' => 0,
            'errors' => 0,
        ];
        
        // If calculateMissingDates is false, only calculate for target date
        if (!$calculateMissingDates) {
            $dateResult = $this->calculateForLoanOnDate($loan, $targetDate, $forceRecalculate, $skipBalanceUpdate);
            
            if ($dateResult['calculated']) {
                $result['processed']++;
                $result['dates_calculated'][] = $targetDate->format('Y-m-d');
                $result['total_interest'] += $dateResult['interest_amount'];
                $result['total_processing_fees'] += $dateResult['processing_fee_amount'];
            }
            
            if ($dateResult['error']) {
                $result['errors']++;
            }
            
            return $result;
        }
        
        // Determine start date for calculation
        $startDate = $this->getCalculationStartDate($loan);
        
        // If no dates need calculation
        if ($startDate->gt($targetDate)) {
            Log::info("Loan {$loan->loannumber} already calculated up to {$targetDate->format('Y-m-d')}");
            return $result;
        }
        
        Log::info("Calculating for loan {$loan->loannumber} from {$startDate->format('Y-m-d')} to {$targetDate->format('Y-m-d')}");
        
        // Process each date in the range
        $currentDate = $startDate->copy();
        while ($currentDate->lte($targetDate)) {
            try {
                DB::beginTransaction();
                
                $dateResult = $this->calculateForLoanOnDate($loan, $currentDate, $forceRecalculate, $skipBalanceUpdate);
                
                if ($dateResult['calculated']) {
                    $result['processed']++;
                    $result['dates_calculated'][] = $currentDate->format('Y-m-d');
                    $result['total_interest'] += $dateResult['interest_amount'];
                    $result['total_processing_fees'] += $dateResult['processing_fee_amount'];
                }
                
                DB::commit();
                
            } catch (\Exception $e) {
                DB::rollBack();
                $result['errors']++;
                Log::error("Error calculating for loan {$loan->loannumber} on {$currentDate->format('Y-m-d')}: " . $e->getMessage());
            }
            
            $currentDate->addDay();
        }
        
        // Update loan's last calculation dates
        if ($result['processed'] > 0) {
            $this->updateLastCalculationDates($loan, $targetDate);
        }
        
        return $result;
    }

    /**
     * Determine the start date for calculation
     * Uses the most recent of: last_interest_calculation_date, last_processing_fee_calculation, or disburseddate
     */
    private function getCalculationStartDate($loan)
    {
        $dates = [];
        
        // Add interest calculation date
        if ($loan->last_interest_calculation_date) {
            $dates[] = Carbon::parse($loan->last_interest_calculation_date)->addDay();
        }
        
        // Add processing fee calculation date
        if ($loan->last_processing_fee_calculation) {
            $dates[] = Carbon::parse($loan->last_processing_fee_calculation)->addDay();
        }
        
        // If no calculation dates, start from disbursement date
        if (empty($dates)) {
            return Carbon::parse($loan->disburseddate);
        }
        
        // Get the earliest date (most behind)
        return min($dates);
    }

    /**
     * Update loan's last calculation dates
     */
    private function updateLastCalculationDates($loan, $date)
    {
        $loan->last_interest_calculation_date = $date;
        $loan->last_processing_fee_calculation = $date;
        $loan->save();
        
        Log::info("Updated last calculation dates for loan {$loan->loannumber} to {$date->format('Y-m-d')}");
    }

    /**
     * Calculate for a single loan on a specific date
     */
    private function calculateForLoanOnDate($loan, $calculationDate, $forceRecalculate, $skipBalanceUpdate)
    {
        $result = [
            'calculated' => false,
            'interest_amount' => 0,
            'processing_fee_amount' => 0,
            'error' => null,
        ];
        
        // Skip if date is before disbursement
        if ($calculationDate->lt($loan->disburseddate)) {
            return $result;
        }
        
        // Skip if loan is closed and date is after closed date
        if ($loan->closeddate && $calculationDate->gt($loan->closeddate)) {
            return $result;
        }
        
        // Skip if loan is not active on this date
        if (!$this->isLoanActiveOnDate($loan, $calculationDate)) {
            return $result;
        }
        
        // Get historical principal balance for this date
        $historicalPrincipal = $this->getHistoricalPrincipalBalance($loan->loanid, $calculationDate);
        
        if ($historicalPrincipal <= 0) {
            Log::debug("Loan {$loan->loannumber} had zero principal balance on {$calculationDate->format('Y-m-d')}");
            return $result;
        }
        
        $loanCalculated = false;
        $interestAmount = 0;
        $processingFeeAmount = 0;
        
        // Calculate interest
        $interestResult = $this->calculateDailyInterestForLoan($loan, $calculationDate, $historicalPrincipal, $forceRecalculate, $skipBalanceUpdate);
        if ($interestResult['calculated']) {
            $loanCalculated = true;
            $interestAmount = $interestResult['amount'];
            
            // ALLOCATE interest to payment schedule
            $this->allocateInterestToSchedule($loan, $calculationDate, $interestAmount);
        }
        
        // Calculate processing fee
        $processingFeeResult = $this->calculateDailyProcessingFeeForLoan($loan, $calculationDate, $historicalPrincipal, $forceRecalculate, $skipBalanceUpdate);
        if ($processingFeeResult['calculated']) {
            $loanCalculated = true;
            $processingFeeAmount = $processingFeeResult['amount'];
            
            // ALLOCATE processing fee to payment schedule
            $this->allocateProcessingFeeToSchedule($loan, $calculationDate, $processingFeeAmount);
        }
        
        if ($loanCalculated) {
            Log::info("Loan {$loan->loannumber} calculated for {$calculationDate->format('Y-m-d')} - Interest: {$interestAmount}, Processing Fee: {$processingFeeAmount}");
            
            // Update loan totals
            if (!$skipBalanceUpdate) {
                $this->updateLoanTotals($loan, $interestAmount, $processingFeeAmount, $calculationDate);
            }
            
            $result['calculated'] = true;
            $result['interest_amount'] = $interestAmount;
            $result['processing_fee_amount'] = $processingFeeAmount;
        }
        
        return $result;
    }

    /**
     * Calculate for a single date (original behavior)
     */
    private function calculateForSingleDate($calculationDate, $forceRecalculate, $skipBalanceUpdate)
    {
        $now = Carbon::now();
        $processedCount = 0;
        $errorCount = 0;
        $interestProcessed = 0;
        $processingFeeProcessed = 0;
        $totalInterestAmount = 0;
        $totalProcessingFeeAmount = 0;

        // Get all active loans that were active on the calculation date
        $activeLoans = DisbursedLoan::where('status', 'active')
            ->whereDate('disburseddate', '<=', $calculationDate)
            ->where(function($query) use ($calculationDate) {
                $query->whereNull('maturitydate')
                      ->orWhereDate('maturitydate', '>=', $calculationDate);
            })
            ->where('principalbalance', '>', 0)
            ->with(['loanType', 'paymentSchedules' => function($query) use ($calculationDate) {
                // Get schedules that are accruing on this date
                $query->where(function($q) use ($calculationDate) {
                    $q->where('accrual_start_date', '<=', $calculationDate)
                      ->where(function($q2) use ($calculationDate) {
                          $q2->where('accrual_end_date', '>=', $calculationDate)
                             ->orWhereNull('accrual_end_date');
                      });
                })
                ->where('status', '!=', 'paid')
                ->orderBy('paymentdate', 'asc');
            }])
            ->get();

        Log::info("Single date calculation for {$calculationDate->format('Y-m-d')} at {$now->format('Y-m-d H:i:s')} for " . count($activeLoans) . ' loans');

        foreach ($activeLoans as $loan) {
            try {
                DB::beginTransaction();

                $loanCalculated = false;
                $loanInterestAmount = 0;
                $loanProcessingFeeAmount = 0;

                // Get historical principal balance for this date
                $historicalPrincipal = $this->getHistoricalPrincipalBalance($loan->loanid, $calculationDate);
                
                if ($historicalPrincipal <= 0) {
                    Log::info("Loan {$loan->loannumber} had zero principal balance on {$calculationDate->format('Y-m-d')}, skipping");
                    DB::rollBack();
                    continue;
                }

                // 1. Calculate DAILY INTEREST
                $interestResult = $this->calculateDailyInterestForLoan($loan, $calculationDate, $historicalPrincipal, $forceRecalculate, $skipBalanceUpdate);
                if ($interestResult['calculated']) {
                    $interestProcessed++;
                    $loanCalculated = true;
                    $loanInterestAmount = $interestResult['amount'];
                    $totalInterestAmount += $loanInterestAmount;
                    
                    // ALLOCATE interest to payment schedule
                    $this->allocateInterestToSchedule($loan, $calculationDate, $loanInterestAmount);
                }

                // 2. Calculate DAILY PROCESSING FEE
                $processingFeeResult = $this->calculateDailyProcessingFeeForLoan($loan, $calculationDate, $historicalPrincipal, $forceRecalculate, $skipBalanceUpdate);
                if ($processingFeeResult['calculated']) {
                    $processingFeeProcessed++;
                    $loanCalculated = true;
                    $loanProcessingFeeAmount = $processingFeeResult['amount'];
                    $totalProcessingFeeAmount += $loanProcessingFeeAmount;
                    
                    // ALLOCATE processing fee to payment schedule
                    $this->allocateProcessingFeeToSchedule($loan, $calculationDate, $loanProcessingFeeAmount);
                }

                if ($loanCalculated) {
                    $processedCount++;
                    Log::info("Loan {$loan->loannumber} for {$calculationDate->format('Y-m-d')} - Interest: {$loanInterestAmount}, Processing Fee: {$loanProcessingFeeAmount}");
                    
                    // Update loan totals
                    if (!$skipBalanceUpdate) {
                        $this->updateLoanTotals($loan, $loanInterestAmount, $loanProcessingFeeAmount, $calculationDate);
                    }
                }

                DB::commit();

            } catch (\Exception $e) {
                DB::rollBack();
                $errorCount++;
                Log::error("Error calculating for loan {$loan->loannumber} on {$calculationDate->format('Y-m-d')}: " . $e->getMessage());
            }
        }

        Log::info(sprintf(
            "Single date calculation completed for %s. Loans: %d, Interest: %d ($%.2f), Processing Fees: %d ($%.2f), Errors: %d",
            $calculationDate->format('Y-m-d'),
            $processedCount,
            $interestProcessed,
            $totalInterestAmount,
            $processingFeeProcessed,
            $totalProcessingFeeAmount,
            $errorCount
        ));
        
        return [
            'date' => $calculationDate->format('Y-m-d'),
            'processed' => $processedCount,
            'interest_processed' => $interestProcessed,
            'processing_fee_processed' => $processingFeeProcessed,
            'total_interest_amount' => $totalInterestAmount,
            'total_processing_fee_amount' => $totalProcessingFeeAmount,
            'errors' => $errorCount,
            'total_loans' => count($activeLoans),
            'timestamp' => $now->format('Y-m-d H:i:s'),
            'status' => 'completed'
        ];
    }

    /**
     * Check if loan was active on a specific date
     */
    private function isLoanActiveOnDate($loan, $date)
    {
        // Loan must be disbursed on or before this date
        if ($date->lt($loan->disburseddate)) {
            return false;
        }
        
        // If loan has maturity date, check if date is before maturity
        if ($loan->maturitydate && $date->gt($loan->maturitydate)) {
            return false;
        }
        
        // If loan is closed, check if date is before closed date
        if ($loan->closeddate && $date->gt($loan->closeddate)) {
            return false;
        }
        
        // Check status
        if ($loan->status !== 'active') {
            return false;
        }
        
        return true;
    }

    // ================= UPDATED METHODS (balance assignments removed) =================

    /**
     * ALLOCATE interest to the appropriate payment schedule
     */
    private function allocateInterestToSchedule($loan, $calculationDate, $interestAmount)
    {
        // Find which schedule this date belongs to
        $schedule = $loan->paymentSchedules()
            ->where(function($query) use ($calculationDate) {
                $query->where('accrual_start_date', '<=', $calculationDate)
                      ->where('accrual_end_date', '>=', $calculationDate);
            })
            ->where('status', '!=', 'paid')
            ->orderBy('paymentdate', 'asc')
            ->first();
        
        if (!$schedule) {
            // If no schedule found, find the next upcoming schedule
            $schedule = $loan->paymentSchedules()
                ->where('paymentdate', '>=', $calculationDate)
                ->where('status', '!=', 'paid')
                ->orderBy('paymentdate', 'asc')
                ->first();
                
            if (!$schedule) {
                // If still no schedule, find the last schedule (for arrears)
                $schedule = $loan->paymentSchedules()
                    ->where('status', '!=', 'paid')
                    ->orderBy('paymentdate', 'desc')
                    ->first();
                    
                if (!$schedule) {
                    Log::warning("No payment schedule found for loan {$loan->loannumber} on {$calculationDate->format('Y-m-d')}");
                    return null;
                }
            }
        }
        
        // Update the schedule with accrued interest
        $schedule->actualaccruedinterest = bcadd($schedule->actualaccruedinterest ?? 0, $interestAmount, 2);
        
        // Update status if overdue
        if ($schedule->paymentdate < $calculationDate && $schedule->balance > 0) {
            $schedule->status = 'overdue';
        }
        
        $schedule->save();
        
        Log::debug("Interest of {$interestAmount} allocated to Schedule #{$schedule->installment_number} (Due: {$schedule->paymentdate}) for loan {$loan->loannumber}");
        
        return $schedule;
    }

    /**
     * ALLOCATE processing fee to the appropriate payment schedule
     */
    private function allocateProcessingFeeToSchedule($loan, $calculationDate, $processingFeeAmount)
    {
        // Find which schedule this date belongs to (same logic as interest)
        $schedule = $loan->paymentSchedules()
            ->where(function($query) use ($calculationDate) {
                $query->where('accrual_start_date', '<=', $calculationDate)
                      ->where('accrual_end_date', '>=', $calculationDate);
            })
            ->where('status', '!=', 'paid')
            ->orderBy('paymentdate', 'asc')
            ->first();
        
        if (!$schedule) {
            // If no schedule found, find the next upcoming schedule
            $schedule = $loan->paymentSchedules()
                ->where('paymentdate', '>=', $calculationDate)
                ->where('status', '!=', 'paid')
                ->orderBy('paymentdate', 'asc')
                ->first();
                
            if (!$schedule) {
                // If still no schedule, find the last schedule
                $schedule = $loan->paymentSchedules()
                    ->where('status', '!=', 'paid')
                    ->orderBy('paymentdate', 'desc')
                    ->first();
                    
                if (!$schedule) {
                    Log::warning("No payment schedule found for loan {$loan->loannumber} on {$calculationDate->format('Y-m-d')}");
                    return null;
                }
            }
        }
        
        // Update the schedule with accrued processing fee
        $schedule->actualaccruedprocessingfee = bcadd($schedule->actualaccruedprocessingfee ?? 0, $processingFeeAmount, 2);
        
        $schedule->save();
        
        Log::debug("Processing fee of {$processingFeeAmount} allocated to Schedule #{$schedule->installment_number} (Due: {$schedule->paymentdate}) for loan {$loan->loannumber}");
        
        return $schedule;
    }

    /**
     * Update loan totals - calculates from schedules
     */
    private function updateLoanTotals($loan, $dailyInterest, $dailyProcessingFee, $calculationDate)
    {
        // Calculate totals from schedules
        $totalInterestFromSchedules = $loan->paymentSchedules()->sum('actualaccruedinterest');
        $totalProcessingFeesFromSchedules = $loan->paymentSchedules()->sum('actualaccruedprocessingfee');
        
        // Get total payments made
        $totalInterestPaid = Repayment::where('rloanid', $loan->loanid)
            ->where('status', 'verified')
            ->sum('rinterest');
            
        $totalProcessingFeesPaid = Repayment::where('rloanid', $loan->loanid)
            ->where('status', 'verified')
            ->sum('processing_fees_amount_paid');
        
        // Calculate balances
        $interestBalance = max(0, bcsub($totalInterestFromSchedules, $totalInterestPaid, 2));
        $processingFeeBalance = max(0, bcsub($totalProcessingFeesFromSchedules, $totalProcessingFeesPaid, 2));
        
        // Update loan
        $loan->interestbalance = $interestBalance;
        $loan->cumulative_interest_charged = $totalInterestFromSchedules;
        $loan->processing_fee_balance = $processingFeeBalance;
        $loan->cumulative_processing_fee_charged = $totalProcessingFeesFromSchedules;
        $loan->last_interest_calculation_date = $calculationDate;
        $loan->last_processing_fee_calculation = $calculationDate;
        
        // Recalculate total balance
        $principalBalance = (float) $loan->principalbalance;
        $totalBalance = bcadd(
            bcadd($principalBalance, $interestBalance, 2),
            $processingFeeBalance,
            2
        );
        
        $loan->totalbalance = $totalBalance;
        $loan->save();
        
        Log::debug("Loan {$loan->loannumber} totals recalculated. Interest: {$interestBalance}, Processing Fees: {$processingFeeBalance}, Total: {$totalBalance}");
        
        return true;
    }

    /**
     * Calculate daily interest for a loan for a specific date
     */
    private function calculateDailyInterestForLoan($loan, $calculationDate, $historicalPrincipal, $forceRecalculate = false, $skipBalanceUpdate = false)
    {
        $result = ['calculated' => false, 'amount' => 0];

        // Check if interest was already calculated for the date
        $existingInterest = InterestIncome::where('iloanid', $loan->loanid)
            ->whereDate('chargeddate', $calculationDate)
            ->first();

        if ($existingInterest && !$forceRecalculate) {
            Log::debug("Interest already calculated for loan {$loan->loannumber} on {$calculationDate->format('Y-m-d')}");
            $result['amount'] = (float) $existingInterest->charge;
            return $result;
        }
        
        // If force recalculate, delete existing entry
        if ($existingInterest && $forceRecalculate) {
            Log::info("Deleting existing interest record for loan {$loan->loannumber} on {$calculationDate->format('Y-m-d')}");
            $this->deleteExistingInterestEntry($loan->loanid, $calculationDate, 'interest');
        }

        // Calculate DAILY interest amount using historical principal
        $interestAmount = $this->calculateDailyInterestAmount($loan, $calculationDate, $historicalPrincipal);

        if ($interestAmount <= 0) {
            Log::debug("Zero or negative interest calculated for loan {$loan->loannumber} on {$calculationDate->format('Y-m-d')}, principal: {$historicalPrincipal}");
            return $result;
        }

        // Create interest income record (DAILY)
        InterestIncome::create([
            'iloanid' => $loan->loanid,
            'customerid' => $loan->customerid,
            'ipid' => $loan->loantypeid,
            'charge' => $interestAmount,
            'chargeddate' => $calculationDate,
            'poster' => 1,
            'branchid' => $loan->branchid,
            'companyid' => $loan->companyid,
            'calculation_date' => $calculationDate,
            'historical_principal' => $historicalPrincipal,
            'interest_rate' => $loan->interestrate,
            'interest_method' => $loan->interest_method,
            'payment_frequency' => $loan->paymentfrequency,
        ]);

        // Create journal entry for interest income (DAILY)
        $this->createInterestJournalEntry($loan, $interestAmount, $calculationDate);

        Log::info("Interest calculated for loan {$loan->loannumber} on {$calculationDate->format('Y-m-d')}: {$interestAmount} (principal: {$historicalPrincipal})");

        $result['calculated'] = true;
        $result['amount'] = $interestAmount;
        return $result;
    }

    /**
     * Calculate daily processing fee for a loan for a specific date
     */
    private function calculateDailyProcessingFeeForLoan($loan, $calculationDate, $historicalPrincipal, $forceRecalculate = false, $skipBalanceUpdate = false)
    {
        $result = ['calculated' => false, 'amount' => 0];
        
        // Get loan type details
        $loanType = $this->getLoanType($loan->loantypeid);
        
        if (!$loanType || (float)$loanType->processing_fee <= 0) {
            return $result; // No processing fee configured
        }

        // Check if processing fee was already calculated for the date
        $existingProcessingFee = ProcessingFee::where('iloanid', $loan->loanid)
            ->whereDate('chargeddate', $calculationDate)
            ->first();
        
        if ($existingProcessingFee && !$forceRecalculate) {
            Log::debug("Processing fee already calculated for loan {$loan->loannumber} on {$calculationDate->format('Y-m-d')}");
            $result['amount'] = (float) $existingProcessingFee->charge;
            return $result;
        }
        
        // If force recalculate, delete existing entry
        if ($existingProcessingFee && $forceRecalculate) {
            Log::info("Deleting existing processing fee record for loan {$loan->loannumber} on {$calculationDate->format('Y-m-d')}");
            $this->deleteExistingInterestEntry($loan->loanid, $calculationDate, 'processing_fee');
        }

        // Calculate DAILY processing fee amount using appropriate base amount
        $processingFeeAmount = $this->calculateDailyProcessingFeeAmount($loan, $loanType, $calculationDate, $historicalPrincipal);

        if ($processingFeeAmount <= 0) {
            return $result;
        }

        // Create processing fee record in processingfees table
        ProcessingFee::create([
            'iloanid' => $loan->loanid,
            'customerid' => $loan->customerid,
            'ipid' => $loan->loantypeid,
            'charge' => $processingFeeAmount,
            'chargeddate' => $calculationDate,
            'poster' => 1,
            'branchid' => $loan->branchid,
            'companyid' => $loan->companyid,
            'calculation_date' => $calculationDate,
            'historical_principal' => $historicalPrincipal,
            'processing_fee_rate' => $loanType->processing_fee,
            'processing_fee_basis' => $loanType->processing_fee_basis,
        ]);

        // Create journal entry for processing fee income
        $this->createProcessingFeeJournalEntry($loan, $loanType, $processingFeeAmount, $calculationDate);

        Log::info("Processing fee calculated for loan {$loan->loannumber} on {$calculationDate->format('Y-m-d')}: {$processingFeeAmount}");

        $result['calculated'] = true;
        $result['amount'] = $processingFeeAmount;
        return $result;
    }

    /**
     * Get historical principal balance as of a specific date
     */
    private function getHistoricalPrincipalBalance($loanId, $asOfDate)
    {
        // Get the loan
        $loan = DisbursedLoan::find($loanId);
        if (!$loan) {
            return 0;
        }
        
        // If asOfDate is before disburseddate, return 0
        $disbursedDate = Carbon::parse($loan->disburseddate);
        if ($asOfDate->lessThan($disbursedDate)) {
            return 0;
        }
        
        // Get total principal payments made up to the asOfDate
        $totalPrincipalPayments = Repayment::where('rloanid', $loanId)
            ->where('status', 'verified')
            ->whereDate('rdate', '<=', $asOfDate)
            ->sum('rprincipal');
        
        // Since there's no separate LoanDisbursement model, use the loan amount
        $totalDisbursements = (float) $loan->amount;
        
        // Calculate historical principal: initial amount - principal payments
        $historicalPrincipal = $totalDisbursements - $totalPrincipalPayments;
        
        // Ensure it's not negative and doesn't exceed original amount
        $historicalPrincipal = max(0, min($historicalPrincipal, $loan->amount));
        
        return (float) $historicalPrincipal;
    }

    /**
     * Calculate daily interest amount using historical principal
     */
    private function calculateDailyInterestAmount($loan, $calculationDate, $historicalPrincipal)
    {
        $periodRate = (float) $loan->interestrate; // Rate per payment period
        $interestMethod = $loan->interest_method;
        $frequency = strtolower($loan->paymentfrequency);

        // Convert period rate to decimal
        $periodRateDecimal = $periodRate / 100;

        // Calculate daily rate based on payment frequency
        $dailyRate = $this->getDailyRateFromPeriodRate($periodRateDecimal, $frequency, $calculationDate);

        // Calculate interest based on method
        if ($interestMethod === 'simple_interest') {
            return $this->calculateSimpleInterest($historicalPrincipal, $dailyRate);
        } else { // reducing_balance
            return $this->calculateReducingBalanceInterest($historicalPrincipal, $dailyRate);
        }
    }

    /**
     * Calculate daily processing fee amount
     */
    private function calculateDailyProcessingFeeAmount($loan, $loanType, $calculationDate, $historicalPrincipal)
    {
        $processingFeeRate = (float) $loanType->processing_fee;
        $processingFeeBasis = $loanType->processing_fee_basis ?: 'initial_amount';
        $frequency = strtolower($loan->paymentfrequency);

        // Calculate base amount based on basis
        if ($processingFeeBasis === 'outstanding_balance') {
            $baseAmount = $historicalPrincipal;
        } else if ($processingFeeBasis === 'daily_balance') {
            $baseAmount = $historicalPrincipal;
        } else { // 'initial_amount' (default)
            $baseAmount = (float) $loan->amount;
        }

        // Get daily rate based on frequency
        $dailyProcessingFeeRate = $this->getDailyRateFromPercentage($processingFeeRate, $frequency, $calculationDate);

        // Calculate daily processing fee: base amount * (daily rate / 100)
        $dailyProcessingFee = $baseAmount * ($dailyProcessingFeeRate / 100);

        return round($dailyProcessingFee, 2);
    }

    /**
     * Get daily rate from period rate based on frequency
     */
    private function getDailyRateFromPeriodRate($periodRateDecimal, $frequency, $date = null)
    {
        $date = $date ? Carbon::parse($date) : Carbon::today();

        switch ($frequency) {
            case 'daily':
                $daysInPeriod = 1;
                break;
                
            case 'weekly':
                $daysInPeriod = 7;
                break;
                
            case 'monthly':
                $daysInMonth = $date->daysInMonth;
                return $periodRateDecimal / $daysInMonth;
                
            default:
                $daysInPeriod = 30;
        }

        return $periodRateDecimal / $daysInPeriod;
    }

    /**
     * Get daily rate from percentage based on frequency
     */
    private function getDailyRateFromPercentage($percentageRate, $frequency, $date = null)
    {
        $date = $date ? Carbon::parse($date) : Carbon::today();

        switch ($frequency) {
            case 'daily':
                return $percentageRate;
                
            case 'weekly':
                return $percentageRate / 7;
                
            case 'monthly':
                $daysInMonth = $date->daysInMonth;
                return $percentageRate / $daysInMonth;
                
            default:
                return $percentageRate / 30;
        }
    }

    /**
     * Calculate simple interest
     */
    private function calculateSimpleInterest($principal, $dailyRate)
    {
        return round($principal * $dailyRate, 2);
    }

    /**
     * Calculate reducing balance interest
     */
    private function calculateReducingBalanceInterest($principal, $dailyRate)
    {
        return round($principal * $dailyRate, 2);
    }

    /**
     * Delete existing interest or processing fee entry
     */
    private function deleteExistingInterestEntry($loanId, $date, $type = 'interest')
    {
        $prefix = $type === 'interest' ? 'INT-' : 'PF-';
        
        // Delete the income record
        if ($type === 'interest') {
            InterestIncome::where('iloanid', $loanId)
                ->whereDate('chargeddate', $date)
                ->delete();
        } else {
            ProcessingFee::where('iloanid', $loanId)
                ->whereDate('chargeddate', $date)
                ->delete();
        }
        
        // Delete associated journal entry by reference prefix
        $loan = DisbursedLoan::find($loanId);
        if ($loan) {
            $journalEntry = JournalEntry::whereDate('entry_date', $date)
                ->where('reference', 'like', $prefix . $loan->loannumber . '%')
                ->first();
            
            if ($journalEntry) {
                JournalEntryItem::where('journal_entry_id', $journalEntry->id)->delete();
                $journalEntry->delete();
            }
        }
    }

    /**
     * Create interest journal entry
     */
    private function createInterestJournalEntry($loan, $interestAmount, $date)
    {
        // Get Loan Disbursement account mapping - ACCOUNT ID 4
        $loanDisbursementMapping = $this->getBusinessProcessMapping('Loan Disbursement', $loan->companyid, $loan->branchid);
            
        if (!$loanDisbursementMapping) {
            throw new \Exception("Loan Disbursement account not configured in business process mappings for company {$loan->companyid}, branch {$loan->branchid}");
        }

        // Get Interest Income account mapping - ACCOUNT ID 12
        $interestIncomeMapping = $this->getBusinessProcessMapping('Interest Income', $loan->companyid, $loan->branchid);
            
        if (!$interestIncomeMapping) {
            throw new \Exception("Interest Income account not configured in business process mappings for company {$loan->companyid}, branch {$loan->branchid}");
        }

        // Get current financial year for the company
        $company = $this->getCompany($loan->companyid);
        $financialYearId = $company->current_financial_year_id ?? null;

        // Get the currency for the journal entry
        $currency = $this->getCurrencyForLoan($loan);
        
        if (!$currency) {
            throw new \Exception("Currency not found for loan {$loan->loannumber}");
        }

        $entryNumber = $this->generateJournalEntryNumber($loan->companyid, $date, 'INT');

        $journalEntry = JournalEntry::create([
            'company_id' => $loan->companyid,
            'branch_id' => $loan->branchid,
            'entry_number' => $entryNumber,
            'entry_date' => $date,
            'financial_year_id' => $financialYearId,
            'reference' => 'INT-' . $loan->loannumber . '-' . $date->format('Ymd'),
            'description' => "Daily interest accrual for loan {$loan->loannumber} on {$date->format('Y-m-d')}",
            'currency_id' => $currency->id,
            'exchange_rate' => 1,
            'total_debit' => $interestAmount,
            'total_credit' => $interestAmount,
            'status' => 'posted',
            'posted_by' => 1,
            'posted_at' => Carbon::now(),
            'created_by' => 1,
            'updated_by' => 1,
            'loanid' => $loan->loanid,
            'calculation_type' => 'interest',
            'calculation_date' => $date,
        ]);

        // DEBIT: Loan Disbursement Account (Account ID 4 - Loan Receivable)
        JournalEntryItem::create([
            'company_id' => $loan->companyid,
            'branch_id' => $loan->branchid,
            'journal_entry_id' => $journalEntry->id,
            'account_id' => $loanDisbursementMapping->accountid,
            'description' => "Daily accrued interest for loan {$loan->loannumber} on {$date->format('Y-m-d')}",
            'debit' => $interestAmount,
            'credit' => 0,
            'created_by' => 1,
        ]);

        // CREDIT: Interest Income Account (Account ID 12)
        JournalEntryItem::create([
            'company_id' => $loan->companyid,
            'branch_id' => $loan->branchid,
            'journal_entry_id' => $journalEntry->id,
            'account_id' => $interestIncomeMapping->accountid,
            'description' => "Daily interest income from loan {$loan->loannumber} on {$date->format('Y-m-d')}",
            'debit' => 0,
            'credit' => $interestAmount,
            'created_by' => 1,
        ]);

        Log::debug("Interest journal entry created: DEBIT Account {$loanDisbursementMapping->accountid}, CREDIT Account {$interestIncomeMapping->accountid}");

        return $journalEntry;
    }

    /**
     * Create processing fee journal entry
     */
    private function createProcessingFeeJournalEntry($loan, $loanType, $amount, $date)
    {
        // Get Processing Fee Income account mapping - ACCOUNT ID 23
        $processingFeeMapping = $this->getBusinessProcessMapping('Processing Fees', $loan->companyid, $loan->branchid);
            
        if (!$processingFeeMapping) {
            throw new \Exception("Processing Fees account not configured in business process mappings for company {$loan->companyid}, branch {$loan->branchid}");
        }

        // Get Loan Disbursement account mapping - ACCOUNT ID 4
        $loanDisbursementMapping = $this->getBusinessProcessMapping('Loan Disbursement', $loan->companyid, $loan->branchid);
            
        if (!$loanDisbursementMapping) {
            throw new \Exception("Loan Disbursement account not configured in business process mappings for company {$loan->companyid}, branch {$loan->branchid}");
        }

        // Get current financial year for the company
        $company = $this->getCompany($loan->companyid);
        $financialYearId = $company->current_financial_year_id ?? null;

        // Get the currency for the journal entry
        $currency = $this->getCurrencyForLoan($loan);
        
        if (!$currency) {
            throw new \Exception("Currency not found for loan {$loan->loannumber}");
        }

        // Generate entry number with PF prefix
        $entryNumber = $this->generateJournalEntryNumber($loan->companyid, $date, 'PF');

        // Create journal entry
        $journalEntry = JournalEntry::create([
            'company_id' => $loan->companyid,
            'branch_id' => $loan->branchid,
            'entry_number' => $entryNumber,
            'entry_date' => $date,
            'financial_year_id' => $financialYearId,
            'reference' => 'PF-' . $loan->loannumber . '-' . $date->format('Ymd'),
            'description' => "Daily processing fee for loan {$loan->loannumber} ({$loanType->product})",
            'currency_id' => $currency->id,
            'exchange_rate' => 1,
            'total_debit' => $amount,
            'total_credit' => $amount,
            'status' => 'posted',
            'posted_by' => 1,
            'posted_at' => Carbon::now(),
            'created_by' => 1,
            'updated_by' => 1,
            'loanid' => $loan->loanid,
            'calculation_type' => 'processing_fee',
            'calculation_date' => $date,
        ]);

        // DEBIT: Loan Disbursement Account (Account ID 4 - Loan Receivable)
        JournalEntryItem::create([
            'company_id' => $loan->companyid,
            'branch_id' => $loan->branchid,
            'journal_entry_id' => $journalEntry->id,
            'account_id' => $loanDisbursementMapping->accountid,
            'description' => "Processing fee accrual for loan {$loan->loannumber} on {$date->format('Y-m-d')}",
            'debit' => $amount,
            'credit' => 0,
            'created_by' => 1,
        ]);

        // CREDIT: Processing Fee Income Account (Account ID 23)
        JournalEntryItem::create([
            'company_id' => $loan->companyid,
            'branch_id' => $loan->branchid,
            'journal_entry_id' => $journalEntry->id,
            'account_id' => $processingFeeMapping->accountid,
            'description' => "Processing fee income from loan {$loan->loannumber} on {$date->format('Y-m-d')}",
            'debit' => 0,
            'credit' => $amount,
            'created_by' => 1,
        ]);

        Log::debug("Processing fee journal entry created: DEBIT Account {$loanDisbursementMapping->accountid}, CREDIT Account {$processingFeeMapping->accountid}");

        return $journalEntry;
    }

    /**
     * Get currency for loan
     */
    private function getCurrencyForLoan($loan)
    {
        if (isset($loan->currency_id) && $loan->currency_id) {
            return $this->getCurrency($loan->currency_id);
        }
        
        $company = $this->getCompany($loan->companyid);
        if ($company && $company->reporting_currency_id) {
            return $this->getCurrency($company->reporting_currency_id);
        }
        
        return Currency::where('code', 'ZMW')->first();
    }

    /**
     * Generate journal entry number
     */
    private function generateJournalEntryNumber($companyId, $date, $type = 'INT')
    {
        $datePrefix = Carbon::parse($date)->format('Ymd');
        $company = $this->getCompany($companyId);
        $companyPrefix = strtoupper(substr(preg_replace('/[^A-Z]/', '', $company->name ?? 'IMM'), 0, 3));
        
        // Get the last entry number for today based on the type prefix in entry_number
        $lastEntry = JournalEntry::where('company_id', $companyId)
            ->whereDate('entry_date', $date)
            ->where('entry_number', 'like', "JNL-{$companyPrefix}-{$datePrefix}-{$type}%")
            ->orderBy('id', 'desc')
            ->first();
        
        if ($lastEntry && preg_match('/-(\d+)$/', $lastEntry->entry_number, $matches)) {
            $sequence = (int)$matches[1] + 1;
        } else {
            $sequence = 1;
        }
        
        $typeCode = $type === 'INT' ? 'INT' : 'PF';
        return "JNL-{$companyPrefix}-{$datePrefix}-{$typeCode}-" . str_pad($sequence, 4, '0', STR_PAD_LEFT);
    }

    /**
     * Get business process mapping with caching
     */
    private function getBusinessProcessMapping($processName, $companyId, $branchId)
    {
        $cacheKey = "{$processName}_{$companyId}_{$branchId}";
        
        if (!isset($this->businessProcessMappings[$cacheKey])) {
            $this->businessProcessMappings[$cacheKey] = BusinessProcessMapping::where('nameofthebusinessprocess', $processName)
                ->where('companyid', $companyId)
                ->where('branchid', $branchId)
                ->first();
        }
        
        return $this->businessProcessMappings[$cacheKey];
    }

    /**
     * Get company with caching
     */
    private function getCompany($companyId)
    {
        if (!isset($this->companies[$companyId])) {
            $this->companies[$companyId] = Company::find($companyId);
        }
        
        return $this->companies[$companyId];
    }

    /**
     * Get currency with caching
     */
    private function getCurrency($currencyId)
    {
        if (!isset($this->currencies[$currencyId])) {
            $this->currencies[$currencyId] = Currency::find($currencyId);
        }
        
        return $this->currencies[$currencyId];
    }

    /**
     * Get loan type with caching
     */
    private function getLoanType($loanTypeId)
    {
        if (!isset($this->loanTypes[$loanTypeId])) {
            $this->loanTypes[$loanTypeId] = LoanType::find($loanTypeId);
        }
        
        return $this->loanTypes[$loanTypeId];
    }
}