<?php

namespace App\Services;

/**
 * LoanService
 * * Handles complex loan amortization including:
 * - Upfront fee deductions (Admin & Insurance)
 * - Periodic Interest (Reducing Balance)
 * - Conditional Processing Fees (Simple vs. Reducing)
 */
class LoanService
{
    /**
     * Generate a complete amortization schedule and financial summary.
     *
     * @param float  $amount           The principal loan amount.
     * @param float  $periodicInterest  The rate per period (e.g., 5.0 for 5%).
     * @param float  $periodicProcessing The processing rate per period.
     * @param int    $term             Total number of periods (days, weeks, or months).
     * @param string $processingBasis  'initial_amount' (Simple) or 'outstanding_balance' (Reducing).
     * @param array  $upfrontFees      ['admin_fee' => x, 'insurance_fee' => y].
     * @return array
     */
    public function generateSchedule($amount, $periodicInterest, $periodicProcessing, $term, $processingBasis = 'outstanding_balance', $upfrontFees = [])
    {
        // 1. Initialize Upfront Fees
        $adminFee = (float) ($upfrontFees['admin_fee'] ?? 0);
        $insuranceFee = (float) ($upfrontFees['insurance_fee'] ?? 0);

        // 2. Convert percentage rates to decimals
        $interestRateDecimal = $periodicInterest / 100;
        $processingRateDecimal = $periodicProcessing / 100;

        // 3. Determine the Periodic Installment
        $totalInstallment = 0;

        if ($processingBasis === 'initial_amount') {
            /**
             * BASIS: INITIAL AMOUNT (Simple Interest Style)
             * Installment = Amortized Principal & Interest + Fixed Processing Fee
             */
            if ($interestRateDecimal > 0) {
                $numerator = $interestRateDecimal * pow(1 + $interestRateDecimal, $term);
                $denominator = pow(1 + $interestRateDecimal, $term) - 1;
                $baseAmortizedInstallment = $amount * ($numerator / $denominator);
            } else {
                $baseAmortizedInstallment = $amount / $term;
            }

            $fixedProcessingFeePerPeriod = $amount * $processingRateDecimal;
            $totalInstallment = $baseAmortizedInstallment + $fixedProcessingFeePerPeriod;

        } else {
            /**
             * BASIS: OUTSTANDING AMOUNT (Reducing Balance Style)
             * Combine Interest + Processing into a single effective rate for amortization
             */
            $combinedRateDecimal = $interestRateDecimal + $processingRateDecimal;

            if ($combinedRateDecimal > 0) {
                $numerator = $combinedRateDecimal * pow(1 + $combinedRateDecimal, $term);
                $denominator = pow(1 + $combinedRateDecimal, $term) - 1;
                $totalInstallment = $amount * ($numerator / $denominator);
            } else {
                $totalInstallment = $amount / $term;
            }
        }

        // 4. Generate the Period-by-Period Schedule
        $schedule = [];
        $remainingBalance = $amount;
        $cumulativeInterest = 0;
        $cumulativeProcessing = 0;

        for ($i = 1; $i <= $term; $i++) {
            // Calculate components based on current remaining balance
            $interestComponent = $remainingBalance * $interestRateDecimal;
            
            $processingComponent = ($processingBasis === 'initial_amount') 
                ? ($amount * $processingRateDecimal) 
                : ($remainingBalance * $processingRateDecimal);

            // Principal portion is what remains of the installment after interest and fees
            $principalComponent = $totalInstallment - ($interestComponent + $processingComponent);
            
            // Adjust for last period rounding errors
            if ($i === $term) {
                $principalComponent = $remainingBalance;
                $totalInstallment = $principalComponent + $interestComponent + $processingComponent;
            }

            $remainingBalance -= $principalComponent;
            
            $cumulativeInterest += $interestComponent;
            $cumulativeProcessing += $processingComponent;

            $schedule[] = [
                'period'          => $i,
                'installment'     => round($totalInstallment, 2),
                'principal'       => round($principalComponent, 2),
                'interest'        => round($interestComponent, 2),
                'processing_fee'  => round($processingComponent, 2),
                'balance'         => max(0, round($remainingBalance, 2)),
            ];
        }

        // 5. Final Summary Object
        return [
            'summary' => [
                'loan_principal'       => round($amount, 2),
                'admin_fee_upfront'    => round($adminFee, 2),
                'insurance_fee_upfront'=> round($insuranceFee, 2),
                'total_deductions'     => round($adminFee + $insuranceFee, 2),
                'net_disbursement'     => round($amount - ($adminFee + $insuranceFee), 2),
                'periodic_installment' => round($totalInstallment, 2),
                'total_interest'       => round($cumulativeInterest, 2),
                'total_processing_fee' => round($cumulativeProcessing, 2),
                'total_repayable'      => round($totalInstallment * $term, 2),
                'processing_basis'     => $processingBasis
            ],
            'schedule' => $schedule
        ];
    }
}