<?php

namespace App\Services;

use App\Models\DisbursedLoan;
use App\Models\PaymentSchedule;
use App\Models\Application;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Throwable;

class PaymentScheduleService
{
    /**
     * Generate and save the final payment schedule for a disbursed loan.
     * Incorporates Interest Methods and Processing Fee Basis.
     * Now includes accrual date initialization for the new accrual system.
     * 
     * @param DisbursedLoan $loan
     * @return int
     */
    public function generatePaymentSchedule(DisbursedLoan $loan): int
    {
        try {
            DB::beginTransaction();

            // 0. Initial Setup - delete existing schedules
            PaymentSchedule::where('loanid', $loan->loanid)->delete();

            $schedules = [];
            $principalBalance = $loan->amount;
            $tenure = (int)$loan->loanterm;
            $frequency = $loan->paymentfrequency;
            
            // Terms locked at disbursement (used for calculation)
            $interestMethod = $loan->interest_method ?? 'reducing_balance';
            $interestRateFactor = (float)$loan->interestrate / 100;
            $feeRateFactor = (float)$loan->processing_fee / 100;
            $installmentAmount = (float)$loan->installmentamount;
            
            $paymentsGenerated = 0;
            $cumulativeInterest = 0;
            $cumulativeFee = 0;
            
            $disbursementDate = Carbon::parse($loan->disburseddate);

            // --- 1. HANDLE ONE-OFF PAYMENT (Bullet Payment) ---
            if ($frequency === 'one_off') {
                $interestAmount = $installmentAmount - $principalBalance;
                $paymentDate = $disbursementDate->copy()->addDays($tenure); 
                
                // For one-off, fee is typically a flat charge factored into installmentAmount 
                $totalCharges = $installmentAmount - $principalBalance;
                $feeComponent = 0; // The allocation logic will be complex here, simplifying for direct service implementation
                
                $schedules[] = $this->prepareScheduleData(
                    $loan, 
                    1, 
                    $paymentDate, 
                    $principalBalance, 
                    $interestAmount, 
                    $feeComponent, 
                    $installmentAmount, 
                    0.00,
                    $interestAmount,
                    $feeComponent,
                    $disbursementDate, // accrual_start_date
                    $paymentDate       // accrual_end_date (FIXED)
                );
                $paymentsGenerated = 1;

            } else {
                // --- 2. HANDLE PERIODIC PAYMENTS ---
                $paymentDate = Carbon::parse($loan->firstpaymentdate);

                for ($i = 1; $i <= $tenure; $i++) {
                    $interestPortion = 0;
                    $managementFee = 0;
                    $currentOutstanding = $principalBalance;
                    
                    // A. FEE CALCULATION
                    if ($loan->processing_fee > 0) {
                        if ($loan->processing_fee_basis === 'initial_amount') {
                            // Fixed fee based on initial amount
                            $managementFee = round($loan->amount * $feeRateFactor, 2);
                        }
                    }

                    // B. INTEREST CALCULATION
                    if ($interestMethod === 'simple_interest') {
                        // Simple Interest: (Principal * Rate * Tenure) / Tenure (Periodic component)
                        $interestPortion = round(($loan->amount * $interestRateFactor * $tenure) / $tenure, 2);
                        
                        // If processing fee is outstanding basis for Simple Interest, it's periodic
                        if ($loan->processing_fee_basis === 'outstanding_balance') {
                             $managementFee = round(($loan->amount * $feeRateFactor * $tenure) / $tenure, 2);
                        }
                    } else {
                        // Reducing Balance: Interest is calculated on the remaining principal
                        $interestPortion = round($currentOutstanding * $interestRateFactor, 2);

                        // If basis is 'outstanding_balance', fee behaves like interest on reducing balance
                        if ($loan->processing_fee_basis === 'outstanding_balance') {
                            $managementFee = round($currentOutstanding * $feeRateFactor, 2);
                        }
                    }

                    // C. PRINCIPAL CALCULATION
                    // Principal = Total Installment (pre-calculated in controller) - Interest - Management Fee
                    $principalComponent = $installmentAmount - $interestPortion - $managementFee;
                    
                    // D. FINAL INSTALLMENT ADJUSTMENT
                    if ($i == $tenure || ($currentOutstanding - $principalComponent) < 0) {
                        $principalComponent = $currentOutstanding;
                        $principalBalance = 0;
                        $installmentAmount = $principalComponent + $interestPortion + $managementFee; // Adjust total
                    } else {
                        $principalBalance -= $principalComponent;
                    }

                    // E. DATE LOGIC: Advance the date based on frequency
                    if ($i == 1) {
                        $paymentDate = $paymentDate->copy(); // Use the first payment date as is
                    } else {
                        $paymentDate = $this->calculateNextPaymentDate($paymentDate, $frequency);
                    }

                    // F. Determine accrual period
                    if ($i == 1) {
                        // First schedule accrues from disbursement date to first payment date
                        $accrualStartDate = $disbursementDate->format('Y-m-d');
                    } else {
                        // Subsequent schedules start day after previous schedule's payment date
                        $previousSchedulePaymentDate = Carbon::parse($schedules[$i-2]['paymentdate']);
                        $accrualStartDate = $previousSchedulePaymentDate->copy()->addDay()->format('Y-m-d');
                    }
                    
                    $accrualEndDate = $paymentDate->format('Y-m-d'); // FIXED - does not change

                    // G. CUMULATIVE TRACKING
                    $cumulativeInterest += $interestPortion;
                    $cumulativeFee += $managementFee;
                    
                    // H. Prepare schedule data with fixed accrual period
                    $schedules[] = $this->prepareScheduleData(
                        $loan, 
                        $i, 
                        $paymentDate, 
                        $principalComponent, 
                        $interestPortion, 
                        $managementFee, 
                        $installmentAmount, 
                        $principalBalance,
                        $cumulativeInterest,
                        $cumulativeFee,
                        $accrualStartDate, // FIXED accrual start
                        $accrualEndDate    // FIXED accrual end
                    );

                    $paymentsGenerated++;
                    if ($principalBalance <= 0) break;
                }
            }

            // Insert all schedules
            if (!empty($schedules)) {
                PaymentSchedule::insert($schedules);
            }
            
            DB::commit();
            return $paymentsGenerated;

        } catch (Throwable $e) {
            DB::rollBack();
            Log::error("Schedule Generation Error for Loan {$loan->loannumber}: " . $e->getMessage());
            throw new \Exception("Schedule Generation Failed: " . $e->getMessage());
        }
    }

    /**
     * Generate a temporary amortization schedule for a Loan Application (Pre-Disbursement View).
     * Used by ApplicationController@viewSchedule.
     * 
     * @param Application $application
     * @return array
     */
    public function generateAmortizationScheduleForApplication(Application $application): array
    {
        $schedule = [];
        $principalBalance = (float)$application->loan_amount;
        $tenure = (int)$application->loan_tenure;
        $frequency = $application->payment_frequency;
        $interestMethod = $application->interest_method ?? 'reducing_balance';

        $interestRateFactor = (float)$application->interest_rate / 100;
        $feeRateFactor = (float)$application->processing_fee / 100;
        $installmentAmount = (float)$application->installment_amount;
        $disbursementDate = Carbon::parse(now()->format('Y-m-d')); // Use today's date for projection

        // --- Disbursement Entry ---
        $schedule[] = [
            'date' => $disbursementDate->format('M d, Y'),
            'type' => 'Disbursement',
            'total_due' => 0.00,
            'principal' => 0.00,
            'interest' => 0.00,
            'fee' => 0.00,
            'balance' => round($principalBalance, 2),
            'accrual_start_date' => $disbursementDate->format('Y-m-d'),
            'accrual_end_date' => null,
            'actual_accrued_interest' => 0.00,
            'actual_accrued_processing_fee' => 0.00,
        ];

        // --- Periodic Payment Logic ---
        $firstPaymentDate = $this->calculateFirstPaymentDate($disbursementDate, $frequency, $tenure);
        $paymentDate = $firstPaymentDate->copy();

        for ($i = 1; $i <= $tenure; $i++) {
            $interestPortion = 0;
            $managementFee = 0;
            $currentOutstanding = $principalBalance;

            // A. FEE CALCULATION
            if ($application->processing_fee > 0) {
                if ($application->processing_fee_basis === 'initial_amount') {
                    $managementFee = round($application->loan_amount * $feeRateFactor, 2);
                }
            }

            // B. INTEREST CALCULATION
            if ($interestMethod === 'simple_interest') {
                $interestPortion = round(($application->loan_amount * $interestRateFactor * $tenure) / $tenure, 2);
                if ($application->processing_fee_basis === 'outstanding_balance') {
                     $managementFee = round(($application->loan_amount * $feeRateFactor * $tenure) / $tenure, 2);
                }
            } else {
                $interestPortion = round($currentOutstanding * $interestRateFactor, 2);
                if ($application->processing_fee_basis === 'outstanding_balance') {
                    $managementFee = round($currentOutstanding * $feeRateFactor, 2);
                }
            }

            // C. PRINCIPAL CALCULATION
            $principalComponent = $installmentAmount - $interestPortion - $managementFee;
            
            // D. FINAL INSTALLMENT ADJUSTMENT
            if ($i == $tenure || ($currentOutstanding - $principalComponent) < 0) {
                $principalComponent = $currentOutstanding;
                $principalBalance = 0;
                $totalDue = $principalComponent + $interestPortion + $managementFee;
            } else {
                $principalBalance -= $principalComponent;
                $totalDue = $installmentAmount;
            }

            // E. DATE LOGIC: Advance the date based on frequency
            if ($i > 1) {
                $paymentDate = $this->calculateNextPaymentDate($paymentDate, $frequency);
            }

            // F. Calculate accrual period for display (simulated)
            $accrualStartDate = null;
            $accrualEndDate = null;
            
            if ($i == 1) {
                $accrualStartDate = $disbursementDate->format('Y-m-d');
            } else {
                $accrualStartDate = Carbon::parse($paymentDate)->subDays(30)->format('Y-m-d');
            }
            $accrualEndDate = $paymentDate->format('Y-m-d');

            $schedule[] = [
                'date' => $paymentDate->format('M d, Y'),
                'type' => 'Payment ' . $i,
                'total_due' => round($totalDue, 2),
                'principal' => round($principalComponent, 2),
                'interest' => round($interestPortion, 2),
                'fee' => round($managementFee, 2),
                'balance' => max(0, round($principalBalance, 2)),
                'accrual_start_date' => $accrualStartDate,
                'accrual_end_date' => $accrualEndDate,
                'actual_accrued_interest' => 0.00,
                'actual_accrued_processing_fee' => 0.00,
            ];

            if ($principalBalance <= 0) break;
        }

        return $schedule;
    }
    
    /**
     * Structure the data for the database insert.
     * Now includes accrual fields initialization.
     */
    private function prepareScheduleData($loan, $index, $paymentDate, $principal, $interest, $fee, $total, $balance, 
                                         $cumulativeInterest, $cumulativeFee, $accrualStartDate, $accrualEndDate)
    {
        return [
            'loannumber' => $loan->loannumber,
            'loanid' => $loan->loanid,
            'interestamount' => round($interest, 2),
            'principalamount' => round($principal, 2),
            'processingfeesamount' => round($fee, 2), // Scheduled fee amount
            'totalamount' => round($total, 2),
            'paymentdate' => $paymentDate->format('Y-m-d'),
            'transactiontype' => 'scheduled_payment',
            'status' => 'scheduled',
            'installment_number' => $index,
            'outstanding_principal' => max(0, round($balance, 2)),
            'outstanding_interest' => 0, // Calculated dynamically
            'outstanding_processingfees_amoun' => 0, // Calculated dynamically
            'cumulative_principal_paid' => round($loan->amount - $balance, 2),
            'cumulative_interest_paid' => round($cumulativeInterest, 2),
            'cumulative_processingfees_amoun' => round($cumulativeFee, 2),
            'branchid' => $loan->branchid,
            'companyid' => $loan->companyid,
            'createdby' => $loan->createdby,
            'updatedby' => $loan->updatedby,
            'created_at' => now(),
            'updated_at' => now(),
            
            // New accrual fields - initialize to 0
            'actualaccruedinterest' => 0.00,
            'actualaccruedprocessingfee' => 0.00,
            
            // FIXED accrual dates
            'accrual_start_date' => $accrualStartDate,
            'accrual_end_date' => $accrualEndDate,
        ];
    }

    /**
     * Helper to determine the date of the first payment.
     */
    private function calculateFirstPaymentDate(Carbon $disbursementDate, string $frequency, int $tenure): Carbon
    {
        $date = $disbursementDate->copy();
        return match ($frequency) {
            'one_off' => $date->addDays($tenure),
            'daily'   => $date->addDay(),
            'weekly'  => $date->addWeek(),
            'monthly' => $date->addMonth(),
            default   => $date->addMonth(),
        };
    }
    
    /**
     * Helper to advance the date based on frequency.
     */
    private function calculateNextPaymentDate(Carbon $currentDate, string $frequency): Carbon
    {
        $nextDate = $currentDate->copy();

        switch ($frequency) {
            case 'daily':
                return $nextDate->addDay();
            case 'weekly':
                return $nextDate->addWeek();
            case 'monthly':
            default:
                return $nextDate->addMonth();
        }
    }
    
    /**
     * Create disbursement entry in payment schedule.
     * Updated to include accrual fields.
     */
    public function createDisbursementEntry(DisbursedLoan $loan)
    {
        return PaymentSchedule::create([
            'loannumber' => $loan->loannumber,
            'loanid' => $loan->loanid,
            'interestamount' => 0,
            'principalamount' => $loan->amount,
            'processingfeesamount' => 0, // No scheduled fee at disbursement
            'totalamount' => $loan->amount,
            'paymentdate' => $loan->disburseddate,
            'transactiontype' => 'Disbursement',
            'status' => 'paid',
            'installment_number' => 0,
            'outstanding_principal' => $loan->amount,
            'outstanding_interest' => 0,
            'outstanding_processingfees_amoun' => 0,
            'cumulative_principal_paid' => 0,
            'cumulative_interest_paid' => 0,
            'cumulative_processingfees_amoun' => 0,
            'paid_date' => $loan->disburseddate,
            'paid_principal' => $loan->amount,
            'paid_interest' => 0,
            'paid_penalty' => 0,
            'paid_processingefees_amount' => 0,
            'total_paid' => $loan->amount,
            'branchid' => $loan->branchid,
            'companyid' => $loan->companyid,
            'createdby' => $loan->createdby,
            'updatedby' => $loan->updatedby,
            
            // New accrual fields for disbursement entry
            'actualaccruedinterest' => 0.00,
            'actualaccruedprocessingfee' => 0.00,
            'accrual_start_date' => $loan->disburseddate,
            'accrual_end_date' => $loan->disburseddate,
        ]);
    }
}