<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Process Account Mapping | {{ config('app.name', 'IMMIA FINANCE') }}</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <script>
        tailwind.config = {
            theme: {
                extend: {
                    colors: {
                        primary: '#146CAA',
                        primaryLight: '#E1F0FA',
                        secondary: '#6b7280',
                        success: '#107C10',
                        warning: '#F68C1E',
                        danger: '#D83B01',
                        info: '#0078D4',
                        gray50: '#FAF9F8',
                        gray100: '#F3F2F1',
                        gray200: '#EDEBE9',
                        gray300: '#E1DFDD',
                        gray400: '#C8C6C4',
                        gray500: '#A19F9D',
                        gray600: '#605E5C',
                        gray700: '#323130',
                        gray800: '#201F1E',
                        gray900: '#000000'
                    }
                }
            }
        }
    </script>
    <style>
        .quickbooks-table {
            border-collapse: separate;
            border-spacing: 0;
            width: 100%;
        }
        .quickbooks-table th {
            background-color: #F3F2F1;
            color: #323130;
            font-weight: 600;
            font-size: 13px;
            text-transform: uppercase;
            padding: 12px 16px;
            border-bottom: 1px solid #EDEBE9;
            text-align: left;
        }
        .quickbooks-table td {
            padding: 12px 16px;
            border-bottom: 1px solid #EDEBE9;
            font-size: 14px;
            color: #323130;
        }
        .quickbooks-table tbody tr:hover {
            background-color: #F3F2F1;
        }
        .quickbooks-badge {
            display: inline-flex;
            align-items: center;
            padding: 4px 8px;
            border-radius: 12px;
            font-size: 12px;
            font-weight: 600;
        }
        .action-btn {
            width: 32px;
            height: 32px;
            border-radius: 4px;
            display: inline-flex;
            align-items: center;
            justify-content: center;
            margin: 0 2px;
            transition: all 0.2s;
        }
        .action-btn:hover {
            transform: translateY(-1px);
        }
        .action-btn-view {
            background-color: #F3F2F1;
            color: #323130;
        }
        .action-btn-view:hover {
            background-color: #146CAA;
            color: white;
        }
        .action-btn-edit {
            background-color: #E1F0FA;
            color: #146CAA;
        }
        .action-btn-edit:hover {
            background-color: #146CAA;
            color: white;
        }
        .action-btn-delete {
            background-color: #FEE8E6;
            color: #D83B01;
        }
        .action-btn-delete:hover {
            background-color: #D83B01;
            color: white;
        }
        .modal-overlay {
            display: none;
            position: fixed;
            inset: 0;
            background: rgba(0, 0, 0, 0.5);
            backdrop-filter: blur(4px);
            z-index: 1000;
            align-items: center;
            justify-content: center;
            padding: 20px;
        }
        
        .modal-overlay.active {
            display: flex !important;
        }
        
        .modal-content {
            background: white;
            border-radius: 8px;
            box-shadow: 0 8px 32px rgba(0, 0, 0, 0.1);
            width: 100%;
            max-width: 500px;
            margin: 1rem;
            max-height: 90vh;
            overflow-y: auto;
            position: relative;
            z-index: 1001;
            transform: translateZ(0);
            animation: modalSlideIn 0.3s ease-out;
        }
        
        @keyframes modalSlideIn {
            from {
                opacity: 0;
                transform: translateY(-20px) scale(0.95);
            }
            to {
                opacity: 1;
                transform: translateY(0) scale(1);
            }
        }

        @keyframes fadeIn {
            from { opacity: 0; transform: translateY(10px); }
            to { opacity: 1; transform: translateY(0); }
        }

        .animate-fade-in {
            animation: fadeIn 0.5s ease-in-out;
        }

        body.no-scroll {
            overflow: hidden;
        }

        .quickbooks-input {
            border: 1px solid #E1DFDD;
            border-radius: 4px;
            padding: 8px 12px;
            font-size: 14px;
            transition: border-color 0.2s;
            width: 100%;
        }
        .quickbooks-input:focus {
            outline: none;
            border-color: #146CAA;
            box-shadow: 0 0 0 2px rgba(20, 108, 170, 0.1);
        }
        
        .process-badge {
            display: inline-flex;
            align-items: center;
            padding: 4px 12px;
            border-radius: 12px;
            font-size: 12px;
            font-weight: 600;
            text-transform: uppercase;
        }
        
        .badge-general {
            background-color: #F3F2F1;
            color: #605E5C;
        }
    </style>
</head>
<body class="bg-gray50">
    @include('accounting.partials.sidebar')

    <div class="lg:ml-64">
        @include('accounting.partials.header', ['title' => 'Process Account Mapping', 'subtitle' => 'Map business processes to general ledger accounts'])

        <main class="p-6">
            <!-- Success/Error Messages -->
            @if(session('success'))
                <div class="mb-6 bg-green-50 border border-green-200 rounded-lg p-4 animate-fade-in">
                    <div class="flex items-center justify-between">
                        <div class="flex items-center">
                            <i class="fas fa-check-circle text-green-500 mr-2"></i>
                            <span class="text-green-800 font-medium">{{ session('success') }}</span>
                        </div>
                        <button onclick="this.parentElement.parentElement.remove()" class="text-green-600 hover:text-green-800">
                            <i class="fas fa-times"></i>
                        </button>
                    </div>
                </div>
            @endif

            @if(session('error'))
                <div class="mb-6 bg-red-50 border border-red-200 rounded-lg p-4 animate-fade-in">
                    <div class="flex items-center justify-between">
                        <div class="flex items-center">
                            <i class="fas fa-exclamation-circle text-red-500 mr-2"></i>
                            <span class="text-red-800 font-medium">{{ session('error') }}</span>
                        </div>
                        <button onclick="this.parentElement.parentElement.remove()" class="text-red-600 hover:text-red-800">
                            <i class="fas fa-times"></i>
                        </button>
                    </div>
                </div>
            @endif

            <!-- Action Bar -->
            <div class="bg-white rounded-lg border border-gray200 p-4 mb-6">
                <div class="flex flex-col md:flex-row md:items-center justify-between space-y-4 md:space-y-0">
                    <div>
                        <h2 class="text-lg font-semibold text-gray900">Business Process Mappings</h2>
                        <p class="text-gray600 text-sm mt-1">{{ $mappings->count() }} mapping(s) configured</p>
                    </div>
                    <div class="flex space-x-3">
                        <!-- Search Box -->
                        <div class="relative">
                            <input type="text" 
                                   id="searchInput" 
                                   placeholder="Search mappings..." 
                                   class="quickbooks-input pl-10 w-64">
                            <i class="fas fa-search absolute left-3 top-3 text-gray500"></i>
                        </div>
                        <button onclick="openCreateModal()" 
                                class="bg-primary text-white px-4 py-2 rounded text-sm font-medium hover:bg-primary/90 flex items-center gap-2">
                            <i class="fas fa-plus"></i>
                            New Mapping
                        </button>
                    </div>
                </div>
            </div>

            <!-- Mappings Table -->
            <div class="bg-white rounded-lg border border-gray200 overflow-hidden">
                <div class="overflow-x-auto">
                    <table class="quickbooks-table">
                        <thead>
                            <tr>
                                <th>Business Process</th>
                                <th>Description</th>
                                <th>GL Account</th>
                                <th>Created By</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody id="mappingsTableBody">
                            @forelse($mappings as $mapping)
                            <tr class="hover:bg-gray100 transition-colors" 
                                data-process-name="{{ strtolower($mapping->nameofthebusinessprocess) }}">
                                <td>
                                    <div class="flex items-center gap-3">
                                        <div class="w-10 h-10 rounded-lg bg-gray100 flex items-center justify-center">
                                            <i class="fas fa-sitemap text-gray600"></i>
                                        </div>
                                        <div>
                                            <div class="font-medium text-gray900">{{ $mapping->nameofthebusinessprocess }}</div>
                                            <div class="text-xs text-gray500">
                                                <span class="process-badge badge-general">
                                                    {{ $mapping->process_type ?? 'general' }}
                                                </span>
                                            </div>
                                        </div>
                                    </div>
                                </td>
                                <td>
                                    <div class="text-sm text-gray700 max-w-xs" title="{{ $mapping->businessprocessdescription ?? 'No description' }}">
                                        {{ $mapping->businessprocessdescription ? (strlen($mapping->businessprocessdescription) > 80 ? substr($mapping->businessprocessdescription, 0, 80) . '...' : $mapping->businessprocessdescription) : 'No description provided' }}
                                    </div>
                                </td>
                                <td>
                                    <div class="flex items-center gap-3">
                                        <div class="w-10 h-10 rounded-lg bg-green-100 flex items-center justify-center">
                                            <i class="fas fa-chart-line text-green-600"></i>
                                        </div>
                                        <div>
                                            <div class="font-medium text-gray900">{{ $mapping->account->code ?? 'N/A' }}</div>
                                            <div class="text-xs text-gray500">{{ $mapping->account->name ?? 'N/A' }}</div>
                                        </div>
                                    </div>
                                </td>
                                <td>
                                    <div class="text-sm text-gray900">{{ $mapping->creator->name ?? 'System' }}</div>
                                    <div class="text-xs text-gray500">{{ $mapping->created_at->format('M d, Y') }}</div>
                                </td>
                                <td>
                                    <div class="flex items-center">
                                        <button onclick="openEditModal({{ $mapping->businessmappingid }})" 
                                                class="action-btn action-btn-edit"
                                                title="Edit Mapping">
                                            <i class="fas fa-edit"></i>
                                        </button>
                                        <button onclick="openDeleteModal({{ $mapping->businessmappingid }}, '{{ $mapping->nameofthebusinessprocess }}')" 
                                                class="action-btn action-btn-delete"
                                                title="Delete Mapping">
                                            <i class="fas fa-trash"></i>
                                        </button>
                                    </div>
                                </td>
                            </tr>
                            @empty
                            <tr>
                                <td colspan="5" class="text-center py-12">
                                    <div class="py-8 text-center">
                                        <div class="w-16 h-16 bg-gray100 rounded-full flex items-center justify-center mx-auto mb-4">
                                            <i class="fas fa-sitemap text-gray500 text-2xl"></i>
                                        </div>
                                        <h3 class="text-lg font-medium text-gray900 mb-2">No process mappings found</h3>
                                        <p class="text-gray600 mb-6 max-w-md mx-auto">
                                            Get started by creating your first process account mapping
                                        </p>
                                        <button onclick="openCreateModal()" 
                                                class="bg-primary text-white px-4 py-2 rounded text-sm font-medium hover:bg-primary/90 flex items-center gap-2 mx-auto">
                                            <i class="fas fa-plus"></i>
                                            Create First Mapping
                                        </button>
                                    </div>
                                </td>
                            </tr>
                            @endforelse
                        </tbody>
                    </table>
                </div>
            </div>
        </main>
    </div>

    <!-- Create Modal -->
    <div id="createModal" class="modal-overlay">
        <div class="modal-content">
            <div class="flex items-center justify-between p-6 border-b border-gray200">
                <div class="flex items-center">
                    <div class="w-10 h-10 bg-primaryLight rounded-lg flex items-center justify-center mr-3">
                        <i class="fas fa-plus text-primary"></i>
                    </div>
                    <div>
                        <h3 class="text-lg font-semibold text-gray900">Create Process Mapping</h3>
                        <p class="text-gray600 text-sm mt-1">Map a business process to a GL account</p>
                    </div>
                </div>
                <button onclick="closeCreateModal()" class="text-gray500 hover:text-gray700">
                    <i class="fas fa-times"></i>
                </button>
            </div>
            
            <div class="p-6">
                <form id="createForm" method="POST" action="{{ route('admin.accounting.settings.business-process-mappings.store') }}">
                    @csrf
                    <div class="space-y-4">
                        <!-- Business Process -->
                        <div>
                            <label for="create_process_name" class="block text-sm font-medium text-gray700 mb-2">Business Process *</label>
                            <select name="nameofthebusinessprocess" 
                                    id="create_process_name"
                                    class="quickbooks-input"
                                    required>
                                <option value="">Select a business process</option>
                                @foreach($processNames as $key => $value)
                                    <option value="{{ $value }}">{{ $value }}</option>
                                @endforeach
                            </select>
                        </div>

                        <!-- GL Account -->
                        <div>
                            <label for="create_account_id" class="block text-sm font-medium text-gray700 mb-2">General Ledger Account *</label>
                            <select name="accountid" 
                                    id="create_account_id"
                                    class="quickbooks-input"
                                    required>
                                <option value="">Select a GL account</option>
                                @foreach($accounts as $account)
                                    <option value="{{ $account->id }}">{{ $account->code }} - {{ $account->name }}</option>
                                @endforeach
                            </select>
                        </div>

                        <!-- Description -->
                        <div>
                            <label for="create_description" class="block text-sm font-medium text-gray700 mb-2">Description</label>
                            <textarea name="businessprocessdescription" 
                                      id="create_description"
                                      rows="2"
                                      class="quickbooks-input resize-none"
                                      placeholder="Optional description of how this process relates to the GL account"></textarea>
                        </div>
                    </div>

                    <div class="flex justify-end gap-3 mt-6 pt-6 border-t border-gray200">
                        <button type="button" 
                                onclick="closeCreateModal()" 
                                class="px-4 py-2 text-gray700 hover:bg-gray100 rounded text-sm font-medium">
                            Cancel
                        </button>
                        <button type="submit" 
                                id="createSubmitBtn"
                                class="bg-primary text-white px-4 py-2 rounded text-sm font-medium hover:bg-primary/90 flex items-center gap-2">
                            <i class="fas fa-plus"></i>
                            Create Mapping
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- Edit Modal -->
    <div id="editModal" class="modal-overlay">
        <div class="modal-content">
            <div class="flex items-center justify-between p-6 border-b border-gray200">
                <div class="flex items-center">
                    <div class="w-10 h-10 bg-primaryLight rounded-lg flex items-center justify-center mr-3">
                        <i class="fas fa-edit text-primary"></i>
                    </div>
                    <div>
                        <h3 class="text-lg font-semibold text-gray900">Edit Process Mapping</h3>
                        <p class="text-gray600 text-sm mt-1">Update the process to account mapping</p>
                    </div>
                </div>
                <button onclick="closeEditModal()" class="text-gray500 hover:text-gray700">
                    <i class="fas fa-times"></i>
                </button>
            </div>
            
            <div class="p-6">
                <form id="editForm" method="POST">
                    @csrf
                    @method('PUT')
                    <div class="space-y-4">
                        <!-- Business Process -->
                        <div>
                            <label for="edit_process_name" class="block text-sm font-medium text-gray700 mb-2">Business Process *</label>
                            <select name="nameofthebusinessprocess" 
                                    id="edit_process_name"
                                    class="quickbooks-input"
                                    required>
                                <option value="">Select a business process</option>
                                @foreach($processNames as $key => $value)
                                    <option value="{{ $value }}">{{ $value }}</option>
                                @endforeach
                            </select>
                        </div>

                        <!-- GL Account -->
                        <div>
                            <label for="edit_account_id" class="block text-sm font-medium text-gray700 mb-2">General Ledger Account *</label>
                            <select name="accountid" 
                                    id="edit_account_id"
                                    class="quickbooks-input"
                                    required>
                                <option value="">Select a GL account</option>
                                @foreach($accounts as $account)
                                    <option value="{{ $account->id }}">{{ $account->code }} - {{ $account->name }}</option>
                                @endforeach
                            </select>
                        </div>

                        <!-- Description -->
                        <div>
                            <label for="edit_description" class="block text-sm font-medium text-gray700 mb-2">Description</label>
                            <textarea name="businessprocessdescription" 
                                      id="edit_description"
                                      rows="2"
                                      class="quickbooks-input resize-none"
                                      placeholder="Optional description of how this process relates to the GL account"></textarea>
                        </div>
                    </div>

                    <div class="flex justify-end gap-3 mt-6 pt-6 border-t border-gray200">
                        <button type="button" 
                                onclick="closeEditModal()" 
                                class="px-4 py-2 text-gray700 hover:bg-gray100 rounded text-sm font-medium">
                            Cancel
                        </button>
                        <button type="submit" 
                                id="editSubmitBtn"
                                class="bg-primary text-white px-4 py-2 rounded text-sm font-medium hover:bg-primary/90 flex items-center gap-2">
                            <i class="fas fa-save"></i>
                            Update Mapping
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- Delete Confirmation Modal -->
    <div id="deleteModal" class="modal-overlay">
        <div class="modal-content">
            <div class="text-center p-6">
                <div class="mx-auto flex items-center justify-center h-16 w-16 rounded-full bg-red-100 mb-4">
                    <i class="fas fa-exclamation-triangle text-red-600 text-2xl"></i>
                </div>
                <h3 class="text-lg font-semibold text-gray900 mb-2">Delete Process Mapping</h3>
                <p class="text-gray600 text-sm mb-6">
                    Are you sure you want to delete the mapping for 
                    "<span id="deleteProcessName" class="font-semibold text-gray900"></span>"? 
                    This action cannot be undone.
                </p>
                
                <form id="deleteForm" method="POST">
                    @csrf
                    @method('DELETE')
                    <div class="flex justify-center gap-3">
                        <button type="button" 
                                onclick="closeDeleteModal()" 
                                class="px-4 py-2 text-gray700 hover:bg-gray100 rounded text-sm font-medium">
                            Cancel
                        </button>
                        <button type="submit" 
                                id="deleteSubmitBtn"
                                class="bg-danger text-white px-4 py-2 rounded text-sm font-medium hover:bg-danger/90 flex items-center gap-2">
                            <i class="fas fa-trash"></i>
                            Delete Mapping
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <script>
        // Base URL configuration
        const BASE_URL = '{{ url('/') }}';

        // Modal Functions
        function openCreateModal() {
            const modal = document.getElementById('createModal');
            modal.classList.add('active');
            document.body.classList.add('no-scroll');
            
            setTimeout(() => {
                const processNameInput = document.getElementById('create_process_name');
                if (processNameInput) processNameInput.focus();
            }, 100);
        }

        function closeCreateModal() {
            const modal = document.getElementById('createModal');
            modal.classList.remove('active');
            document.body.classList.remove('no-scroll');
            const createForm = document.getElementById('createForm');
            if (createForm) createForm.reset();
        }

        function openEditModal(id) {
            const modal = document.getElementById('editModal');
            modal.classList.add('active');
            document.body.classList.add('no-scroll');
            
            // Show loading state
            const editSubmitBtn = document.getElementById('editSubmitBtn');
            if (editSubmitBtn) {
                editSubmitBtn.disabled = true;
                editSubmitBtn.innerHTML = '<i class="fas fa-spinner fa-spin mr-1"></i> Loading...';
            }
            
            // Clear previous form data
            const processNameInput = document.getElementById('edit_process_name');
            const descriptionInput = document.getElementById('edit_description');
            const accountInput = document.getElementById('edit_account_id');
            
            if (processNameInput) processNameInput.value = '';
            if (descriptionInput) descriptionInput.value = '';
            if (accountInput) accountInput.value = '';
            
            // Set form action
            const editForm = document.getElementById('editForm');
            if (editForm) {
                editForm.action = `${BASE_URL}/admin/accounting/settings/business-process-mappings/${id}`;
            }
            
            // Fetch mapping data
            fetch(`${BASE_URL}/admin/accounting/settings/business-process-mappings/${id}/data`)
                .then(response => {
                    if (!response.ok) {
                        throw new Error(`HTTP error! status: ${response.status}`);
                    }
                    return response.json();
                })
                .then(data => {
                    if (data.success) {
                        const mapping = data.mapping;
                        
                        // Populate form fields
                        if (processNameInput) processNameInput.value = mapping.nameofthebusinessprocess;
                        if (descriptionInput) descriptionInput.value = mapping.businessprocessdescription || '';
                        if (accountInput) accountInput.value = mapping.accountid;
                        
                        // Reset button state
                        if (editSubmitBtn) {
                            editSubmitBtn.disabled = false;
                            editSubmitBtn.innerHTML = '<i class="fas fa-save mr-1"></i> Update Mapping';
                        }
                        
                        setTimeout(() => {
                            if (processNameInput) processNameInput.focus();
                        }, 100);
                    } else {
                        throw new Error(data.error || 'Failed to load mapping data');
                    }
                })
                .catch(error => {
                    console.error('Error loading mapping data:', error);
                    showNotification('Error loading mapping data: ' + error.message, 'error');
                    closeEditModal();
                    
                    // Reset button state
                    if (editSubmitBtn) {
                        editSubmitBtn.disabled = false;
                        editSubmitBtn.innerHTML = '<i class="fas fa-save mr-1"></i> Update Mapping';
                    }
                });
        }

        function closeEditModal() {
            const modal = document.getElementById('editModal');
            modal.classList.remove('active');
            document.body.classList.remove('no-scroll');
        }

        function openDeleteModal(id, processName) {
            const modal = document.getElementById('deleteModal');
            modal.classList.add('active');
            document.body.classList.add('no-scroll');
            
            // Set form action and process name
            const deleteForm = document.getElementById('deleteForm');
            const processNameSpan = document.getElementById('deleteProcessName');
            
            if (deleteForm) {
                deleteForm.action = `${BASE_URL}/admin/accounting/settings/business-process-mappings/${id}`;
            }
            if (processNameSpan) {
                processNameSpan.textContent = processName;
            }
        }

        function closeDeleteModal() {
            const modal = document.getElementById('deleteModal');
            modal.classList.remove('active');
            document.body.classList.remove('no-scroll');
        }

        // Close modals with Escape key only
        document.addEventListener('keydown', function(e) {
            if (e.key === 'Escape') {
                closeCreateModal();
                closeEditModal();
                closeDeleteModal();
            }
        });

        // Search functionality
        const searchInput = document.getElementById('searchInput');
        if (searchInput) {
            searchInput.addEventListener('input', function(e) {
                const searchTerm = e.target.value.toLowerCase();
                const rows = document.querySelectorAll('#mappingsTableBody tr');
                
                rows.forEach(row => {
                    const processName = row.getAttribute('data-process-name') || '';
                    const textContent = row.textContent.toLowerCase();
                    
                    if (textContent.includes(searchTerm) || processName.includes(searchTerm)) {
                        row.style.display = '';
                    } else {
                        row.style.display = 'none';
                    }
                });
            });
        }

        // Form submission handlers
        const createForm = document.getElementById('createForm');
        if (createForm) {
            createForm.addEventListener('submit', function(e) {
                const submitBtn = document.getElementById('createSubmitBtn');
                const processName = document.getElementById('create_process_name').value;
                const accountId = document.getElementById('create_account_id').value;
                
                if (!processName || !accountId) {
                    e.preventDefault();
                    showNotification('Please fill in all required fields.', 'error');
                    return false;
                }
                
                if (submitBtn) {
                    submitBtn.disabled = true;
                    submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin mr-1"></i> Creating...';
                }
            });
        }

        const editForm = document.getElementById('editForm');
        if (editForm) {
            editForm.addEventListener('submit', function(e) {
                const submitBtn = document.getElementById('editSubmitBtn');
                const processName = document.getElementById('edit_process_name').value;
                const accountId = document.getElementById('edit_account_id').value;
                
                if (!processName || !accountId) {
                    e.preventDefault();
                    showNotification('Please fill in all required fields.', 'error');
                    return false;
                }
                
                if (submitBtn) {
                    submitBtn.disabled = true;
                    submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin mr-1"></i> Updating...';
                }
            });
        }

        const deleteForm = document.getElementById('deleteForm');
        if (deleteForm) {
            deleteForm.addEventListener('submit', function(e) {
                const submitBtn = document.getElementById('deleteSubmitBtn');
                if (submitBtn) {
                    submitBtn.disabled = true;
                    submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin mr-1"></i> Deleting...';
                }
            });
        }

        // Notification function
        function showNotification(message, type = 'info') {
            const existing = document.querySelector('.quickbooks-notification');
            if (existing) existing.remove();

            const colors = {
                success: 'bg-success',
                error: 'bg-danger',
                info: 'bg-primary',
                warning: 'bg-warning'
            };

            const icons = {
                success: 'fa-check-circle',
                error: 'fa-exclamation-circle',
                info: 'fa-info-circle',
                warning: 'fa-exclamation-triangle'
            };

            const notification = document.createElement('div');
            notification.className = `quickbooks-notification fixed top-4 right-4 ${colors[type]} text-white px-4 py-3 rounded shadow-lg z-50 flex items-center animate-fade-in`;
            notification.innerHTML = `
                <i class="fas ${icons[type]} mr-2"></i>
                <span>${message}</span>
            `;

            document.body.appendChild(notification);

            setTimeout(() => {
                notification.classList.add('animate-fade-out');
                setTimeout(() => {
                    notification.remove();
                }, 300);
            }, 3000);
        }

        // Add animation styles
        const style = document.createElement('style');
        style.textContent = `
            @keyframes fadeIn {
                from { opacity: 0; transform: translateY(-10px); }
                to { opacity: 1; transform: translateY(0); }
            }
            @keyframes fadeOut {
                from { opacity: 1; transform: translateY(0); }
                to { opacity: 0; transform: translateY(-10px); }
            }
            .animate-fade-in {
                animation: fadeIn 0.3s ease-out;
            }
            .animate-fade-out {
                animation: fadeOut 0.3s ease-in;
            }
            .quickbooks-notification {
                animation: fadeIn 0.3s ease-out;
            }
        `;
        document.head.appendChild(style);

        // Auto-hide alerts after 5 seconds
        document.addEventListener('DOMContentLoaded', function() {
            setTimeout(() => {
                const alerts = document.querySelectorAll('.bg-green-50, .bg-red-50');
                alerts.forEach(alert => {
                    if (alert) {
                        alert.style.opacity = '0';
                        alert.style.transition = 'opacity 0.5s ease';
                        setTimeout(() => {
                            if (alert.parentNode) {
                                alert.parentNode.removeChild(alert);
                            }
                        }, 500);
                    }
                });
            }, 5000);
        });

        console.log('Business Process Mapping JavaScript loaded successfully');
    </script>
</body>
</html>